#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <lara.h>
#include <getopt.h>
#include <asm/io.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>
#include <linux/i2c-slave.h>

char* device = NULL;

static const char *usage_msg =
"\n"
"   Options:\n"
"     -d device               ..  I2C device (default: /dev/i2c-0)\n"
"     -f                      ..  flush the i2c slave buffer\n"
"     -m slave_addr           ..  master operation with slave address\n"
"     -s                      ..  slave operation\n"
"     -a                      ..  send address on master operations\n"
"     -r count                ..  read count bytes\n"
"     -p timeout              ..  poll <timeout> seconds before read\n"
"     -q timeout              ..  select <timeout> seconds before read\n"
"     -w string               ..  write string as data\n"
"\n";

int do_read_i2c = 0;
int do_write_i2c = 0;
int do_integer_scan = 0;
int do_flush = 0;
int do_poll = 0;
int do_select = 0;

int	i2c_addr = 0;
int	reg_addr = 0;
int 	reg_val  = 0;
int	write_num = 0;
int	is_master = 0;
int	is_slave = 0;
int	send_address = 0;
int	read_count = 0;
int	poll_timeout = 0;

static void
dump_iic_buf(const char* buf, int len)
{
	int i;
	printf("Bytes:\n");
	for (i = 0; i < len; i++) {
		printf("0x%02x\n", buf[i] & 0xff);
	}
}

int
main(int argc, char ** argv)
{
    int fd = 0;
    int errflag = 0;
    int c;
    char *endptr;
    unsigned char buf[4096];      
    int retval=0;
    int mode = 0;
    int ret;
    
    memset(buf, 0, sizeof(buf));
    
    while ((c = getopt(argc, argv, "d:fm:saIr:p:q:w:?"))
	   != -1)
	switch (c) {
	  case 'd': 	device = strdup(optarg);
	      break;
	  case 'f':	do_flush = 1;
	      break;
	  case 'm': 	is_master = 1; i2c_addr = strtol(optarg, &endptr, 16);
	      break;
	  case 's':	is_slave = 1;
	      break;
	  case 'a':	send_address = 1;
	      break;
	  case 'I': 	do_integer_scan=1;
	      break;
	  case 'r':	do_read_i2c = 1; read_count = strtol(optarg, &endptr, 10);
	      break;
	  case 'p':	do_poll = 1; poll_timeout = strtol(optarg, &endptr, 10);
	      break;
	  case 'q':	do_select = 1; poll_timeout = strtol(optarg, &endptr, 10);
	      break;
	  case 'w':	do_write_i2c = 1; strncpy(buf, optarg, sizeof(buf) - 1);
	      break;   
	  case '?':
	      errflag++;
	}
	
    if (!is_master && !is_slave) {
    	fprintf(stderr, "Error: no mode (master/slave) specified!\n");
    	errflag++;
    }
  	
    if (errflag) {
	printf(usage_msg);
	retval = 3;
	goto out;
    }
 	
    /* Open the I2C interface */
    
    if (device == NULL) {
    	device = strdup("/dev/i2c-0");
    }
    
    if ((fd = open(device, O_RDWR)) < 0) {
	/* ERROR HANDLING; you can check errno to see what went wrong */
	fprintf(stderr, "Error Opening I2C device %i.\n",errno);
	retval = 1;
	goto out;
    }
    printf("Device %s opened, fd=%d.\n", device, fd);
    
    if (do_flush) {
    	if (ioctl(fd, I2C_SLAVE_FLUSH_BUFFER, 0) < 0) {
    	    fprintf(stderr, "Error flushing i2c buffer: %i.\n", errno);
    	}
    	else {
    	    printf("i2c buffer flushed.\n");
    	}
	goto out;
    }
    
    /* set the operation mode */
    if (is_master) {
    	mode = I2C_MODE_MASTER;
    	if (send_address) {
    	    mode |= I2C_MODE_MASTER_ADDRESS;
    	}
	if (ioctl(fd, I2C_SLAVE, i2c_addr) < 0) {
	    fprintf(stderr, "IOCTL I2C device I2C_SLAVE: %i.\n",errno);
	    retval = 1;
	    goto out;
	}
	printf("Master mode, I2C Addr: 0x%02x\n", i2c_addr);
    }
    else {
    	mode = I2C_MODE_SLAVE;
	printf("Slave mode\n");
    }
    if (ioctl(fd, I2C_DEVICE_SET_TRANSMITTER_MODE, mode) < 0) {
	fprintf(stderr, "IOCTL I2C device I2C_DEVICE_SET_TRANSMITTER_MODE: %i.\n",errno);
	retval = 1;
	goto out;
    }
    if (ioctl(fd, I2C_DEVICE_SET_RECEIVER_MODE, mode) < 0) {
	fprintf(stderr, "IOCTL I2C device I2C_DEVICE_SET_RECEIVER_MODE: %i.\n",errno);
	retval = 1;
	goto out;
    }
    
    /* Read or write registers with addressing */

    if (do_read_i2c && !do_write_i2c) {
	if (do_poll) {
	    struct pollfd pfd;
	    
	    pfd.fd = fd;
	    pfd.events = POLLIN;
	    
	    ret = poll(&pfd, 1, poll_timeout * 1000);
	    
	    if (ret == 0) {
	    	printf("poll() timed out, no data available\n");
	    	goto out;
	    }
	    else if (ret < 0) {
	    	fprintf(stderr, "Error in poll() call!\n");
	    	retval = 1;
	    	goto out;
	    }
	    else {
	    	printf("poll() signalled that data is available.\n");
	    }
	}
	else if (do_select) {
	    struct timeval to;
	    fd_set fdset;
	    
	    to.tv_sec = poll_timeout;
	    to.tv_usec = 0;
	    FD_ZERO(&fdset);
	    FD_SET(fd, &fdset);
	    
	    ret = select(fd + 1, &fdset, NULL, NULL, &to);
	    
	    if (ret < 0) {
	    	fprintf(stderr, "Error in select() call!\n");
	    	retval = 1;
	    	goto out;
	    }
	    if(!FD_ISSET(fd, &fdset)) { // timeout
	    	printf("select() timed out, no data available\n");
	    	goto out;
	    }
	    else {
	    	printf("select() signalled that data is available.\n");
	    }
	    	
    	}
	ret = read(fd, buf, read_count);
	if (ret == 0) {
	    printf("Could not read data, EOF?\n");
	}
	else if (ret < 0) {
	    printf("Error reading data: %i\n", errno);
	}
	else {
	    printf("Read %i bytes.\n", ret);
	    dump_iic_buf(buf, ret);
	}
	goto out;
    }
	
    if (!do_read_i2c && do_write_i2c) {
	ret = write(fd, buf, strlen(buf));
	if (ret == 0) {
	    printf("Could not write data, EOF?\n");
	}
	else if (ret < 0) {
	    printf("Error writing data: %i\n", errno);
	}
	else {
	    printf("Wrote %i bytes.\n", ret);
	    dump_iic_buf(buf, ret);
	}
	goto out;
    }

out:
    if (fd) close(fd);
    if (device) free(device);
    
    return retval;
}


