/*******************************************************************************
********************************************************************************
***                                                                           **
***                    COPYRIGHT 1998 INTEL CORPORATION                       **
***                                                                           **
***                INTEL CORPORATION PROPRIETARY INFORMATION                  **
***                                                                           **
***      This software is supplied under the terms of a license agreement     **
***      or non-disclosure agreement with Intel Corporation and may not be    **
***      copied or disclosed except in accordance with the terms of that      **
***      agreement.                                                           **
***                                                                           **
***      This Source Code is provided "as is" without any warranty of any     **
***      kind, express, implied, statutory or otherwise, and Intel            **
***      specifically disclaims any implied warranties for any particular     **
***      purpose.                                                             **
***                                                                           **
********************************************************************************
******** 10 ****** 20 ****** 30 ****** 40 ****** 50 ****** 60 ****** 70 *******/

/******************************************************************************

   Module description:

   This is the source code for a utility to send raw commands to a BMC or other
   micro.

*******************************************************************************/


/*----------------------------------------------------------
* Copyright (c) 2002 by National Semiconductor Corporation
* All rights reserved.
*-----------------------------------------------------------
* File Contents:
*     Added SMB interface for mBMC, this interface is default
* Project:  PC87431
*---------------------------------------------------------*/
/* version 3.0 made isa_sms interface the default interface*/



/* Includes */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/stat.h>
#include "common.h"


/*******************************************************************************
*  Routine: ExitWithUsage
*
*  Arguments:
*     N/A
*
*  Returns:
*     N/A
*
*  Description:
*     This routine exits back to the OS with a message displaying the correct
*     usage for the command line arguments.
*
*******************************************************************************/

void ExitWithUsage(void)
{
/*
   printf("\n");
   printf("IpmiTool\n");
   PrintStandardDisclaimer();
   printf("Usage: IpmiTool [-i9][-if interface] IPMBAddr NetFnLUN Cmd [Data1 ... DataN]\n");
   printf("  -i9      Use IPMI 0.9 commands instead of IPMI 1.0 Send Msg/Get Msg (default)\n");
   printf("  -if <if> Select BMC interface to use (default:isa_sms)\n");
   printf("           Supported interfaces: isa_sms, isa_smm, smb\n");
   printf("  IPMBAddr IPMB address of microcontroller (e.g., BMC = 20, HSC = C0, mBMC = 84)\n");
   printf("  NetFnLUN Command net function (high 6 bits) and LUN (low 2 bits)\n");
   printf("  Cmd      Command micro to perform\n");
   printf("  Data1..N Command data\n");
   printf("Note: IpmiTool returns the completion code in the DOS errorlevel\n");
*/
   printf("\n");
   printf("DOS tool for local configuration of KIRA-based products.\n");
   printf("\n");
   printf("Usage: kiratool [-v] [-asmi] <cmd> [<cmd args>]\n");
   printf("Commands\n");
   printf("  ver..................queries firmware version\n");
   printf("  serial...............queries serial number\n");
   printf("  serial <serial#>.....sets serial number\n");
   printf("  mac..................queries mac address\n");
   printf("  mac <addr>...........sets mac address\n");
   printf("  upgrade <filename>...upgrades firmware\n");
   printf("  defaults.............resets to factory defaults\n");
   printf("Options\n");
   printf("  -v...................verbose\n");
   printf("  -asmi................use on ASMI modules\n");
   printf("\n");
   exit(0xFFFF);
}

void SendRecv(MicroReq_t *cmd, MicroResp_t *resp, int pollonly)
{
    if ((SendRequestToMicro( cmd, resp ) != 0) && (!pollonly))
    {
      printf("Error sending command to BMC.\n");
      exit(0xFFFF);
    }

    resp->ipmbAddr = cmd->ipmbAddr;

    if ((ReadResponseFromMicro( resp )) && (!pollonly))
    {
        printf("Error reading from micro\n");
        exit(0xFFFF);
    }
}

void ErrExit(int cc)
{
    printf("ERROR (%02xh)\n", cc);
    exit(cc);
}

void ver(void)
{
    MicroReq_t  cmd;
    MicroResp_t resp;

    cmd.bmcReq.netFnLn = 0x3c << 2; // netfn/lun: PPOEM/0
    cmd.bmcReq.cmd = 0x20; // cmd: Get FW Ver
    cmd.ipmbAddr = 0x20;
    cmd.length = 0;

    SendRecv(&cmd, &resp, 0);

    switch (resp.bmcResp.cCode) {
        case 0x00: {
                char *tag, *oem;
                unsigned long *ver_data;
                int oem_len;

                ver_data = (unsigned long*) resp.bmcResp.data;
                printf("F/W ver: %lu.%lu.%lu build %lu\n",
                    ver_data[0], ver_data[1], ver_data[2], ver_data[3]);

                printf("H/W ID:  0x%02x\n", resp.bmcResp.data[16]);

                tag = resp.bmcResp.data + 17;
                printf("F/W Tag: '%s'\n", tag);

                oem = tag + strlen(tag) + 1;
                oem_len = resp.respLen - (oem - resp.bmcResp.data);
                printf("OEM:     '%.*s'\n", oem_len, oem);
                break;
            }
        default:
            ErrExit(resp.bmcResp.cCode);
    }

    exit( resp.bmcResp.cCode );
}

void getsn(void)
{
    MicroReq_t  cmd;
    MicroResp_t resp;

    cmd.bmcReq.netFnLn = 0x3c << 2; // netfn/lun: PPOEM/0
    cmd.bmcReq.cmd = 0x41; // cmd: Get Serial Number
    cmd.ipmbAddr = 0x20;
    cmd.length = 0;

    SendRecv(&cmd, &resp, 0);

    switch (resp.bmcResp.cCode) {
        case 0x00:
            printf("Serial number: '%.*s'\n", resp.respLen, resp.bmcResp.data);
            break;
        case 0x80:
            printf("Serial number not yet set\n");
            break;
        default:
            ErrExit(resp.bmcResp.cCode);
    }

    exit( resp.bmcResp.cCode );
}

void setsn(const char* sn)
{
    MicroReq_t  cmd;
    MicroResp_t resp;

    if (!sn) ExitWithUsage();

    cmd.bmcReq.netFnLn = 0x3c << 2; // netfn/lun: PPOEM/0
    cmd.bmcReq.cmd = 0x42; // cmd: Set Serial Number
    strcpy(cmd.bmcReq.data, sn); // TODO: not save!
    cmd.ipmbAddr = 0x20;
    cmd.length = strlen(sn);

    SendRecv(&cmd, &resp, 0);

    switch (resp.bmcResp.cCode) {
        case 0x00:
            printf("Serial number successfully set\n");
            break;
        case 0x80:
            printf("Serial number already set, can not be modified\n");
            break;
        default:
            ErrExit(resp.bmcResp.cCode);
    }

    exit( resp.bmcResp.cCode );
}

void getmac(void)
{
    MicroReq_t  cmd;
    MicroResp_t resp;

    cmd.bmcReq.netFnLn = 0x0c << 2; // netfn/lun: Transport/0
    cmd.bmcReq.cmd = 0x02; // cmd: Get LAN Config Params
    cmd.bmcReq.data[0] = 1; // chan: LAN
    cmd.bmcReq.data[1] = 5; // param: MAC addr
    cmd.bmcReq.data[2] = 0; // set
    cmd.bmcReq.data[3] = 0; // block
    cmd.ipmbAddr = 0x20;
    cmd.length = 4;

    SendRecv(&cmd, &resp, 0);

    switch (resp.bmcResp.cCode) {
        case 0x00:
            printf("MAC address: %02x:%02x:%02x:%02x:%02x:%02x\n",
                   resp.bmcResp.data[1], resp.bmcResp.data[2],
                   resp.bmcResp.data[3], resp.bmcResp.data[4],
                   resp.bmcResp.data[5], resp.bmcResp.data[6]);
            break;
        case 0x80:
            printf("MAC address not available\n");
            break;
        default:
            ErrExit(resp.bmcResp.cCode);
    }

    exit( resp.bmcResp.cCode );
}

void setmac(const char* mac)
{
    MicroReq_t  cmd;
    MicroResp_t resp;
    int addr[6];

    if (!mac || 6 != sscanf(mac, "%02x:%02x:%02x:%02x:%02x:%02x",
        &addr[0], &addr[1], &addr[2], &addr[3], &addr[4], &addr[5])) {
        printf("Malformed MAC address (format xx:xx:xx:xx:xx:xx)\n");
        exit(0xffff);
    }

    cmd.bmcReq.netFnLn = 0x0c << 2; // netfn/lun: Transport/0
    cmd.bmcReq.cmd = 0x01; // cmd: Set LAN Config Params
    cmd.bmcReq.data[0] = 1; // chan: LAN
    cmd.bmcReq.data[1] = 5; // param: MAC addr
    cmd.bmcReq.data[2] = (BYTE)addr[0];
    cmd.bmcReq.data[3] = (BYTE)addr[1];
    cmd.bmcReq.data[4] = (BYTE)addr[2];
    cmd.bmcReq.data[5] = (BYTE)addr[3];
    cmd.bmcReq.data[6] = (BYTE)addr[4];
    cmd.bmcReq.data[7] = (BYTE)addr[5];
    cmd.ipmbAddr = 0x20;
    cmd.length = 8;

    SendRecv(&cmd, &resp, 0);

    switch (resp.bmcResp.cCode) {
        case 0x00:
            printf("MAC address successfully set\n");
            break;
        case 0x80:
            printf("MAC address not available\n");
            break;
        case 0x82:
            printf("MAC address can not be changed\n");
            break;
        default:
            ErrExit(resp.bmcResp.cCode);
    }

    exit( resp.bmcResp.cCode );
}

void upgrade(const char* fname)
{
    MicroReq_t  cmd;
    MicroResp_t resp;
    struct stat s;
    int retries;
    FILE *f;
    unsigned short res_id;
    long chunk_size, offs;
    clock_t tm;

    if (!fname) ExitWithUsage();

    tm = clock();

    f = fopen(fname, "rb");
    if (!f) {
        printf("Can't open firmeware file '%s'\n", fname);
        exit(0xffff);
    }
    if (fstat(fileno(f), &s) < 0) {
        printf("Can't get file size of '%s'\n", fname);
        exit(0xffff);
    }

    printf("Starting firmware upgrade...");

    cmd.bmcReq.netFnLn = 0x3c << 2; // netfn/lun: PPOEM/0
    cmd.bmcReq.cmd = 0x21; // cmd: Start FW Upgrade
    *(unsigned long*)cmd.bmcReq.data = s.st_size;
    cmd.ipmbAddr = 0x20;
    cmd.length = sizeof(unsigned long);

    SendRecv(&cmd, &resp, 0);

    if (resp.bmcResp.cCode != 0) ErrExit(resp.bmcResp.cCode);

    printf("done\n");

    res_id = *(unsigned short*)(resp.bmcResp.data + 0);
    chunk_size = (long)*(unsigned long*)(resp.bmcResp.data + 2);

#define MAX_CHUNK_SIZE 1000
    if (chunk_size > MAX_CHUNK_SIZE) chunk_size = MAX_CHUNK_SIZE;

    printf("\n");
    for (offs = 0; offs < (long)s.st_size; offs += chunk_size) {
        unsigned long p;
        size_t count = (size_t) min(chunk_size, (long)s.st_size - offs);
        size_t chunk_len = fread(cmd.bmcReq.data + 6, 1, count, f);
        if (chunk_len < count) {
            if (ferror(f)) {
                printf("\nFailed to read '%s' at offset %d\n", fname, offs);
            }
            if (feof(f)) {
                printf("\nFound unexpected EOF in '%s' at offset %ld\n", fname, offs+chunk_len);
            }
            ErrExit(-1);
        }

        p = (long)offs * 100UL / (long)s.st_size + 1;
        printf("\rUploading firmware...%ld/%ld (%ld%%)",
               offs / 1000L, (long)s.st_size / 1000L, p);
        fflush(stdout);

        cmd.bmcReq.netFnLn = 0x3c << 2; // netfn/lun: PPOEM/0
        cmd.bmcReq.cmd = 0x22; // cmd: Upload FW
        *(unsigned short*)(cmd.bmcReq.data + 0) = res_id;
        *(unsigned long*)(cmd.bmcReq.data + 2) = (unsigned long)offs;
        cmd.ipmbAddr = 0x20;
        cmd.length = (int)(6 + chunk_len);

        SendRecv(&cmd, &resp, 0);

        if (resp.bmcResp.cCode != 0) {
            printf("\nFailed to upload firmware at offset %d\n", offs);
            ErrExit(resp.bmcResp.cCode);
        }
    }

    printf("\rUploading firmware...done                        \n");

    tm = clock() - tm;
    printf("Upload time: %ds\n", tm / 1000);

    fclose(f);

    printf("Flashing firmware...");

    cmd.bmcReq.netFnLn = 0x3c << 2; // netfn/lun: PPOEM/0
    cmd.bmcReq.cmd = 0x23; // cmd: Flash FW
    *(unsigned short*)(cmd.bmcReq.data + 0) = res_id;
    cmd.bmcReq.data[2] = 0x00; // flags
    cmd.ipmbAddr = 0x20;
    cmd.length = 3;

    SendRecv(&cmd, &resp, 0);

    if (resp.bmcResp.cCode != 0) ErrExit(resp.bmcResp.cCode);

    printf("done\n");

    //FW flash will at least take a minute, so wait
    printf("Waiting for FW update to complete...");
    msDelay(60000);
    printf("done.\n");

    printf("Checking FW update status and reboot device...");
    // poll the BMC in 10 second intervals
    // if it does not get ready after 5 minutes (30 runs),
    // we suppose the FW flash failed
    retries = 30;
    while (retries > 0) {
        cmd.bmcReq.netFnLn = 0x3c << 2; // netfn/lun: PPOEM/0
        cmd.bmcReq.cmd = 0x25; // cmd: Finalize FW upgrade
        *(unsigned short*)(cmd.bmcReq.data + 0) = res_id;
        cmd.ipmbAddr = 0x20;
        cmd.length = 2;

        SendRecv(&cmd, &resp, 1);

        if (resp.bmcResp.cCode == 0x00) {
            printf("done.\n");
            retries = 0;
        } else if (resp.bmcResp.cCode == 0x83) {
            printf("FW flash failed: '%.*s'\n", resp.respLen, resp.bmcResp.data);
            retries = 0;
        } else {
            retries--;
            msDelay(10000);
        }
    }

    exit( resp.bmcResp.cCode );
}

void defaults(void)
{
    MicroReq_t  cmd;
    MicroResp_t resp;

    cmd.bmcReq.netFnLn = 0x3c << 2; // netfn/lun: PPOEM/0
    cmd.bmcReq.cmd = 0x40; // cmd: Reset To Factory Defaults
    cmd.ipmbAddr = 0x20;
    cmd.length = 0;

    SendRecv(&cmd, &resp, 0);

    switch (resp.bmcResp.cCode) {
        case 0x00:
            printf("Command successfully sent, device is resetting now.\n");
            break;
        default:
            ErrExit(resp.bmcResp.cCode);
    }

    exit( resp.bmcResp.cCode );
}

/*******************************************************************************
*  Routine: main
*
*  Arguments:
*     argc - argument count
*     argv - array of arguments
*
*  Returns:
*     Errorlevel of 0 to O/S if all went well
*     Nonzero errorlevel otherwise
*
*  Description:
*     This is the entry point for the utility to download an SDR file to the BMC
*
*******************************************************************************/

void main(int argc, char *argv[])
{
    int             i;
    int             printTime = 0;
    clock_t         startTime;
    BYTE            CmdBuffer[256];
    int             temp;
    BYTE            ResponseBuffer[256];
    char *          ifName="isa_sms";
    MicroReq_t *    cmd = (MicroReq_t *) CmdBuffer;
    MicroResp_t *   resp = (MicroResp_t *) ResponseBuffer;

    argv++;
    argc--;

    // printf( "CLOCKS_PER_SEC = %d\n", CLOCKS_PER_SEC );
    //
    // make sure we have enough to handle interface request
    //
    if ( argc < 1 )
    {
        ExitWithUsage();
    }
    //
    // parse our switches
    //
    while ( *argv[0] == '-' )
    {
        //
        // check for debug option
        //
        if ( stricmp( *argv, "-v" ) == 0 )
        {
            Debug = 1;
            argv++;
            argc--;
        }
        //
        // use special KCS address for ASMI
        //
        else if ( stricmp( *argv, "-asmi" ) == 0 )
        {
            ifName = "isa_sms:A02";
            argv++;
            argc--;
        }
        //
        // see if he wants to print the command execution time
        //
        else if ( stricmp( *argv, "-t" ) == 0 )
        {
            printTime = 1;
            argv++;
            argc--;
        }
        //
        // see if he wants to print the command execution time
        //
        else if ( stricmp( *argv, "-i9" ) == 0 )
        {
            SetIPMIversion( IPMI_0_9 );
            argv++;
            argc--;
        }
        //
        // check for requested interface
        //
        else if ( stricmp( *argv, "-if" ) == 0 )
        {
            if( (ifName = *++argv) == NULL )
            {
                ExitWithUsage();
            }
            argv++;
            argc -= 2;
        }
    }
    if ( FindInterface( ifName ) == FALSE )
    {
        fprintf( stderr, "Can't find an interface to use\n");
        exit( 1 );
    }

    if (argc < 1) ExitWithUsage();
    if (stricmp(*argv, "ver") == 0) ver();
    if (stricmp(*argv, "serial") == 0) {
        argv++; argc--;
        if (argc > 0) setsn(*argv);
        else getsn();
    }
    if (stricmp(*argv, "mac") == 0) {
        argv++; argc--;
        if (argc > 0) setmac(*argv);
        else getmac();
    }
    if (stricmp(*argv, "upgrade") == 0) {
        argv++; argc--;
        if (argc > 0) upgrade(*argv);
        else ExitWithUsage();
    }
    if (stricmp(*argv, "defaults") == 0) defaults();

    //
    // require at least SA, netfnLun, cmd
    //
    if ( argc < 3 )
    {
        ExitWithUsage();
    }
    //
    // get slave address
    //
    if ( sscanf( *argv++, "%x", &temp) != 1 )
    {
        ExitWithUsage();
    }
    cmd->ipmbAddr = (BYTE)(temp & 0xFF);
    //
    // collect netfnLun, cmd, and data
    //
    for ( i = 0; *argv; i++)
    {
        if( sscanf( *argv++, "%x", &temp) != 1 )
        {
            ExitWithUsage();
        }
        ((BYTE *) & cmd->bmcReq)[ i ] = (BYTE)(temp & 0xFF);
    }
    cmd->length = i - 2;    // adjust for the netfn and cmd bytes
    //
    // Send the command to the micro
    //
    startTime = clock();

    if( SendRequestToMicro( cmd, resp ) != 0 )
    {
      printf("Error sending command to BMC.\n");
      exit(0xFFFF);
    }
    resp->ipmbAddr = cmd->ipmbAddr;


    if( ReadResponseFromMicro( resp ) )
    {
        printf("Error reading from micro\n");
        exit(0xFFFF);
    }

    //
    // print out the buffer, skipping the first byte (length)
    // there are 5 bytes not accounted for by the respLen:
    //   respLen, ipmbAddr, nfLun, cmd, cCode
    //
    for ( i = 1; i < resp->respLen + 5 ; i++)
    {
        if(i != 1)
            printf(" ");

        printf( "%02x", ResponseBuffer[i]);
    }
    printf("\n");

    if( printTime )
        printf( "%ld ms\n", clock() - startTime );

    exit( resp->bmcResp.cCode );
}
