/*******************************************************************************
********************************************************************************
***                                                                           **
***                    COPYRIGHT 1998 INTEL CORPORATION                       **
***                                                                           **
***                INTEL CORPORATION PROPRIETARY INFORMATION                  **
***                                                                           **
***      This software is supplied under the terms of a license agreement     **
***      or non-disclosure agreement with Intel Corporation and may not be    **
***      copied or disclosed except in accordance with the terms of that      **
***      agreement.                                                           **
***                                                                           **
***      This Source Code is provided "as is" without any warranty of any     **
***      kind, express, implied, statutory or otherwise, and Intel            **
***      specifically disclaims any implied warranties for any particular     **
***      purpose.                                                             **
***                                                                           **
********************************************************************************
******** 10 ****** 20 ****** 30 ****** 40 ****** 50 ****** 60 ****** 70 *******/


/*----------------------------------------------------------
* Copyright (c) 2002 by National Semiconductor Corporation
* All rights reserved.
*-----------------------------------------------------------
* File Contents:
*     Added SMB interface for mBMC
* Project:  PC87431
*---------------------------------------------------------*/

/*******************************************************************************

    Source Control Information:

    $Workfile:   isa.c  $

    $Revision: 1.18 $

    $Date: 2003/07/10 13:33:30Z $

    $Modtime:   Nov 04 1999 17:41:12  $

    $Archive:   R:/FW/NVSTLTS.PRJ/ISA.C_V  $

*******************************************************************************/ 

/*******************************************************************************

   Module description: 

   This is the source code for the ISA interface (used in Nightshade & Sitka)

*******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>                         
#include <conio.h>
#include "pci.h"

#define dbgFp	stdout

//
// turn off the generic definition of InterfaceHandle
//
#define INTERFACE_PRIVATE   1 
#define TIMEOUT_COUNT       2000
#define NUM_RETRIES         5
//
// Overload the interface to be a pointer to an IsaInterface structure
//
typedef struct _IsaInterface * InterfaceHandle;

#include "common.h"
//
// private ISA interface data structure
//
typedef struct _IsaInterface
{
	InterfaceApi	interface;	// array of function pointers
	WORD			dataAddr;	// address of interface
	clock_t			reqTime;	// time it took last send, not valid between send and recv
	WORD            Controller; // Controller type
	WORD            ReqBMCAddr; // BMC Address in request

} IsaInterface;

#include "isa.h"                           
                     
/*******************************************************************************
*  Routine: printISAStatus
*
*  Arguments:
*     N/A
*
*  Returns:
*     N/A
*
*  Description:
*     This routine prints info. about the current state of the ISA interface.
*     
*******************************************************************************/

void printISAStatus( BYTE ISACmdStatus )
{
   fprintf( dbgFp, "ISA Registers\n");
   fprintf( dbgFp, "\n");

   fprintf( dbgFp, "Command/Status:     %02Xh\n", ISACmdStatus);
   fprintf( dbgFp, "-------------------------\n");
   fprintf( dbgFp, "   S1:              %02Xh\n", (ISACmdStatus & ISA_S1_FLAG));
   fprintf( dbgFp, "   S0:              %02Xh\n", (ISACmdStatus & ISA_S0_FLAG));
   fprintf( dbgFp, "   RESERVED:        %02Xh\n", (ISACmdStatus & 0x20));
   fprintf( dbgFp, "   RESERVED:        %02Xh\n", (ISACmdStatus & 0x10));
   fprintf( dbgFp, "   C/D#:            %02Xh\n", (ISACmdStatus & ISA_CD_FLAG));
   fprintf( dbgFp, "   SMS_ATN/SMM_ATN: %02Xh\n", (ISACmdStatus & ISA_SMS_MSG_FLAG));
   fprintf( dbgFp, "   IBF:             %02Xh\n", (ISACmdStatus & ISA_IBF_FLAG));
   fprintf( dbgFp, "   OBF:             %02Xh\n", (ISACmdStatus & ISA_OBF_FLAG));
   fprintf( dbgFp, "\n");
   fprintf( dbgFp, "   STATE (S1 + SO): ");

   switch(ISACmdStatus & ISA_STATE_MASK)
   {
      case ISA_IDLE_STATE:
         fprintf( dbgFp, "IDLE\n");
         break;
      case ISA_READ_STATE:
         fprintf( dbgFp, "READ\n");
         break;
      case ISA_WRITE_STATE:
         fprintf( dbgFp, "WRITE\n");
         break;
      case ISA_ERROR_STATE:
         fprintf( dbgFp, "ERROR\n");
         break;
      default:
         break;
   }
}

/* -----------------------------------------------------------------------------
Name:    delay
Purpose: Waits 15 usec 
         timeout expires, returns 0. returns -1 if timeout occurs.
Returns: none
Parameters: none
Notes:   Patch to cover problem with CL10 ISA interface
----------------------------------------------------------------------------- */
static void delay (void)
{ 
   usDelay(15);
}

/* -----------------------------------------------------------------------------
Name:    wait_while_ibf
Purpose: Waits timeout ms for IBF to be low. If IBF goes low before
         timeout expires, returns 0. returns -1 if timeout occurs.
Returns: 0 success, -1 failure
Parameters:
         timeout - millisec to wait
Notes:
----------------------------------------------------------------------------- */
static int wait_while_sms_ibf( WORD addr, WORD timeout)
{           
   clock_t overallTimeout;

   SetMsTimeout( &overallTimeout, timeout);

   while ( ISA_IBF(addr) )
   {     
      if ( CheckMsTimeout(overallTimeout) ) 
      {
         printISAStatus( ISA_STATUS(addr) );
         return(-1); /* timed out */
      }              
   }

   return(0);        /* have message, no timeout */
}
//
// busy wait for 'timeout' us
//
#define SMM_QUANTUM	1

static int wait_while_smm_ibf( WORD addr, WORD timeout )
    {           
    for ( ; ISA_IBF(addr) ; timeout -= SMM_QUANTUM )
    {
        usDelay( SMM_QUANTUM );

        if ( timeout <= SMM_QUANTUM ) 
        {
            printISAStatus( ISA_STATUS(addr) );
            return -1;
        }              
    }
    return 0;
}

/* -----------------------------------------------------------------------------
Name:    wait_til_obf
Purpose: Waits timeout ms for OBF to be high. If OBF goes high before
      timeout expires, returns 0. returns -1 if timeout occurs.
Returns: 0 success, -1 failure
Parameters:
      timeout - millisec to wait
Notes:
----------------------------------------------------------------------------- */

static int wait_til_obf( WORD addr, WORD timeout)
{       
   clock_t overallTimeout;

   SetMsTimeout( &overallTimeout, timeout);

   while ( !ISA_OBF(addr) )
   {        
      if ( CheckMsTimeout(overallTimeout) ) 
      {
         return(-1); /* timed out */
      }              
   }                                                    

   return(0);        /* have message, no timeout */
}

/* -----------------------------------------------------------------------------
Name:    checkMessageReady
Purpose: Checks for ready flag from target. If the target is the BMC, the
      ready flag is PMOBF. Otherwise, the flag is ISA_SMS_MSG
Returns: 0 success, -1 failure
Parameters:
         target - IPMB I.D. of destination
      timeout - millisec to wait
Notes:
----------------------------------------------------------------------------- */

static int checkMessageReady( WORD addr, BYTE target, WORD timeout)
{       
   clock_t			overallTimeout;
   volatile BYTE	status;
   
   if (target == BMC_SLAVE_ADDR)
   {               
      SetMsTimeout( & overallTimeout, timeout );
      do
      {        
         if ( CheckMsTimeout(overallTimeout) ) 
         {
            return(-1); /* timed out */
         }              
         status = ISA_STATUS(addr);
      }                                                    
      while ( (status & 0x34) != 0 || (status & ISA_OBF_FLAG) == 0 );
   
      delay();

      return(0);        /* have message, no timeout */
   }
   else
   {            
      SetMsTimeout ( &overallTimeout, timeout);

      while ( ! ISA_MSG(addr) )
      {
         if ( CheckMsTimeout(overallTimeout) ) 
         {                
            return(-1); /* timed out */
         }
      }
      return(0);        /* have message, no timeout */
   }
}

/* -----------------------------------------------------------------------------
Name:    IsaSmsSendRequest
Purpose: Sends cmds to the BMC
Returns: 0 success, -1 failure
Parameters:
         msgBuf - pointer to source buffer
Notes:
----------------------------------------------------------------------------- */

static int IsaSmsSendRequest( IsaInterface * intf, BmcReq_t *req, int length )
    { 
    BYTE *msgBuf = (byte *) req;
    int    i;
    BYTE    state;
    BYTE    status;

    length += MIN_BMC_REQ_SIZE;

    dprintf( dbgFp, "IsaSmsSendRequest: intf struct at 0x%x, 0x%x bytes\n",
        (DWORD) intf, length );

    intf->reqTime = clock();
    dprintf( dbgFp, "req start time %lX\n", intf->reqTime );
    //
    // wait for BMC to read whatever might be in the input buffer
    //
    if ( wait_while_sms_ibf( intf->dataAddr, ISA_SMS_TIMEOUT ) )
    {
        fprintf( dbgFp, "ERROR: Timeout waiting for IBF to clear before sending WRITE START.\n");
        return( -1 );                               
    }
    //
    // clear any OBF
    //
    status = ISA_STATUS(intf->dataAddr);

    if ( status & ISA_OBF_FLAG )                 /* flush output buffer */
    {
        ISA_READ_BMC_DATA(intf->dataAddr);
    }
    //
    // make sure we're starting in idle or error state
    //
    state = status & ISA_STATE_MASK;

    if (state != ISA_ERROR_STATE && state != ISA_IDLE_STATE)
    {
        fprintf( dbgFp, "ERROR: BMC Interface not in idle or error state\n");

        if ( Debug )
            printISAStatus( status );
        //
        // try forcing into error state
        //
        ISA_WRITE_BMC_CMD( intf->dataAddr, ISA_ERROR_STATE );

        if ( wait_while_sms_ibf( intf->dataAddr, ISA_SMS_TIMEOUT ) )
        {
            fprintf( dbgFp, "ERROR: Timeout waiting for IBF to clear after sending ERROR.\n");
            return( -1 );                               
        }
        //
        // flush any outpound data
        //
        if( ISA_OBF(intf->dataAddr) )
            (void) ISA_READ_BMC_DATA(intf->dataAddr);

        // return -1;
    }
    //
    // get the BMC's attention and wait until he reads it
    // 
    ISA_WRITE_BMC_CMD( intf->dataAddr, ISA_WRITE_START );

    if ( wait_while_sms_ibf( intf->dataAddr, ISA_SMS_TIMEOUT ) )
    {
        fprintf( dbgFp, "ERROR: Timeout waiting for IBF to clear after sending WRITE START.\n");
        return( -1 );                               
    }
    //
    // send each message byte out
    //
    for ( i = 0; i < length - 1; i++ )              /* send the message */
    {                   
        status = ISA_STATUS(intf->dataAddr);

        if ( status & ISA_OBF_FLAG )                 /* flush output buffer */
        {
            ISA_READ_BMC_DATA(intf->dataAddr);
        }

        if ( (status & ISA_STATE_MASK) != ISA_WRITE_STATE )  /* should have gone to */
        {                                                    /* WRITE_STATE */
            fprintf( dbgFp, "ERROR: BMC did not enter WRITE STATE.\n");
            return( -1 );                               
        }

        ISA_WRITE_BMC_DATA( intf->dataAddr, msgBuf[i] );             /* output data byte */

        if ( wait_while_sms_ibf( intf->dataAddr, ISA_SMS_TIMEOUT ) )         /* wait til BMC reads */
        {
            fprintf( dbgFp, "ERROR: Timeout waiting for IBF to clear after sending data byte.\n");
            return( -1 );                               
        }
    }

    ISA_WRITE_BMC_CMD( intf->dataAddr, ISA_WRITE_END );             /* flag next byte is last */

    if ( wait_while_sms_ibf( intf->dataAddr, ISA_SMS_TIMEOUT ) )            /* wait til BMC reads */
    {                                               /* WRITE_END control code */
        fprintf( dbgFp, "ERROR: Timeout waiting for IBF to clear after sending WRITE END.\n");
        return( -1 );                               
    }
    status = ISA_STATUS(intf->dataAddr);

    if ( status & ISA_OBF_FLAG )                 /* flush output buffer */
    {
        ISA_READ_BMC_DATA(intf->dataAddr);
    }

    if ( (status & ISA_STATE_MASK) != ISA_WRITE_STATE )  /* should have stayed in */
    {                                                    /* WRITE_STATE */
        fprintf( dbgFp, "ERROR: BMC did not stay in WRITE STATE.\n");
        return( -1 );                               
    }


    ISA_WRITE_BMC_DATA( intf->dataAddr, msgBuf[i] );                /* output last data byte */

    if ( wait_while_sms_ibf( intf->dataAddr, ISA_SMS_TIMEOUT ) )            /* wait til BMC reads */
    {                                               /* data byte */
        fprintf( dbgFp, "ERROR: Timeout waiting for IBF to clear after sending last data byte.\n");
        return( -1 );                               
    }

    if ( ISA_STATE(intf->dataAddr) != ISA_READ_STATE ) /* and verify BMC went */
    {                                                  /* into READ_STATE */
        fprintf( dbgFp, "ERROR: BMC did not enter READ STATE.\n");
        return( -1 );                               
    }
    else
        return( 0 );
}

/* -----------------------------------------------------------------------------
Name:    IsaSmsReadResponse
Purpose: Reads response data through the SMS/BMC ISA interface.
Returns: count - number of data bytes read (first three are NetFn,system sw ID, and cmd)
Parameters:
         msgBuf - pointer to destination buffer
Notes:
----------------------------------------------------------------------------- */

static int IsaSmsReadResponse( IsaInterface * intf, BmcResp_t * resp )
    {
    BYTE *	msgBuf = (BYTE *) resp;                
    int		i;
    BYTE	state;
    BYTE	status;

    dprintf( dbgFp, "IsaSmsReadResponse: intf struct at 0x%x, dataAddr %xh\n", (DWORD) intf, intf->dataAddr );

    i = 0;

    while( i < MAX_ISA_LENGTH )
    {
        //
        // make sure the BMC saw our last request
        //
        if ( wait_while_sms_ibf( intf->dataAddr, ISA_SMS_TIMEOUT) )
        {                           
            intf->reqTime = 0;
            dprintf( dbgFp, "IsaSmsReadResponse: IBF timeout, respLen %d\n", i );

            return -2;
        }
        //
        // test for state
        //
        status = ISA_STATUS(intf->dataAddr);

        // dprintf( dbgFp, "OW-S%02X ", status);

        state = (BYTE)(status & ISA_STATE_MASK);
        //
        // check to see if we're done
        //
        if ( state == ISA_IDLE_STATE )
        {
            //
            // read and ignore our handshake (clears OBF)
            //
            status = ISA_READ_BMC_DATA(intf->dataAddr);

            dprintf( dbgFp, "req end time %lX\n", clock() );

            intf->reqTime = clock() - intf->reqTime;	// calculate duration

            dprintf( dbgFp, "IsaSmsReadResponse: read %d bytes\n", i );

            return i;
        }
        //
        // any state other than read is an error
        //
        else if( state != ISA_READ_STATE )
        {
            dprintf( dbgFp, "IsaSmsReadResponse: intf in wrong state (%02Xh), respLen %d\n",
                                status, i);
            intf->reqTime = 0;

            return -1;
        }
        //
        // we're in read state
        // wait until the BMC gives us our data
        //
        if ( wait_til_obf( intf->dataAddr, ISA_SMS_TIMEOUT) )
        {
            intf->reqTime = 0;
            dprintf( dbgFp, "IsaSmsReadResponse: OBF timeout, respLen %d\n", i );

            return -3;
        }
        //
        // capture our data and ask for the next
        //
        msgBuf[i++] = ISA_READ_BMC_DATA(intf->dataAddr);

        ISA_WRITE_BMC_DATA( intf->dataAddr, ISA_READ );
    }
    //
    // too many data bytes, complain vociferously
    //
    dprintf( dbgFp, "IsaSmsReadResponse: buffer overrun\n" ); 

    intf->reqTime = 0;

    return -3;
}

static void IsaIntfConfigure( IsaInterface * intf, WORD addr )
{
	intf->dataAddr		= addr;
}

static WORD IsaSmsCapabilities( IsaInterface *intf )
{
	return IF_IPMB_ACCESS;
}

static byte IsaBmcAddr( IsaInterface *intf )
{
	return BMC_DEV_ADDR;
}

static void IsaSmsSetTimeout( IsaInterface *intf, int which, int what )
{
	;	// not yet
}

static int IsaSmsReady( IsaInterface * intf )
{
	return ISA_MSG(intf->dataAddr);
}

static clock_t IsaLastReqTime( IsaInterface * intf )
{
	return intf->reqTime;
}
////////////////////////////////////////////////////////////////////////
// Isa SMM routines
//

static WORD IsaSmmCapabilities( IsaInterface * intf )
{
	return 0;
}

static void IsaSmmSetTimeout( IsaInterface * intf, int which, int what )
{
	;
}
static int IsaSmmSendRequest( IsaInterface * intf, BmcReq_t * req, int length )
    {
    byte *	msgBuf = (byte *) req;
    int		i;

    length += MIN_BMC_REQ_SIZE;
    //
    // wait until we can stuff our request in the buffer
    //
    if ( wait_while_smm_ibf( intf->dataAddr, ISA_SMM_TIMEOUT ) )                    /* BMC could be busy */
    {
        fprintf( stderr, "ERROR: Timeout waiting for IBF to clear before sending WRITE START.\n");
        return -1;                               
    }
    ISA_WRITE_BMC_CMD( intf->dataAddr, ISA_WRITE_START );           /* get the BMC's attention */
    //
    // make sure he gets the write request
    //
    if ( wait_while_smm_ibf( intf->dataAddr, ISA_SMM_TIMEOUT ) )            /* wait til BMC reads */
    {
        fprintf( stderr, "ERROR: Timeout waiting for IBF to clear after sending WRITE START.\n");
        return -1;                               
    }
    //
    // write out our message
    //
    for ( i = 0; i < length - 1; i++ )              /* send the message */
    {                   
        BYTE status = ISA_STATUS(intf->dataAddr);

        if ( status & ISA_OBF_FLAG )                 /* flush output buffer */
            ISA_READ_BMC_DATA(intf->dataAddr);
        //
        // make sure the BMC entered WRITE state
        //
        if ( (status & ISA_STATE_MASK) != ISA_WRITE_STATE )
        {
            fprintf( stderr, "ERROR: BMC did not enter WRITE STATE.\n");
            return -1;                               
        }
        //
        // give the BMC our data byte
        //
        ISA_WRITE_BMC_DATA( intf->dataAddr, msgBuf[i] );
        //
        // make sure it takes it
        //
        if ( wait_while_smm_ibf( intf->dataAddr, ISA_SMM_TIMEOUT ) )
        {
            fprintf( stderr, "ERROR: Timeout waiting for IBF to clear after sending data byte.\n");
            return -1;                               
        }
    }
    //
    // tell the BMC that the next byte is the last in the request
    // and wait for it to get the command
    //
    ISA_WRITE_BMC_CMD( intf->dataAddr, ISA_WRITE_END );

    if ( wait_while_smm_ibf( intf->dataAddr, ISA_SMM_TIMEOUT ) )
    {
      fprintf( stderr, "ERROR: Timeout waiting for IBF to clear after sending WRITE END.\n");
      return -1;                               
    }
    //
    // give the BMC the last byte
    // and wait for it to take it
    //
    ISA_WRITE_BMC_DATA( intf->dataAddr, msgBuf[i] );

    if ( wait_while_smm_ibf( intf->dataAddr, ISA_SMM_TIMEOUT ) )
    {
      fprintf( stderr, "ERROR: Timeout waiting for IBF to clear after sending last data byte.\n");
      return -1;                               
    }
    //
    // make sure the BMC goes into READ state
    //
    if ( ISA_STATE(intf->dataAddr) != ISA_READ_STATE )
    {
        fprintf( stderr, "ERROR: BMC did not enter READ STATE.\n");
        return -1;                               
    }
    else
        return 0;
}

static int IsaSmmReadResponse( IsaInterface * intf, BmcResp_t * resp )  
{
    byte *	msgBuf = (byte *) resp;
    int		i;
    BYTE	state;
    BYTE	status;
    WORD	loopCnt;

    for (i = 0; i < MAX_ISA_LENGTH; i++)
    {
        //
        // get a byte
        //
        loopCnt = 1000;

        do
        {
            //
            // wait until there's data to read or the interface
            // is idle or has an error
            //
            status = ISA_STATUS(intf->dataAddr);

            state = (BYTE)(status & ISA_STATE_MASK);

            if (state == ISA_ERROR_STATE)
            {
                fprintf( stderr, "Interface error after %d bytes read\n", i);
                printISAStatus( ISA_STATUS(intf->dataAddr) );
                return -1;
            }
            else if (state == ISA_IDLE_STATE)
            {
                // printf( "Interface idle after %d bytes read\n", i);

                return i;
            }
            else if ( loopCnt-- <= 0 )
            {
                fprintf( stderr, "Interface timeout after %d bytes read\n", i);
                printISAStatus( ISA_STATUS(intf->dataAddr) );
                return -1;
            }
        }
        while ( (status & ISA_OBF_FLAG) == 0 );

        msgBuf[i] = ISA_READ_BMC_DATA(intf->dataAddr);

        //
        // wait for 10 us and then make sure that the BMC has
        // taken our last command byte
        //
        // usDelay( (WORD) 10 );

        //for (  ISA_SMM_IBF ; loopCnt-- )
        //	;
        //if ( ISA_SMM_IBF )
        //{
        //	printISAStatus( ISA_SMM_STATUS );
        //	return -1;
        //}
        //
        // ask for the next byte
        //
        //ISA_SMM_WRITE_BMC_DATA( ISA_READ );
    }
    printf( "Receive buffer overflow\n" );
    return -1;
}

static void SMBIntfConfigure( IsaInterface * intf, WORD addr )
{

    intf->dataAddr    = addr;
}

static void findSMBcontroller (IsaInterface * intf)
{
    int  found;
    unsigned long devid, devaddr;
    unsigned char dev, func;
   
    // Find PCI SMBus controller
    /* Check PCI identifier */
    found = 0;
    for (dev=0;dev<32;dev++) {
     for (func=0;func<8;func++) {
        switch (devid = get_PCI_devid(dev,func)) {
        case ID_8111:
            printf("AMD 8111xx SMBus controller,");
            found = 1;
            break;
        case ID_81801:
            printf("Intel 82801xx SMBus controller,");
            found = 1;
            break;
        case ID_81801AB:
            printf("Intel 82801AB (ICH0) SMBus controller,");
            found = 1;
            break;
        case ID_82801BA:
            printf("Intel 82801BA (ICH2) SMBus controller,");
            found = 1;
            break;
        case ID_82801CA:
            printf("Intel 82801BA (ICH3) SMBus controller,");
            found = 1;
            break;
        case ID_ICH4:
            printf("Intel ICH4 SMBus controller,");
            found = 1;
            break;
        case ID_ICH5:
            printf("Intel ICH5 SMBus controller,");
            found = 1;
            break;
         case Hance_Rapids:
         	printf("Intel Hance Rapids SMBus controller");
         	found=1;
         	break;
        case ID_CSB5:
            printf("ServerWorks CSB5 SMBus controller,");
            found = 1;
            break;
        case ID_CSB6:
            printf("ServerWorks CSB6 SMBus controller,");
            found = 1;
            break;
        case ID_OSB4:
            printf("ServerWorks OSB4 SMBus controller,");
            found = 1;
            break;
        default:
//            if (devid != 0xFFFFFFFF)
//                printf("devid is %lx\n", devid);
            found = 0;
            break;
        }
        if (found == 1) break;
      }
      if (found == 1) break;
    }
    if (found == 0) {
        fprintf( stderr, "ERROR: SMBus controller isn't found\n");
        return;
    }
    
    devaddr  = 0x80000000 | ((dev<<11)&0xFF00);
    devaddr |= ((func<<8)&0x0700);
    printf(" dev = %lx, ", devaddr);
    // Read SMBus controller base address
    switch (devid) {
    
    case ID_8111:
        /* SMBus 2.0 handling */
        intf->Controller = AMD_SMBC;
        intf->dataAddr = (WORD)(PCIC_IN_32(devaddr, AMD_SMB_BASE_2) & 0xFFFE);
        PCIC_OUT_32(devaddr, AMD_SMB_BASE_2_EN, 1);
        break;
    case ID_81801:
    case ID_81801AA:
    case ID_81801AB:
    case ID_82801BA:
    case ID_82801CA:
    case ID_ICH4:
    case ID_ICH5:
    case Hance_Rapids:
        intf->Controller = INTEL_SMBC;
        intf->dataAddr = (WORD)(PCIC_IN_32(devaddr, ICH_SMB_BASE) & 0xFFFE);
        // Enable SMBus Host Controller, and set SMBus behavior
        func = (PCIC_IN_8(devaddr, ICH_HOSTC) & 0xFFFE);
        func |= ICH_HOSTC_HST_EN;
        func &= ~ICH_HOSTC_I2C_EN;
        PCIC_OUT_8(devaddr, ICH_HOSTC, func);
        break;
	case ID_OSB4:
    case ID_CSB5:
    case ID_CSB6:
        intf->Controller = SW_SMBC;
        intf->dataAddr = (WORD)(PCIC_IN_32(devaddr, 0x90) & 0xFFFE);
        break;
    default:
        fprintf( stderr, "ERROR: Wrong SMBus controller - %lx\n", devid);
        break;
    }
    printf("I/O address = %x\n", intf->dataAddr);
}

static byte SMBBmcAddr( IsaInterface *intf )
{
    return 0x00;
}
int amd_ec_wait_write(IsaInterface * intf)
{
    int timeout = 500;

    while (timeout-- && (inp(intf->dataAddr + AMD_EC_SC) & AMD_EC_SC_IBF))
        usDelay(1);

    if (!timeout) {
        printf("i2c-amd8111.c: Timeout while waiting for IBF to clear\n");
        return -1;
    }

    return 0;
}

int amd_ec_wait_read(IsaInterface * intf)
{
    int timeout = 500;

    while (timeout-- && (~inp(intf->dataAddr + AMD_EC_SC) & AMD_EC_SC_OBF))
        usDelay(1);

    if (!timeout) {
        printf("i2c-amd8111.c: Timeout while waiting for OBF to set\n");
        return -1;
    }

    return 0;
}

int amd_ec_read(IsaInterface * intf, unsigned char address, unsigned char *data)
{
    if (amd_ec_wait_write(intf))
        return -1;
    outp(intf->dataAddr + AMD_EC_CMD,AMD_EC_CMD_RD);

    if (amd_ec_wait_write(intf))
        return -1;
    outp(intf->dataAddr + AMD_EC_DATA,address);

    if (amd_ec_wait_read(intf))
        return -1;
    *data = inp(intf->dataAddr + AMD_EC_DATA);

    return 0;
}

int amd_ec_write(IsaInterface * intf, unsigned char address, unsigned char data)
{
    if (amd_ec_wait_write(intf))
        return -1;
    outp( intf->dataAddr + AMD_EC_CMD, AMD_EC_CMD_WR);

    if (amd_ec_wait_write(intf))
        return -1;
    outp(intf->dataAddr + AMD_EC_DATA,address);

    if (amd_ec_wait_write(intf))
        return -1;
    outp(intf->dataAddr + AMD_EC_DATA,data);

    return 0;
    }


static int SMBSendRequest( IsaInterface * intf, BmcReq_t * req, int length)
{
    unsigned char data, status, num_retries;
    IpmbMsg_t *msgBuf = (IpmbMsg_t *) req;
    int    i, timeout=0;

    // Find SMBus controller and base address
    findSMBcontroller(intf);

    intf->reqTime = clock();
    dprintf( dbgFp, "req start time %lX\n", intf->reqTime );

    length+=2;    
    // Build KCS message that will be sent via SMBus
    memcpy(&msgBuf->rsSa, &msgBuf->rqSa, length-3);
    length-=3;
    memcpy(&msgBuf->cSum1, &msgBuf->cmd, length-4);
    length-=4;
    intf->ReqBMCAddr = *(BYTE *)((BYTE *)req + 0);

    num_retries=NUM_RETRIES;
  // Handle the AMD chipset
  if (intf->Controller == AMD_SMBC){

      /* SMBus 2.0 handling */
          do {
            
            if (amd_ec_write(intf,(byte)AMD_SMB_STS, (byte)0)) break;      
            if (amd_ec_write(intf,(byte)AMD_SMB_ADDR,(byte)intf->ReqBMCAddr & 0xfe)) break;
            if (amd_ec_write(intf,(byte)AMD_SMB_CMD,(byte)0x02)) break;
            if (amd_ec_write(intf,(byte)AMD_SMB_BCNT,(byte)(length-1))) break;
            
            for (i = 1; i < length; i++) {
                amd_ec_write(intf, (byte)(AMD_SMB_DATA+i-1), *(byte*)((byte*)req + i));
            }
            if (amd_ec_write(intf, (byte)AMD_SMB_PRTCL, AMD_SMB_BLOCK_WRITE)) break;
           
            do {
                amd_ec_read(intf, (byte)AMD_SMB_STS, &data);
             }
            while ((timeout++<TIMEOUT_COUNT)&&(~data & AMD_SMB_STS_DONE));

            msDelay(80);
            timeout=0;
            }
            while ((((~data & AMD_SMB_STS_DONE) || (data & AMD_SMB_STS_STATUS)))  && (num_retries-->0));
            if ((~data & AMD_SMB_STS_DONE)|| (data & AMD_SMB_STS_STATUS)){
                fprintf( stderr, "REQUEST ERROR: SMBus error detected, status = %x\n", data);
                return -1;
            }
            
      }
  else {
  // Handle the Intel and ServerWorks chipsets
  do{
    // Clear all status bits, Host Status Register
    switch (intf->Controller) {
        case INTEL_SMBC:
            status = ICH_HST_STA_ALL_ERRS|ICH_HST_STA_INTR|ICH_HST_STA_BYTE_DONE_STS;
            break;
        case SW_SMBC:
            status = ICH_HST_STA_ALL_ERRS|ICH_HST_STA_INTR;
            break;
    }
    outp(intf->dataAddr+ICH_HST_STA, status);

    // Block protocol, Host Control Register
    outp(intf->dataAddr+ICH_HST_CNT, ICH_HST_CNT_SMB_CMD_BLOCK);

    // Slave address, Host Address Register
    outp(intf->dataAddr+ICH_XMIT_SLVA, intf->ReqBMCAddr & 0xfe);
    // IPMI or Device specific command command, Host Command Register
    outp(intf->dataAddr+ICH_HST_CMD, 0x02);
    // Block length, Host DATA0 Register
    outp(intf->dataAddr+ICH_D0, length-1);
    switch (intf->Controller) {
    
        case INTEL_SMBC:
            // the first byte
            outp(intf->dataAddr+ICH_BLOCK_DB, *(BYTE *)((BYTE *)req + 1));
            // Block protocol and Start, Host Control Register
            outp(intf->dataAddr+ICH_HST_CNT, ICH_HST_CNT_START | ICH_HST_CNT_SMB_CMD_BLOCK);

            // Send byte by byte
            for ( i = 2; i < length; i++ ) {
                do {
                    // Read host status register
                    data = inp(intf->dataAddr+ICH_HST_STA);
                    usDelay(1000);        
                // Any error or Interrupt bit or byte done bit set ?
                } while ((timeout++<TIMEOUT_COUNT)&&(data & status) == 0);
                
                /* Check for byte completion in block transfer */
                if ((data & ICH_HST_STA_BYTE_DONE_STS) != ICH_HST_STA_BYTE_DONE_STS) break;

                outp(intf->dataAddr+ICH_BLOCK_DB, *(BYTE *)((BYTE *)req + i));
                // Clear status bits, Host Status Register
                outp(intf->dataAddr+ICH_HST_STA, data & status);
            }
            break;
            
        case SW_SMBC:
            // Block length, Host DATA1 Register
            outp(intf->dataAddr+ICH_D1, length-1);
            // Reset block index
            data = inp(intf->dataAddr+ICH_HST_CNT);
            // Put all bytes of the message to Block Data Register
            for ( i = 1; i < length; i++ )
                outp(intf->dataAddr+ICH_BLOCK_DB, *(BYTE *)((BYTE *)req + i));
            // Block protocol and Start, Host Control Register
            outp(intf->dataAddr+ICH_HST_CNT, ICH_HST_CNT_START | ICH_HST_CNT_SMB_CMD_BLOCK);
            do {
                // Read host status register
                data = inp(intf->dataAddr+ICH_HST_STA);
            // Any error or Interrupt bit set ?
            } while ((timeout++<TIMEOUT_COUNT)&&(data & status) == 0);
            break;
    }
    // Clear status bits, Host Status Register
    outp(intf->dataAddr+ICH_HST_STA, data & status);
    msDelay(50);
    timeout=0;
    
     }
    while (((data & ICH_HST_STA_ALL_ERRS) != 0) && (num_retries-->0));
   
    if ((data & ICH_HST_STA_ALL_ERRS) != 0) {
        fprintf( stderr, "REQUEST ERROR: SMBus error detected, status = %x\n", data);
        return -1;
    }
   }     

    return 0;
}

static int SMBReadResponse( IsaInterface * intf, BmcResp_t * resp)  
{
    unsigned char data, status, num_retries;
    BYTE * msgBuf = (BYTE *) resp;                
    int    i, timeout=0, length;

   num_retries=NUM_RETRIES;
   
   
   if (intf->Controller == AMD_SMBC){
         /* SMBus 2.0 handling */
      do {
        if (amd_ec_write(intf, AMD_SMB_STS, (byte)0)) break;
        if (amd_ec_write(intf, AMD_SMB_ADDR, (byte)intf->ReqBMCAddr)) break;
        if (amd_ec_write(intf, AMD_SMB_CMD, (byte)0x03)) break;
        if (amd_ec_write(intf, AMD_SMB_PRTCL, AMD_SMB_BLOCK_READ)) break;

        do {
            amd_ec_read(intf, AMD_SMB_STS, &data);
         }
        while ((timeout++<TIMEOUT_COUNT)&&(~data & AMD_SMB_STS_DONE));
        
        if (amd_ec_read(intf, AMD_SMB_BCNT, &length)) break;
        if (length > 32) length=32;
        
        for (i = 0; i < length; i++) {
            amd_ec_read(intf, (byte)(AMD_SMB_DATA + i),&msgBuf[i]);
            }
        msDelay(50);
        timeout=0;
            
        }
         while ((~data & AMD_SMB_STS_DONE) && (num_retries-->0));
            intf->reqTime = clock() - intf->reqTime;    // calculate duration                
            if ((~data & AMD_SMB_STS_DONE)||
                data & AMD_SMB_STS_STATUS){
              fprintf( stderr, "RESPONSE ERROR: SMBus error detected, status = %x\n", data);
              return 0;
          }
   }
     // Handle the Intel and ServerWorks chipsets
   else {
   do {
    // Clear all status bits, Host Status Register
    switch (intf->Controller) {
        case INTEL_SMBC:
            status = ICH_HST_STA_ALL_ERRS|ICH_HST_STA_INTR|ICH_HST_STA_BYTE_DONE_STS;
            break;
        case SW_SMBC:
            status = ICH_HST_STA_ALL_ERRS|ICH_HST_STA_INTR;
            break;
    }
    outp(intf->dataAddr+ICH_HST_STA, status);

    // Block protocol, Host Control Register
    outp(intf->dataAddr+ICH_HST_CNT, ICH_HST_CNT_SMB_CMD_BLOCK);

    // Slave address, Host Address Register
    outp(intf->dataAddr+ICH_XMIT_SLVA, intf->ReqBMCAddr|ICH_XMIT_SLVA_READ);
    // IPMI or Device Specific command, Host Command Register
    outp(intf->dataAddr+ICH_HST_CMD, 0x03);
    // Reset block index
    data = inp(intf->dataAddr+ICH_HST_CNT);
    // Block protocol and Start, Host Control Register
    outp(intf->dataAddr+ICH_HST_CNT, ICH_HST_CNT_START | ICH_HST_CNT_SMB_CMD_BLOCK);

    do {
        // Read host status register
        data = inp(intf->dataAddr+ICH_HST_STA);
        msDelay(10);        
    } while ((timeout++<TIMEOUT_COUNT) &&((data & status) == 0));    // Any error or Interrupt bit set ?
    // Clear status bits, Host Status Register
    outp(intf->dataAddr+ICH_HST_STA, data & status);
  
    
   if ((data & ICH_HST_STA_ALL_ERRS) == 0) {

    // Block length, Host DATA0 Register
    if ((length = (int)inp(intf->dataAddr+ICH_D0)) > MAX_ISA_LENGTH)
        length = MAX_ISA_LENGTH;

    switch (intf->Controller) {
    
        case INTEL_SMBC:
            // Read the first byte
            msgBuf[0] = (BYTE)inp(intf->dataAddr+ICH_BLOCK_DB);

            // Put all bytes of the message to Block Data Register
            for ( i = 1; i < length; i++ ) {
                if (i == (length-1)) {
                    // Set Last Byte bit
                    outp(intf->dataAddr+ICH_HST_CNT, inp(intf->dataAddr+ICH_HST_CNT)|ICH_HST_CNT_LAST_BYTE);
                }
                // Clear status bits, Host Status Register
                outp(intf->dataAddr+ICH_HST_STA, data & status);

                do {
                    // Read host status register
                    data = inp(intf->dataAddr+ICH_HST_STA);
                    usDelay(1000);        
                } while ((data & status) == 0);    // Any error or Interrupt bit set ?
                if ((data & ICH_HST_STA_ALL_ERRS) != 0) {
                    break;
                }

                msgBuf[i] = (BYTE)inp(intf->dataAddr+ICH_BLOCK_DB);
            }
            break;
            
        case SW_SMBC:
            // Put all bytes of the message to Block Data Register
            for ( i = 0; i < length; i++ ) {
                msgBuf[i] = (BYTE)inp(intf->dataAddr+ICH_BLOCK_DB);
            }
            break;
        }
    }
    // Clear status bits, Host Status Register
    outp(intf->dataAddr+ICH_HST_STA, data & status);

    msDelay(80);        
    timeout=0;
    
    }
   while (((data & ICH_HST_STA_ALL_ERRS) != 0) && (num_retries-- >0));
    intf->reqTime = clock() - intf->reqTime;    // calculate duration
  if ((data & ICH_HST_STA_ALL_ERRS) != 0) {
        fprintf( stderr, "RESPONSE ERROR: SMBus error detected, status = %x, retries = %x\n", data, num_retries);
        return 0;
    }
}
    dprintf( dbgFp, "req end time %lX\n", clock() );

    return length;
}


typedef struct
{
	char *			ifName;
	WORD			dataAddr;
	InterfaceApi *	interface;
	
} IsaIfInfo;


//
// Interface structure for the ISA SMS interface
//
static InterfaceApi IsaSmsInterface = {
	IsaIntfConfigure,
	IsaSmsCapabilities,
	IsaBmcAddr,
	IsaSmsSetTimeout,
	IsaSmsSendRequest,
	IsaSmsReadResponse,
	IsaSmsReady,
	IsaLastReqTime,
	"isa_sms"
};
//
// Interface structure for the ISA SMM interface
//
static InterfaceApi IsaSmmInterface = {
    IsaIntfConfigure,
    IsaSmmCapabilities,
    IsaBmcAddr,
    IsaSmmSetTimeout,
    IsaSmmSendRequest,
    IsaSmmReadResponse,
    NULL,
    IsaLastReqTime,
    "isa_smm"
};

//
// Interface structure for the SMBus interface (431 mBMC)
//
static InterfaceApi SMBInterface = {
    SMBIntfConfigure,
    IsaSmmCapabilities,
    SMBBmcAddr,
    IsaSmmSetTimeout,
    SMBSendRequest,
    SMBReadResponse,
    NULL,
    IsaLastReqTime,
    "smb"
};

static IsaIfInfo IsaInterfaces[] =
    {
    { "smb",     0x000,	& SMBInterface     },
    { "isa",	 0xCA2,	& IsaSmsInterface  }, 	// alias for isa_sms
    { "isa_sms", 0xCA2,	& IsaSmsInterface  },
    { "isa_smm", 0xCA4,	& IsaSmmInterface  },
    { NULL,		0,		NULL }
};


InterfaceHandle IsaIntfPresent( char * str )
    {
    UINT	i;
    UINT	dataAddr = 0;
    char *	addrStr;
    IsaInterface *intf;
    //
    // see if this is a generic search
    //	
    if ( str == NULL )
    {
        //
        // only search for the standard SMS if.
        //
        i = 0;
        dprintf( stderr, "IsaIntfPresent: NULL name, will test for first if\n");
    }
    else
    {
        //
        // look for an address component to the name
        //
        dprintf( stderr, "IsaIntfPresent: looking for name, \"%s\"\n", str );

        if ( addrStr = strchr( str, ':' ) )
        {
            *addrStr++ = '\0';
            (void) sscanf( addrStr, "%X", & dataAddr );
        }
        //
        // look for the interface name
        //
        for( i = 0 ; IsaInterfaces[i].ifName ; i++ )
        {
            if( strcmp( str, IsaInterfaces[i].ifName) == 0 )
                break;
        }
        //
        // check for failure to find the name
        //
        if( IsaInterfaces[i].ifName == NULL )
        {
            dprintf( stderr, "IsaIntfPresent: can't find name\n");
            return NULL;
        }
        dprintf( stderr, "IsaIntfPresent: found name\n", str );

    }
    if( dataAddr == 0 )
        dataAddr = IsaInterfaces[i].dataAddr;
    //
    // see if the interface is there
    //
    //	if( ISA_STATUS(dataAddr) == 0xFF && ISA_READ_BMC_DATA(dataAddr) == 0xFF )
    //	{
    //		dprintf( stderr, "IsaIntfPresent: can't find intf at %xh\n", dataAddr );
    //		return NULL;
    //	}
    dprintf( stderr, "IsaIntfPresent: found intf at %xh\n", dataAddr );
    //
    // create an interface structure to pass back
    //
    intf = (IsaInterface *) malloc( sizeof(IsaInterface) );

    intf->interface = * IsaInterfaces[i].interface;
    intf->dataAddr	= dataAddr;

    dprintf( stderr, "IsaIntfPresent: intf struct at 0x%x\n", (DWORD) intf );

    return (InterfaceHandle) intf;
}
