/******************************************************************/
/*                                                                */
/* Module:       jb_io.c                                          */
/*                                                                */
/* Descriptions: Manages I/O related routines, file and string    */
/*               processing functions.                            */
/*                                                                */
/* Revisions:    1.0 02/22/02                                     */
/*                                                                */
/******************************************************************/

#include <stdio.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include <lara.h>
#include <pp/base.h>

#include "jb_io.h"

#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifndef LINUX_PARPORT
// #include <sys/io.h>
#else
 #include <linux/ppdev.h>
 #include <sys/ioctl.h>
 #include <fcntl.h>
 #include <unistd.h>
#endif

/* global definitions */

#define PORT_IO_BUFFER_SIZE		256
#define MAX_FILE_LINE_LENGTH            160


/* global Variables */

#ifndef LINUX_PARPORT
unsigned short	 lpt_addr;
#else
static int	 ppfp			= 0;
char 	devicepath[256];
#endif
int		 port_io_buffer_count	= 0;
int		 port_io_buffer[PORT_IO_BUFFER_SIZE];
int		 totalrd		= 0;
int		 totalwr		= 0;
int		 gpio_fd		= -1;

void InitLinuxDriver()
{
#ifndef LINUX_PARPORT

    
#ifdef PPC405
    /* opening the gpio directly here to avoid
       the indirection through the base lib */
    if ((gpio_fd = open(PP_GPIO_FPGA_DEVNAME, O_RDWR)) < 0) {
	D( D_ERROR, "Error: cannot open GPIO device.\n" );
	exit(1);
    }

    set_tck(0);
    set_tms(1);
    set_tdi(1);

#else
                D(D_BLABLA, "Attempting to gain IO permissions on 0x%02x to 0x%02x\n",lpt_addr,lpt_addr+3); /* DEBUG */
                if (ioperm(lpt_addr, 3, 1)) {
                  perror("initialize_jtag_hardware failed");
                  exit(errno);
                }
#endif         
#else

	int ret = 0;
	ppfp = open (devicepath, O_RDWR);
	if (!ppfp) {
		D( D_ERROR, "Error: Device open failed. exiting.\n" );
		exit(1);
	}
	ret = ioctl (ppfp, PPCLAIM);
	D (D_ERROR, "Note: ioctl claim returned stat: %i (err: %s)\n", ret, strerror (errno));
	if (ret != 0) {
		D( D_ERROR, "Error: Driver initialization failed. check permissions. exiting.\n" );
		close (ppfp);
		exit(1);
	}
#ifdef PARPORT_EXCLUSIVE
  	ret = ioctl (ppfp, PPEXCL);
	D (D_ERROR, "Note: ioctl excl returned stat: %i (err: %s)\n", ret, strerror (errno));
	if (ret != 0) {
		D (D_ERROR, "Warning: Device exclusive access failed.\n");
	}
#endif /* EXCLUSIVE */
#endif /* PARPORT*/
}
void CloseLinuxDriver()
{
#ifndef LINUX_PARPORT
#ifndef PPC405
	ioperm(lpt_addr, 3, 0);
	close(gpio_fd);
#endif
#else
	if (ppfp) close(ppfp);
	D(D_NOTICE,"Total bytes transfered [r:%i|w:%i]\n", totalrd,totalwr);
#endif /* PARPORT*/
}

void flush_ports(void)
{	
#ifndef LINUX_PARPORT
	;;
#else
	int i,ret=0;
//	ret=write(ppfp,&port_io_buffer,port_io_buffer_count*sizeof(int));
	for (i=0; ret == 0 && i < port_io_buffer_count ; i++) {
		ret = ioctl (ppfp, PPWDATA, &(port_io_buffer[i]));
		totalwr++;
		D(D_NOTICE, "%i - %i\n",ret,port_io_buffer[i]);
	}

	if ( ret<0) 
	{
		D(D_ERROR, "I/O Error:  Cannot flush buffers to ByteBlaster hardware (%i,%i)\n",ret,errno);
		CloseLinuxDriver();
		exit(1);
	}
	port_io_buffer_count = 0;
#endif /* PARPORT*/
}
int VerifyHardware()
{
	int verify_ok = 0;
	int test_count = 0;
	int read_data = 0;
	
#ifdef PPC405
	return verify_ok;
#endif

	for ( test_count = 0; test_count < 2; test_count++ )
	{
		/* Write '0' to Pin 7 and Pin 9 (Data5,7) for the first test and '1' for the second test */
		int vector = (test_count) ? 0xA0 : 0x0;/* 1010 0000:0000 0000... drive to Port0 */
		int expect = (test_count) ? 0x60 : 0x0;/* X11X XXXX:X00X XXXX... expect from Port1 */

		WritePort( 0, vector, 1 );
		
		/* Expect '0' at Pin 10 and Pin 12 (Ack and Paper End) for the first test and '1' for the second test */
		read_data = ReadPort( 1 ) & 0x60;

		verify_ok = verify_ok && ( read_data == expect ? 1:0 );
	}

	if ( !verify_ok )
		D( D_NOTICE, "Info: Verifying hardware: Hardware found...\n" );
	else
		D( D_ERROR, "Error: Verifying hardware: Hardware not found or not installed properly...\n" );

	return verify_ok;
}

/******************************************************************/
/* Name:         ReadPort                                         */
/*                                                                */
/* Parameters:   port                                             */
/*               -the index of port from the parallel port base   */
/*                address.                                        */
/*                                                                */
/* Return Value: Value of the port.                               */
/*               		                                          */
/* Descriptions: Read the value of the port registers.            */
/*                                                                */
/******************************************************************/
int ReadPort(int port)
{
	int data = 0;

#ifndef LINUX_PARPORT
          D(D_VERBOSE, "Attempting to read port 0x%03x\n",port+lpt_addr); /*DEBUG*/
          data = inb_ppc405(port + lpt_addr);
          D(D_VERBOSE, "Read data == 0x%02x\n",data); /*DEBUG*/
#else
	int ret = -1;
	if      (port == 0) ret = ioctl (ppfp, PPRDATA, &data);
	else if (port == 1) ret = ioctl (ppfp, PPRSTATUS, &data);
	else if (port == 2) ret = ioctl (ppfp, PPRCONTROL, &data);
	D(D_VERBOSE,"Debug: Device read access [p:%i|d:%i] (err: %i).\n",port,data,ret);
	if (ret != 0) {
		D(D_ERROR,"Warning: Device read access failed (err: %i).\n",ret);
	} else totalrd++;
#endif /* PARPORT */

	return (data & 0xff);
}

/******************************************************************/
/* Name:         WritePort                                        */
/*                                                                */
/* Parameters:   port,data,test                                   */
/*               -port is the index from the parallel port base   */
/*                address.                                        */
/*               -data is the value to dump to the port.          */
/*               -purpose of write.                               */
/*                                                                */
/* Return Value: None.                                            */
/*               		                                          */
/* Descriptions: Write "data" to "port" registers. When dump to   */
/*               port 0,if "test"=0,processes in "port_io_buffer" */
/*               are flushed when "PORT_IO_BUFFER_SIZE" is        */
/*               reached. if "test"=1,"data" is dumped to port 0  */
/*               at once.                                         */
/*                                                                */
/******************************************************************/
void WritePort(int port,int data,int test UNUSED)
{

#ifndef LINUX_PARPORT
          D(D_VERBOSE, "writing 0x%02x to address 0x%03x\n", data, port + lpt_addr); /* DEBUG */
        outb_ppc405(data, port + lpt_addr);
#else
	int ret = -1;
/*
	if (port == 0 && test == 0)
	{
		port_io_buffer[port_io_buffer_count] = data;
		++port_io_buffer_count;

		if (port_io_buffer_count >= PORT_IO_BUFFER_SIZE) flush_ports();
		ret=0;
	}
*/
	if (port == 0) ret = ioctl (ppfp, PPWDATA, &data);
	else if (port == 2) ret = ioctl (ppfp, PPWCONTROL, &data);
	D(D_BLABLA,"Debug: Device write access [p:%i|d:%i|t:%i] (err: %i).\n",port,data,test,ret);
	if (ret != 0) {
		D(D_ERROR,"Warning: Device write access failed (err: %i).\n",ret);
	} else totalwr++;
#endif /* PARPORT */
}


/***************************************************************************
*
*   bitwise functions
*
***************************************************************************/

void set_tck(unsigned char val) {
    gpio_ioctl_set_bit(gpio_fd, PP_GPIO_FPGA_TCK, val);
}

void set_tms(unsigned char val) {
    gpio_ioctl_set_bit(gpio_fd, PP_GPIO_FPGA_TMS, val);
}

void set_tdi(unsigned char val) {
    gpio_ioctl_set_bit(gpio_fd, PP_GPIO_FPGA_TDI, val);
}

void set_tdi_byte(unsigned char val) {
    if (ioctl(eric_fd, ERICIOCFPGAWRITEBYTE, &val) == -1) {
	/* should never happen -> exit */
	pp_log_err(" ioctl(ERICIOCFPGAWRITEBYTE) failed");
	exit(1);
    }
}

unsigned int get_tdo(void) {
    //return pp_gpio_bit_get(PP_GPIO_FPGA_DEV, PP_GPIO_FPGA_TDO);
    return gpio_ioctl_get_bit(gpio_fd, PP_GPIO_FPGA_TDO);
}

/***************************************************************************
*
*   macros for bitwise work
*
***************************************************************************/

#define SET_TCK  set_tck(1)
#define CLR_TCK  set_tck(0)

#define SET_TDI  set_tdi(1)
#define CLR_TDI  set_tdi(0)

#define SET_TMS  set_tms(1)
#define CLR_TMS  set_tms(0)


int   inb_ppc405(int port)
{
	int local_port, ret_val=0;
	
	local_port = port & 0x07;
	
	switch (local_port)
	{
		case 0:	
			break;
		case 1:	
			
			if (get_tdo() == 1) ret_val = 0;
			else ret_val = 1;
			ret_val = get_tdo();

			if (ret_val == 1) ret_val = 0;
			else ret_val = 1;

			D(D_BLABLA, "TDO %01i \n", ret_val ? 1:0);
			ret_val = ret_val << 7; /* shift this value to the right bit position */
			break;
		case 2:	
			D(D_NOTICE,"INB Port Offset %i Data %02x\n",local_port, ret_val);
			break;
		case 3:	
			D(D_NOTICE,"INB Port Offset %i Data %02x\n",local_port, ret_val);
			break;
		default:
			break;

	}
	
	return ret_val;	
}


void  outb_ppc405(int data, int port)
{
	int local_port, out_val=0;
	
	local_port = port & 0x07;
	
	switch (local_port)
	{
		case 0:	
			out_val = data & 0x01;
			set_tck(out_val);
			D(D_ALWAYS, "TCK %01i ", out_val ? 1:0);
			out_val = data >> 1;
			out_val = out_val & 0x01;
			D(D_ALWAYS, "TMS %01i ", out_val ? 1:0);
			set_tms(out_val);
			
			out_val = data >> 6;
			out_val = out_val & 0x01;
			D(D_ALWAYS, "TDI %01i\n", out_val ? 1:0);
			set_tdi(out_val);
			
			break;
		case 1:	
			D(D_ALWAYS,"OUTB Port Offset %i Data %02x\n",local_port, out_val);
			break;
		case 2:	
			D(D_ALWAYS,"OUTB Port Offset %i Data %02x\n",local_port, out_val);
			break;
		case 3:	
			D(D_ALWAYS,"OUTB Port Offset %i Data %02x\n",local_port, out_val);
			break;
		default:
			break;

	}
	
}






/*****************************/
/*                           */
/* File processing functions */
/*                           */
/*****************************/

int jb_fopen(const char* argv,const char* mode)
{
	FILE* file_id;

	file_id = fopen( argv, mode );

	return (int) file_id;
}

int	jb_fclose(int file_id)
{
	fclose( (FILE*) file_id);

	return 0;
}

int jb_fseek(int finputid,int start,int end)
{
	int seek_position;

	seek_position = fseek( (FILE*) finputid, start, end );

	return seek_position;
}

int jb_ftell(int finputid)
{
	int file_size;

	file_size = ftell( (FILE*) finputid );

	return file_size;
}

int jb_fgetc(int finputid)
{
	int one_byte;

	one_byte = fgetc( (FILE*) finputid );

	return one_byte;
}

char* jb_fgets(char* buffer, int finputid)
{
	char* test;
	test=fgets(buffer,MAX_FILE_LINE_LENGTH,(FILE*) finputid);

	return test;
}

/*******************************/
/*                             */
/* String processing functions */
/*                             */
/*******************************/

/******************************************************************/
/* Name:         jb_grabdata                                      */
/*                                                                */
/* Parameters:   buffer, start_byte, term, str                    */
/*               -buffer is the line buffer stored when reading   */
/*                the CDF.                                        */
/*               -start_byte is the byte to start with on the     */
/*                buffer parsed from CDF                          */
/*               -term is how many terms to look into from the    */
/*                start_byte parsed from CDF			  */
/*		 -str is the string from start_byte till term in  */
/*		  the buffer parsed from CDF			  */
/*                                                                */
/* Return Value: mark - "0" if all spaces in the buffer, else "1" */
/*               		                                  */
/******************************************************************/
/* [sbng,4/12/02,jb_io.c ver1.1] New function added */
int jb_grabdata(char* buffer,int start_byte,int term,char* str)
{
	unsigned i=0,j=0;
	int mark=0;
	int space=0;

	if(start_byte<0 || start_byte>=(int)(strlen(buffer)-1))
	{		
		str[0]='\0';
		return (-1);
	}

	for(i=start_byte;i<strlen(buffer);i++)
	{
		if(mark==0)
		{
			if( buffer[i]!=' ' && buffer[i]!='\t' && buffer[i]!='\n' )
			{
				if(space==term-1)
					str[j++]=buffer[i];
				mark=1;
			}
			else if( buffer[i]==' ' || buffer[i]=='\t' )
				mark=0;
			else
			{
				if(!((buffer[i]==' ' || buffer[i]=='\t') && (buffer[(i>0)? i-1:0]==' ' || buffer[i]=='\t' )))
					space++;
				if(space>term-1)
					break;
			}
		}
		else if(mark==1)
		{
			if( buffer[i]!=' ' && buffer[i]!='\t' && buffer[i]!='\n' )
			{
				if(space==term-1)
					str[j++]=buffer[i];
			}
			else
			{
				if(!((buffer[i]==' ' || buffer[i]=='\t' ) && (buffer[(i>0)? i-1:0]==' ' || buffer[i]=='\t' )))
					space++;
				if(space>term-1)
					break;
			}
		}
	}

	str[j]='\0';

	return mark;
}

void jb_strcpy(char* a,const char* b)
{
	strcpy(a,b);
}

int jb_str_cmp(const char* charset,const char* buffer)
{
	char* pc;
	char  temp[MAX_FILE_LINE_LENGTH+1],store;
	unsigned int i;

	for(i=0;i<strlen(buffer);i++)
	{
		strcpy(temp,buffer);
		pc = &temp[i];

		if(*pc==*charset)
		{
			store = temp[i+strlen(charset)];
			temp[i+strlen(charset)]='\0';

			if(!strncmp(pc,charset,strlen(charset)))
			{
				temp[i+strlen(charset)]=store;
				return i+1;
			}
			
			temp[i+strlen(charset)]=store;
		}

		pc++;		
	}

	return 0;
}

int jb_strlen(const char* str)
{
	return strlen(str);
}

int jb_strcmp(const char* a,const char* b)
{
	return strcasecmp(a,b);
}

void jb_strcat(char* dst,const char* src)
{
	strcat(dst,src);
}

int jb_atoi(const char* number)
{
	return atoi(number);
}

void jb_toupper(char* str)
{
	char* pstr;

	pstr = str;

	while(*pstr)
	{
		if(*pstr>='a' && *pstr<='z')
			*pstr -= 0x20;

		pstr++;
	}
}
