/**
 * esb2_tpt.h
 *
 * KIRA100 FML (slave) driver for Intel's ESB2 TCP Pass-Trough
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#ifndef __ESB2_TPT_H__
#define __ESB2_TPT_H__

#define ESB2_TPT_STAT_CLOSED            0x0
#define ESB2_TPT_STAT_LISTENING         0x1
#define ESB2_TPT_STAT_CONNECTED         0x2
#define ESB2_TPT_STAT_RECV_TIMEOUT      0x3
#define ESB2_TPT_STAT_TRANSM_TIMEOUT    0x4
#define ESB2_TPT_STAT_TERMINATED        0x5
#define ESB2_TPT_STAT_STANDBY           0x6
#define ESB2_TPT_STAT_LOST_PACKET       0x7
#define ESB2_TPT_STAT_DISABLED          0x8
#define ESB2_TPT_STAT_AUTH_FAILED       0x9
#define ESB2_TPT_STAT_INTEGRITY_FAILED  0xa
#define ESB2_TPT_STAT_DECRYPT_FAILED    0xb
#define ESB2_TPT_STAT_NO_LINK           0xc

#define ESB2_TPT_IS_CONNECTED(s) ((s) == ESB2_TPT_STAT_CONNECTED)
#define ESB2_TPT_IS_LISTENING(s) \
    ((s) == ESB2_TPT_STAT_LISTENING || \
     (s) == ESB2_TPT_STAT_RECV_TIMEOUT || \
     (s) == ESB2_TPT_STAT_TRANSM_TIMEOUT || \
     (s) == ESB2_TPT_STAT_TERMINATED || \
     (s) == ESB2_TPT_STAT_AUTH_FAILED || \
     (s) == ESB2_TPT_STAT_INTEGRITY_FAILED || \
     (s) == ESB2_TPT_STAT_DECRYPT_FAILED)
#define ESB2_TPT_IS_CLOSED(s) (!ESB2_TPT_IS_CONNECTED((s)) && !ESB2_TPT_IS_LISTENING((s)))

typedef struct {
    unsigned char chan;
    unsigned char status;
    unsigned char remote_ip[4];
} esb2_tpt_ioctl_tpt_stat_t;

typedef struct {
    unsigned char chan;
    unsigned char timeout; // secs
} esb2_tpt_ioctl_recv_timeout_t;

typedef struct {
    int reg;
    unsigned long val;
} esb2_tpt_ioctl_reg_t;

typedef struct {
    /* irq counting */
    int intr_cnt;
    int fml_intr_cnt;
    int slave_intr_cnt;
    int bytes_sent;
    int bytes_received;
} esb2_tpt_ioctl_hk_t;

typedef struct {
} esb2_tpt_ioctl_connlost_t;

#define ESB2_TPT_IOCTL_MAGIC 'F'
#define ESB2_TPT_IOCTL_GET_TPT_STAT     _IOWR(ESB2_TPT_IOCTL_MAGIC, 1,  esb2_tpt_ioctl_tpt_stat_t)
#define ESB2_TPT_IOCTL_SET_RECV_TIMEOUT _IOW (ESB2_TPT_IOCTL_MAGIC, 2,  esb2_tpt_ioctl_recv_timeout_t)
#define ESB2_TPT_IOCTL_GET_REG          _IOWR(ESB2_TPT_IOCTL_MAGIC, 90, esb2_tpt_ioctl_reg_t)
#define ESB2_TPT_IOCTL_SET_REG          _IOW (ESB2_TPT_IOCTL_MAGIC, 91, esb2_tpt_ioctl_reg_t)
#define ESB2_TPT_IOCTL_GET_HK           _IOR (ESB2_TPT_IOCTL_MAGIC, 99, esb2_tpt_ioctl_hk_t)
#define ESB2_TPT_IOCTL_SET_CONNLOST     _IOW (ESB2_TPT_IOCTL_MAGIC, 92, esb2_tpt_ioctl_connlost_t)

/*
 * Helper function to cover IOCTL (only in user mode)
 */
#ifndef __KERNEL__
#include <sys/ioctl.h>

static __inline__ int esb2_tpt_ioctl_get_tpt_stat(int fd, esb2_tpt_ioctl_tpt_stat_t* status)
{
    return ioctl(fd, ESB2_TPT_IOCTL_GET_TPT_STAT, status);
}

static __inline__ int esb2_tpt_ioctl_set_recv_timeout(int fd, esb2_tpt_ioctl_recv_timeout_t* timeout)
{
    return ioctl(fd, ESB2_TPT_IOCTL_SET_RECV_TIMEOUT, timeout);
}

static __inline__ int esb2_tpt_ioctl_set_reg(int fd, int reg, unsigned char val)
{
    esb2_tpt_ioctl_reg_t ioc = { .reg = reg, .val = val, };
    return ioctl(fd, ESB2_TPT_IOCTL_SET_REG, &ioc);
}

static __inline__ int esb2_tpt_ioctl_get_reg(int fd, int reg)
{
    esb2_tpt_ioctl_reg_t ioc = { .reg = reg, };
    int r = ioctl(fd, ESB2_TPT_IOCTL_GET_REG, &ioc);
    return r < 0 ? r : (int)ioc.val;
}

static __inline__ int esb2_tpt_ioctl_get_hk(int fd, esb2_tpt_ioctl_hk_t* hk)
{
    return  ioctl(fd, ESB2_TPT_IOCTL_GET_HK, hk);
}

static __inline__ int esb2_tpt_ioctl_connlost(int fd)
{
    return ioctl(fd, ESB2_TPT_IOCTL_SET_CONNLOST);
}

#endif /* !__KERNEL__ */

#endif /* !__ESB2_TPT_H__ */
