/**
 * esb2_tpt_mod.c
 *
 * KIRA100 FML (slave) driver for Intel's ESB2 TCP Pass-Trough
 *
 * (c) 2005 Peppercon AG, 2005/07/05, Ralf Guenther <rgue@peppecon.de>
 */

#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/netdevice.h>
#include <linux/ioctl.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/poll.h>
#include <linux/pci.h>
#include <linux/wait.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/version.h>
#include <linux/module.h>
#include <asm/semaphore.h>
#include <asm/io.h>

#ifdef __arm__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <asm/arch/cpe/peppercon.h>
#define cond_resched() if (current->need_resched) schedule()
#else
#include <asm/arch/platform/peppercon.h>
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif
#endif

#include "esb2_tpt.h"
#include "fml_esb2.h"
#include "fml_test.h"
#include "fmlcore.h"

#define SUCCESS 0

#define FML_IO_SIZE 16384
#define RECV_BUF_SIZE 8192
#define NAME                    "esb2_tpt"
#define LOG_COLORED
#define noLOG_BLACK_BG
#define noLOG_DBG
#include "log.h"

#define ESB2_TPT_MAJOR ((uint8_t)249)

// PCI dev and INTMUX not used by KIRA
#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI) || defined (LARA_KIMAMD) || defined (LARA_KIMINTEL)
#  define LPC_USE_PCI
#  define LPC_USE_INTMUX
#endif

#ifdef LPC_USE_PCI
#  define KIRA100_PCI_VENDOR_ID  0x1743
#  define KIRA100_PCI_DEVICE_ID  0x1003
#  define KIRA100_PCI_BAR_NO     1
#endif

/* KIRA100 IO addresses/bits */

uint32_t kira100_io_base = 0;
//#define FML_IO_SIZE             (20 * sizeof(uint32_t)) /* regs are dword alligned */

#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI) || defined (LARA_KIMAMD) || defined (LARA_KIMINTEL)
#  define FML_IO_BASE_OFFS      0x9b0000
#  define DEF_IRQ               27
#endif

#ifdef PP_BOARD_KIRA
#  define FML_IO_BASE_OFFS      CPE_FML_BASE
#  define DEF_IRQ               29
#endif

#ifdef LPC_USE_INTMUX
#  define INTMUX_IO_BASE_OFFS   0x9a0000
#  define INTMUX_IO_SIZE        (1 * sizeof(uint32_t)) /* 1 dword */
#  define FML_INT               ((uint16_t)(1 << 14)) // interrupt flag bits (INTMUX register)
#endif

#define FML_SLAVE_ADDR          0x55
#define TPT_CHAN                1

/* ------------------------------------------------------------------------- *
 * private data
 * ------------------------------------------------------------------------- */

typedef struct {
    atomic_t    refcnt;
    void        *fml_addr;
    void        *intmux_addr;
    int         isr_registered;
    int         conn_lost;

    unsigned char recv_timeout; /* receive timeout in secs set by IOCTL */

    struct semaphore    sem;        /* serialize ESB2 write ops */
    wait_queue_head_t   wait;       /* receivers wait queue */
    unsigned char       recv_buf[RECV_BUF_SIZE]; /* receive data buffer */
 
    uint8_t send_buf[SBUF_SIZE];
    unsigned int send_fill;
    wait_queue_head_t  send_wait;
    unsigned int data_to_send;
    unsigned int send_thread_running;

    int                 recv_size;
    spinlock_t          recv_lock;

    struct fmlcore_private  fml;
    int                     fml_init;

    esb2_tpt_ioctl_hk_t hk;
} esb2_tpt_dev_t;

static esb2_tpt_dev_t esb2_tpt_dev;

/* ------------------------------------------------------------------------- *
 * device driver prototypes
 * ------------------------------------------------------------------------- */

static int  esb2_tpt_mod_init(void);
static void esb2_tpt_mod_cleanup(void);
static int  esb2_tpt_mod_ioctl(struct inode *, struct file *, uint, ulong);
static int  esb2_tpt_mod_open(struct inode *, struct file *);
static int  esb2_tpt_mod_release(struct inode *, struct file *);
static int  esb2_tpt_mod_read(struct file *, char *, size_t, loff_t *);
static int  esb2_tpt_mod_write(struct file *, const char *, size_t, loff_t *);
static unsigned int esb2_tpt_mod_poll(struct file *, poll_table *);

static irqreturn_t esb2_tpt_mod_isr(int irq, void *dev_id, struct pt_regs *regs);
static int recv_cb(struct fmlcore_private *fml);
static int send_collector_thread(void * arg);

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Ralf Guenther <rgue@peppercon.de>");
MODULE_DESCRIPTION("KIRA100 FML (slave) driver for Intel's ESB2 TCP Pass-Trough");

module_init(esb2_tpt_mod_init);
module_exit(esb2_tpt_mod_cleanup);

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations esb2_tpt_mod_ops;

/* ------------------------------------------------------------------------- *
 * driver initialization
 * ------------------------------------------------------------------------- */

int __init
esb2_tpt_mod_init(void)
{
    int ret = SUCCESS, r;
    uint32_t fml_io_base;
#ifdef LPC_USE_INTMUX
    uint32_t intmux_io_base;
#endif

    /* init jump table */
    memset(&esb2_tpt_mod_ops, 0, sizeof(esb2_tpt_mod_ops));
    esb2_tpt_mod_ops.ioctl   = esb2_tpt_mod_ioctl;
    esb2_tpt_mod_ops.open    = esb2_tpt_mod_open;
    esb2_tpt_mod_ops.release = esb2_tpt_mod_release;
    esb2_tpt_mod_ops.read    = esb2_tpt_mod_read;
    esb2_tpt_mod_ops.write   = esb2_tpt_mod_write;
    esb2_tpt_mod_ops.poll    = esb2_tpt_mod_poll;
    SET_MODULE_OWNER(&esb2_tpt_mod_ops);

    /* init dev's private data */
    memset(&esb2_tpt_dev, 0, sizeof(esb2_tpt_dev));
    init_waitqueue_head(&esb2_tpt_dev.wait);
    atomic_set(&esb2_tpt_dev.refcnt, 0);
    spin_lock_init(&esb2_tpt_dev.recv_lock);
    sema_init(&esb2_tpt_dev.sem, 1);
    esb2_tpt_dev.recv_timeout = 60; /* 1 min by default */

    if (fml_test_slave_init(&esb2_tpt_dev.fml)) {
        ERR("failed to fml test slave part.");
        ret = -ENODEV;
        goto fail;
    }

    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(ESB2_TPT_MAJOR, NAME, &esb2_tpt_mod_ops)) < 0) {
        ERR("failed to register device (err %d)", r);
        ret = -ENODEV;
        goto fail;
    }

#ifdef LPC_USE_PCI
    {
        /* find hardware */
        struct pci_dev *pci = pci_find_device(KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID, NULL);
        if (pci == NULL) {
            ERR("failed to find PCI device (%04x:%04x) containing FML core",
                KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID);
            ret = -ENODEV;
            goto fail;
        }
        kira100_io_base = pci_resource_start(pci, KIRA100_PCI_BAR_NO);
    }
#endif

    fml_io_base = kira100_io_base + FML_IO_BASE_OFFS;
    if (!(esb2_tpt_dev.fml_addr = ioremap_nocache(fml_io_base, FML_IO_SIZE))) {
        ERR("failed to map FML core IO space at %08x-%08x to memory",
            fml_io_base, fml_io_base + FML_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }

#ifdef __arm__
    if (KIRA_MAJOR(pp_kira_get_revision()) < 2) {
        INFO("FML slave support is not available in KIRA100 R01.");
        ret = -ENODEV;
        goto fail;
    }
#endif

#ifdef LPC_USE_INTMUX
    intmux_io_base = kira100_io_base + INTMUX_IO_BASE_OFFS;
    if (!(esb2_tpt_dev.intmux_addr = ioremap_nocache(intmux_io_base, INTMUX_IO_SIZE))) {
        ERR("failed to map INTMUX IO space at %08x-%08x to memory",
            intmux_io_base, intmux_io_base + INTMUX_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }
#endif

#ifdef LPC_USE_INTMUX
    INFO("successfully loaded. (io: %08x, intmux: %08x, irq: %d)",
        fml_io_base, intmux_io_base, DEF_IRQ);
#else // !LPC_USE_INTMUX
    INFO("successfully loaded. (io: %08x, irq: %d)",
        fml_io_base, DEF_IRQ);
#endif

    return ret;
fail:
    esb2_tpt_mod_cleanup();
    return ret;
}

static void
esb2_tpt_mod_cleanup(void)
{
    int r;
    if ((r = unregister_chrdev(ESB2_TPT_MAJOR, NAME)) < 0) {
        ERR("failed to unregister device (err %d)", r);
    }
#ifdef LPC_USE_INTMUX
    if (esb2_tpt_dev.intmux_addr) {
        iounmap(esb2_tpt_dev.intmux_addr);
        esb2_tpt_dev.intmux_addr = NULL;
    }
#endif
    if (esb2_tpt_dev.fml_addr) {
        iounmap(esb2_tpt_dev.fml_addr);
        esb2_tpt_dev.fml_addr = NULL;
    }
    
    fml_test_slave_cleanup();
}


/* ------------------------------------------------------------------------- *
 * the driver operations
 * ------------------------------------------------------------------------- */

static int
esb2_tpt_mod_open(struct inode *inode, struct file *file)
{
    int ret = SUCCESS;
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;
    MOD_INC_USE_COUNT;

    if (atomic_inc_return(&dev->refcnt) == 1) {
        /* first time init */
        int r;
        uint32_t fml_io_base = kira100_io_base + FML_IO_BASE_OFFS;

        DBG("load");

        if ((r = check_mem_region(fml_io_base, FML_IO_SIZE))) {
            ERR("IO space %08x-%08x already in use (err %d)",
                fml_io_base, fml_io_base + FML_IO_SIZE - 1, r);
            ret = -EBUSY;
            goto fail;
        }
        request_mem_region(fml_io_base, FML_IO_SIZE, "fml");

        /* fill fml struct */
        dev->fml.name     = NAME;
        dev->fml.io_base  = (u32)dev->fml_addr;
        dev->fml.fml_clk  = 8000; /* 8Mhz */
#ifdef PP_BOARD_KIRA
        dev->fml.sys_clk  = 100000; /* 100 Mhz */
#else 
	dev->fml.sys_clk  = 66666; /* 66 MHz */
#endif
        dev->fml.irq      = DEF_IRQ;
        dev->fml.slave    = 1;
        dev->fml.saddr    = FML_SLAVE_ADDR << 1;
        dev->fml.slave_req_callback  = recv_cb;

        if (fmlcore_init(&dev->fml) < 0) {
            ret = -ENODEV;
            goto fail;
        }
        dev->fml_init = 1;

        if ((r = request_irq(DEF_IRQ, esb2_tpt_mod_isr, SA_SHIRQ, NAME, dev)) < 0) {
            ERR("failed to request irq %d (err %d)", DEF_IRQ, r);
            ret = -EBUSY;
            goto fail;
        }
        dev->isr_registered = 1;

        //start the write collector thread
        dev->send_fill = 0;
        dev->send_thread_running = 1;
        init_waitqueue_head(&dev->send_wait);
        if ((r = kernel_thread(send_collector_thread, NULL, 0)) < 0) {
            ERR("failed to initialize send collector thread (err %d)", r);
            ret = -EBUSY;
            goto fail;
        }

        // tell the ESB2 our in buffer space and request data
        if (esb2_fml_read_req(&dev->fml, sizeof(dev->recv_buf) - dev->recv_size) < 0) {
            ERR("failed to initially report free TPT receive buffer size");
        }
    }

    DBG("opened");
    return ret;
fail:
    esb2_tpt_mod_release(inode, file);
    return ret;
}

static int
esb2_tpt_mod_release(struct inode *inode, struct file *file)
{
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;

    if (atomic_dec_return(&dev->refcnt) == 0) {
        DBG("unload");
        /* least time cleanup */
	dev->recv_size = 0;
	dev->send_fill = 0;
	dev->send_thread_running = 0;
        wake_up(&dev->send_wait);
        if (dev->isr_registered) free_irq(DEF_IRQ, dev); dev->isr_registered = 0;
        if (dev->fml_init); fmlcore_release(&dev->fml); dev->fml_init = 0;
        release_mem_region(kira100_io_base + FML_IO_BASE_OFFS, FML_IO_SIZE);
    }

    DBG("closed");
    MOD_DEC_USE_COUNT;
    return SUCCESS;
}

static int
esb2_tpt_mod_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;

    switch (cmd) {
    case ESB2_TPT_IOCTL_GET_TPT_STAT:
    {
        esb2_tpt_ioctl_tpt_stat_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        if (esb2_tpt_get_status(&dev->fml, d.chan, &d.status, d.remote_ip) < 0) {
            ERR("failed to get TPT status");
            return -EIO;
        }
        if (copy_to_user((void*)arg, &d, sizeof(d))) return -EFAULT;
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_SET_RECV_TIMEOUT:
    {
        esb2_tpt_ioctl_recv_timeout_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        dev->recv_timeout = d.timeout; /* chan is ignored */
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_GET_REG:
    {
        esb2_tpt_ioctl_reg_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        d.val = dev->fml.read_reg(&dev->fml, d.reg);
        if (copy_to_user((void*)arg, &d, sizeof(d))) return -EFAULT;
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_SET_REG:
    {
        esb2_tpt_ioctl_reg_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        dev->fml.write_reg(d.val, &dev->fml, d.reg);
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_GET_HK:
    {
        if (copy_to_user((void*)arg, &dev->hk, sizeof(esb2_tpt_ioctl_hk_t))) return -EFAULT;
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_SET_CONNLOST:
    {
	dev->conn_lost = 1;
	wake_up(&dev->wait);
	return SUCCESS;
    }
    default:
        /* invalid cmd */
        return -EINVAL;
    }
}

static int
esb2_tpt_mod_read(struct file* file, char *buffer, size_t size, loff_t *offset)
{
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;
    u_long flags;
    int i, newsize;

    if (dev->conn_lost && (dev->recv_size == 0)) {
	//special case: connection was lost, in this case we only send EOF
	dev->conn_lost = 0;
	return 0;
    }

    // blocking read, timeout set by IOCTL (default: 1 min)
    for (i = 0; dev->recv_size == 0 && i < dev->recv_timeout; i++) {
        /* after 1 sec, every 10 secs */
        if ((i % 10) == 1) {
            /* use time to check connections status */
            unsigned char stat = ESB2_TPT_STAT_CONNECTED; // optimistic point of view ;)

            if (down_interruptible(&dev->sem) < 0) return -EINTR;
            if (esb2_tpt_get_status(&dev->fml, TPT_CHAN, &stat, NULL) < 0)
                WARN("failed to query connection state while waiting for data");
            up(&dev->sem);

            if (!ESB2_TPT_IS_CONNECTED(stat)) return -ENOTCONN;
        }

        wait_event_interruptible_timeout(dev->wait, dev->recv_size, HZ);
    }
    if (dev->recv_size == 0) return -EAGAIN;

    spin_lock_irqsave(&dev->recv_lock, flags);

    /* we cannot read more than received */
    if (size > dev->recv_size) size = dev->recv_size;

    /* get the data */
    memcpy(buffer, dev->recv_buf, size);

    /* copy the rest to buffer head */
    if (size < dev->recv_size) memmove(dev->recv_buf, dev->recv_buf + size, dev->recv_size - size);

    /* reduce remaining recv buf size */
    dev->recv_size -= size;
    newsize = sizeof(dev->recv_buf) - dev->recv_size;

    spin_unlock_irqrestore(&dev->recv_lock, flags);

    DBG("(%d lost) TPT packet read (%d bytes): %.2x %.2x %.2x ...", esb2_tpt_blk_lost, size,
        (unsigned char)buffer[0], (unsigned char)buffer[1], (unsigned char)buffer[2]);

    /* inform FML master about new recv buf size */
    if (size > 0 && dev->recv_size == 0) {
        if (down_interruptible(&dev->sem) < 0) return -EINTR;
        if (esb2_fml_read_req(&dev->fml, newsize) < 0) {
            WARN("failed to report free TPT receive buffer size");
        }
        up(&dev->sem);
    }

    return size;
}


static int
esb2_tpt_mod_write(struct file* file, const char *buffer, size_t size, loff_t *offset)
{
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;
    int len = 0, c, count;
    int retval = 0;

    DBG("(%d lost) sending TPT packet (%d bytes): %.2x %.2x %.2x ...", esb2_tpt_blk_lost, size,
        (unsigned char)buffer[0], (unsigned char)buffer[1], (unsigned char)buffer[2]);

    if (down_interruptible(&dev->sem) < 0) return -EINTR;
    while (len < (int)size) {
        count = min(size-len, MESB2_MAX_BLK_LEN-(dev->send_fill));
        copy_from_user(dev->send_buf+dev->send_fill, buffer+len, count);

	if ((dev->send_fill+count) >= MESB2_MAX_BLK_LEN) {
            //buffer full, send it out
            c = esb2_tpt_send(&dev->fml, TPT_CHAN, dev->send_buf, MESB2_MAX_BLK_LEN);
            dev->send_fill = 0;
            dev->hk.bytes_sent += c;
        } else {
            //we have got not enough data to send it out, so wake the collector thread
            c = count;
            dev->send_fill += count;
            dev->data_to_send = count;
            wake_up(&dev->send_wait);
        }
        len += count;
        if (c <= 0) {
            ERR("failed to send TPT packet (%d)", c);
            retval = -EIO;
            goto exit;
        }
   }
    retval = len;
exit:
    up(&dev->sem);
    return retval;
}

static unsigned int
esb2_tpt_mod_poll(struct file *file, poll_table *wait)
{
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;
    unsigned int mask = POLLOUT | POLLWRNORM;
    if (dev->recv_size || dev->conn_lost) mask |= POLLIN | POLLRDNORM;
    poll_wait(file, &dev->wait, wait);
    return mask;
}

/* ------------------------------------------------------------------------- *
 * interrupt service routine
 * ------------------------------------------------------------------------- */

static irqreturn_t
esb2_tpt_mod_isr(int irq, void *dev_id, struct pt_regs *regs)
{
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;

    /* statistics */
    dev->hk.intr_cnt++;
#ifdef LPC_USE_INTMUX
    {
        unsigned int intmux = readl(dev->intmux_addr);
        if (intmux & FML_INT) dev->hk.fml_intr_cnt++;
    }
#endif

    /* call the FML core driver's IRQ handler */
//WARN("enter isr");
    fmlcore_irq_handler(irq, &dev->fml, regs);
//WARN("leave isr");
    return IRQ_HANDLED;
}

static int
send_collector_thread(void * arg)
{
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;
    int ret, c;

    while (dev->send_thread_running) {
        ret = wait_event_interruptible_timeout(dev->send_wait, (dev->data_to_send > 0), HZ/20);
        if (ret <= 0) {
            //timeout, send immediately
            if (down_interruptible(&dev->sem) < 0) return -EINTR;
            if (dev->send_fill > 0) {
                if ((c = esb2_tpt_send(&dev->fml, TPT_CHAN, dev->send_buf, dev->send_fill)) <= 0) {
                    ERR("failed to send from collector thread");
		} else {
                    dev->send_fill = 0;
 		    dev->hk.bytes_sent += c;
                }
            }
            up(&dev->sem);
        } 
        dev->data_to_send = 0;
	cond_resched();
    }
    return 0;
}

/* will be called back by FML core driver's IRQ handler in case of any notification IRQ */
static int
recv_cb(struct fmlcore_private *fml)
{
//WARN("enter cb");
    esb2_tpt_dev_t *dev = &esb2_tpt_dev;
    int size;
    unsigned char chan;

    dev->hk.slave_intr_cnt++;

    // any op according to recv_buf must be locked against READ
    spin_lock(&dev->recv_lock);

    size = esb2_tpt_recv(&dev->fml,
                         &chan,
                         dev->recv_buf + dev->recv_size,
                         sizeof(dev->recv_buf) - dev->recv_size);
//DBG("TPT packet received (%d bytes): %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x ...", size,
//dev->recv_buf[dev->recv_size], dev->recv_buf[dev->recv_size + 1], dev->recv_buf[dev->recv_size + 2],
//dev->recv_buf[dev->recv_size + 3], dev->recv_buf[dev->recv_size + 4], dev->recv_buf[dev->recv_size + 5],
//dev->recv_buf[dev->recv_size + 6], dev->recv_buf[dev->recv_size + 7], dev->recv_buf[dev->recv_size + 8]);

    if (size > 0 && chan == TPT_CHAN) {
        dev->recv_size += size;
        dev->hk.bytes_received += size;
    }

    spin_unlock(&dev->recv_lock);

    if (size > 0 && chan == TPT_CHAN) {
        //DBG("TPT packet received (%d bytes)", size);
    } else {
        if (size < 0) ERR("failed to read received TPT packet");
        else if (size > 0 && chan != TPT_CHAN)
            ERR("received data at wrong TPT chan: expected=%d is=%d", TPT_CHAN, chan);
    }

    // wake any pending pollers
    if (dev->recv_size > 0) wake_up(&dev->wait);

//WARN("leave cb");
    return 0;
}
