/**
 * fml_esb2.c
 *
 * FML TCP Pass Through sideband protocol of
 * Intel ESB2 (Southbridge)
 *
 * (c) 2005 Peppercon AG, 2005/07/21, Ralf Guenther <rgue@peppecon.de>
 */

#include <linux/kernel.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <asm/byteorder.h>

#include "fmlcore.h"
#include "fml_esb2.h"
#include "fml_test.h"

#define SUCCESS 0

#define NAME                    "fml_esb2"
#define LOG_COLORED
#define noLOG_BLACK_BG
#define noLOG_DBG
#include "log.h"

#define ESB2_OP_REQ_NONE            0x00
#define ESB2_OP_REQ_READ            0x40
#define ESB2_OP_REQ_STATUS          0x80

#define ESB2_OP_LONG_TRANS          0x10

#define ESB2_OP_MASK                0x0f
#define ESB2_OP_STATUS              0x00
#define ESB2_OP_IPMI                0x04
#define ESB2_OP_TPT                 0x05
#define ESB2_OP_RESET_FML           0x0d
#define ESB2_OP_CLEAR_FML_ERR       0x0e

#define ESB2_LEN_LONG_TRANS         0xfc

#define ESB2_STATUS_MASK            0xf0
#define ESB2_STATUS_OUT_DATA_AVAIL  0x80
#define ESB2_STATUS_IN_BUF_FULL     0x40

#define ESB2_ERR_MASK               0x0f
#define ESB2_ERR_NONE               0x00
#define ESB2_ERR_IN_DATA_OVERRUN    0x01
#define ESB2_ERR_READ_EMPTY_BUF     0x02
#define ESB2_ERR_FML_RESET          0x03
#define ESB2_ERR_FML_BUS_FAILURE    0x04

#define ESB2_SHORT_DATA_SIZE        252
#define ESB2_LONG_DATA_SIZE         SBUF_SIZE // 2048 bytes

#define ESB2_TPT_CMD_GET_STATUS     0
#define ESB2_TPT_CMD_SEND           1
#define ESB2_TPT_CMD_RECV           2

#define BMC 26

#if BMC <= 18
#  define ESB2_TPT_START_DATA       0
#  define ESB2_TPT_MIDDLE_DATA      0
#  define ESB2_TPT_END_DATA         0
#  define ESB2_TPT_SINGLE_DATA      0
#else
#  define ESB2_TPT_START_DATA       (0 << 6)
#  define ESB2_TPT_MIDDLE_DATA      (1 << 6)
#  define ESB2_TPT_END_DATA         (2 << 6)
#  define ESB2_TPT_SINGLE_DATA      (3 << 6)
#endif

#define ESB2_TPT_CURR_CHAN          0x0e

#define PACKED __attribute__((__packed__))

typedef struct {
    uint8_t data[ESB2_SHORT_DATA_SIZE];
} PACKED short_data_t;

typedef struct {
    uint16_t ext_data_len;
    uint8_t data[ESB2_LONG_DATA_SIZE];
} PACKED long_data_t;

typedef struct {
    uint8_t op;
    uint8_t data_len;
    uint16_t read_buf_len;
    union {
        short_data_t s;
        long_data_t l;
    } data PACKED;
} PACKED esb2_write_transaction_t;

#define ESB2_WRITE_HEADER_SIZE      \
    (sizeof(esb2_write_transaction_t) \
    - sizeof(((esb2_write_transaction_t*)0)->data) \
    + sizeof(((esb2_write_transaction_t*)0)->data.s) \
    - sizeof(((esb2_write_transaction_t*)0)->data.s.data))

#define ESB2_LONG_WRITE_HEADER_SIZE \
    (sizeof(esb2_write_transaction_t) \
    - sizeof(((esb2_write_transaction_t*)0)->data) \
    + sizeof(((esb2_write_transaction_t*)0)->data.l) \
    - sizeof(((esb2_write_transaction_t*)0)->data.l.data))

typedef struct {
    uint8_t op;
    uint8_t data_len;
    uint8_t status;
    union {
	short_data_t s;
        long_data_t l;
    } data PACKED;
} PACKED esb2_read_transaction_t;

#define ESB2_READ_HEADER_SIZE      \
    (sizeof(esb2_read_transaction_t) \
    - sizeof(((esb2_read_transaction_t*)0)->data) \
    + sizeof(((esb2_read_transaction_t*)0)->data.s) \
    - sizeof(((esb2_read_transaction_t*)0)->data.s.data))

#define ESB2_LONG_READ_HEADER_SIZE \
    (sizeof(esb2_read_transaction_t) \
    - sizeof(((esb2_read_transaction_t*)0)->data) \
    + sizeof(((esb2_read_transaction_t*)0)->data.l) \
    - sizeof(((esb2_read_transaction_t*)0)->data.l.data))

static const char* fml_err_str(int err)
{
    switch (err & ESB2_ERR_MASK) {
    case ESB2_ERR_NONE:             return "no error";
    case ESB2_ERR_IN_DATA_OVERRUN:  return "input buffer overflow";
    case ESB2_ERR_READ_EMPTY_BUF:   return "output buffer underflow";
    case ESB2_ERR_FML_RESET:        return "has been reset";
    case ESB2_ERR_FML_BUS_FAILURE:  return "bus error, unexpected NAK";
    default:                        return "unknown";
    }
}

/**************************************
 * ESB2 FML protocol
 **************************************/

static int max_recv_len = SBUF_SIZE;
static unsigned char last_tpt_blk_no = 0;
int esb2_tpt_blk_lost = 0;

int esb2_fml_read_req(struct fmlcore_private *fml, int max_len)
{
    esb2_write_transaction_t p;
    max_recv_len = max_len;

    p.op = ESB2_OP_TPT | ESB2_OP_REQ_READ;
    p.data_len = sizeof(p.read_buf_len);
    p.read_buf_len = __cpu_to_le16(max_recv_len);

    return fmlcore_slave_write(fml, (void*)&p, ESB2_WRITE_HEADER_SIZE) != ESB2_WRITE_HEADER_SIZE ? -1 : 0;
}
int esb2_fml_stat_req(struct fmlcore_private *fml)
{
    esb2_write_transaction_t p;

    p.op = ESB2_OP_TPT | ESB2_OP_REQ_STATUS;
    p.data_len = sizeof(p.read_buf_len);
    p.read_buf_len = __cpu_to_le16(max_recv_len);

    return fmlcore_slave_write(fml, (void*)&p, ESB2_WRITE_HEADER_SIZE) != ESB2_WRITE_HEADER_SIZE ? -1 : 0;
}

int esb2_fml_reset(struct fmlcore_private *fml)
{
    esb2_write_transaction_t p;
    p.op = ESB2_OP_RESET_FML | ESB2_OP_REQ_NONE;
    p.data_len = sizeof(p.read_buf_len);
    p.read_buf_len = __cpu_to_le16(max_recv_len);

    return fmlcore_slave_write(fml, (void*)&p, ESB2_WRITE_HEADER_SIZE) != ESB2_WRITE_HEADER_SIZE ? -1 : 0;
}

int esb2_fml_clr_err(struct fmlcore_private *fml)
{
    esb2_write_transaction_t p;
    p.op = ESB2_OP_CLEAR_FML_ERR | ESB2_OP_REQ_NONE;
    p.data_len = sizeof(p.read_buf_len);
    p.read_buf_len = __cpu_to_le16(max_recv_len);

    return fmlcore_slave_write(fml, (void*)&p, ESB2_WRITE_HEADER_SIZE) != ESB2_WRITE_HEADER_SIZE ? -1 : 0;
}

int esb2_fml_write(struct fmlcore_private *fml, const unsigned char *data, int len)
{
    int l;
    esb2_write_transaction_t p;
    p.op = ESB2_OP_TPT | ESB2_OP_LONG_TRANS | ESB2_OP_REQ_READ;
    p.data_len = ESB2_LEN_LONG_TRANS;
    p.read_buf_len = __cpu_to_le16(max_recv_len);
    p.data.l.ext_data_len =
        __cpu_to_le16(sizeof(p.read_buf_len) + sizeof(p.data.l.ext_data_len) + len);
    memcpy(p.data.l.data, data, len);

    l = fmlcore_slave_write(fml, (void*)&p, ESB2_LONG_WRITE_HEADER_SIZE + len);
    if (l < ESB2_LONG_WRITE_HEADER_SIZE) return -1;
    return l - ESB2_LONG_WRITE_HEADER_SIZE;
}

/**************************************
 * ESB2 TPT protocol
 **************************************/

static struct {
    unsigned char stat;
    unsigned char ip[4];
    unsigned char waiting;
    wait_queue_head_t wait;
} get_stat[2] = {
    { .wait = __WAIT_QUEUE_HEAD_INITIALIZER(get_stat[0].wait), },
    { .wait = __WAIT_QUEUE_HEAD_INITIALIZER(get_stat[1].wait), },
};

int esb2_tpt_get_status(struct fmlcore_private *fml,
                        unsigned char chan,
                        unsigned char *pstat,
                        unsigned char *remote_ip)
{
    esb2_write_transaction_t p;

    if (chan > 1) return -1;

    p.op = ESB2_OP_TPT | ESB2_OP_REQ_READ;
    p.data_len = sizeof(p.read_buf_len) + 2;
    p.read_buf_len = __cpu_to_le16(max_recv_len);
    p.data.s.data[0] = ESB2_TPT_CMD_GET_STATUS;
    p.data.s.data[1] = chan & 0x0f;

    get_stat[chan].waiting = 1;

    if ((fmlcore_slave_write(fml, (void*)&p, ESB2_WRITE_HEADER_SIZE + 2)) < ESB2_WRITE_HEADER_SIZE + 2) {
        WARN("sending 'Get TPT Status' request failed");
        return -1;
    }

    wait_event_interruptible_timeout(get_stat[chan].wait, !get_stat[chan].waiting, 2*HZ);

    if (get_stat[chan].waiting) {
	ERR("got no TPT status response");
	return -1;
    }
    if (pstat) *pstat = get_stat[chan].stat;
    if (remote_ip) *(long*)remote_ip = *(long*)get_stat[chan].ip;
    return 0;
}

int esb2_tpt_send(struct fmlcore_private *fml,
                  unsigned char chan,
                  const unsigned char *data,
                  int len)
{
    int c, header_size, retries = 10;
    esb2_write_transaction_t p;
    if (2 + len < ESB2_LEN_LONG_TRANS) {
        // fill FML header
        p.op = ESB2_OP_TPT | ESB2_OP_REQ_READ;
        p.data_len = sizeof(p.read_buf_len) + 2 + len;
        p.read_buf_len = __cpu_to_le16(max_recv_len);

        // fill TPT header + packet
        p.data.s.data[0] = ESB2_TPT_CMD_SEND;
        p.data.s.data[1] = (chan & 0x0f) | ESB2_TPT_SINGLE_DATA;
        memcpy(p.data.s.data + 2, data, len);

        header_size = ESB2_WRITE_HEADER_SIZE + 2;
    } else {
        // check data len and truncate if too long
        if (len > sizeof(p.data.l.data) - ESB2_LONG_WRITE_HEADER_SIZE - 2)
            len = sizeof(p.data.l.data) - ESB2_LONG_WRITE_HEADER_SIZE - 2;

        // fill FML header
        p.op = ESB2_OP_TPT | ESB2_OP_LONG_TRANS | ESB2_OP_REQ_READ;
        p.data_len = ESB2_LEN_LONG_TRANS;
        p.read_buf_len = __cpu_to_le16(max_recv_len);
        p.data.l.ext_data_len =
            __cpu_to_le16(sizeof(p.read_buf_len) + sizeof(p.data.l.ext_data_len) + 2 + len);

        // fill TPT header + packet
        p.data.l.data[0] = ESB2_TPT_CMD_SEND;
        p.data.l.data[1] = (chan & 0x0f) | ESB2_TPT_SINGLE_DATA;
        memcpy(p.data.l.data + 2, data, len);

	header_size = ESB2_LONG_WRITE_HEADER_SIZE + 2;
    }

    c = -1;
    while ((c < 0) && (retries > 0)) {
        c = fmlcore_slave_write(fml, (void*)&p, len + header_size) - header_size;
        retries--;
    }
    if (c < 0) ERR("esb2_tpt_send failed (%d)", c);
    return c < 0 ? -1 : c;
}

int esb2_tpt_recv(struct fmlcore_private *fml,
                  unsigned char *pchan,
                  unsigned char *data,
                  int bufsize)
{
    esb2_read_transaction_t p;
    int size, data_size, len = 0, test_ret;
//memset(&p, 0xfc, sizeof(p));

    // copy received packet from device
    size = fmlcore_slave_read(fml, (void*)&p, sizeof(p));
    
    // FIXME: put the test slave part to a better place
    test_ret = fml_test_slave(fml, (char *)&p, size);
    if (test_ret <= 0) {
        // error or handled test call
        return test_ret;
    }

    // check packet size
    if (size < ESB2_READ_HEADER_SIZE
        || ((p.op & ESB2_OP_LONG_TRANS) && size < ESB2_LONG_READ_HEADER_SIZE)) {
        ERR("FML packet header truncated: %d bytes received but %d bytes expect", size,
            p.op & ESB2_OP_LONG_TRANS ? ESB2_LONG_READ_HEADER_SIZE : ESB2_READ_HEADER_SIZE);
        return -1;
    }
    DBG("FML packet received: size=%d op=0x%.2x len=%d status=0x%.2x ext_len=%d", size, p.op, p.data_len, p.status, data_size = __le16_to_cpu(p.data.l.ext_data_len));

    // calc payload size
    data_size = p.op & ESB2_OP_LONG_TRANS
                ? __le16_to_cpu(p.data.l.ext_data_len) - 3
                : p.data_len - 1;
    size -= p.op & ESB2_OP_LONG_TRANS ? ESB2_LONG_READ_HEADER_SIZE : ESB2_READ_HEADER_SIZE;

    if (data_size != size) {
        ERR("FML packet payload size mismatch: len=%d space=%d", data_size, size);
        size = min(size, data_size);
    }

    if ((p.status & ESB2_ERR_MASK) != ESB2_ERR_NONE) {
        WARN("FML error detected: '%s' (err=%d)",
             fml_err_str(p.status), p.status & ESB2_ERR_MASK);
    }

    switch (p.op & ESB2_OP_MASK)
    {
    case ESB2_OP_STATUS:
        //DBG("received FML op ESB2_OP_STATUS");
        break;
    case ESB2_OP_TPT:
        //DBG("received FML op ESB2_OP_TPT");
        {
            unsigned char *buf = p.op & ESB2_OP_LONG_TRANS ? p.data.l.data : p.data.s.data;

            // must contain at least a TPT cmd
            if (size < 1) return -1;

            switch (buf[0])
            {
            case ESB2_TPT_CMD_GET_STATUS:
                //DBG("received TPT cmd ESB2_TPT_CMD_GET_STATUS");
                {
                    // no real payload, but TPT Get Status response
                    unsigned char chan;
                    if (size < 7) return -1;
                    chan = buf[1] & 0x0f;
                    if (chan > 1) return -1;

                    get_stat[chan].stat = buf[2] & 0x0f;
                    *(long*)get_stat[chan].ip = *(long*)&buf[3];
                    get_stat[chan].waiting = 0;
                    wake_up(&get_stat[chan].wait);
                }
                break;
            case ESB2_TPT_CMD_RECV:
                //DBG("received TPT cmd ESB2_TPT_CMD_RECV (chan=%d blk#=%d)", buf[1], buf[2]);
                if (size < 3) return -1;

                // return chan
                if (pchan) *pchan = buf[1] & 0x0f;

                // check block number
                if (buf[2] != 0 && last_tpt_blk_no != 0) { // use block 0 for sync
                    if (buf[2] != last_tpt_blk_no + 1) {
                        esb2_tpt_blk_lost += buf[2] - last_tpt_blk_no - 1;
                        ERR("received TPT block out of order: expected=%d is=%d",
                            last_tpt_blk_no + 1, buf[2]);
                    }
                }
                last_tpt_blk_no = buf[2];

                // remove cmd, chan, and block number
                len = size - 3;

/*
if (len == 16 || (buf[3] == 0x05 && (buf[9] | buf[10]) != 0))
INFO("### ESB2_TPT_CMD_RECV %d bytes: %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x", len,
buf[3], buf[4], buf[5], buf[6],
buf[7], buf[8], buf[9], buf[10],
buf[11], buf[12], buf[13], buf[14],
buf[15], buf[16], buf[17], buf[18]);
*/

                if (len > bufsize) {
                    ERR("received TPT block too large: max=%d is=%d", bufsize, len);
                    len = bufsize;
                }
                memcpy(data, buf + 3, len);
                break;
            default:
                ERR("unknown TPT cmd %d received", buf[0]);
            }
        }
        break;
    default:
        ERR("unknown FML op %d received", p.op & ESB2_OP_MASK);
    }
    return len;
}
