/**
 * fml_ophir.h
 *
 * FML Advanced Pass Through sideband protocol of
 * Intel 82571 Gigabit Ethernet Controller (Ophir)
 *
 * (c) 2004 Peppercon AG, 2004/12/20, Ralf Guenther <rgue@peppecon.de>
 */

#ifndef __FML_OPHIR_H__
#define __FML_OPHIR_H__

#include <linux/delay.h>

#include "tco_core.h"

/* write commands */
#define OPHIR_CMD_XMIT_PKT      ((uint8_t)0x04)
#define OPHIR_CMD_WRITE_CFG     ((uint8_t)0x06)
#define OPHIR_CMD_RECV_EN       ((uint8_t)0x0a)
#define OPHIR_CMD_UPDATE_FILTER ((uint8_t)0x0c)
#define OPHIR_CMD_REQ_STAT      ((uint8_t)0x1d)

/* read commands */
#define OPHIR_CMD_RECV_ANY      ((uint8_t)0x00)
#define OPHIR_CMD_READ_FILTER   ((uint8_t)0x0d)
#define OPHIR_CMD_RECV_PKT      ((uint8_t)0x10)
#define OPHIR_CMD_READ_CFG      ((uint8_t)0x12)
#define OPHIR_CMD_READ_RECV_EN  ((uint8_t)0x1a)
#define OPHIR_CMD_READ_STAT     ((uint8_t)0x1e)
#define OPHIR_CMD_GET_MAC_ADDR  ((uint8_t)0x14)

/* read op-codes */
#define OPHIR_OP_RECV_PKT       ((uint8_t)0x10)
#define OPHIR_OP_READ_CFG       ((uint8_t)0x12)
#define OPHIR_OP_READ_RECV_EN   ((uint8_t)0x1a)
#define OPHIR_OP_READ_STAT      ((uint8_t)0x1d)
#define OPHIR_OP_GET_MAC_ADDR   ((uint8_t)0x14)

/* Receive Enable flags */
#define OPHIR_RCV_EN            ((uint8_t)1 << 0)
#define OPHIR_RCV_ALL           ((uint8_t)1 << 1)
#define OPHIR_EN_STA            ((uint8_t)1 << 2)
#define OPHIR_EN_ARP_RES        ((uint8_t)1 << 3)
#define OPHIR_NM_MASK           ((uint8_t)3 << 4)
#define OPHIR_NM_SMB_ALERT      ((uint8_t)0 << 4)
#define OPHIR_NM_ASYNC_NOTIFY   ((uint8_t)1 << 4)
#define OPHIR_NM_DIRECT_RECV    ((uint8_t)2 << 4)
#define OPHIR_RPSTL             ((uint8_t)1 << 6)
#define OPHIR_CBDM              ((uint8_t)1 << 7)

/* Read Status flags */
#define OPHIR_CMD_ABORTED       ((uint8_t)1 << (6 + 8))
#define OPHIR_LINK_STATE        ((uint8_t)1 << (5 + 8))
#define OPHIR_INIT              ((uint8_t)1 << (3 + 8))
#define OPHIR_POWER_STATE_MASK  ((uint8_t)3 << (0 + 8))
#define OPHIR_POWER_STATE_DR    ((uint8_t)0 << (0 + 8))
#define OPHIR_POWER_STATE_D0U   ((uint8_t)1 << (0 + 8))
#define OPHIR_POWER_STATE_D0    ((uint8_t)2 << (0 + 8))
#define OPHIR_POWER_STATE_D3    ((uint8_t)3 << (0 + 8))

#define OPHIR_DRV_VALID         ((uint8_t)1 << (3 + 0))
#define OPHIR_INTR_PENDING      ((uint8_t)1 << (2 + 0))
#define OPHIR_ICR_REG_READ      ((uint8_t)1 << (1 + 0))

/* Packet sequence flags */
#define OPHIR_CMD_MASK          ((uint8_t)0x3f)
#define OPHIR_SEQ_MASK          ((uint8_t)0xc0)
#define OPHIR_SEQ_SINGLE        ((uint8_t)0xc0)
#define OPHIR_SEQ_FIRST         ((uint8_t)0x80)
#define OPHIR_SEQ_LAST          ((uint8_t)0x40)
#define OPHIR_SEQ_MIDDLE        ((uint8_t)0x00)

/* MANC register flags */
#define OHPIR_MANC_FLEXPORT0_EN ((uint32_t)1 << 3)
#define OHPIR_MANC_FLEXPORT1_EN ((uint32_t)1 << 4)
#define OHPIR_MANC_FLEXPORT2_EN ((uint32_t)1 << 5)
#define OHPIR_MANC_FLEXTCO1_EN  ((uint32_t)1 << 6)
#define OHPIR_MANC_FLEXTCO0_EN  ((uint32_t)1 << 7)
#define OHPIR_MANC_RMCP_EN      ((uint32_t)1 << 8)
#define OHPIR_MANC_0298_EN      ((uint32_t)1 << 9)
#define OHPIR_MANC_ARP_REQ_EN   ((uint32_t)1 << 13)
#define OHPIR_MANC_NEIGHBOR_EN  ((uint32_t)1 << 14)
#define OHPIR_MANC_ARP_RES_EN   ((uint32_t)1 << 15)
#define OHPIR_MANC_MNG2HOST_EN  ((uint32_t)1 << 21)
#define OHPIR_MANC_IP_ADDR_EN   ((uint32_t)1 << 22)
#define OHPIR_MANC_XSUM_EN      ((uint32_t)1 << 23)
#define OHPIR_MANC_BCAST_EN     ((uint32_t)1 << 24)

/* MNG2HOST register flags */
#define OHPIR_MNG2HOST_FP0      ((uint32_t)1 << 0)
#define OHPIR_MNG2HOST_FP1      ((uint32_t)1 << 1)
#define OHPIR_MNG2HOST_FP2      ((uint32_t)1 << 2)
#define OHPIR_MNG2HOST_FT0      ((uint32_t)1 << 3)
#define OHPIR_MNG2HOST_FT1      ((uint32_t)1 << 4)
#define OHPIR_MNG2HOST_026F     ((uint32_t)1 << 5)
#define OHPIR_MNG2HOST_0298     ((uint32_t)1 << 6)
#define OHPIR_MNG2HOST_ARP_REQ  ((uint32_t)1 << 7)
#define OHPIR_MNG2HOST_ARP_RES  ((uint32_t)1 << 8)
#define OHPIR_MNG2HOST_BCAST    ((uint32_t)1 << 9)
#define OHPIR_MNG2HOST_VT0      ((uint32_t)1 << 11)
#define OHPIR_MNG2HOST_VT1      ((uint32_t)1 << 12)
#define OHPIR_MNG2HOST_VT2      ((uint32_t)1 << 13)
#define OHPIR_MNG2HOST_VT3      ((uint32_t)1 << 14)
#define OHPIR_MNG2HOST_MCA16    ((uint32_t)1 << 15)

#define OPHIR_MAX_DATA_SIZE     240 // absolute maximum, specific core data size may be less!

/*
 * receive packet layout
 */

typedef struct tco_recv_pkt_s {
//    uint8_t count; -> not part of the packet, but is return value of recv func
    uint8_t op_code;
    uint8_t data[OPHIR_MAX_DATA_SIZE];
} __attribute__ ((packed)) tco_recv_pkt_t;

/*
 * write commands
 */
 
int ophir_xmit_packet(tco_t *tco, uint8_t seq, uint8_t len, uint8_t *data);
int ophir_write_config(tco_t *tco, uint16_t addr, uint32_t data);
int ophir_recv_enable(tco_t *tco, uint8_t ctrl);
int ophir_recv_enable_ex(tco_t *tco,
                       uint8_t ctrl,
                       uint8_t mac_addr[6],
                       uint8_t ip_addr[4],
                       uint8_t bmc_smbus_addr,
                       uint8_t if_data,
                       uint8_t alert_data);
int ophir_update_filter(tco_t *tco, uint8_t number, uint8_t len, uint8_t *data);
int ophir_request_status(tco_t *tco);

/*
 * read commands
 */

int ophir_recv_any(tco_t *tco, uint8_t *count, tco_recv_pkt_t *recv_pkt);
int ophir_read_filter(tco_t *tco, uint8_t number, uint8_t *len, uint8_t *data);
int ophir_recv_packet(tco_t *tco, uint8_t *seq, uint8_t *len, uint8_t *data); /* last frame: len=1 data[0]=status */
int ophir_read_config(tco_t *tco, uint16_t addr, uint32_t *data);
int ophir_read_recv_enable(tco_t *tco,
                         uint8_t *ctrl,
                         uint8_t *mac_addr,
                         uint8_t *ip_addr,
                         uint8_t *bmc_smbus_addr,
                         uint8_t *if_data,
                         uint8_t *alert_data);
int ophir_read_status(tco_t *tco, uint16_t *status);
int ophir_get_sys_mac_addr(tco_t *tco, uint8_t *addr);

#endif /* !__FML_OPHIR_H__ */
