/*********************************************************************
 **                                                                 
 **  File          : fmlcore.c                                       
 **  Project       : FML-Core Driver for Linux 2.6.x                
 **  Purpose       : usage of the FML-Bus to access Intels 82570 MAC
 **  Contents      : defines for the register access to the         
 **                  FML-Core and declarations of the basic routines
 **  Author        : Marko R�ler <marr@infotech.tu-chemnitz.de>    
 **  Last modified : <11.03.2005 14:56:35 marr>                     
 **                                                                 
 *********************************************************************/

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>

#include <linux/module.h>
// #include <linux/moduleparam.h>
#include <linux/sched.h>

#include <linux/init.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/version.h>
#include <asm/semaphore.h>
#include <asm/irq.h>
#include <asm/io.h>

#ifdef __powerpc__
#include <asm/ppc405_pcimcs.h>
#endif

#ifdef __arm__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <asm/arch/cpe/peppercon.h>
#else
#include <asm/arch/platform/peppercon.h>
#endif
#include <asm/arch/net_lock.h>
#endif

#include "fmlcore.h"
#include "fml_ophir.h"

#define NAME "fmlcore"
#define LOG_COLORED
#define noLOG_DBG
#include "log.h"

// detailed debug logging
//#define DBG2 DBG
//#define REGS regs
#define DBG2(...)
#define REGS(...)

//timeout values for wait_while_tip
#define MASTER_TIMEOUT (HZ / 5)
#define SLAVE_TIMEOUT (HZ * 2)

#ifdef __arm__
#define LOCK_NET if (KIRA_MAJOR(fml_priv->kira_rev) < 2) ahbfix_lock_net()
#define UNLOCK_NET if (KIRA_MAJOR(fml_priv->kira_rev) < 2) ahbfix_unlock_net()
#else 
#define LOCK_NET
#define UNLOCK_NET
#endif

#define BUFF_ADDR(priv, num) (priv->buf_offset+(num*priv->buf_size))
#define ISSUE_COMMAND(priv, val) reset_finished(priv); \
				 priv->write_reg(val, priv, MCR0_ADDR)

#ifdef LOG_DBG
static void regs(struct fmlcore_private* fml, const char* txt)
{
    int mgr  = fml->read_reg(fml, MGR_ADDR);
    int mcr0 = fml->read_reg(fml, MCR0_ADDR);
    int mgsr = fml->read_statusreg(fml);
    int prer = fml->read_reg(fml, MPRER_ADDR);
    int tout = fml->read_reg(fml, MTOUT_ADDR);
    int id1  = fml->read_reg(fml, MID1_ADDR);
    int id2  = fml->read_reg(fml, MID2_ADDR);
    int srbc = fml->read_reg(fml, SRBC_ADDR);
    int swbc = fml->read_reg(fml, SWBC_ADDR);
    int sar = fml->read_reg(fml, SAR_ADDR);
    int core_state = fml->read_reg(fml, CORESTAT_ADDR);
    int byte_state = fml->read_reg(fml, BYTESTAT_ADDR);
    int bit_state = fml->read_reg(fml, BITSTAT_ADDR);
    printk("%s:\n"
           "  mgr  = %08x\n  mcr0 = %08x\n"
           "  mgsr = %08x\n  prer = %08x\n"
           "  tout = %08x\n  id   = '%.4s%.4s' (%08x %08x)\n"
           "  sar = %08x\n"
           "  srbc = %08x\n  swbc = %08x\n"
           "  corestate = %08x bytestate = %08x bitstate = %08x\n",
           txt, mgr, mcr0, mgsr, prer, tout, 
           (char*)&id1, (char*)&id2, id1, id2, sar, srbc, swbc, 
           core_state, byte_state, bit_state);
}
#endif

#ifdef __arm__

/***
    here is everything needed for the DMA/MMU hack
***/

void fixed_read_block(u32 *buf, struct fmlcore_private *fml_priv, u32 offset, int cnt)
{
    ahbfix_read_block(buf, fml_priv->phys_base, offset, cnt, 0);
}

void fixed_write_block(u32 *vals, struct fmlcore_private *fml_priv, u32 offset, int cnt)
{
    ahbfix_write_block(vals, fml_priv->phys_base, offset, cnt, 0);
}

u32 fixed_read(struct fmlcore_private* priv, u32 offset)
{
    u32 val;
    ahbfix_read_block(&val, priv->phys_base, offset, 1, 0);
    return val;
}

u32 fixed_read_status(struct fmlcore_private *fml_priv)
{
    u32 mgsr, mgsr2;
    int count = 0;
    
    ahbfix_lock_net();

    mgsr = fixed_read(fml_priv, MGSR_ADDR);
    mgsr2 = fixed_read(fml_priv, MGSR_ADDR);
    while ((mgsr != mgsr2) && (count < 10)) {
        mgsr = mgsr2;
        mgsr2 = fixed_read(fml_priv, MGSR_ADDR);
        count++;
    } 
    
    ahbfix_unlock_net();
    return (count == 10) ? 0xffffffff : mgsr;
}

void fixed_write(u32 val, struct fmlcore_private* priv, u32 offset)
{
    ahbfix_write_block(&val, priv->phys_base, offset, 1, 0);
    return;
}

#endif 

/***
    the normal register read/write functions
***/
    
u32 direct_read(struct fmlcore_private * fml_priv, u32 offset)
{
    return readl(fml_priv->io_base + (offset << 2));
}

u32 direct_read_status(struct fmlcore_private * fml_priv)
{
    u32 mgsr, mgsr2;
    int count = 0;
    
    mgsr = readl(fml_priv->io_base + (MGSR_ADDR << 2));
    mgsr2 = readl(fml_priv->io_base + (MGSR_ADDR << 2));
    while ((mgsr != mgsr2) && (count < 10)) {
        mgsr = mgsr2;
        mgsr2 = readl(fml_priv->io_base + (MGSR_ADDR << 2));
        count++;
    } 
    return (count == 10) ? 0xffffffff : mgsr;
    
   // return readl(fml_priv->io_base + (MGSR_ADDR << 2));
}

void direct_write(u32 val, struct fmlcore_private * fml_priv, u32 offset)
{
    writel(val, fml_priv->io_base + (offset << 2));
}

void direct_read_block(u32 *buf, struct fmlcore_private * fml_priv, u32 offset, int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) buf[i] = readl(fml_priv->io_base + ((offset+i) << 2));
}

void direct_write_block(u32 *vals, struct fmlcore_private * fml_priv, u32 offset, int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) writel(vals[i], fml_priv->io_base + ((offset+i) << 2));
}

/*********************************************************************
 * calulates the fmlclock for a given prescale and sysclock value
 *********************************************************************/
static int get_fmlclk(int prescale , int sys_clk)
{
  int factor;
  switch (prescale)
    {
    case 0:
      factor = 6;
      break;
    case 1:
      factor = 5;
      break;
    default:
      factor = 4;
    }
  return ( sys_clk / ((prescale+1) * factor));
}

/* this function reset the finished indicator */
static inline void reset_finished(struct fmlcore_private* fml_priv)
{
    spin_lock(&fml_priv->fin_lock);
    fml_priv->finished = 0;
    spin_unlock(&fml_priv->fin_lock);
}

/*********************************************************************
 *function handles and clears a core interrupt
 *********************************************************************/
irqreturn_t fmlcore_irq_handler(int this_irq, void *dev_id, struct pt_regs *regs)
{
    struct fmlcore_private * fml_priv = (struct fmlcore_private *) dev_id;
    u32 temp;

    /*read status register */
    temp = fml_priv->read_statusreg(fml_priv);
    DBG2("FML: interrupt handler called, MGSR=0x%08x MCR0=0x%08x",
        temp, fml_priv->read_reg(fml_priv,MCR0_ADDR));

    if (temp & MCORE_IRQ) {
        /* it is a core interrupt, wake who ever is waiting for that */
        fml_priv->write_reg(MCOREIACK, fml_priv,MIACK_ADDR);
        /* set finished value */
        spin_lock(&fml_priv->fin_lock);
        fml_priv->finished = 1;
        spin_unlock(&fml_priv->fin_lock);
        wake_up_interruptible(&fml_priv->wait);      
    }

    if (temp & MFMLSL_IRQ){
      /* fml-slave requests a transaction, call the handler... */

      /* to handling a slave request takes long time! therefore first
       * disable those ints, handle them smothly and finaly enable them again...
       */
        
      /* clear interrupt source */
      fml_priv->slave_req_callback(fml_priv);      
      fml_priv->write_reg(MFMLIACK, fml_priv,MIACK_ADDR);
    }

    wake_up_interruptible(&fml_priv->wait);      

    if (temp & MERR) {
        DBG("FML timeout! (%08x)", temp);

        // generate stop condition if we are master
	if (!fml_priv->slave) {
	    fml_priv->write_reg(MSPM, fml_priv, MCR0_ADDR);
	    udelay(20); // can't use wait_while_tip here!
	}

        // core reset
ERR("ISR: timeout!!!\n");
        unsigned long mgr = fml_priv->read_reg(fml_priv,MGR_ADDR);
        fml_priv->write_reg(MRST,fml_priv,MGR_ADDR);
        fml_priv->write_reg(mgr, fml_priv, MGR_ADDR);
        fml_priv->write_reg(fml_priv->saddr, fml_priv, SAR_ADDR);
    }

    return IRQ_HANDLED;    
}

/*********************************************************************
 * function puts itself a sleep, it will be woken by the int-handler 
 * (in case interrupts are enabled, otherwise busy waiting for done
 * bit ni mgsr register...)
 *********************************************************************/
static int wait_while_tip(struct fmlcore_private *fml_priv, int timeout, volatile u32* pmgsr) 
{
        if (wait_event_interruptible_timeout(fml_priv->wait, fml_priv->finished, timeout) <= 0)
            return -1;

        if (pmgsr) {
	    *pmgsr = fml_priv->read_statusreg(fml_priv);
	    DBG2("returning status register value 0x%08x", *pmgsr);
	}

	return 0;
}

/*********************************************************************
 * function reads a fml-frame from slave and stores it in buffer1
 *
 * Note: buffer has to be at least 256 byte long!
 *********************************************************************/
static int fmlcore_read_block_internal(struct fmlcore_private * fml_priv,
		                       u8 command, u8 slave_addr,
		                       u8 *byte_count, u8 *data,
		                       int do_send_command)
{
    int ret=0;
    volatile u32 temp;
    u32 buf[256 / 4];
    int i;
    int buf_size;
    u8 *data_end;
    
    *byte_count=0;

    if (do_send_command) {
        DBG("FML: fmlcore_read_block() called with command 0x%02x", command);
            
        /* fill the command buffer */
        /* 32-bit mem cell has to look like:            */
        /*                                              */
        /* bitnum:     31    16|15    8|7           1|0 */
        /* content:    not used|command|slave address|0 */
        temp = 0
          | (slave_addr & 0x7f) << 1 //mask 7 bit and shift one left
          | command << 8;
    
        LOCK_NET;
    
        fml_priv->write_reg(temp,fml_priv, BUFF_ADDR(fml_priv, 1));
    
        /* set command reg */
        ISSUE_COMMAND(fml_priv, (MSTC|MWRC|(2<<MBC_SHIFT)|(0x1<<MBUFSEL_SHIFT)));
        
        /*wait for the Core to finish */
        if (wait_while_tip(fml_priv, MASTER_TIMEOUT, &temp) < 0) {
            WARN("FML: read_block: wait_while_tip failed while writing command");
            ret = -1;
            goto exit_locked;
        }
    
        if((temp & MERR)|| (temp & MAB)|| (GET_MBWD(temp) != 2)) {
            WARN("FML: read_block failed, slave did not accept command (MGSR 0x%08x)", temp);
            ret=-1;
            goto exit_locked;
        }
    } else {
        DBG("FML: fmlcore_read_block() called without command");

        LOCK_NET;
    }

    /* start slave-master transaction */
    /* 32-bit mem cell has to look like:            */
    /*                                              */
    /* bitnum:     31     8|7           1|0 */
    /* content:    not used|slave address|1 */
    temp = ((slave_addr & 0x7f) << 1) |0x1; //mask 7 bit and shift one left
    fml_priv->write_reg(temp,fml_priv, BUFF_ADDR(fml_priv, 1));

    /* set command reg */
    ISSUE_COMMAND(fml_priv, (MSTC|MWRC|(1<<MBC_SHIFT)|(0x1<<MBUFSEL_SHIFT)));

    /*wait for the Core to finish */
    if (wait_while_tip(fml_priv, MASTER_TIMEOUT, &temp) < 0) {
        WARN("FML: read_block: wait_while_tip failed while starting master read");
        ret = -1;
        goto exit_locked;
    }

    if((temp & MERR) || (temp & MAB) || (GET_MBWD(temp) != 1)) {
        WARN("FML: read_block failed, slave did not accept start of read transaction (MGSR 0x%08x)", temp);
        ret=-1;
        goto exit_locked;
    }

    /* set command reg (read, stop, take bytecount from slave, use buffer 1*/
    ISSUE_COMMAND(fml_priv, (MSPM|MRDC|BTCNT_EN|(0x1<<MBUFSEL_SHIFT)));

    // this may take a little bit longer, so don't block network anymore
    UNLOCK_NET;

    /*wait for the Core to finish */
    if (wait_while_tip(fml_priv, MASTER_TIMEOUT, &temp) < 0) {
        WARN("FML: read_block: wait_while_tip failed while master read");
        ret = -1;
        goto exit_unlocked;
    }

    LOCK_NET;

    if((temp & MERR) || (temp & MOFLW)) {
        DBG("FML: read_block failed after byte %d (MGSR 0x%08x)", GET_RD_BNTCNT(temp), temp);
        ret=-1;
        goto exit_locked;
    } 

    *byte_count = GET_RD_BNTCNT(temp) - 1; // minus Byte Count
    buf_size = (*byte_count + 3) >> 2; // u32 buf size
    data_end = data + *byte_count;

    /* sanity check: the maximum packet size we can copy is
        OPHIR_MAX_DATA_SIZE, so the maximum value of
        *byte_count is OPHIR_MAX_DATA_SIZE + 1 byte opcode */
    if (*byte_count > (OPHIR_MAX_DATA_SIZE+1)) {
        WARN("FML: received packet which was too large (%d bytes)", *byte_count);
        *byte_count = (OPHIR_MAX_DATA_SIZE+1);
    }

    /* get received */
    fml_priv->read_block(buf, fml_priv, BUFF_ADDR(fml_priv, 1), ((*byte_count + 3) >> 2));

    /* copy received data to external (callers) buffer
        (convert CPU byte order to LSB) */
    for (i = 0; i < buf_size; i++) {
        *(data++) = buf[i] >> 0;
        if (data < data_end) *(data++) = buf[i] >> 8;
        if (data < data_end) *(data++) = buf[i] >> 16;
        if (data < data_end) *(data++) = buf[i] >> 24;
    }

exit_locked:
    UNLOCK_NET;
exit_unlocked:
    return(ret);
}

int fmlcore_read_block(struct fmlcore_private * fml_priv, u8 command,
		       u8 slave_addr, u8 *byte_count, u8 *data) {
    return fmlcore_read_block_internal(fml_priv, command, slave_addr, byte_count, data, 1);
}

int fmlcore_read_block_simple(struct fmlcore_private * fml_priv, u8 slave_addr,
		              u8 *byte_count, u8 *data) {
    return fmlcore_read_block_internal(fml_priv, 0, slave_addr, byte_count, data, 0);
}


/*********************************************************************
 * function writes a fml-frame to buffer0 and sends it to the slave
 *********************************************************************/
int fmlcore_write_block(struct fmlcore_private * fml_priv,
			   u8 command,
			   u8 slave_addr,
			   u8 byte_count,
			   const u8 * data)
{
    int buf_size; //in dwords (32_bit)
    int i,ret=0; 

    u32 temp[256 / 4];
    volatile u32 mgsr;
    const u8 * my_data = data, *my_data_end = data + byte_count;

    DBG("FML: fmlcore_write_block() called with command 0x%02x, byte_count 0x%02x",
        command, byte_count);

    /* RAM buffer size calculation */
    //+3 for dword allignment, +3 for addr+cmd+bc!
    buf_size = (byte_count + 3 + 3) >> 2; 

    /* fill the buffer properly */

    /* first 32-bit mem cell has to look like:                           */
    /*                                                                   */
    /* bitnum:        31           24|23      16|15    8|7           1|0 */
    /* content:       first data byte|byte_count|command|slave address|0 */
    temp[0] = ((slave_addr & 0x7f) << 1) //mask 7 bit and shift one left
            | (command << 8)
            | (byte_count << 16)
            | ((*(my_data++) & 0xff) << 24);

    /* copy data at the end of buffer (convert LSB to CPU byte order) */
    for (i = 1; i < buf_size; i++) {
        temp[i]  = (*(my_data++) << 0);
        if (my_data < my_data_end) temp[i] |= (*(my_data++) << 8);
        if (my_data < my_data_end) temp[i] |= (*(my_data++) << 16);
        if (my_data < my_data_end) temp[i] |= (*(my_data++) << 24);
    }

    LOCK_NET;

    fml_priv->write_block(temp, fml_priv, BUFF_ADDR(fml_priv,0), buf_size);

    /* set command reg */
    ISSUE_COMMAND(fml_priv, (MSTC|MWRC|MSPM|((byte_count+3)<<MBC_SHIFT)));
    
    // this may take a little bit longer, so don't block network anymore
    UNLOCK_NET;

    /*wait for the Core to finish */
    if (wait_while_tip(fml_priv, MASTER_TIMEOUT, &mgsr) < 0) {
        WARN("FML: write_block: wait_while_tip failed while master write");
        ret = -1;
        goto exit_with_stop;
    }

    if((mgsr & MERR) || (mgsr & MAB) || (GET_MBWD(mgsr) != byte_count+3)) {
        WARN("FML: write_block failed after byte %d (MGSR 0x%08x)", GET_MBWD(mgsr), mgsr);
        ret=-1;
        goto exit_with_stop;
    }
    goto exit_unlocked; // skip additional stop condition

exit_with_stop:

    LOCK_NET;

    ISSUE_COMMAND(fml_priv, (MSPM));
    if (wait_while_tip(fml_priv, MASTER_TIMEOUT, &mgsr) < 0) {
        WARN("FML: write_block: wait_while_tip failed while generating stop condition");
        ret = -1;
        goto exit_locked;
    }
    
    if((mgsr & MERR) || (mgsr & MAB)) {
        WARN("FML: write_block failed after stop condition (MGSR 0x%08x)", mgsr);
        ret=-1;
        goto exit_locked;
    }

exit_locked:
    UNLOCK_NET;

exit_unlocked:
    return(ret);
}

/*********************************************************************
 * function initiates transfer of data to FML master
 *********************************************************************/
int fmlcore_slave_write(struct fmlcore_private * fml_priv, const unsigned char *data, int size)
{
    int i, retval = 0;
    int timeout = HZ;
    const unsigned long *ddata = (const unsigned long*)data;
    int dsize = size / 4; /* down-rounded! */
    u32 msgr;

    down(&fml_priv->slave_write_sem);

    DBG2("FML slave write %d bytes: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x",
        size, data[0], data[1], data[2], data[3], data[4], data[5], data[6], data[7], data[8], data[9], data[10]);

    if (size > SBUF_SIZE) size = SBUF_SIZE;

    for (i = 0; i < dsize; i++) {
        fml_priv->write_reg(ddata[i], fml_priv, SBUF_WR_OFFSET + i);
    }

    /* not dword-aligned (avoid seg faults) */
    if (size & 3) {
        unsigned long l = 0;
        memcpy(&l, data + (size & ~3), size & 3);
        fml_priv->write_reg(l, fml_priv, SBUF_WR_OFFSET + i);
    }

    if (wait_event_interruptible_timeout(fml_priv->wait, (!(fml_priv->read_statusreg(fml_priv) & MTRANS)), timeout <= 0)) {
        /*wait for potentially running read transaction */
        WARN("FML: fmlcore_slave_write: core busy");
        REGS(fml_priv, "core busy");
	retval = -1;
	goto out;
    }

    REGS(fml_priv, "pre write cmd");

    /* set command reg */
    ISSUE_COMMAND(fml_priv, (MWRC | (size << MBC_SHIFT)));
    REGS(fml_priv, "post write cmd");

    /*wait for the Core to finish */
    if (wait_while_tip(fml_priv, SLAVE_TIMEOUT, &msgr) < 0) {
        //if we timed out here, we have to check if all bytes
        //were sent - in this case no retransmit is neccessary
        REGS(fml_priv, "timeout");
        int swbc;
        retval = ((swbc = fml_priv->read_reg(fml_priv, SWBC_ADDR)) == size) ? size : -1;
        WARN("FML: fmlcore_slave_write timed out (%d bytes sent)", swbc);
        goto out;
    }

    if (msgr & MERR) {
        // core reset
        unsigned long mgr = fml_priv->read_reg(fml_priv,MGR_ADDR);
        fml_priv->write_reg(MRST,fml_priv,MGR_ADDR);
        fml_priv->write_reg(mgr, fml_priv, MGR_ADDR);
        fml_priv->write_reg(fml_priv->saddr, fml_priv, SAR_ADDR);

        WARN("FML: fmlcore_slave_write failed");
        WARN("FML: status reg: 0x%08x, written bytes: %d",
                (unsigned int)msgr, fml_priv->read_reg(fml_priv, SWBC_ADDR));
	retval = -1;
	goto out;
    }

    REGS(fml_priv, "post write cmd");
    retval = fml_priv->read_reg(fml_priv, SWBC_ADDR);
    DBG("fml_slave_write: wrote %d bytes", retval);

out:
    up(&fml_priv->slave_write_sem);
    return retval;
}

/*********************************************************************
 * function returns data from a completed transfer by FML master
 *********************************************************************/
int fmlcore_slave_read(struct fmlcore_private * fml_priv, unsigned char *data, int bufsize)
{
    int i;
    unsigned long size = fml_priv->read_reg(fml_priv, SRBC_ADDR);
    unsigned long *ddata = (unsigned long*)data;
    int dsize = size / 4; /* down-rounded! */

    DBG("FML slave read: size = %lu, bufsize = %d", size, bufsize);

    if (size > bufsize) return -1;

    for (i = 0; i < dsize; i++) {
        ddata[i] = fml_priv->read_reg(fml_priv, SBUF_RD_OFFSET + i);
    }

    /* not dword-aligned (avoid seg faults) */
    if (size & 3) {
        unsigned long l = fml_priv->read_reg(fml_priv, SBUF_RD_OFFSET + i);
        memcpy(data + (size & ~3), &l, size & 3);
    }

    return size;
}

/*********************************************************************
 * function initializes the FML-Core, it takes the resource 
 * information from fml_priv
 *********************************************************************/
int fmlcore_init(struct fmlcore_private * fml_priv)
{
    int prescale = fml_priv->slave ? 0 : 2; // lesser values will not work due to tripple-latched inputs for master mode

#ifdef __arm__    
    fml_priv->kira_rev = pp_kira_get_revision();
    DBG("FML: detected KIRA revision %d.%d", KIRA_MAJOR(fml_priv->kira_rev), KIRA_MINOR(fml_priv->kira_rev));
    if (KIRA_MAJOR(fml_priv->kira_rev) < 2) {
        /* we need the dma or mmu hack in this case */
        WARN("using AHB workaround!");
        fml_priv->write_reg = fixed_write;
        fml_priv->read_reg = fixed_read;
        fml_priv->read_statusreg = fixed_read_status;
        fml_priv->read_block = fixed_read_block;
        fml_priv->write_block = fixed_write_block;

        fml_priv->buf_offset = MONLY_BUF_OFFSET;
        fml_priv->buf_size = MONLY_BUF_SIZE;
    } else
#endif
    {
        fml_priv->write_reg = direct_write;
        fml_priv->read_reg = direct_read;
        fml_priv->read_statusreg = direct_read_status;
        fml_priv->read_block = direct_read_block;
        fml_priv->write_block = direct_write_block;

        fml_priv->buf_offset = MS_BUF_OFFSET;
        fml_priv->buf_size = MS_BUF_SIZE;
    } 

    spin_lock_init(&fml_priv->fin_lock);
    sema_init(&fml_priv->slave_write_sem, 1);
    fml_priv->finished = 1; 
    
    /* probing */
    if (fml_priv->read_reg(fml_priv, MID1_ADDR) != FML_ID1
        || fml_priv->read_reg(fml_priv, MID2_ADDR) != FML_ID2)
        return -1;

    DBG("FML: found FML-Core at 0x%08x", (unsigned int)fml_priv->io_base);

    if (!fml_priv 
	|| !fml_priv->name
	|| !fml_priv->irq
	|| !fml_priv->io_base
	|| (fml_priv->fml_clk <= 0)
	|| (fml_priv->sys_clk <= 0)
	|| !fml_priv->slave_req_callback)
      return -EINVAL; 

    /* prer has to be set according to the following equation     */
    /* prer = (f     / (4 * f   )   ) -1                          */
    /*          core         fml                                  */
    /*                                                            */
    if (get_fmlclk(prescale, fml_priv->sys_clk) < fml_priv->fml_clk) {
      DBG("FML: system clock too low for FML full clock of %dkHz", fml_priv->fml_clk);
    } else {
      while (get_fmlclk(prescale, fml_priv->sys_clk) > fml_priv->fml_clk)
        prescale++;
    }

    DBG("FML: use prescale of %d which leads to a FML-Clock of %d at system clock of %d)",
            prescale, get_fmlclk(prescale, fml_priv->sys_clk), fml_priv->sys_clk);

    init_waitqueue_head(&fml_priv->wait);

    /* reset FSM in Core... just in case... take care not to write
     * anything in the core prior that!
     *
     * FIXME: maybe it would be save to wait a while here...
     */
    fml_priv->write_reg(MRST,fml_priv,MGR_ADDR);
    
    /*set initial register values */
    fml_priv->write_reg(prescale,fml_priv,MPRER_ADDR);     

    /* set the master timeout to ~10msecs */
    if (!fml_priv->slave) {
	fml_priv->write_reg(0xfffff, fml_priv, MTOUT_ADDR);
    }
    
    /*enable FSM in Core and interrupts!... */
    fml_priv->write_reg(MGO | MALLIEN | MCOREIEN | MFMLIEN 
                      | (fml_priv->slave ? 0 : MS), fml_priv, MGR_ADDR);
    fml_priv->write_reg(fml_priv->saddr, fml_priv, SAR_ADDR);
    
    return 0;
}

int fmlcore_release(struct fmlcore_private * fml_priv)
{
    fml_priv->write_reg(MRST,fml_priv, MGR_ADDR);
    return 0;
}

/*
EXPORT_SYMBOL(fml_write_block);
EXPORT_SYMBOL(fml_read_block);
EXPORT_SYMBOL(fml_read_reg);
EXPORT_SYMBOL(fml_write_reg);
EXPORT_SYMBOL(fml_core_release);
EXPORT_SYMBOL(fml_core_init);
EXPORT_SYMBOL(fml_core_probe);
EXPORT_SYMBOL(fml_irq_handler);
*/
