/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  CIM Test Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"
#include "cim.h"
#include "paragon_cim.h"
#include "vm_cim.h"
#include "vm_lpbk.h"

#define DOMINION_CIM                 0
#define PARAGON_CIM                  1


int cim_test_paragon_port( int fd )
{
    int link;
    int line_start, line_end;
    int target_port;
    int type;
    int txpar, rxpar;
    int protocol = FPD_PROTOCOL_PARAGON;
    int retry, passed;
    int rc;
    
    printf("Enter Starting Target Port: ");
    scanf("%d", &line_start);
    printf("Enter Ending Target Port  : ");
    scanf("%d", &line_end);
    if( line_start > line_end ) {
        fprintf(stderr, "ERROR: Invalid params\n");
        return -1;
    }
    printf("Enter 0=DCIM or 1=PCIM    : ");
    scanf("%d", &type);
    if( type == DOMINION_CIM ) {
        txpar = 0;
        rxpar = 0;
    }
    else if( type == PARAGON_CIM ) {
        txpar = 1;
        rxpar = 1;
    }
    else {
        fprintf(stderr, "ERROR: Invalid params\n");
        return -1;
    }

    system("date");
    printf("\n");

    /* turn off RxPkt display */
    paragon_cim_set_display_rxpkt(0);

    for( link = 0; link < max_link_if; link++ ) {
        /* make sure that all links are disconnected */
        cim_disconnect_link(fd, link);

        /* initialize command table for each link */
        paragon_cim_init(fd, link, 0);
    }
    cim_disconnect_link(fd, FPD_LINK_IF_ID_BGND);
    paragon_cim_init(fd, FPD_LINK_IF_ID_BGND, 0);

    for( target_port = line_start; target_port <= line_end; target_port++ ) {
        for( link = 0; link < max_link_if; link++ ) {
            /* connect Link IF to specified target port */
            printf("Switching Link IF %d to Target %d...........", link, target_port);
            if( cim_switch_link(fd, link, target_port, protocol, txpar, rxpar) == 0 ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf("FPD_SWITCH_CHANNEL ioctl failed : %s\n", strerror(errno) );
                }
                return -2;
            }

            /* get Paragon device type */
            retry = 0;
            passed = 0;
            do {
                rc = paragon_cim_send_echo(fd, link, NULL);
                switch( rc ) {
                    case  0:
                        passed = 1;
                        break;
                    case -2:
                        if( retry > 10 ) {
                            printf("Echo request failed\n");
                            cim_disconnect_link(fd, link);
                            return -3;
                        }
                        break;
                    default:
                        printf("Echo request failed\n");
                        cim_disconnect_link(fd, link);
                        return -3;
                }
                retry++;
            } while( !passed );

            /* ask for the channel name */
            if((rc = paragon_cim_get_channel_name(fd, link)) != 0 ) {
                printf("Get Channel Name failed (rc=%d)\n", rc);
                cim_disconnect_link(fd, link);
                return -4;
            }

            /* disconnect Link IF from target port */
            printf("Disconnecting Link IF %d from Target %d.....", link, target_port);
            if( cim_disconnect_link(fd, link) == 0 ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf("FPD_DISCONNECT_CHANNEL ioctl failed : %s\n", strerror(errno) );
                }
                return -5;
            }
        }


        if( bgnd_link_if_present ) {
            link = FPD_LINK_IF_ID_BGND;

            /* connect Link IF to specified target port */
            printf("Switching BGND Link IF to Target %d........", target_port);
            if( cim_switch_link(fd, link, target_port, protocol, txpar, rxpar) == 0 ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf("FPD_SWITCH_CHANNEL ioctl failed : %s\n", strerror(errno) );
                }
                return -2;
            }

            /* get Paragon device type */
            retry = 0;
            passed = 0;
            do {
                rc = paragon_cim_send_echo(fd, link, NULL);
                switch( rc ) {
                    case  0:
                        passed = 1;
                        break;
                    case -2:
                        if( retry > 10 ) {
                            printf("Echo request failed\n");
                            cim_disconnect_link(fd, link);
                            return -3;
                        }
                        break;
                    default:
                        printf("Echo request failed\n");
                        cim_disconnect_link(fd, link);
                        return -3;
                }
                retry++;
            } while( !passed );

            /* ask for the channel name */
            if((rc = paragon_cim_get_channel_name(fd, link)) != 0 ) {
                printf("Get Channel Name failed (rc=%d)\n", rc);
                cim_disconnect_link(fd, link);
                return -4;
            }

            /* disconnect Link IF from target port */
            printf("Disconnecting BGND Link IF from Target %d..", target_port);
            if( cim_disconnect_link(fd, link) == 0 ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf("FPD_DISCONNECT_CHANNEL ioctl failed : %s\n", strerror(errno) );
                }
                return -5;
            }
        }

        if( target_port < line_end ) {
            int done, response;
            
            fflush(stdin);
            do {
                printf("\rDo you want to continue to port %d (y/n): ", target_port+1);
                response = getchar();
                switch( response ) {
                    case 'y':
                    case 'Y':
                    case 'n':
                    case 'N':
                        done = 1;
                        break;
                    default:
                        done = 0;
                        break;
                }
            } while( !done );

            if( response == 'n' || response == 'N' ) {
                break;
            }
        }
    }

    system("date");

    return 0;
}

int cim_test_vm_port_loopback( int fd, int link, int target_port, int btype, direction_t dir, int option )
{
    int rc = 0;
    int protocol = FPD_PROTOCOL_VM;
    unsigned char opt[8];

    /* initialize data pattern */
    vm_lpbk_init();

    if( dir == SEND ) {
        /* connect Link IF to specified target port */
        printf("Switching Link IF %d to Target %d...........", link, target_port);
        if( cim_switch_link(fd, link, target_port, protocol, 0, 0) == 0 ) {
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("FPD_SWITCH_CHANNEL ioctl failed : %s\n", strerror(errno) );
                printf("Make sure that no other Link Interface is using the target\n");
            }
            return -1;
        }

        /* turn off printouts in vm_cim.c */
        vm_cim_set_display(0);

        /* place VM CIM into loopback test mode */
        printf("Placing VM CIM into loopback mode.........");
        opt[0] = LPBK_TEST_CIM_BUFFER;
        if( vm_cim_setup_test_mode(TEST_LOOPBACK, 1, &opt[0]) == 0 ) {
            if( vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_TEST_MODE) == 0 ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                printf("ERROR: Invalid Test.\n");
                return -2;
            }
        }
        else {
            printf("FAILED.\n");
            printf("ERROR: Invalid Test.\n");
            return -3;
        }

        /* run test */
        rc = send_protocol_data_continuously(fd, link, FPD_CIM_BUFFER, option, 1);

        /* take VM CIM out of loopback test mode */
        printf("\nTaking VM CIM out of loopback mode........");
        opt[0] = LPBK_TEST_DISABLED;
        if( vm_cim_setup_test_mode(TEST_LOOPBACK, 1, &opt[0]) == 0 ) {
            vm_cim_setup_buffer(FPD_PRIORITY_BUFFER);
            vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_TEST_MODE);
            vm_cim_setup_buffer(FPD_CIM_BUFFER);
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
            printf("ERROR: Invalid Test.\n");
        }

        /* disconnect Link IF from target port */
        printf("Disconnecting Link IF %d from Target %d.....", link, target_port);
        if( cim_disconnect_link(fd, link) == 0 ) {
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("FPD_DISCONNECT_CHANNEL ioctl failed : %s\n", strerror(errno) );
            }
            return -5;
        }
    }
    else {
#if 1
        /*
         * This is what we want but there is an issue on the VM CIM side
         * at the moment.  It seems like the VM CIM is dropping packets once
         * once in a while.  Will need to be fixed first before enabling this
         * again.
         */
        rc = receive_protocol_data_continuously(fd, link, FPD_CIM_BUFFER, option, 1);
#else
        rc = receive_protocol_data_no_verification(fd, link, FPD_CIM_BUFFER, option, 1);
#endif
    }

    return rc;
}
