/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Virtual Media CIM Loopback Test Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>
#include <time.h>

#include "fpd_ioctl.h"
#include "cim.h"
#include "vm_lpbk.h"
#include "events.h"


static int one_pass_only = 0;
unsigned char ascending[MAX_PROTOCOL_BUFFER_SIZE];
unsigned char descending[MAX_PROTOCOL_BUFFER_SIZE];
unsigned char repeating[MAX_PROTOCOL_BUFFER_SIZE];
static unsigned char test_pattern[] = {0x00, 0x00, 0x00, 0x00,
                                       0x11, 0x11, 0x11, 0x11,
                                       0x22, 0x22, 0x22, 0x22,
                                       0x33, 0x33, 0x33, 0x33,
                                       0x44, 0x44, 0x44, 0x44,
                                       0x55, 0x55, 0x55, 0x55,
                                       0x66, 0x66, 0x66, 0x66,
                                       0x77, 0x77, 0x77, 0x77,
                                       0x88, 0x88, 0x88, 0x88,
                                       0x99, 0x99, 0x99, 0x99,
                                       0xaa, 0xaa, 0xaa, 0xaa,
                                       0xbb, 0xbb, 0xbb, 0xbb,
                                       0xcc, 0xcc, 0xcc, 0xcc,
                                       0xdd, 0xdd, 0xdd, 0xdd,
                                       0xee, 0xee, 0xee, 0xee,
                                       0xee, 0xee, 0xee, 0xee,
                                       0xff, 0xff, 0xff, 0xff,
                                       0xba, 0xbe, 0xfa, 0xce,
                                       0xf0, 0xf0, 0xf0, 0xf0,
                                       0x0f, 0x0f, 0x0f, 0x0f,
                                       0xa5, 0xa5, 0xa5, 0xa5,
                                       0x5a, 0x5a, 0x5a, 0x5a,
                                       0x3c, 0x3c, 0x3c, 0x3c,
                                       0xc3, 0xc3, 0xc3, 0xc3,
                                       0xde, 0xad, 0xba, 0xbe,
                                       0x01, 0x23, 0x45, 0x67,
                                       0x89, 0xab, 0xcd, 0xef,
                                       0xfe, 0xdc, 0xba, 0x98,
                                       0x76, 0x54, 0x32, 0x10,
                                       0xff, 0x00, 0xff, 0x00,
                                       0x00, 0xff, 0x00, 0xff};


int vm_lpbk_init( void )
{
    int num_patterns;
    int i;

    /* initialize patterns */
    num_patterns = sizeof(test_pattern)/sizeof(test_pattern[0]);
    for( i = 0; i < MAX_PROTOCOL_BUFFER_SIZE; i++ ) {
        ascending[i] = i;
        descending[i] = MAX_PROTOCOL_BUFFER_SIZE - 1 - i;
        repeating[i] = test_pattern[i % num_patterns];
    }

    return 0;
}

static void dump_data( unsigned char *bufaddr, int pktlen )
{
    int i;

    fprintf(stderr, "Data:\n");
    for( i = 0; i < pktlen; i++ ) {
        fprintf(stderr, "%02x ", bufaddr[i]);
    }
    fprintf(stderr, "\n");

    return;
}

int verify_data( int type, unsigned char *pbuf, int len, pkt_pattern_t pattern, int round )
{
    int rc, i;

    /* compare contents against expected data */
    switch( pattern ) {
        case PKT_PATTERN_8BIT_ASCENDING:
            rc = memcmp((const void *)pbuf, (const void *)&ascending[0], len);
            if( rc != 0 ) {
                for( i = 0; i < len; i++ ) {
                    if( pbuf[i] != ascending[i] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, ascending[i], pbuf[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_8BIT_DESCENDING:
            rc = memcmp((const void *)pbuf, (const void *)&descending[0], len);
            if( rc != 0 ) {
                for( i = 0; i < len; i++ ) {
                    if( pbuf[i] != descending[i] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, descending[i], pbuf[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_8BIT_REPEATING:
            rc = memcmp((const void *)pbuf, (const void *)&repeating[0], len);
            if( rc != 0 ) {
                for( i = 0; i < len; i++ ) {
                    if( pbuf[i] != repeating[i] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, repeating[i], pbuf[i]);
                    }
                }
            }
            break;

        default:
            fprintf(stderr, "FAILED\n");
            fprintf(stderr, "ERROR: Unknown Packet Pattern!\n");
            return -1;
    }

    if( rc != 0 ) {
        if( !debug_on ) {
            fprintf(stderr, "[%d]Comparing contents with expected data (%d bytes) from %s buffer....FAILED\n",
                    round, len, type?"Priority":"CIM");
        }
        fprintf(stderr, "Expected %d bytes of ", len);
        switch( pattern ) {
            case PKT_PATTERN_8BIT_ASCENDING:
                fprintf(stderr, "8-bit ascending ");
                break;
            case PKT_PATTERN_8BIT_DESCENDING:
                fprintf(stderr, "8-bit descending ");
                break;
            case PKT_PATTERN_8BIT_REPEATING:
                fprintf(stderr, "8-bit repeating ");
                break;
            default:
                break;
        }
        fprintf(stderr, "pattern but received different data\n");

        /* dump data */
        if( len > 0 ) {
            dump_data(pbuf, len);
        }
    }

    return rc;
}

int send_protocol_data_continuously( int fd, unsigned int link_if, int btype, int option, int dopt )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    int rc;
    int pktlen;
    int result = 0;
    time_t start_time, current_time;
    unsigned int loop = 0;

    if( option ) {
        one_pass_only = 1;
    }
    else {
        one_pass_only = 0;
    }

    if( dopt ) {
        printf("\n");
        system("date");
    }

    pdata->link_if = link_if;
    pdata->type = btype;

    while( 1 ) {
        loop++;

        if( !debug_on ) {
            printf("\n[%d]Sending %d packets to %s buffer......",
                   loop, FPD_MAX_PACKET_SIZE, pdata->type?"PRI":"CIM");
            fflush(stdout);
        }

        for( pktlen = 1; pktlen <= FPD_MAX_PACKET_SIZE; pktlen++ ) {
            pdata->requested_len = pktlen;

            /* determine test pattern to use */
            switch( pktlen % 3 ) {
                case 0:
                    pdata->buf = &ascending[0];
                    break;
                case 1:
                    pdata->buf = &descending[0];
                    break;
                default:
                    pdata->buf = &repeating[0];
                    break;
            }

            if( debug_on ) {
                printf("\n[%d]Sending %d bytes to %s buffer....",
                       loop, pdata->requested_len, pdata->type?"Priority":"CIM");
            }
resend:
            rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, pdata);
            if( rc == 0 ) {
                if( pdata->actual_len == pdata->requested_len ) {
                    if( debug_on ) {
                        printf("passed.");
                    }

                    /* reset time of last good send */
                    time(&start_time);
                }
                else {
                    time(&current_time);
                    if(( pdata->actual_len == 0 ) && ( difftime(current_time, start_time) < RETRY_TIMEOUT )) {
                        goto resend;
                    }
                    else {
                        printf("FAILED.\n");
                        if( !debug_on ) {
                            printf("\n[%d]Sending %d bytes to %s buffer....FAILED\n",
                                   loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                        }
                        printf("Actual bytes written %d\n", pdata->actual_len);
                        result = -1;
                        goto send_finish;
                    }
                }
            }
            else {
                if( errno ) {
                    if( errno == ENOSPC ) {
                        time(&current_time);
                        if( difftime(current_time, start_time) < RETRY_TIMEOUT ) {
                            goto resend;
                        }
                        else {
                            printf("FAILED.\n");
                            if( !debug_on ) {
                                printf("\n[%d]Sending %d bytes to %s buffer....FAILED\n",
                                       loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                            }
                            printf("ERROR: %s\n", strerror(errno));
                            result = -1;
                            goto send_finish;
                        }
                    }
                    else if( errno == ENOLINK ) {
                        printf("FAILED.\n");
                        if( !debug_on ) {
                            printf("\n[%d]Sending %d bytes to %s buffer....FAILED\n",
                                   loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                        }
                        printf("ERROR: %s\n", strerror(errno));
                        result = -1;
                        goto send_finish;
                    }
                }
            }
        }

        if( !debug_on ) {
            printf("passed.");
            fflush(stdout);
        }

        if( one_pass_only ) {
            break;
        }
    }

send_finish:
    if( dopt ) {
        printf("\n\n");
        system("date");
    }

    return result;
}

int receive_protocol_data_continuously( int fd, unsigned int link_if, int btype, int option, int dopt )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    unsigned int timeout = POLL_TIMEOUT;
    pkt_pattern_t test_pattern;
    unsigned int loop = 1;
    int pktlen;
    int rx_pdata;
    int more_data_available;
    int result = 0;
    int rc;

    if( option ) {
        one_pass_only = 1;
    }
    else {
        one_pass_only = 0;
    }

    if( dopt ) {
        printf("\n");
        system("date");
    }

    pktlen = 1;

    if( !debug_on ) {
        printf("\n[%d]Reading %d packets from %s buffer....",
               loop, FPD_MAX_PACKET_SIZE, btype?"PRI":"CIM");
        fflush(stdout);
    }
    else {
        printf("\n");
    }

    while(1) {
        more_data_available = 1;
        rx_pdata = 0;

        if( cim_poll_once(fd, timeout, pevt) != 0 ) {
            result = -1;
            goto receive_finish;
        }

        memset(&process, 0, sizeof(rx_notify_t));
        process.link_if = link_if;
        handle_events(fd, pevt, &process);

        /* check if FPGA receive protocol data */
        if( process.rx_cim ) {
            /* CIM Buffer */
            if( btype == FPD_CIM_BUFFER ) {
                rx_pdata = 1;
            }
        }
        else if( process.rx_priority ) {
            /* Priority buffer */
            if( btype == FPD_PRIORITY_BUFFER ) {
                rx_pdata = 1;
            }
        }

        if( rx_pdata ) {
            pdata->link_if = link_if;
            pdata->type = btype;
            do {
                pdata->requested_len = pktlen;
                pdata->buf = protocol_buffer;
                if( debug_on ) {
                    printf("[%d]Reading %d bytes of data from %s buffer....",
                           loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                }
                rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
                if( rc == 0 ) {
                    if( pdata->actual_len == pdata->requested_len ) {
                        if( debug_on ) {
                            printf("passed.\n");
                        }
                    }
                    else if( pdata->actual_len == 0 ) {
                        if( debug_on ) {
                            printf("No data.\n");
                        }
                        more_data_available = 0;
                    }
                    else {
                        printf("FAILED.\n");
                        if( !debug_on ) {
                            printf("[%d]Reading %d bytes of data from %s buffer....FAILED\n",
                                   loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                        }
                        printf("Requested %d, Got %d\n",
                               pdata->requested_len, pdata->actual_len);
                        result = -3;
                        goto receive_finish;
                    }

                    if( pdata->actual_len > 0 ) {
                        if( debug_on ) {
                            printf("Comparing contents with expected data......");
                        }

                        /* determine test pattern used */
                        switch( pktlen % 3 ) {
                            case 0:
                                test_pattern = PKT_PATTERN_8BIT_ASCENDING;
                                break;
                            case 1:
                                test_pattern = PKT_PATTERN_8BIT_DESCENDING;
                                break;
                            default:
                                test_pattern = PKT_PATTERN_8BIT_REPEATING;
                                break;
                        }
                        /* compare contents against expected data */
                        if( verify_data(pdata->type, pdata->buf, pdata->actual_len, test_pattern, loop) == 0 ) {
                            if( debug_on ) {
                                printf("passed.\n");
                            }
                        }
                        else {
                            result = -4;
                            goto receive_finish;
                        }
                    }

                    /* increment packet length */
                    if( pdata->actual_len == pdata->requested_len ) {
                        if( pktlen == FPD_MAX_PACKET_SIZE ) {
                            pktlen = 1;
                            loop++;
                            if( !debug_on ) {
                                printf("passed.");
                                if( !one_pass_only ) {
                                    printf("\n[%d]Reading %d packets from %s buffer....",
                                           loop, FPD_MAX_PACKET_SIZE, pdata->type?"Priority":"CIM");
                                }
                                fflush(stdout);
                            }

                            if( one_pass_only ) {
                                goto receive_finish;
                            } 
                        }
                        else {
                            ++pktlen;
                        }
                    }
                }
                else {
                    printf("FAILED.\n");
                    if( !debug_on ) {
                        printf("[%d]Reading %d bytes of data from %s buffer....FAILED\n",
                               loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                    }
                    if(errno) {
                        printf("ioctl failed : %s\n", strerror(errno));
                    }
                    result = -5;
                    goto receive_finish;
                }
            } while( more_data_available );
        }
    }

receive_finish:
    if( dopt ) {
        printf("\n\n");
        system("date");
    }

    return result;
}

int receive_protocol_data_no_verification( int fd, unsigned int link_if, int btype, int option, int dopt )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    unsigned int timeout = POLL_TIMEOUT;
    unsigned int loop = 1;
    int pktlen;
    int rx_pdata;
    int more_data_available;
    int result = 0;
    int rc;

    if( option ) {
        one_pass_only = 1;
    }
    else {
        one_pass_only = 0;
    }

    if( dopt ) {
        printf("\n");
        system("date");
    }

    pktlen = 1;

    if( !debug_on ) {
        printf("\n[%d]Reading %d packets from %s buffer....",
               loop, FPD_MAX_PACKET_SIZE, btype?"PRI":"CIM");
        fflush(stdout);
    }
    else {
        printf("\n");
    }

    while(1) {
        more_data_available = 1;
        rx_pdata = 0;

        if( cim_poll_once(fd, timeout, pevt) != 0 ) {
            result = -1;
            goto receive_finish2;
        }

        memset(&process, 0, sizeof(rx_notify_t));
        process.link_if = link_if;
        handle_events(fd, pevt, &process);

        /* check if FPGA receive protocol data */
        if( process.rx_cim ) {
            /* CIM Buffer */
            if( btype == FPD_CIM_BUFFER ) {
                rx_pdata = 1;
            }
        }
        else if( process.rx_priority ) {
            /* Priority buffer */
            if( btype == FPD_PRIORITY_BUFFER ) {
                rx_pdata = 1;
            }
        }

        if( rx_pdata ) {
            pdata->link_if = link_if;
            pdata->type = btype;
            do {
                pdata->requested_len = 512;
                pdata->buf = protocol_buffer;
                if( debug_on ) {
                    printf("[%d]Reading %d bytes of data from %s buffer....",
                           loop, pktlen, pdata->type?"Priority":"CIM");
                }
                rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
                if( rc == 0 ) {
                    if( pdata->actual_len > 0 ) {
                        /* increment packet length */
                        if( pktlen == FPD_MAX_PACKET_SIZE ) {
                            pktlen = 1;
                            loop++;
                            if( !debug_on ) {
                                printf("passed.");
                                if( !one_pass_only ) {
                                    printf("\n[%d]Reading %d packets from %s buffer....",
                                           loop, FPD_MAX_PACKET_SIZE, pdata->type?"Priority":"CIM");
                            }
                                fflush(stdout);
                            }
                            
                            if( one_pass_only ) {
                                goto receive_finish2;
                            } 
                        }
                        else {
                            ++pktlen;
                        }
                    }
                    else if( pdata->actual_len == 0 ) {
                        more_data_available = 0;
                    }
                }
                else {
                    printf("FAILED.\n");
                    if( !debug_on ) {
                        printf("[%d]Reading %d bytes of data from %s buffer....FAILED\n",
                               loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                    }
                    if(errno) {
                        printf("ioctl failed : %s\n", strerror(errno));
                    }
                    result = -5;
                    goto receive_finish2;
                }
            } while( more_data_available );
        }
    }

receive_finish2:
    if( dopt ) {
        printf("\n\n");
        system("date");
    }

    return result;
}
