/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Device Diagnostic Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"
#include "fpd-diagnostics.h"
#include "protocol.h"
#include "host.h"
#include "legacy_cim.h"

int debug_on = 0;
pkt_pattern_t pkt_pattern = PKT_PATTERN_8BIT_ASCENDING;
unsigned char *protocol_buffer;
unsigned char ascending[MAX_PROTOCOL_BUFFER_SIZE];
unsigned char descending[MAX_PROTOCOL_BUFFER_SIZE];
unsigned char repeating[MAX_PROTOCOL_BUFFER_SIZE];
unsigned int dma_page_size = 0;
int max_link_if = 0;
char bgnd_link_if_present = 0;

static FPD_msg_t gmsg;
static int max_line_if = 0;
static int max_host_chan = 0;
static unsigned int link_if = 0;
static line_connection_t **line_conn;
static line_connection_t *bgnd_line_conn = NULL;
static unsigned int host_chan = 0;
static unsigned int buffer_type = FPD_CIM_BUFFER;
static int pkt_size = 32;
static unsigned char test_pattern[] = {0x00, 0x00, 0x00, 0x00,
                                       0x11, 0x11, 0x11, 0x11,
                                       0x22, 0x22, 0x22, 0x22,
                                       0x33, 0x33, 0x33, 0x33,
                                       0x44, 0x44, 0x44, 0x44,
                                       0x55, 0x55, 0x55, 0x55,
                                       0x66, 0x66, 0x66, 0x66,
                                       0x77, 0x77, 0x77, 0x77,
                                       0x88, 0x88, 0x88, 0x88,
                                       0x99, 0x99, 0x99, 0x99,
                                       0xaa, 0xaa, 0xaa, 0xaa,
                                       0xbb, 0xbb, 0xbb, 0xbb,
                                       0xcc, 0xcc, 0xcc, 0xcc,
                                       0xdd, 0xdd, 0xdd, 0xdd,
                                       0xee, 0xee, 0xee, 0xee,
                                       0xee, 0xee, 0xee, 0xee,
                                       0xff, 0xff, 0xff, 0xff,
                                       0xba, 0xbe, 0xfa, 0xce,
                                       0xf0, 0xf0, 0xf0, 0xf0,
                                       0x0f, 0x0f, 0x0f, 0x0f,
                                       0xa5, 0xa5, 0xa5, 0xa5,
                                       0x5a, 0x5a, 0x5a, 0x5a,
                                       0x3c, 0x3c, 0x3c, 0x3c,
                                       0xc3, 0xc3, 0xc3, 0xc3,
                                       0xde, 0xad, 0xba, 0xbe,
                                       0x01, 0x23, 0x45, 0x67,
                                       0x89, 0xab, 0xcd, 0xef,
                                       0xfe, 0xdc, 0xba, 0x98,
                                       0x76, 0x54, 0x32, 0x10,
                                       0xff, 0x00, 0xff, 0x00,
                                       0x00, 0xff, 0x00, 0xff};


int fpd_init_diag( int fd )
{
    line_connection_t *pconn;
    int link;
    int num_patterns;
    int rc;
    int i;

    memset(&gmsg, 0, sizeof(FPD_msg_t));

    /* determine Link and Line Interface count */
    rc = ioctl(fd, FPD_GET_INFO, &gmsg);
    if( rc == 0 ) {
        max_link_if = gmsg.info.link_if_cnt;
        max_line_if = gmsg.info.line_cnt;
        max_host_chan = gmsg.info.host_buffer_cnt;
        dma_page_size = gmsg.info.host_buffer_pagesize;
        bgnd_link_if_present = gmsg.info.bgnd_link_if_present;
    }

    /* allocate memory for line connections */
    if( bgnd_link_if_present ) {
        bgnd_line_conn = malloc(sizeof(line_connection_t));
        if( bgnd_line_conn == NULL ) {
            fprintf(stderr, "bgnd_line_conn not allocated\n");
            return -1;
        }
    }

    line_conn = malloc(max_link_if * sizeof(line_connection_t *));
    if( line_conn ) {
        for( i = 0; i < max_link_if; i++ ) {
            line_conn[i] = malloc(sizeof(line_connection_t));
            if( line_conn[i] == NULL ) {
                fprintf(stderr, "line_conn[%d] not allocated\n", i);
                free(line_conn);
                free(bgnd_line_conn);
                return -3;
            }
        }
    }
    else {
        fprintf(stderr, "line_conn not allocated\n");
        free(bgnd_line_conn);
        return -2;
    }

    /* allocate buffer */
    protocol_buffer = (unsigned char *)malloc(MAX_PROTOCOL_BUFFER_SIZE);
    if( !protocol_buffer ) {
        for( i = 0; i < max_link_if; i++ ) {
            free(line_conn[i]);
        }
        free(line_conn);
        free(bgnd_line_conn);
        return -5;
    }

    /* initialize patterns */
    num_patterns = sizeof(test_pattern)/sizeof(test_pattern[0]);
    for( i = 0; i < MAX_PROTOCOL_BUFFER_SIZE; i++ ) {
        ascending[i] = i;
        descending[i] = MAX_PROTOCOL_BUFFER_SIZE - 1 - i;
        repeating[i] = test_pattern[i % num_patterns];
    }

    /* initialize host module */
    rc = init_host_module( fd, gmsg.info.host_buffer_memsize );
    if( rc < 0 ) {
        fprintf(stderr, "Host Module init failed\n");
        free(protocol_buffer);
        for( i = 0; i < max_link_if; i++ ) {
            free(line_conn[i]);
        }
        free(line_conn);
        free(bgnd_line_conn);
        return -4;
    }

    /* find out current switch info */
    rc = ioctl(fd, FPD_GET_SWITCH_INFO, &gmsg);
    if( rc == 0 ) {
        for( link = 0; link < max_link_if; link++ ) {
            pconn = line_conn[link];
            if( gmsg.switch_info.link[link].enabled ) {
                pconn->target_port = gmsg.switch_info.link[link].target_port;
                pconn->protocol = gmsg.switch_info.link[link].protocol;
                pconn->tx_parity = gmsg.switch_info.link[link].tx_parity;
                pconn->rx_parity = gmsg.switch_info.link[link].rx_parity;
            }
            else {
                pconn->target_port = -1;
                pconn->protocol = -1;
                pconn->tx_parity = -1;
                pconn->rx_parity = -1;
            }
        }

        if( bgnd_link_if_present ) {
            if( gmsg.switch_info.bgnd_link.enabled ) {
                bgnd_line_conn->target_port = gmsg.switch_info.bgnd_link.target_port;
                bgnd_line_conn->protocol = gmsg.switch_info.bgnd_link.protocol;
                bgnd_line_conn->tx_parity = gmsg.switch_info.bgnd_link.tx_parity;
                bgnd_line_conn->rx_parity = gmsg.switch_info.bgnd_link.rx_parity;
            }
            else {
                bgnd_line_conn->target_port = -1;
                bgnd_line_conn->protocol = -1;
                bgnd_line_conn->tx_parity = -1;
                bgnd_line_conn->rx_parity = -1;
            }
        }
    }
    else {
        for( link = 0; link < max_link_if; link++ ) {
            pconn = line_conn[link];
            pconn->target_port = -1;
            pconn->protocol = -1;
            pconn->tx_parity = -1;
            pconn->rx_parity = -1;
        }

        if( bgnd_link_if_present ) {
            bgnd_line_conn->target_port = -1;
            bgnd_line_conn->protocol = -1;
            bgnd_line_conn->tx_parity = -1;
            bgnd_line_conn->rx_parity = -1;
        }
    }

    return 0;
}

int fpd_cleanup_diag( void )
{
    int i;

    /* free allocated memory */
    for( i = 0; i < max_link_if; i++ ) {
        free(line_conn[i]);
    }
    free(line_conn);
    free(bgnd_line_conn);
    free(protocol_buffer);
    cleanup_host_module();

    return 0;
}

int fpd_get_current_link_if( void )
{
    return link_if;
}

int fpd_get_link_if_protocol( void )
{
    line_connection_t *pconn;

    if( link_if < max_link_if ) {
        pconn = line_conn[link_if];
    }
    else if ( link_if == FPD_LINK_IF_ID_BGND ) {
        pconn = bgnd_line_conn;
    }
    else {
        return -1;
    }

    return pconn->protocol;
}

void fpd_display_info( int fd )
{
    int rc;

    rc = ioctl(fd, FPD_GET_INFO, &gmsg);
    if( rc == 0 ) {
        printf("Protocol Version                    : %d.%d.%d\n"
               "FPGA Version                        : %d.%d.%d\n"
               "Driver Version                      : %d.%d.%d\n"
               "Number of Link Interfaces           : %d\n"
               "Background Link Interface Present   : %s\n"
               "Number of Lines                     : %d\n"
               "Number of Host Buffers per Link IF  : %d\n"
               "Host Buffers Memory Size            : %d\n"
               "Number of Pages used in Host Buffer : %d\n"
               "Page size used in Host Buffer       : %d\n",
               gmsg.info.protocol_version.major,
               gmsg.info.protocol_version.minor,
               gmsg.info.protocol_version.patch,
               gmsg.info.fpga_version.major,
               gmsg.info.fpga_version.minor,
               gmsg.info.fpga_version.patch,
               gmsg.info.driver_version.major,
               gmsg.info.driver_version.minor,
               gmsg.info.driver_version.patch,
               gmsg.info.link_if_cnt,
               (gmsg.info.bgnd_link_if_present ? "yes" : "no"),
               gmsg.info.line_cnt,
               gmsg.info.host_buffer_cnt,
               gmsg.info.host_buffer_memsize,
               gmsg.info.host_buffer_pagecnt,
               gmsg.info.host_buffer_pagesize);
    }
    else {
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
        else {
            printf( "ioctl failed : Invalid argument\n" );
        }
    }

    return;
}

void fpd_display_pci_cfg( int fd )
{
    int rc;
    FPD_pcicfg_t *pci = &gmsg.pcicfg;

    rc = ioctl(fd, FPD_GET_PCI_CFG, &gmsg);
    if( rc == 0 ) {
        printf("PCI_VENDOR_ID          : %04x\n", pci->vendor_id);
        printf("PCI_DEVICE_ID          : %04x\n", pci->device_id);
        printf("PCI_COMMAND            : %04x\n", pci->command);
        printf("PCI_STATUS             : %04x\n", pci->status);
        printf("PCI_REVISION_ID        : %02x\n", pci->revision_id);
        printf("PCI_CLASS_PROG         : %02x\n", pci->prog_if);
        printf("PCI_CLASS_DEVICE       : %02x\n", pci->sub_class_code);
        printf("PCI_CLASS_CODE         : %02x\n", pci->base_class_code);
        printf("PCI_CACHE_LINE_SIZE    : %02x\n", pci->cache_line_size);
        printf("PCI_LATENCY_TIMER      : %02x\n", pci->latency_timer);
        printf("PCI_HEADER_TYPE        : %02x\n", pci->header_type);
        printf("PCI_BIST               : %02x\n", pci->bist);
        printf("PCI_BASE_ADDRESS_0     : %08x\n", pci->bar0);
        printf("PCI_BASE_ADDRESS_1     : %08x\n", pci->bar1);
        printf("PCI_BASE_ADDRESS_2     : %08x\n", pci->bar2);
        printf("PCI_BASE_ADDRESS_3     : %08x\n", pci->bar3);
        printf("PCI_BASE_ADDRESS_4     : %08x\n", pci->bar4);
        printf("PCI_BASE_ADDRESS_5     : %08x\n", pci->bar5);
        printf("PCI_CARDBUS_CIS        : %08x\n", pci->cardbus_cis_ptr);
        printf("PCI_SUBSYSTEM_VENDOR_ID: %04x\n", pci->subsystem_vendor_id);
        printf("PCI_SUBSYSTEM_ID       : %04x\n", pci->subsystem_id);
        printf("PCI_ROM_ADDRESS        : %08x\n", pci->rom_bar);
        printf("PCI_CAPABILITY_LIST    : %02x\n", pci->cap_ptr);
        printf("PCI_INTERRUPT_LINE     : %02x\n", pci->interrupt_line);
        printf("PCI_INTERRUPT_PIN      : %02x\n", pci->interrupt_pin);
        printf("PCI_MIN_GNT            : %02x\n", pci->min_grant);
        printf("PCI_MAX_LAT            : %02x\n", pci->max_latency);
    }
    else {
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
        else {
            printf( "ioctl failed : Invalid argument\n" );
        }
    }

    return;
}

static int get_read_reg_param( FPD_rw_reg_t *cfg )
{
    printf("\nEnter Parameters\n");
    printf("Register: 0x");
    scanf("%x", &cfg->reg);

    return 0;
}

static int get_write_reg_param( FPD_rw_reg_t *cfg )
{
    printf("\nEnter Parameters\n");
    printf("Register: 0x");
    scanf("%x", &cfg->reg);
    printf("Value   : 0x");
    scanf("%x", &cfg->value);

    return 0;
}

int fpd_read_reg( int fd )
{
    int rc;

    get_read_reg_param(&gmsg.rwreg);

    rc = ioctl(fd, FPD_READ_REG, &gmsg);
    if( rc == 0 ) {
        printf("Value   : %08x\n", gmsg.rwreg.value);
    }
    else {
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
        else {
            printf( "ioctl failed : Invalid argument\n" );
        }
    }

    return 0;
}

int fpd_write_reg( int fd )
{
    int rc;

    get_write_reg_param(&gmsg.rwreg);
    rc = ioctl(fd, FPD_WRITE_REG, &gmsg);
    if( rc < 0 ) {
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
        else {
            printf( "ioctl failed : Invalid argument\n" );
        }
    }

    return 0;
}

int fpd_get_line_connection_info( int link_if, line_connection_t **pconn )
{
    if( link_if < max_link_if ) {
        *pconn = line_conn[link_if];
    }
    else if( link_if == FPD_LINK_IF_ID_BGND ) {
        *pconn = bgnd_line_conn;
    }
    else {
        fprintf(stderr, "\nERROR: Invalid Link Interface specified\n");
        *pconn = NULL;
        return -1;
    }

    return 0;
}

int fpd_set_line_connection_info( int link_id, int target, int protocol, int txpar, int rxpar )
{
    line_connection_t *pconn;

    if( link_id < max_link_if ) {
        pconn = line_conn[link_id];
    }
    else if ( link_id == FPD_LINK_IF_ID_BGND ) {
        pconn = bgnd_line_conn;
    }
    else {
        return -1;
    }

    pconn->target_port = target;
    pconn->protocol = protocol;
    pconn->tx_parity = txpar;
    pconn->rx_parity = rxpar;

    return 0;
}

int fpd_disconnect_link( int fd, int link_id )
{
    int rc;

    gmsg.disc.link_if = link_id;
    gmsg.disc.driver_txbuf_wait_time = 0;
    gmsg.disc.fpga_txbuf_wait_time = 0;

    rc = ioctl(fd, FPD_DISCONNECT_CHANNEL, &gmsg);
    if( rc == 0 ) {
        fpd_set_line_connection_info(link_id, -1, -1, -1, -1);
    }

    return rc;
}

int fpd_switch_link( int fd, int link_id, int target, int protocol, int txpar, int rxpar )
{
    int rc;

    gmsg.conn.link_if = link_id;
    gmsg.conn.target_port = target;
    gmsg.conn.protocol = protocol;
    gmsg.conn.tx_parity = txpar;
    gmsg.conn.rx_parity = rxpar;
    gmsg.conn.driver_txbuf_wait_time = 0;
    gmsg.conn.fpga_txbuf_wait_time = 0;

    rc = ioctl(fd, FPD_SWITCH_CHANNEL, &gmsg);

    if( rc == 0 ) {
        fpd_set_line_connection_info(link_id, target, protocol, txpar, rxpar);
    }

    return rc;
}

int fpd_reset_linkif( int fd, int link_id )
{
    gmsg.reset_link.link_if = link_id;
    return( ioctl(fd, FPD_RESET_LINK_INTERFACE, &gmsg) );
}

int fpd_select_masterslave( int fd )
{
    int rc;
    unsigned int val;

    /* find out current setting */
    rc = ioctl(fd, FPD_GET_LINK0_MODE, &gmsg);
    if( rc == 0 ) {
        printf("\nCurrent setting                      : ");
        if( gmsg.link0mode.mode == FPD_LINK0_MODE_MASTER )
            printf("Master\n");
        else
            printf("Slave\n");
    }
    else {
        if(errno) {
            printf("FPD_GET_LINK0_MODE ioctl failed : %s\n", strerror(errno));
        }
        return -1;
    }

    printf("Enter Master [1] or Slave [0] Setting: ");
    scanf("%d", &val);
    if(( val == 0 ) || ( val == 1 ) ) {
        if( val == 0 )
            gmsg.link0mode.mode = FPD_LINK0_MODE_SLAVE;
        else
            gmsg.link0mode.mode = FPD_LINK0_MODE_MASTER;
        rc = ioctl(fd, FPD_SET_LINK0_MODE, &gmsg);
        if( rc != 0 ) {
            if(errno) {
                printf("FPD_SET_LINK0_MODE ioctl failed : %s\n", strerror(errno));
            }
            return -1;
        }
    }
    else {
        printf("\nERROR: Invalid entry\n");
        printf("Please re-do!!!\n");
        return -1;
    }

    return 0;
}

int fpd_get_switch_info( int fd )
{
    FPD_switchinfo_t *pinfo;
    int id;
    int rc;

    rc = ioctl(fd, FPD_GET_SWITCH_INFO, &gmsg);
    if (rc != 0) {
        return -1;
    }

    pinfo = &gmsg.switch_info;

    printf("\nSwitch Information:\n");
    for( id = 0; id < pinfo->link_if_cnt; id++ ) {
        printf("Link IF %x    <--> Target %d ",
               pinfo->link[id].link_if,
               pinfo->link[id].target_port);
        if( pinfo->link[id].enabled ) {
            printf("[");
            switch( pinfo->link[id].protocol ) {
                case FPD_PROTOCOL_VM:
                    printf("VM protocol");
                    break;
                case FPD_PROTOCOL_PARAGON:
                    printf("Paragon protocol, ");
                    switch( pinfo->link[id].tx_parity ) {
                        case FPD_PARAGON_PARITY_EVEN:
                            printf("Even TX Parity, ");
                            break;
                        case FPD_PARAGON_PARITY_ODD:
                            printf("Odd TX Parity, ");
                            break;
                        default:
                            printf("Unknown TX Parity, ");
                            break;
                    }
                    switch( pinfo->link[id].rx_parity ) {
                        case FPD_PARAGON_PARITY_EVEN:
                            printf("Even RX Parity");
                            break;
                        case FPD_PARAGON_PARITY_ODD:
                            printf("Odd RX Parity");
                            break;
                        default:
                            printf("Unknown RX Parity");
                            break;
                    }
                    break;
                default:
                    printf("Unspecified protocol");
                    break;
            }
            printf("]\n");
        }
        else {
            printf("(disabled)\n");
        }
    }

    if( pinfo->bgnd_link_if_present ) {
        printf("BGND Link IF <--> Target %d ",
               pinfo->bgnd_link.target_port);
        if( pinfo->bgnd_link.enabled ) {
            printf("[");
            switch( pinfo->bgnd_link.protocol ) {
                case FPD_PROTOCOL_VM:
                    printf("VM protocol");
                    break;
                case FPD_PROTOCOL_PARAGON:
                    printf("Paragon protocol, ");
                    switch( pinfo->bgnd_link.tx_parity ) {
                        case FPD_PARAGON_PARITY_EVEN:
                            printf("Even TX Parity, ");
                            break;
                        case FPD_PARAGON_PARITY_ODD:
                            printf("Odd TX Parity, ");
                            break;
                        default:
                            printf("Unknown TX Parity, ");
                            break;
                    }
                    switch( pinfo->bgnd_link.rx_parity ) {
                        case FPD_PARAGON_PARITY_EVEN:
                            printf("Even RX Parity");
                            break;
                        case FPD_PARAGON_PARITY_ODD:
                            printf("Odd RX Parity");
                            break;
                        default:
                            printf("Unknown RX Parity");
                            break;
                    }
                    break;
                default:
                    printf("Unspecified protocol");
                    break;
            }
            printf("]\n");
        }
        else {
            printf("(disabled)\n");
        }
    }

    return 0;
}

static void display_link_properties( line_connection_t *plink )
{
    switch( plink->protocol ) {
        case FPD_PROTOCOL_VM:
            printf("VM\n");
            break;
        case FPD_PROTOCOL_PARAGON:
            printf("Paragon\n");
            printf("Current TX Parity selected                             : ");
            switch( plink->tx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                    printf("Even\n");
                    break;
                case FPD_PARAGON_PARITY_ODD:
                    printf("Odd\n");
                    break;
                default:
                    printf("Unknown\n");
                    break;
            }
            printf("Current RX Parity selected                             : ");
            switch( plink->rx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                    printf("Even\n");
                    break;
                case FPD_PARAGON_PARITY_ODD:
                    printf("Odd\n");
                    break;
                case FPD_PARAGON_PARITY_IGNORE:
                    printf("Ignored\n");
                    break;
                default:
                    printf("Unknown\n");
                    break;
            }
            break;
        default:
            printf("Unknown!\n");
            break;
    }

    return;
}

void fpd_display_current_setup( void )
{
    line_connection_t *pconn;

    printf("\nCurrent Link Interface selected                        : ");
    if( link_if < max_link_if ) {
        printf("%d\n", link_if);
        pconn = line_conn[link_if];
    }
    else if ( link_if == FPD_LINK_IF_ID_BGND ) {
        printf("Background\n");
        pconn = bgnd_line_conn;
    }
    else {
        printf("None\n");
        pconn = NULL;
    }

    printf("Current Protocol Type selected                         : ");
    if( link_if < max_link_if || link_if == FPD_LINK_IF_ID_BGND ) {
        display_link_properties(pconn);
    }
    else {
        printf("None\n");
    }

    printf("Current Line selected [-1 = Disconnected]              : ");
    if( link_if < max_link_if || link_if == FPD_LINK_IF_ID_BGND ) {
        printf("%d\n", pconn->target_port);
    }
    else {
        printf("None\n");
    }

    return;
}

int fpd_select_linkif( void )
{
    unsigned int val;

    printf("\nEnter Link Interface [0-%d] or Background Link [%d]   : ",
           max_link_if - 1, FPD_LINK_IF_ID_BGND);
    scanf("%d", &val);
    if( val < max_link_if ) {
        link_if = val;
    }
    else if ( val == FPD_LINK_IF_ID_BGND ) {
        if( bgnd_link_if_present ) {
            link_if = val;
        }
        else {
            printf("\nERROR: Background Link Interface is not present\n");
            printf("Please re-do!!!\n");
            return -1;
        }
    }
    else {
        printf("\nERROR: Invalid Link Interface ID %d\n", val);
        printf("Please re-do!!!\n");
        return -1;
    }

    return 0;
}

int fpd_select_linkif_and_line( int fd, int link_id )
{
    unsigned int val;
    unsigned int cim_protocol, tx_parity, rx_parity;
    int line;

    if( link_id >= max_link_if ) {
        if ( link_id == FPD_LINK_IF_ID_BGND ) {
            if( !bgnd_link_if_present ) {
                printf("\nERROR: No BGND Link Interface found\n");
                printf("Please re-do!!!\n");
                return -1;
            }
        }
        else {
            printf("\nERROR: Invalid Link Interface ID %d\n", link_id);
            printf("Please re-do!!!\n");
            return -1;
        }
    }

    printf("Enter Protocol Type [0=VM, 1=Paragon]                  : ");
    scanf("%d", &val);
    switch( val ) {
        case FPD_PROTOCOL_VM:
            cim_protocol = FPD_PROTOCOL_VM;
            break;
        case FPD_PROTOCOL_PARAGON:
            cim_protocol = FPD_PROTOCOL_PARAGON;
            printf("Enter TX Parity [0=Even, 1=Odd]                        : ");
            scanf("%d", &tx_parity);
            switch( tx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                case FPD_PARAGON_PARITY_ODD:
                    break;
                default:
                    printf("\nERROR: Invalid Parity\n");
                    printf("Please re-do!!!\n");
                    return -3;
            }

            printf("Enter RX Parity [0=Even, 1=Odd, 2=Ignore]              : ");
            scanf("%d", &rx_parity);
            switch( rx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                case FPD_PARAGON_PARITY_ODD:
                case FPD_PARAGON_PARITY_IGNORE:
                    break;
                default:
                    printf("\nERROR: Invalid Parity\n");
                    printf("Please re-do!!!\n");
                    return -4;
            }
            break;
        default:
            printf("\nERROR: Invalid Protocol Type\n");
            printf("Please re-do!!!\n");
            return -2;
    }

    printf("Enter Line for connection [0-%02d] or -1 to disconnect   : ",
           max_line_if - 1);
    scanf("%d", &line);

    if(( line >= max_line_if ) || ( line < -1 )) {
        printf("\nERROR: Invalid Line ID %d\n", line);
        printf("Please re-do!!!\n");
        return -5;
    }

    /* process switch or disconnect */
    if( line == -1 ) {
        if( link_id < max_link_if ) {
            printf("\nDisconnecting Link IF %d....", link_id);
        }
        else {
            printf("\nDisconnecting Background Link IF....");
        }

        if( fpd_disconnect_link(fd, link_id) == 0 ) {
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("FPD_DISCONNECT_CHANNEL ioctl failed : %s\n", strerror(errno) );
            }
            return -6;
        }
    }
    else {
        if( link_id < max_link_if ) {
            printf("\nSwitching Link IF %d to Line %d....", link_id, line);
        }
        else {
            printf("\nSwitching Background Link IF to Line %d....", line);
        }

        if( fpd_switch_link(fd, link_id, line, cim_protocol, tx_parity, rx_parity) == 0 ) {
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("FPD_SWITCH_CHANNEL ioctl failed : %s\n", strerror(errno) );
            }
            return -7;
        }
    }

    return 0;
}

int fpd_write_buffer( int fd )
{
    int btype;
    int rc;

    printf("Enter Buffer Type (0=CIM, 1=Priority, 2=Host): ");
    scanf("%d", &btype);

    switch( btype ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            rc = write_protocol_data_manually(fd, link_if, btype);
            break;
        case FPD_HOST_BUFFER:
            rc = 0;
            break;
        default:
            rc = -1;
    }

    return rc;
}

int fpd_read_buffer( int fd )
{
    int btype;
    int rc;

    printf("\nEnter Buffer Type (0=CIM, 1=Priority, 2=Host): ");
    scanf("%d", &btype);

    switch( btype ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            rc = read_protocol_data_manually(fd, link_if, btype);
            break;
        case FPD_HOST_BUFFER:
            rc = read_host_data_manually(fd, link_if, host_chan);
            break;
        default:
            rc = -1;
    }

    return rc;
}

unsigned int fpd_get_buffer_type( void )
{
    return buffer_type;
}

void fpd_display_packet_info( int fd )
{
    printf("\nCurrent Buffer                 : ");
    switch( buffer_type ) {
        case FPD_CIM_BUFFER:
            printf("CIM");
            break;
        case FPD_PRIORITY_BUFFER:
            printf("Priority");
            break;
        case FPD_HOST_BUFFER:
            printf("Host");
            get_host_configuration(fd, link_if, host_chan);
            break;
        default:
            printf("UNKNOWN");
            break;
    }
    printf("\nCurrent Packet Size            : ");
    switch( buffer_type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
        case FPD_HOST_BUFFER:
            printf("%d", pkt_size);
            break;
        default:
            printf("Invalid buffer");
            break;
    }
    printf("\nCurrent Packet Pattern         : ");
    switch( pkt_pattern ) {
        case PKT_PATTERN_8BIT_ASCENDING:
            printf("8-bit ascending\n");
            break;
        case PKT_PATTERN_8BIT_DESCENDING:
            printf("8-bit descending\n");
            break;
        case PKT_PATTERN_8BIT_REPEATING:
            printf("8-bit repeating\n");
            break;
        case PKT_PATTERN_16BIT_ASCENDING:
            printf("16-bit ascending\n");
            break;
        case PKT_PATTERN_16BIT_DESCENDING:
            printf("16-bit descending\n");
            break;
        case PKT_PATTERN_32BIT_ASCENDING:
            printf("32-bit ascending\n");
            break;
        case PKT_PATTERN_32BIT_DESCENDING:
            printf("32-bit descending\n");
            break;
        default:
            printf("UNKNOWN\n");
            break;
    }

    return;
}

int fpd_change_buffer_type( int fd )
{
    unsigned int val;
    int opt;
    int response;
    int done;
    int i = 0;

    printf("\nBuffer Type:");
    printf("\n============");
    printf("\n\t %d: CIM", i++);
    printf("\n\t %d: Priority", i++);
    printf("\n\t %d: Host", i++);
    printf("\n\nPlease choose: ");
    scanf("%d", &opt);

    switch( opt ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            break;
        case FPD_HOST_BUFFER:
            printf("\nEnter Host Channel ID (0-based, %d-channel(s))   : ",
                   max_host_chan);
            scanf("%d", &val);
            if( val >= max_host_chan ) {
                fprintf(stderr, "\nERROR: Invalid channel ID\n");
                return -1;
            }
            host_chan = val;

            fflush(stdin);
            do {
                printf("\rDo you want to change Host Buffer settings (y/n): ");
                response = getchar();
                switch( response ) {
                    case 'y':
                    case 'Y':
                    case 'n':
                    case 'N':
                        done = 1;
                        break;
                    default:
                        done = 0;
                        break;
                }
            } while( !done );

            if( response == 'y' || response == 'Y' ) {
                if(set_host_configuration(fd, link_if, host_chan) != 0) {
                    fprintf(stderr, "\nERROR: Failed to config Host\n");
                    return -2;
                }
            }
            break;
        default:
            fprintf(stderr, "\nERROR: Invalid buffer selection!\n");
            return -3;
    }
    buffer_type = opt;

    return 0;
}

int fpd_change_packet( int fd )
{
    unsigned int val;
    int i = 0;
    int opt;
    int max_size;

    switch( buffer_type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            max_size = FPD_MAX_PACKET_SIZE;
            break;
        case FPD_HOST_BUFFER:
            max_size = MAX_HOST_BUFFER_SIZE;
            break;
        default:
            fprintf(stderr, "\nERROR: Invalid buffer selection!\n");
            return -1;
    }

    printf("\nEnter New Packet Size (%d max)  : ", max_size);
    scanf("%d", &val);
    if( val > max_size ) {
        printf("\nERROR: Invalid size\n");
        return -2;
    }
    pkt_size = val;

    i = 0;
    printf("\nPacket Pattern:");
    printf("\n===============");
    printf("\n\t %d: 8-bit Ascending", i++);
    printf("\n\t %d: 8-bit Descending", i++);
    printf("\n\t %d: 8-bit Repeating", i++);
    if( buffer_type == FPD_HOST_BUFFER ) {
        printf("\n\t %d: 16-bit Ascending", i++);
        printf("\n\t %d: 16-bit Descending", i++);
        printf("\n\t %d: 32-bit Ascending", i++);
        printf("\n\t %d: 32-bit Descending", i++);
    }
    printf("\n\nPlease choose: ");
    scanf("%d", &opt);

    switch( opt ) {
        case PKT_PATTERN_8BIT_ASCENDING:
        case PKT_PATTERN_8BIT_DESCENDING:
        case PKT_PATTERN_8BIT_REPEATING:
            pkt_pattern = opt;
            break;
        case PKT_PATTERN_16BIT_ASCENDING:
        case PKT_PATTERN_16BIT_DESCENDING:
        case PKT_PATTERN_32BIT_ASCENDING:
        case PKT_PATTERN_32BIT_DESCENDING:
            if( buffer_type == FPD_HOST_BUFFER ) {
                pkt_pattern = opt;
            }
            else {
                printf("\nERROR: Invalid Pattern selection!\n");
                return -5;
            }
            break;
        default:
            printf("\nERROR: Invalid Pattern selection!\n");
            return -5;
    }

    return 0;
}

int fpd_poll_once( int fd, FPD_event_t *pevent )
{
    unsigned int timeout = -1;
    struct pollfd fds;
    int rc;

    fds.fd = fd;
    fds.events = POLLIN;

    fprintf(stderr, "\nwaiting for events ... ");

    rc = poll(&fds, 1, timeout);
    if ((rc == 0) || (rc < 0 && errno == EINTR)) {
        printf("timedout\n");
        return -1;
    }
    else if( rc < 0 ) {
        fprintf(stderr, "poll() returned %d\n", rc);
        return -2;
    }

    fprintf(stderr, "receive notifications\n");
    fprintf(stderr, "getting events ... ");
    rc = ioctl(fd, FPD_CHECK_EVENTS, pevent);
    fprintf(stderr, "done\n");
    if (rc < 0) {
        return -1;
    }

    return 0;
}

int fpd_send_pkt( int fd )
{
    int rc;

    switch( buffer_type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            rc = send_protocol_data(fd, link_if, buffer_type, pkt_size);
            break;
        case FPD_HOST_BUFFER:
            rc = send_host_data(fd, link_if, host_chan, pkt_size);
            break;
        default:
            rc = -1;
    }

    return rc;
}

int fpd_receive_pkt( int fd )
{
    int rc;

    switch( buffer_type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            rc = receive_protocol_data(fd, link_if, buffer_type, pkt_size);
            break;
        case FPD_HOST_BUFFER:
            rc = receive_host_data(fd, link_if, host_chan, pkt_size);
            break;
        default:
            rc = -1;
    }

    return rc;
}

static void display_stats( FPD_stats_t *pinfo )
{
    if( pinfo->link_if == FPD_LINK_IF_ID_BGND ) {
        printf("\nBackground Link IF:\n");
    }
    else {
        printf("\nLink IF %d:\n", pinfo->link_if);
    }

    printf("  RX Statistics\n");
    printf("\tBuffer Full   : %d\n", pinfo->rx.buf_full_cnt);
    printf("\tProtocol Error: %d\n", pinfo->rx.protocol_err_cnt);
    printf("\tCRC Error     : %d\n", pinfo->rx.crc_err_cnt);
    printf("\tInvalid Packet: %d\n", pinfo->rx.inv_pkt_cnt);
    printf("\tParity Error  : %d\n", pinfo->rx.parity_err_cnt);
    printf("\tSequence Error: %d\n", pinfo->rx.seq_err_cnt);
    printf("\tTimeout       : %d\n", pinfo->rx.timeout_cnt);
    printf("\tNoise         : %d\n", pinfo->rx.noise_cnt);

    printf("  TX Statistics\n");
    printf("\tMax Retry     : %d\n", pinfo->tx.max_retry);
    printf("\tBuffer Full   : %d\n", pinfo->tx.buf_full_cnt);
    printf("\tProtocol Error: %d\n", pinfo->tx.protocol_err_cnt);
    printf("\tCRC Error     : %d\n", pinfo->tx.crc_err_cnt);

    return;
}

int fpd_get_stats( int fd )
{
    int rc;

    gmsg.stats.link_if = link_if;
    rc = ioctl(fd, FPD_GET_STATISTICS, &gmsg);
    if (rc == 0) {
        /* display info */
        display_stats(&gmsg.stats);
    }
    else {
        printf("FAILED.\n");
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
    }

    return 0;
}

static void display_events( FPD_event_t *pevt )
{
    int link;
    int id;

    printf("\nEvents:\n");

    /* Link Interfaces */
    for( link = 0; link < max_link_if; link++ ) {
        /* Host buffer */
        for( id = 0; id < FPD_HOST_CNT; id++ ) {
            if( pevt->link_if[link].host[id].has_data ) {
                printf("Link IF %d require Host Buffer %d processing\n",
                       link, id);
            }
        }

        /* Priority buffer */
        if( pevt->link_if[link].rx_priority ) {
            printf("Link IF %d receive Priority Buffer data\n", link);
        }

        /* CIM buffer */
        if( pevt->link_if[link].rx_cim ) {
            printf("Link IF %d receive CIM Buffer data\n", link);
        }

        /* error detected */
        if( pevt->link_if[link].error ) {
            printf("Link IF %d receive errors\n", link);
        }

        /* echo response detected */
        if( pevt->link_if[link].echo_rsp ) {
            printf("\tLink IF %d receive echo response\n", link);
        }
    }

    /* Background Link Interface events */
    if( pevt->bgnd_link_if.rx_cim ) {
        printf("Bacground Link IF receive CIM Buffer data\n");
    }

    if( pevt->bgnd_link_if.error ) {
        printf("Background Link IF receive errors\n");
    }

    if( pevt->bgnd_link_if.echo_rsp ) {
        printf("\tBackground Link IF receive echo response\n");
    }

    if( pevt->pci_error ) {
        printf("Detected PCI error\n");
    }

    if( pevt->cim_detect ) {
        printf("Detected changes to CIM topology\n");
    }

    return;
}

int fpd_check_events( int fd )
{
    int rc;

    rc = ioctl(fd, FPD_CHECK_EVENTS, &gmsg);
    if (rc == 0) {
        /* display info */
        display_events(&gmsg.event);
    }
    else {
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
    }

    return 0;
}

static int fpd_get_invalid_cmd( int fd, int link_if )
{
    FPD_rxinvcmd_t invalid_cmd;
    int rc;

    invalid_cmd.link_if = link_if;
    rc = ioctl(fd, FPD_GET_RX_INVALID_CMD, &invalid_cmd);
    if (rc == 0) {
        if( invalid_cmd.n_entries > 0 ) {
            int i;

            for( i = 0; i < invalid_cmd.n_entries; i++ ) {
                printf("\t\t\tCMD 0x%02x\n", invalid_cmd.inv_cmd[i]);
            }
        }
        else {
            printf("\t\t\tWEIRD!! No entries found.\n");
        }
    }
    else {
        if(errno) {
            printf("FPD_GET_RX_INVALID_CMD ioctl failed : %s\n", strerror(errno));
        }
    }

    return 0;
}

static void display_error( int fd, FPD_error_t *pinfo )
{
    switch( pinfo->type ) {
        case FPD_ERR_TYPE_PCI:
            printf("\nPCI Error: %08x\n", pinfo->error);
            if( pinfo->error ) {
                if( pinfo->error & PCI_ERR_PARITY ) {
                    printf("       Data Parity Detected\n");
                }
                else if( pinfo->error & PCI_ERR_SIG_TARGET_ABORT ) {
                    printf("       Signaled Target Abort\n");
                }
                else if( pinfo->error & PCI_ERR_RCV_TARGET_ABORT ) {
                    printf("       Received Target Abort\n");
                }
                else if( pinfo->error & PCI_ERR_RCV_MASTER_ABORT ) {
                    printf("       Received Master Abort\n");
                }
                else if( pinfo->error & PCI_ERR_SIG_SYSTEM_ERROR ) {
                    printf("       Signaled System Error\n");
                }
                else if( pinfo->error & PCI_ERR_DETECTED_PARITY ) {
                    printf("       Detected Parity Error\n");
                }
            }
            else {
                printf("       NONE\n");
            }
            break;

        case FPD_ERR_TYPE_LINK_IF:
            if( pinfo->link_if == FPD_LINK_IF_ID_BGND ) {
                printf("\nBackground Link IF Error: %08x\n", pinfo->error);
            }
            else {
                printf("\nLink IF %d Error: %08x\n",
                       pinfo->link_if, pinfo->error);
            }

            if( pinfo->error ) {
                if( pinfo->error & FPD_ERR_RX_INV_CMD ) {
                    printf("\t\tFPD_ERR_RX_INV_CMD\n");

                    /* retrieve the invalid command */
                    fpd_get_invalid_cmd(fd, pinfo->link_if);
                }
                if( pinfo->error & FPD_ERR_UPDATE_INV_PKT ) {
                    printf("\t\tFPD_ERR_UPDATE_INV_PKT\n");
                }
                if( pinfo->error & FPD_ERR_RX_PRI_RD_EMPTY_BUF ) {
                    printf("       FPD_ERR_RX_PRI_RD_EMPTY_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_SYNC ) {
                    printf("       FPD_ERR_TX_PRI_SYNC\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_NO_EOP ) {
                    printf("       FPD_ERR_TX_PRI_NO_EOP\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_WR_FULL_BUF ) {
                    printf("       FPD_ERR_TX_PRI_WR_FULL_BUF\n");
                }
                if( pinfo->error & FPD_ERR_RX_CIM_RD_EMPTY_BUF ) {
                    printf("       FPD_ERR_RX_CIM_RD_EMPTY_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_SYNC ) {
                    printf("       FPD_ERR_TX_CIM_SYNC\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_NO_EOP ) {
                    printf("       FPD_ERR_TX_CIM_NO_EOP\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_WR_FULL_BUF ) {
                    printf("       FPD_ERR_TX_CIM_WR_FULL_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_TIMEOUT ) {
                    printf("       FPD_ERR_TX_PRI_TIMEOUT\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_TIMEOUT ) {
                    printf("       FPD_ERR_TX_CIM_TIMEOUT\n");
                }
                if( pinfo->error & FPD_ERR_TX_HOST_TIMEOUT_0 ) {
                    printf("       FPD_ERR_TX_HOST_TIMEOUT_0\n");
                }
                if( pinfo->error & FPD_ERR_TX_HOST_TIMEOUT_1 ) {
                    printf("       FPD_ERR_TX_HOST_TIMEOUT_1\n");
                }
                if( pinfo->error & FPD_ERR_RX_DMA_NO_BUF_0 ) {
                    printf("       FPD_ERR_RX_DMA_NO_BUF_0\n");
                }
                if( pinfo->error & FPD_ERR_RX_DMA_NO_BUF_1 ) {
                    printf("       FPD_ERR_RX_DMA_NO_BUF_1\n");
                }
            }
            else {
                printf("       NONE\n");
            }
            break;

        default:
            break;
    }

    return;
}

int fpd_get_link_errors( int fd )
{
    int rc;

    gmsg.error.type = FPD_ERR_TYPE_LINK_IF;
    gmsg.error.link_if = link_if;
    rc = ioctl(fd, FPD_GET_ERROR, &gmsg);
    if (rc == 0) {
        /* display info */
        display_error(fd, &gmsg.error);
    }
    else {
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
    }

    return 0;
}

int fpd_get_pci_errors( int fd )
{
    int rc;

    gmsg.error.type = FPD_ERR_TYPE_PCI;
    gmsg.error.link_if = 0; /* don't care */
    rc = ioctl(fd, FPD_GET_ERROR, &gmsg);
    if (rc == 0) {
        /* display info */
        display_error(fd, &gmsg.error);
    }
    else {
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
    }

    return 0;
}

int fpd_set_dma_burst_size( int fd )
{
    int rc;
    int val;

    printf("\nEnter DMA burst size (8, 16, 32, 64 words) : ");
    scanf("%d", &val);
    gmsg.dma_burst.size = val;
    rc = ioctl(fd, FPD_SET_DMA_BURST_SIZE, &gmsg);
    if (rc != 0) {
        if(errno) {
            fprintf(stderr, "ERROR: ioctl failed : %s\n", strerror(errno));
        }
        return -1;
    }

    return 0;
}

int fpd_get_dma_burst_size( int fd )
{
    int rc;

    rc = ioctl(fd, FPD_GET_DMA_BURST_SIZE, &gmsg);
    if (rc == 0) {
        printf("\nDMA burst size : %d\n", gmsg.dma_burst.size);
    }
    else {
        if(errno) {
            fprintf(stderr, "ERROR: ioctl failed : %s\n", strerror(errno));
        }
        return -1;
    }

    return 0;
}

static void fpd_display_diag_errors( unsigned int result )
{
    switch( result ) {
        case FPD_DIAG_ERROR_NONE:
            printf("No errors found.\n");
            break;
        case FPD_DIAG_ERROR_INVALID_PKT_LENGTH:
            printf("ERROR: Invalid Packet Length specified\n");
            break;
        case FPD_DIAG_ERROR_INVALID_BUFFER_TYPE:
            printf("ERROR: Invalid Buffer Type specified\n");
            break;
        case FPD_DIAG_ERROR_INVALID_LINKIF:
            printf("ERROR: Invalid Link Interface specified\n");
            break;
        case FPD_DIAG_ERROR_INVALID_TEST:
            printf("ERROR: Invalid Diagnostics specified\n");
            break;
        case FPD_DIAG_ERROR_MEM_ALLOC_FAILED:
            printf("ERROR: Memory allocation failed\n");
            break;
        case FPD_DIAG_ERROR_USER_MEM_FAULT:
            printf("ERROR: User memory fault\n");
            break;
        case FPD_DIAG_ERROR_FPGA_BUFFER_NO_SPACE:
            printf("ERROR: No space left in FPGA buffer\n");
            break;
        case FPD_DIAG_ERROR_TX_TIMEOUT_NOT_SET:
            printf("ERROR: TX CIM/PRI TIMEOUT bit did not get set\n");
            break;
        case FPD_DIAG_ERROR_INCORRECT_TX_BYTE_AVAIL:
            printf("ERROR: Incorrect TX byte available\n");
            break;
        case FPD_DIAG_ERROR_WRONG_TX_CRC_CNT:
            printf("ERROR: Wrong TX CRC count\n");
            break;
        case FPD_DIAG_ERROR_TX_CRC_CNT_NOT_CLEARED:
            printf("ERROR: TX CRC count did not get cleared\n");
            break;
        case FPD_DIAG_ERROR_WRONG_MAX_RETRY:
            printf("ERROR: Wrong Max Retry count\n");
            break;
        case FPD_DIAG_ERROR_MAX_RETRY_NOT_CLEARED:
            printf("ERROR: Max retry count did not get cleared\n");
            break;
        case FPD_DIAG_ERROR_WRONG_TX_PROT_ERR_CNT:
            printf("ERROR: Wrong TX Protocol error count\n");
            break;
        case FPD_DIAG_ERROR_TX_PROT_ERR_CNT_NOT_CLEARED:
            printf("ERROR: TX Protocol error count did not get cleared\n");
            break;
        case FPD_DIAG_ERROR_WRONG_RX_TIMEOUT_CNT:
            printf("ERROR: Wrong RX Timeout count\n");
            break;
        case FPD_DIAG_ERROR_RX_TIMEOUT_CNT_NOT_CLEARED:
            printf("ERROR: RX Timeout count did not get cleared\n");
            break;
        case FPD_DIAG_ERROR_TX_BUF_NOT_SET:
            printf("ERROR: TX CIM/PRI BUF bit did not get set\n");
            break;
        case FPD_DIAG_ERROR_TX_BYTE_AVAIL_NOT_512:
            printf("ERROR: TX byte available did not go back to 512\n");
            break;
        case FPD_DIAG_ERROR_WRONG_TX_BUF_FULL_CNT:
            printf("ERROR: Wrong TX Buffer full count\n");
            break;
        case FPD_DIAG_ERROR_TX_BUF_FULL_CNT_NOT_CLEARED:
            printf("ERROR: TX Buffer full count did not get cleared\n");
            break;
        default:
            printf("ERROR: Unknown fault\n");
    }

    return;
}

static int fpd_run_diag_receive_packet( int fd, int btype, int pktlen, int diag_code )
{
    FPD_event_t *pevt;
    int rx_pdata = 0;
    int num_rcv_pkt = 0;
    int rc;

    /*
     *  Expected to receive ascending pattern
     */
wait_event:
    if( fpd_poll_once(fd, &gmsg.event) == 0 ) {
        /* check if FPGA receive protocol data */
        pevt = &gmsg.event;

        if( link_if == FPD_LINK_IF_ID_BGND ) {
            if( btype == FPD_CIM_BUFFER ) {
                /* CIM Buffer */
                if( pevt->bgnd_link_if.rx_cim ) {
                    rx_pdata = 1;
                }
            }
        }
        else {
            if( btype == FPD_CIM_BUFFER ) {
                /* CIM Buffer */
                if( pevt->link_if[link_if].rx_cim ) {
                    rx_pdata = 1;
                }
            }
            else {
                /* Priority buffer */
                if( pevt->link_if[link_if].rx_priority ) {
                    rx_pdata = 1;
                }
            }
        }
    }

    if( rx_pdata ) {
get_packet:
        ++num_rcv_pkt;
        gmsg.pdata.link_if = link_if;
        gmsg.pdata.requested_len = pktlen;
        gmsg.pdata.type = btype;
        gmsg.pdata.buf = protocol_buffer;
        memset(protocol_buffer, 0, MAX_PROTOCOL_BUFFER_SIZE);
        rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, &gmsg);
        if( rc == 0 ) {
            int i;

            if( gmsg.pdata.actual_len != gmsg.pdata.requested_len ) {
                if(( diag_code == FPD_DIAG_TEST_BUFFER_FULL ) && ( num_rcv_pkt > 1 )) {
                    /* need to read twice but there's no more data, poll again */
                    goto wait_event;
                }

                printf("FAILED.\n");
                printf("Requested %d, Got %d\n",
                       gmsg.pdata.requested_len, gmsg.pdata.actual_len);
                return -2;
            }

            /* compare contents against expected data */
            printf("Comparing contents with expected data......");
            rc = memcmp((const void *)protocol_buffer, (const void *)&ascending[0], pktlen);

            if( rc == 0 ) {
                printf("passed.\n");

                if(( diag_code == FPD_DIAG_TEST_BUFFER_FULL ) && ( num_rcv_pkt < 2 )) {
                    /* read twice */
                    goto get_packet;
                }
            }
            else {
                printf("FAILED.\n");

                /* do a byte-by-byte comparison */
                for( i = 0; i < pktlen; i++ ) {
                    if( protocol_buffer[i] != ascending[i] ) {
                        printf("Index %d: Expected %02x, Got %02x\n", i, ascending[i], protocol_buffer[i]);
                    }
                }
                return -3;
            }
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("Reason: %s\n", strerror(errno));
            }
            return -4;
        }
    }
    else {
        /* different event occurred */
        return -5;
    }

    return 0;
}

int fpd_run_diag( int fd, int diag_code, direction_t direction, int enable )
{
    int pktlen = 256;
    int rc;

    switch( buffer_type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            break;

        case FPD_HOST_BUFFER:
            fprintf(stderr, "\nThis test is not implemented in Host Buffer.\n");
            fprintf(stderr, "Please run it using CIM or Priority Buffer.\n");
            return -1;

        default:
            return -2;
    }

    /*
     * Send/Receive a 256 byte packet of ascending pattern
     */
    if(( direction != RECEIVE  ) && ( direction != SEND ))
        return -1;

    switch( diag_code ) {
        case FPD_DIAG_TEST_CRC_ERROR:
            fprintf(stderr, "\n%s Packets with CRC Error %s %s buffer....",
                    (direction == SEND)?"Sending":"Receiving",
                    (direction == SEND)?"to":"from",
                    (buffer_type == FPD_CIM_BUFFER)?"CIM":"Priority");
            break;
        case FPD_DIAG_TEST_PROTOCOL_ERROR:
            fprintf(stderr, "\n%s Packets with Protocol Error %s %s buffer....",
                    (direction == SEND)?"Sending":"Receiving",
                    (direction == SEND)?"to":"from",
                    (buffer_type == FPD_CIM_BUFFER)?"CIM":"Priority");
            break;
        case FPD_DIAG_TEST_INVALID_PKT:
            fprintf(stderr, "\n%s Invalid Packets %s %s buffer....",
                    (direction == SEND)?"Sending":"Receiving",
                    (direction == SEND)?"to":"from",
                    (buffer_type == FPD_CIM_BUFFER)?"CIM":"Priority");
            break;
        case FPD_DIAG_TEST_BUFFER_FULL:
            if( direction == SEND ) {
                fprintf(stderr, "\nTesting %s Buffer Full (%s-side)....",
                        (buffer_type == FPD_CIM_BUFFER)?"CIM":"Priority",
                        (direction == SEND)?"tx":"rx");
            }
            else {
                fprintf(stderr, "\n%s testing %s Buffer Full (%s-side)....",
                        enable?"Initiate":"Complete",
                        (buffer_type == FPD_CIM_BUFFER)?"CIM":"Priority",
                        (direction == SEND)?"tx":"rx");
            }
            break;
        default:
            return -1;
    }

    if( direction == SEND ) {
        /* message construction */
        gmsg.diag.diag_code = diag_code;
        gmsg.diag.param1 = direction;
        gmsg.diag.param2 = enable;
        gmsg.diag.pdata.link_if = link_if;
        gmsg.diag.pdata.requested_len = pktlen;
        gmsg.diag.pdata.type = buffer_type;
        gmsg.diag.pdata.buf = protocol_buffer;
        memset(gmsg.diag.pdata.buf, 0, MAX_PROTOCOL_BUFFER_SIZE);
        memcpy(gmsg.diag.pdata.buf, &ascending[0], pktlen);

        rc = ioctl(fd, FPD_RUN_DIAGNOSTICS, &gmsg);
        if( rc == 0 ) {
            if( gmsg.diag.result == 0 ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                fpd_display_diag_errors(gmsg.diag.result);
            }
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf( "ioctl failed : %s\n", strerror(errno) );
            }
        }
    }
    else {
        if( diag_code == FPD_DIAG_TEST_BUFFER_FULL ) {
            /* message construction */
            gmsg.diag.diag_code = diag_code;
            gmsg.diag.param1 = direction;
            gmsg.diag.param2 = enable;
            gmsg.diag.pdata.link_if = link_if;
            gmsg.diag.pdata.requested_len = pktlen;
            gmsg.diag.pdata.type = buffer_type;
            gmsg.diag.pdata.buf = protocol_buffer;
            memset(gmsg.diag.pdata.buf, 0, MAX_PROTOCOL_BUFFER_SIZE);
            memcpy(gmsg.diag.pdata.buf, &ascending[0], pktlen);

            rc = ioctl(fd, FPD_RUN_DIAGNOSTICS, &gmsg);
            if( rc == 0 ) {
                if( gmsg.diag.result == 0 ) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                    fpd_display_diag_errors(gmsg.diag.result);
                    return -1;
                }
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf( "ioctl failed : %s\n", strerror(errno) );
                }
                return -1;
            }

            if( enable ) {
                /* read the stats to clear it before starting test */
                gmsg.stats.link_if = link_if;
                rc = ioctl(fd, FPD_GET_STATISTICS, &gmsg);
                if (rc != 0) {
                    printf("FAILED.\n");
                    if(errno) {
                        printf("ioctl failed : %s\n", strerror(errno));
                    }
                    return -1;
                }
                return 0;
            }
        }
        else {
            /* read the stats to clear it before starting test */
            gmsg.stats.link_if = link_if;
            rc = ioctl(fd, FPD_GET_STATISTICS, &gmsg);
            if (rc != 0) {
                printf("FAILED.\n");
                if(errno) {
                    printf("ioctl failed : %s\n", strerror(errno));
                }
                return -1;
            }
        }

        /* expect to receive packet */
        if( fpd_run_diag_receive_packet(fd, buffer_type, pktlen, diag_code) == 0 ) {
            FPD_stats_t *pinfo = &gmsg.stats;
            int error = 0;

            /* get stats */
            gmsg.stats.link_if = link_if;
            rc = ioctl(fd, FPD_GET_STATISTICS, &gmsg);
            if (rc == 0) {
                switch( diag_code ) {
                    case FPD_DIAG_TEST_CRC_ERROR:
                        /* RX_CRC_CNT in RXSTATS1 should be 255 */
                        if( pinfo->rx.crc_err_cnt != 255 ) {
                            fprintf(stderr, "ERROR: Received %d CRC errors (but less than 255)\n", pinfo->rx.crc_err_cnt);
                            error = 1;
                        }

                        /* check that all other RX stats are 0 */
                        if( pinfo->rx.buf_full_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d buffer full\n", pinfo->rx.buf_full_cnt);
                        }
                        if( pinfo->rx.protocol_err_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d protocol errors\n", pinfo->rx.protocol_err_cnt);
                        }
                        if( pinfo->rx.inv_pkt_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d invalid packets\n", pinfo->rx.inv_pkt_cnt);
                        }
                        if( pinfo->rx.timeout_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d timeout\n", pinfo->rx.timeout_cnt);
                        }
                        if( pinfo->rx.noise_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d noise\n", pinfo->rx.noise_cnt);
                        }
                        break;
                    case FPD_DIAG_TEST_PROTOCOL_ERROR:
                        /* RX_PROT_ERR_CNT in RXSTATS1 should be 255 */
                        if( gmsg.stats.rx.protocol_err_cnt != 255 ) {
                            fprintf(stderr, "ERROR: Received %d Protocol errors (but less than 255)\n", pinfo->rx.protocol_err_cnt);
                            error = 1;
                        }

                        /* check that all other RX stats are 0 */
                        if( pinfo->rx.buf_full_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d buffer full\n", pinfo->rx.buf_full_cnt);
                        }
                        if( pinfo->rx.crc_err_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d CRC errors\n", pinfo->rx.crc_err_cnt);
                        }
                        if( pinfo->rx.inv_pkt_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d invalid packets\n", pinfo->rx.inv_pkt_cnt);
                        }
                        if( pinfo->rx.timeout_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d timeout\n", pinfo->rx.timeout_cnt);
                        }
                        if( pinfo->rx.noise_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d noise\n", pinfo->rx.noise_cnt);
                        }
                        break;
                    case FPD_DIAG_TEST_INVALID_PKT:
                        /* RX_INV_PKT_CNT in RXSTATS1 should be 255 */
                        if( gmsg.stats.rx.inv_pkt_cnt != 255 ) {
                            fprintf(stderr, "ERROR: Received %d invalid packets (but less than 255)\n", pinfo->rx.inv_pkt_cnt);
                            error = 1;
                        }

                        /* check that all other RX stats are 0 */
                        if( pinfo->rx.buf_full_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d buffer full\n", pinfo->rx.buf_full_cnt);
                        }
                        if( pinfo->rx.crc_err_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d CRC errors\n", pinfo->rx.crc_err_cnt);
                        }
                        if( pinfo->rx.protocol_err_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d protocol errors\n", pinfo->rx.protocol_err_cnt);
                        }
                        if( pinfo->rx.timeout_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d timeout\n", pinfo->rx.timeout_cnt);
                        }
                        if( pinfo->rx.noise_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d noise\n", pinfo->rx.noise_cnt);
                        }
                        break;
                    case FPD_DIAG_TEST_BUFFER_FULL:
                        /* RX_BUF_FULL_CNT in RXSTATS1 should be 255 */
                        if( gmsg.stats.rx.buf_full_cnt != 255 ) {
                            fprintf(stderr, "ERROR: Received %d buffer full (but less than 255)\n", pinfo->rx.buf_full_cnt);
                            error = 1;
                        }

                        /* check that all other RX stats are 0 */
                        if( pinfo->rx.crc_err_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d CRC errors\n", pinfo->rx.crc_err_cnt);
                        }
                        if( pinfo->rx.protocol_err_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d protocol errors\n", pinfo->rx.protocol_err_cnt);
                        }
                        if( pinfo->rx.inv_pkt_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d invalid packets\n", pinfo->rx.inv_pkt_cnt);
                        }
                        if( pinfo->rx.timeout_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d timeout\n", pinfo->rx.timeout_cnt);
                        }
                        if( pinfo->rx.noise_cnt != 0 ) {
                            fprintf(stderr, "WARNING: Received %d noise\n", pinfo->rx.noise_cnt);
                        }
                        break;
                    default:
                        return -1;
                }
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf( "ioctl failed : %s\n", strerror(errno) );
                }
            }


            if( error )
                return -1;

            /* verify that stats cleared */
            gmsg.stats.link_if = link_if;
            rc = ioctl(fd, FPD_GET_STATISTICS, &gmsg);
            if (rc == 0) {
                /* check that all other RX stats are 0 */
                if( pinfo->rx.buf_full_cnt != 0 ) {
                    fprintf(stderr, "ERROR: Still receiving %d buffer full\n", pinfo->rx.buf_full_cnt);
                }
                if( pinfo->rx.crc_err_cnt != 0 ) {
                    fprintf(stderr, "ERROR: Still receiving %d CRC errors\n", pinfo->rx.crc_err_cnt);
                }
                if( pinfo->rx.protocol_err_cnt != 0 ) {
                    fprintf(stderr, "ERROR: Still receiving %d protocol errors\n", pinfo->rx.protocol_err_cnt);
                }
                if( pinfo->rx.inv_pkt_cnt != 0 ) {
                    fprintf(stderr, "ERROR: Still receiving %d invalid packets\n", pinfo->rx.inv_pkt_cnt);
                }
                if( pinfo->rx.timeout_cnt != 0 ) {
                    fprintf(stderr, "ERROR: Still receiving %d timeout\n", pinfo->rx.timeout_cnt);
                }
                if( pinfo->rx.noise_cnt != 0 ) {
                    fprintf(stderr, "ERROR: Still receiving %d noise\n", pinfo->rx.noise_cnt);
                }
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf( "ioctl failed : %s\n", strerror(errno) );
                }
            }
        }
    }

    return 0;
}

int fpd_send_pkts_continuously( int fd )
{
    int option;
    int rc;

    switch( buffer_type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            fprintf(stderr, "\nChoose One pass (1) or forever loop (0): ");
            scanf("%d", &option);
            rc = send_protocol_data_continuously(fd, link_if, buffer_type, option, 1);
            break;
        case FPD_HOST_BUFFER:
            rc = send_host_data_continuously(fd, link_if, host_chan);
            break;
        default:
            rc = -1;
    }

    return rc;
}

int fpd_receive_pkts_continuously( int fd )
{
    int option;
    int rc;

    switch( buffer_type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            fprintf(stderr, "\nChoose One pass (1) or forever loop (0): ");
            scanf("%d", &option);
            rc = receive_protocol_data_continuously(fd, link_if, buffer_type, option, 1);
            break;
        case FPD_HOST_BUFFER:
            rc = receive_host_data_continuously(fd, link_if, host_chan);
            break;
        default:
            rc = -1;
    }

    return rc;
}

int fpd_set_cable_type( int fd )
{
    int rc;
    unsigned int val;

    /* find out current setting */
    gmsg.rwreg.reg = 0x84; /* LED Control Register */
    rc = ioctl(fd, FPD_READ_REG, &gmsg);
    if( rc == 0 ) {
        printf("\nCurrent cable setting                    : ");
        if( gmsg.rwreg.value & 0x40000000 )
            printf("Cross-over (D+/D-) CAT5\n");
        else
            printf("Regular CAT5\n");
    }
    else {
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
        return -1;
    }

    printf("Enter Regular [0] or Cross-over [1] Cable: ");
    scanf("%d", &val);
    if(( val == 0 ) || ( val == 1 ) ) {
        gmsg.rwreg.reg = 0x84; /* LED Control Register */
        if( val == 0 )
            gmsg.rwreg.value &= ~0x40000000;
        else
            gmsg.rwreg.value |= 0x40000000;
        rc = ioctl(fd, FPD_WRITE_REG, &gmsg);
        if( rc != 0 ) {
            if(errno) {
                printf( "ioctl failed : %s\n", strerror(errno) );
            }
            return -1;
        }
    }
    else {
        printf("\nERROR: Invalid entry\n");
        printf("Please re-do!!!\n");
        return -1;
    }

    return 0;
}

int fpd_test_kxgen2_vm_port( int fd )
{
    int link;
    int line_start, line_end;
    int target_port;
    int protocol = FPD_PROTOCOL_VM;
    int rc;
    
    printf("Enter Starting Target Port: ");
    scanf("%d", &line_start);
    printf("Enter Ending Target Port  : ");
    scanf("%d", &line_end);
    if( line_start > line_end ) {
        fprintf(stderr, "ERROR: Invalid params\n");
        return -1;
    }

    system("date");

    /* disconnect all links */
    for( link = 0; link < max_link_if; link++ ) {
        fpd_disconnect_link(fd, link);
    }
    fpd_disconnect_link(fd, FPD_LINK_IF_ID_BGND);

    for( target_port = line_start; target_port <= line_end; target_port++ ) {
        for( link = 0; link < max_link_if; link++ ) {
            /* connect Link IF to specified target port */
            printf("Switching Link IF %d to Target %d...........", link, target_port);
            if( fpd_switch_link(fd, link, target_port, protocol, 0, 0) == 0 ) {
                printf("passed.");
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf("FPD_SWITCH_CHANNEL ioctl failed : %s\n", strerror(errno) );
                }
                return -2;
            }

            /* send data using CIM Buffer */
            rc = send_protocol_data_continuously(fd, link, FPD_CIM_BUFFER, 1, 0);
            if( rc != 0 ) {
                fpd_disconnect_link(fd, link);
                return -3;
            }

            /* receive data using CIM Buffer */
            rc = receive_protocol_data_continuously(fd, link, FPD_CIM_BUFFER, 1, 0);
            if( rc != 0 ) {
                fpd_disconnect_link(fd, link);
                return -4;
            }

            /* send data using Priority Buffer */
            rc = send_protocol_data_continuously(fd, link, FPD_PRIORITY_BUFFER, 1, 0);
            if( rc != 0 ) {
                fpd_disconnect_link(fd, link);
                return -5;
            }

            /* receive data using Priority Buffer */
            rc = receive_protocol_data_continuously(fd, link, FPD_PRIORITY_BUFFER, 1, 0);
            if( rc != 0 ) {
                fpd_disconnect_link(fd, link);
                return -6;
            }

            /* disconnect Link IF from target port */
            printf("\nDisconnecting Link IF %d from Target %d.....", link, target_port);
            if( fpd_disconnect_link(fd, link) == 0 ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf("FPD_DISCONNECT_CHANNEL ioctl failed : %s\n", strerror(errno) );
                }
                return -7;
            }
        }


        if( bgnd_link_if_present ) {
            link = FPD_LINK_IF_ID_BGND;

            /* connect Link IF to specified target port */
            printf("Switching BGND Link IF to Target %d........", target_port);
            if( fpd_switch_link(fd, link, target_port, protocol, 0, 0) == 0 ) {
                printf("passed.");
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf("FPD_SWITCH_CHANNEL ioctl failed : %s\n", strerror(errno) );
                }
                return -2;
            }

            /* send data using CIM Buffer */
            rc = send_protocol_data_continuously(fd, link, FPD_CIM_BUFFER, 1, 0);
            if( rc != 0 ) {
                fpd_disconnect_link(fd, link);
                return -3;
            }

            /* receive data using CIM Buffer */
            rc = receive_protocol_data_continuously(fd, link, FPD_CIM_BUFFER, 1, 0);
            if( rc != 0 ) {
                fpd_disconnect_link(fd, link);
                return -4;
            }

            /* disconnect Link IF from target port */
            printf("\nDisconnecting BGND Link IF from Target %d..", target_port);
            if( fpd_disconnect_link(fd, link) == 0 ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf("FPD_DISCONNECT_CHANNEL ioctl failed : %s\n", strerror(errno) );
                }
                return -7;
            }
        }

        if( target_port < line_end ) {
            int done, response;
            
            fflush(stdin);
            do {
                printf("\rDo you want to continue to port %d (y/n): ", target_port+1);
                response = getchar();
                switch( response ) {
                    case 'y':
                    case 'Y':
                    case 'n':
                    case 'N':
                        done = 1;
                        break;
                    default:
                        done = 0;
                        break;
                }
            } while( !done );

            if( response == 'n' || response == 'N' ) {
                break;
            }
        }
    }

    system("date");

    return 0;
}

int fpd_test_receiver_vm_port( int fd )
{
    int link = 0;
    int target_port = 0;
    int protocol = FPD_PROTOCOL_VM;
    int i, pass, rc;

    printf("Enter Number of Test Pass : ");
    scanf("%d", &pass);

    system("date");

    /* connect Link IF to specified target port */
    printf("Switching Link IF %d to Target %d...........", link, target_port);
    if( fpd_switch_link(fd, link, target_port, protocol, 0, 0) == 0 ) {
        printf("passed.");
    }
    else {
        printf("FAILED.");
        if(errno) {
            printf("FPD_SWITCH_CHANNEL ioctl failed : %s\n", strerror(errno) );
        }
        return -1;
    }

    for( i = 0; i < pass; i++ ) {
        /* receive data using CIM Buffer */
        rc = receive_protocol_data_continuously(fd, link, FPD_CIM_BUFFER, 1, 0);
        if( rc != 0 ) {
            return -2;
        }

        /* send data using CIM Buffer */
        rc = send_protocol_data_continuously(fd, link, FPD_CIM_BUFFER, 1, 0);
        if( rc != 0 ) {
            return -3;
        }

        /* Every 6th pass is the BGND Link IF */
        if( (i+1) % 6 ) {
            /* receive data using Priority Buffer */
            rc = receive_protocol_data_continuously(fd, link, FPD_PRIORITY_BUFFER, 1, 0);
            if( rc != 0 ) {
                return -4;
            }

            /* send data using Priority Buffer */
            rc = send_protocol_data_continuously(fd, link, FPD_PRIORITY_BUFFER, 1, 0);
            if( rc != 0 ) {
                return -5;
            }
        }
    }

    /* wait until all data is sent before disconnecting */
    sleep(1);

    /* disconnect Link IF from target port */
    printf("\nDisconnecting Link IF %d from Target %d.....", link, target_port);
    if( fpd_disconnect_link(fd, link) == 0 ) {
        printf("passed.\n");
    }
    else {
        printf("FAILED.\n");
        if(errno) {
            printf("FPD_DISCONNECT_CHANNEL ioctl failed : %s\n", strerror(errno) );
        }
        return -6;
    }

    system("date");

    return 0;
}
