/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Device Diagnostic Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>
#include <time.h>

#include "fpd_ioctl.h"
#include "fpd-diagnostics.h"

#define STATIC_SIZE_TEST_LOOP        1000


static void *host_buffer;
static int host_buffer_mmap_len = 0;
static int one_pass_only = 0;
static unsigned char *repeating16;
static unsigned char *ascending16;
static unsigned char *descending16;


static void dump_data( unsigned char *bufaddr, int pktlen )
{
    int i;

    for( i = 0; i < pktlen; i++ ) {
        if((i % 0x10) == 0) {
            printf("\n%08x: ", i);
        }
        printf("%02x ", bufaddr[i]);
    }
    printf("\n");

    return;
}

static void dump_data32( unsigned char *bufaddr, int pktlen )
{
    unsigned int *buf = (unsigned int *)bufaddr;
    int remainder;
    int size;
    int i;

    remainder = pktlen & 0x3;
    size = pktlen - remainder;
    size >>= 2;

    for( i = 0; i < size; i++ ) {
        if((i % 4) == 0) {
            printf("\n%08x: ", i*4);
        }
        printf("%08x ", buf[i]);
    }

    if( remainder ) {
        size = pktlen - remainder;
        for( i = size; i < pktlen; i++ ) {
            printf("%02x ", bufaddr[i]);
        }
    }
    printf("\n");

    return;
}

static int fill_data( unsigned char *bufaddr, int pkt_size, pkt_pattern_t pattern )
{
    unsigned int *buf32addr;
    int revsize;
    int i;

    /* fill in data */
    switch( pattern ) {
        case PKT_PATTERN_8BIT_ASCENDING:
            for( i = 0; i < pkt_size; i++ ) {
                bufaddr[i] = i & 0xff;
            }
            break;

        case PKT_PATTERN_8BIT_DESCENDING:
            for( i = 0; i < pkt_size; i++ ) {
                bufaddr[i] = (0xff - i) % 256;
            }
            break;

        case PKT_PATTERN_8BIT_REPEATING:
            memcpy(bufaddr, repeating16, pkt_size);
            break;

        case PKT_PATTERN_16BIT_ASCENDING:
            memcpy(bufaddr, ascending16, pkt_size);
            break;

        case PKT_PATTERN_16BIT_DESCENDING:
            memcpy(bufaddr, descending16, pkt_size);
            break;

        case PKT_PATTERN_32BIT_ASCENDING:
            buf32addr = (unsigned int *)bufaddr;
            revsize = pkt_size >> 2;
            /* in case, size is not four-byte aligned */
            if( pkt_size & 0x3 ) {
                ++revsize;
            }
            for( i = 0; i < revsize; i++ ) {
                buf32addr[i] = i;
            }
            break;

        case PKT_PATTERN_32BIT_DESCENDING:
            buf32addr = (unsigned int *)bufaddr;
            revsize = pkt_size >> 2;
            /* in case, size is not four-byte aligned */
            if( pkt_size & 0x3 ) {
                ++revsize;
            }
            for( i = 0; i < revsize; i++ ) {
                buf32addr[i] = (0xffffffff - i);
            }
            break;

        default:
            printf("ERROR: Unknown Packet Pattern!\n");
            return -1;
    }

    return 0;
}

static void display_verify_data_error( FPD_host_rxparam_t *prxinfo, unsigned int host_chan, int len, int round )
{
    fprintf(stderr, "FAILED\n");
    if( !debug_on ) {
        fprintf(stderr, "[%d]Comparing contents with expected data (%d bytes) from Host buffer %d....FAILED\n",
                round, len, host_chan);
    }
    fprintf(stderr, "BufID %d, startBD %d, endBD %d, offset %d\n",
            prxinfo->buf_id, prxinfo->startbd, prxinfo->endbd, prxinfo->offset);

    return;
}

static int verify_data( FPD_hostdata_t *pkt, int len, int offset, pkt_pattern_t pattern, int round )
{
    FPD_host_rxparam_t *prxinfo;
    unsigned char *bufaddr;
    unsigned short *buf16addr;
    unsigned int *buf32addr;
    int pktsize = len;
    int rc = 0;
    int i;

    /* compute buffer address */
    prxinfo = &pkt->info.rx;
    bufaddr = (unsigned char *)host_buffer + (prxinfo->buf_id * prxinfo->buf_size) + prxinfo->offset;
    buf16addr = (unsigned short *)bufaddr;
    buf32addr = (unsigned int *)bufaddr;

    /* compare contents against expected data */
    switch( pattern ) {
        case PKT_PATTERN_8BIT_ASCENDING:
            for( i = 0; i < pktsize; i++ ) {
                if( bufaddr[i] != ((i+offset) & 0xff) ) {
                    if( rc == 0 ) {
                        display_verify_data_error(prxinfo, pkt->host_chan, pkt->requested_len, round);
                    }
                    fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n", i, (i+offset) & 0xff, bufaddr[i]);
                    rc = -1;
                }
            }
            break;

        case PKT_PATTERN_8BIT_DESCENDING:
            for( i = 0; i < pktsize; i++ ) {
                if( bufaddr[i] != (unsigned char)((0xff - (i+offset)) % 256) ) {
                    if( rc == 0 ) {
                        display_verify_data_error(prxinfo, pkt->host_chan, pkt->requested_len, round);
                    }
                    fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n", i, (unsigned char)((0xff - (i+offset)) % 256), bufaddr[i]);
                    rc = -1;
                }
            }
            break;

        case PKT_PATTERN_8BIT_REPEATING:
            rc = memcmp((const void *)bufaddr, (const void *)&repeating16[offset], len);
            if( rc != 0 ) {
                display_verify_data_error(prxinfo, pkt->host_chan, pkt->requested_len, round);
                for( i = 0; i < pktsize; i++ ) {
                    if( bufaddr[i] != repeating16[i+offset] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, repeating16[i+offset], bufaddr[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_16BIT_ASCENDING:
            rc = memcmp((const void *)bufaddr, (const void *)&ascending16[offset], len);
            if( rc != 0 ) {
                display_verify_data_error(prxinfo, pkt->host_chan, pkt->requested_len, round);
                for( i = 0; i < pktsize; i++ ) {
                    if( bufaddr[i] != ascending16[i+offset] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, ascending16[i+offset], bufaddr[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_16BIT_DESCENDING:
            rc = memcmp((const void *)bufaddr, (const void *)&descending16[offset], len);
            if( rc != 0 ) {
                display_verify_data_error(prxinfo, pkt->host_chan, pkt->requested_len, round);
                for( i = 0; i < pktsize; i++ ) {
                    if( bufaddr[i] != descending16[i+offset] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, descending16[i+offset], bufaddr[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_32BIT_ASCENDING:
            pktsize >>= 2;
            if( offset > 0 ) {
                offset >>= 2;
            }
            for( i = 0; i < pktsize; i++ ) {
                if( buf32addr[i] != (i+offset) ) {
                    if( rc == 0 ) {
                        display_verify_data_error(prxinfo, pkt->host_chan, pkt->requested_len, round);
                    }
                    fprintf(stderr, "Offset %d: Expected %08x, Rcvd %08x\n", i*4, i+offset, buf32addr[i]);
                    rc = -1;
                }
            }
            break;

        case PKT_PATTERN_32BIT_DESCENDING:
            pktsize >>= 2;
            if( offset > 0 ) {
                offset >>= 2;
            }
            for( i = 0; i < pktsize; i++ ) {
                if( buf32addr[i] != (0xffffffff - (i+offset)) ) {
                    if( rc == 0 ) {
                        display_verify_data_error(prxinfo, pkt->host_chan, pkt->requested_len, round);
                    }
                    fprintf(stderr, "Offset %d: Expected %08x, Rcvd %08x\n", i*4, (0xffffffff - (i+offset)), buf32addr[i]);
                    rc = -1;
                }
            }
            break;

        default:
            printf("ERROR: Unknown Packet Pattern!\n");
            rc = -1;
    }

    if( rc != 0 ) {
        printf("Expected %d bytes of ", len);
        switch( pattern ) {
            case PKT_PATTERN_8BIT_ASCENDING:
                printf("8-bit ascending ");
                break;
            case PKT_PATTERN_8BIT_DESCENDING:
                printf("8-bit descending ");
                break;
            case PKT_PATTERN_8BIT_REPEATING:
                printf("8-bit repeating ");
                break;
            case PKT_PATTERN_16BIT_ASCENDING:
                printf("16-bit ascending ");
                break;
            case PKT_PATTERN_16BIT_DESCENDING:
                printf("16-bit descending ");
                break;
            case PKT_PATTERN_32BIT_ASCENDING:
                printf("32-bit ascending ");
                break;
            case PKT_PATTERN_32BIT_DESCENDING:
                printf("32-bit descending ");
                break;
            default:
                break;
        }
        printf("pattern but received different data\n");

        /* dump data */
        if( pkt->actual_len > 0 ) {
            dump_data32(bufaddr, pkt->actual_len);
        }
    }

    return rc;
}

static int release_rx_buffer( int fd, unsigned int link_if, unsigned int host_chan, FPD_host_rxparam_t *prxinfo )
{
    FPD_host_rxclear_t rxclear, *prx = &rxclear;
    int rc;

    prx->link_if = link_if;
    prx->host_chan = host_chan;
    prx->info = *prxinfo;
    if( debug_on ) {
        printf("Releasing Host buffer %d....", host_chan);
    }
    rc = ioctl(fd, FPD_RELEASE_RX_HOST_DATA, prx);
    if( rc == 0 ) {
        if( debug_on ) {
            printf("passed.\n");
        }
    }
    else {
        fprintf(stderr, "FAILED.\n");
        if( !debug_on ) {
            fprintf(stderr, "Releasing Host buffer %d....FAILED.\n", host_chan);
        }

        if(errno) {
            fprintf(stderr, "ioctl failed : %s\n", strerror(errno));
        }
    }

    return rc;
}

static int get_rxhost_params( FPD_hostdata_t *pkt, unsigned int link_if, unsigned int host_chan )
{
    printf("\nEnter Parameters\n");

    if( link_if <= 4 ) {
        printf("Link IF                     : %d\n", link_if);
    }
    else {
        return -1;
    }
    pkt->link_if = link_if;
    printf("Host Buffer                 : %d\n", host_chan);
    pkt->host_chan = host_chan;
    printf("Data Length                 : ");
    scanf("%d", &pkt->requested_len);

    if( pkt->requested_len > MAX_HOST_BUFFER_SIZE )
        return -1;

    return 0;
}

int init_host_module( int fd, int overall_buf_len )
{
    unsigned char *bufaddr;
    unsigned short *buf16addr;
    int size;
    int i;

    host_buffer_mmap_len = overall_buf_len;

    host_buffer = (void *) mmap(NULL,
                                host_buffer_mmap_len,
                                PROT_READ | PROT_WRITE,
                                MAP_SHARED,
                                fd,
                                0 );
    fprintf(stderr, "host_buffer = %p\n", host_buffer);
    if( host_buffer == (void *) -1 ) {
        return -1;
    }

    /* allocate test pattern buffers */
    repeating16 = (unsigned char *)malloc(MAX_HOST_BUFFER_SIZE);
    ascending16 = (unsigned char *)malloc(MAX_HOST_BUFFER_SIZE);
    descending16 = (unsigned char *)malloc(MAX_HOST_BUFFER_SIZE);

    if( !repeating16 || !ascending16 || !descending16 ) {
        fprintf(stderr, "ERROR: Unable to allocate pattern buffers\n");
        return -2;
    }

    /* fill data: repeating pattern */
    bufaddr = repeating16;
    size = MAX_HOST_BUFFER_SIZE;
    for( i = 0; i < size; i++ ) {
        bufaddr[i] = repeating[i % MAX_PROTOCOL_BUFFER_SIZE];
    }

    /* fill data: 16-bit ascending pattern */
    buf16addr = (unsigned short *)ascending16;
    size = MAX_HOST_BUFFER_SIZE >> 1;
    for( i = 0; i < size; i++ ) {
        buf16addr[i] = (unsigned short)(i & 0xffff);
    }

    /* fill data: 16-bit descending pattern */
    buf16addr = (unsigned short *)descending16;
    size = MAX_HOST_BUFFER_SIZE >> 1;
    for( i = 0; i < size; i++ ) {
        buf16addr[i] = (unsigned short)((0xffff - i) % MAX_HOST_BUFFER_SIZE);
    }

    return 0;
}

void cleanup_host_module( void )
{
    munmap(host_buffer, host_buffer_mmap_len);
    return;
}

int set_host_configuration( int fd, unsigned int link_if, unsigned int host_chan )
{
    FPD_hostcfg_t cfg, *pcfg = &cfg;
    int i = 0;
    int rc;

    pcfg->link_if = link_if;
    pcfg->host_chan = host_chan;
    printf("DMA Size (64 or 512 bytes)                      : ");
    scanf("%d", &pcfg->dma_size);
    printf("Rx Pkt Size Notification                        : ");
    scanf("%d", &pcfg->rx_pkt_size_notification);
    printf("Host Function");
    printf("\n=============");
    printf("\n\t %d: None", i++);
    printf("\n\t %d: Transparent", i++);
    printf("\n\t %d: Virtual Media\n", i++);
    printf("Host Function                                   : ");
    scanf("%d", &pcfg->function);

    rc = ioctl(fd, FPD_SET_HOST_BUFFER_CFG, pcfg);
    if (rc < 0) {
        if(errno) {
            fprintf(stderr, "FPD_SET_HOST_BUFFER_CFG ioctl failed : %s\n", strerror(errno));
        }
        else {
            fprintf(stderr, "FPD_SET_HOST_BUFFER_CFG ioctl failed : Invalid argument\n");
        }
    }

    return rc;
}

int get_host_configuration( int fd, unsigned int link_if, unsigned int host_chan )
{
    FPD_hostcfg_t cfg, *pcfg = &cfg;

    pcfg->link_if = link_if;
    pcfg->host_chan = host_chan;
    if( ioctl(fd, FPD_GET_HOST_BUFFER_CFG, pcfg) == 0 ) {
        printf("\nHost Buffer Channel            : %d", pcfg->host_chan);
        printf("\nDMA Size                       : %d", pcfg->dma_size);
        printf("\nRx Pkt Size Notification       : %d", pcfg->rx_pkt_size_notification);
        printf("\nHost Function                  : ");
        switch(pcfg->function) {
            case FPD_HOST_FUNCTION_NONE:
                printf("Disabled");
                break;
            case FPD_HOST_FUNCTION_TRANSPARENT:
                printf("Transparent");
                break;
            case FPD_HOST_FUNCTION_VIRTUAL_MEDIA:
                printf("Virtual Media");
                break;
        }
    }
    else {
        printf("\nHost Buffer Configuration      : UNKNOWN");
    }

    return 0;
}

int read_host_data_manually( int fd, unsigned int link_if, unsigned int host_chan )
{
    FPD_hostdata_t host_data, *pkt = &host_data;
    FPD_host_rxparam_t *prxinfo;
    unsigned char *bufaddr;
    int rc;

    if((rc = get_rxhost_params(pkt, link_if, host_chan)) == 0 ) {
        printf("\nReading %d bytes of data from Host buffer %d....",
               pkt->requested_len, host_chan);
        rc = ioctl(fd, FPD_RECEIVE_HOST_DATA, pkt);
        if( rc == 0 ) {
            /* compute buffer address */
            prxinfo = &pkt->info.rx;
            bufaddr = (unsigned char *)host_buffer + (prxinfo->buf_id * prxinfo->buf_size) + prxinfo->offset;

            if( pkt->actual_len == pkt->requested_len ) {
                printf("passed.\n");
            }
            else {
                printf("incomplete.\n");
                printf("Requested %d, Got %d\n",
                       pkt->requested_len, pkt->actual_len);
                fprintf(stderr, "BufID %d, startBD %d, endBD %d, offset %d\n",
                        prxinfo->buf_id, prxinfo->startbd, prxinfo->endbd, prxinfo->offset);
            }

            if( pkt->actual_len > 0 ) {
                dump_data(bufaddr, pkt->actual_len);
            }

            /* release RX buffer */
            rc = release_rx_buffer(fd, link_if, host_chan, prxinfo);
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf( "ioctl failed : %s\n", strerror(errno) );
            }
        }
    }

    return rc;
}

int send_host_data( int fd, unsigned int link_if, unsigned int host_chan, int pkt_size )
{
    FPD_hostdata_t host_data, *pkt = &host_data;
    FPD_hostbuf_t hostbuf, *pbuf = &hostbuf;
    unsigned char *bufaddr;
    int rc;

    /* get a buffer */
    pbuf->link_if = link_if;
    pbuf->host_chan = host_chan;
    rc = ioctl(fd, FPD_GET_HOST_BUFFER, pbuf);
    if( rc == 0 ) {
        /* compute buffer address */
        bufaddr = (unsigned char *)host_buffer + (pbuf->buf_id * pbuf->buf_size);
        fprintf(stderr, "\nbufID=%d, baseAddr=%p, bufAddr=%p", pbuf->buf_id, host_buffer, bufaddr);

        /* fill in data */
        if( fill_data(bufaddr, pkt_size, pkt_pattern) != 0 ) {
            return -1;
        }

        /* tell driver to process the buffer */
        pkt->link_if = link_if;
        pkt->host_chan = host_chan;
        pkt->requested_len = pkt_size;
        pkt->info.tx.buf_id = pbuf->buf_id;
        printf("\nSending %d bytes of data to Host buffer %d....",
               pkt->requested_len, pkt->host_chan);
        rc = ioctl(fd, FPD_SEND_HOST_DATA, pkt);
        if( rc == 0 ) {
            if( pkt->actual_len == pkt->requested_len )
                printf("passed.\n");
            else {
                printf("FAILED.\n");
                printf("Actual bytes written %d\n", pkt->actual_len);
            }
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf( "ioctl failed : %s\n", strerror(errno) );
            }
        }
    }
    else {
        fprintf(stderr, "Failed to get host buffer\n");
    }

    return rc;
}

int receive_host_data( int fd, unsigned int link_if, unsigned int host_chan, int pkt_size )
{
    FPD_hostdata_t host_data, *pkt = &host_data;
    FPD_host_rxparam_t *prxinfo;
    FPD_event_t event, *pevt = &event;
    unsigned char *bufaddr;
    int rx_data = 0;
    int complete = 0;
    int offset = 0;
    int accepted_len = 0;
    int rc;

    pkt->link_if = link_if;
    pkt->host_chan = host_chan;
    pkt->requested_len = pkt_size;

    do {
        if( fpd_poll_once(fd, pevt) == 0 ) {
            /* error detected */
            if( pevt->pci_error ) {
                fprintf(stderr, "\nDetected PCI errors");
                fpd_get_pci_errors( fd );
            }

            if( pevt->link_if[link_if].error ) {
                fprintf(stderr, "\nLink IF %d receive errors", link_if);
                fpd_get_link_errors( fd );
            }

            /* check if FPGA receive host data */
            if( pevt->link_if[link_if].host[host_chan].has_data ) {
                rx_data = 1;
            }
        }

        if( rx_data ) {
            prxinfo = &pkt->info.rx;

            printf("Reading %d bytes of data from Host buffer %d....",
                   pkt->requested_len, host_chan);
            rc = ioctl(fd, FPD_RECEIVE_HOST_DATA, pkt);
            if( rc == 0 ) {
                if( pkt->actual_len == pkt->requested_len ) {
                    complete = 1;
                    printf("passed.\n");
                }
                else {
                    printf("incomplete.\n");
                    printf("Requested %d, Got %d\n",
                           pkt->requested_len, pkt->actual_len);

                    fprintf(stderr, "BufID %d, startBD %d, endBD %d, offset %d\n",
                            prxinfo->buf_id, prxinfo->startbd, prxinfo->endbd, prxinfo->offset);

                    if( !prxinfo->short_pkt && !prxinfo->reached_page_end ) {
                        continue;
                    }
                    else if( prxinfo->reached_page_end ) {
                        accepted_len += pkt->actual_len;
                    }
                    else {
                        fprintf(stderr, "Reading %d bytes of data from Host buffer %d....FAILED\n",
                                pkt_size, host_chan);
                        fprintf(stderr, "Requested %d, Got %d\n",
                                pkt->requested_len, pkt->actual_len);
                        fprintf(stderr, "BufID %d, startBD %d, endBD %d, offset %d, short_pkt %d, page_end %d\n",
                                prxinfo->buf_id, prxinfo->startbd, prxinfo->endbd, prxinfo->offset,
                                prxinfo->short_pkt, prxinfo->reached_page_end);

                        /* dump data */
                        if( pkt->actual_len > 0 ) {
                            /* compute buffer address */
                            bufaddr = (unsigned char *)host_buffer + (prxinfo->buf_id * prxinfo->buf_size) + prxinfo->offset;
                            dump_data32(bufaddr, pkt->actual_len);
                        }

                        /* release RX buffer */
                        release_rx_buffer(fd, link_if, host_chan, prxinfo);

                        return -3;
                    }
                }

                /* compare contents against expected data */
                fprintf(stderr, "Comparing contents with expected data......");
                if( verify_data(pkt, pkt->actual_len, offset, pkt_pattern, 1) == 0 ) {
                    fprintf(stderr, "passed.\n");
                }
                else {
                    /* release RX buffer */
                    release_rx_buffer(fd, link_if, host_chan, prxinfo);
                    return -1;
                }

                /* release RX buffer */
                rc = release_rx_buffer(fd, link_if, host_chan, prxinfo);
            }
            else {
                printf("FAILED.\n");
                if(errno) {
                    printf( "ioctl failed : %s\n", strerror(errno) );
                }
            }

            if( !complete ) {
                offset = accepted_len;
                pkt->requested_len = pkt_size - accepted_len;
            }
        }
    } while( !complete );

    return 0;
}

static int send_incrementing_pkt( int fd, unsigned int link_if, unsigned int host_chan, int loop )
{
    FPD_hostdata_t host_data, *pkt = &host_data;
    FPD_hostbuf_t hostbuf, *pbuf = &hostbuf;
    unsigned char *bufaddr;
    int rc;
    int pktlen;
    pkt_pattern_t test_pattern;
    int result = 0;
    time_t start_time, current_time;

    pkt->link_if = link_if;
    pkt->host_chan = host_chan;

    /*
      for( pktlen = 1; pktlen <= 2048; pktlen++ ) {
    */
    for( pktlen = 1; pktlen <= dma_page_size; pktlen++ ) {
        /* get a buffer */
        pbuf->link_if = link_if;
        pbuf->host_chan = host_chan;

        if( debug_on ) {
            fprintf(stderr, "\n[%d]Acquiring Host%d TX DMA Buffer.............................",
                    loop, pkt->host_chan);
        }

host_get_buffer:
        rc = ioctl(fd, FPD_GET_HOST_BUFFER, pbuf);
        if( rc == 0 ) {
            if( debug_on ) {
                fprintf(stderr, "passed.");
            }

            /* determine test pattern to use */
            switch( pktlen % 3 ) {
                case 0:
                    test_pattern = PKT_PATTERN_8BIT_REPEATING;
                    break;
                case 1:
                    test_pattern = PKT_PATTERN_16BIT_ASCENDING;
                    break;
                default:
                    test_pattern = PKT_PATTERN_16BIT_DESCENDING;
                    break;
            }

            /* compute buffer address */
            bufaddr = (unsigned char *)host_buffer + (pbuf->buf_id * pbuf->buf_size);

            /* fill in data */
            if( fill_data(bufaddr, pktlen, test_pattern) != 0 ) {
                return -1;
            }

            /* tell driver to process the buffer */
            pkt->requested_len = pktlen;
            pkt->info.tx.buf_id = pbuf->buf_id;
            if( debug_on ) {
                fprintf(stderr, "\n[%d]Sending %d bytes to Host buffer %d....",
                        loop, pkt->requested_len, pkt->host_chan);
            }

host_resend:
            rc = ioctl(fd, FPD_SEND_HOST_DATA, pkt);
            if( rc == 0 ) {
                if( pkt->actual_len == pkt->requested_len ) {
                    if( debug_on ) {
                        fprintf(stderr, "passed.");
                    }

                    /* reset time of last good send */
                    time(&start_time);
                }
                else {
                    time(&current_time);
                    if(( pkt->actual_len == 0 ) && ( difftime(current_time, start_time) < RETRY_TIMEOUT )) {
                        goto host_resend;
                    }
                    else {
                        fprintf(stderr, "FAILED.\n");
                        if( !debug_on ) {
                            fprintf(stderr, "[%d]Sending %d bytes to Host buffer %d....FAILED\n",
                                    loop, pkt->requested_len, pkt->host_chan);
                        }
                        fprintf(stderr, "Actual bytes written %d\n", pkt->actual_len);
                        return -1;
                    }
                }
            }
            else {
                if( errno ) {
                    if( errno == ENOSPC ) {
                        time(&current_time);
                        if( difftime(current_time, start_time) < RETRY_TIMEOUT ) {
                            goto host_resend;
                        }
                        else {
                            fprintf(stderr, "FAILED.\n");
                            if( !debug_on ) {
                                fprintf(stderr, "[%d]Sending %d bytes to Host buffer %d....FAILED\n",
                                        loop, pkt->requested_len, pkt->host_chan);
                            }
                            fprintf(stderr, "ERROR: %s\n", strerror(errno));
                            return -1;
                        }
                    }
                }
            }
        }
        else {
            time(&current_time);
            if( difftime(current_time, start_time) < RETRY_TIMEOUT ) {
                goto host_get_buffer;
            }
            else {
                fprintf(stderr, "FAILED.\n");
                if( !debug_on ) {
                    fprintf(stderr, "[%d]Acquiring Host%d TX DMA Buffer (%d bytes)..................FAILED.\n",
                            loop, pkt->host_chan, pktlen);
                }
                return -1;
            }
        }
    }

    return result;
}

static int send_static_size_pkt( int fd, unsigned int link_if, unsigned int host_chan, int pktlen, int loop )
{
    FPD_hostdata_t host_data, *pkt = &host_data;
    FPD_hostbuf_t hostbuf, *pbuf = &hostbuf;
    unsigned char *bufaddr;
    int rc;
    pkt_pattern_t test_pattern;
    int result = 0;
    time_t start_time, current_time;
    int endloop = STATIC_SIZE_TEST_LOOP;
    int i;

    pkt->link_if = link_if;
    pkt->host_chan = host_chan;

    /*
      for( pktlen = 1; pktlen <= 2048; pktlen++ ) {
    */
    for( i = 1; i <= endloop; i++ ) {
        /* get a buffer */
        pbuf->link_if = link_if;
        pbuf->host_chan = host_chan;

        if( debug_on ) {
            fprintf(stderr, "\n[%d]Acquiring Host%d TX DMA Buffer.............................",
                    loop, pkt->host_chan);
        }

host_get_buffer2:
        rc = ioctl(fd, FPD_GET_HOST_BUFFER, pbuf);
        if( rc == 0 ) {
            if( debug_on ) {
                fprintf(stderr, "passed.");
            }

            /* determine test pattern to use */
            switch( i % 3 ) {
                case 0:
                    test_pattern = PKT_PATTERN_8BIT_REPEATING;
                    break;
                case 1:
                    test_pattern = PKT_PATTERN_16BIT_ASCENDING;
                    break;
                default:
                    test_pattern = PKT_PATTERN_16BIT_DESCENDING;
                    break;
            }

            /* compute buffer address */
            bufaddr = (unsigned char *)host_buffer + (pbuf->buf_id * pbuf->buf_size);

            /* fill in data */
            if( fill_data(bufaddr, pktlen, test_pattern) != 0 ) {
                return -1;
            }

            /* tell driver to process the buffer */
            pkt->requested_len = pktlen;
            pkt->info.tx.buf_id = pbuf->buf_id;
            if( debug_on ) {
                fprintf(stderr, "\n[%d]Sending (%d) %d bytes to Host buffer %d....",
                        loop, i, pkt->requested_len, pkt->host_chan);
            }

host_resend2:
            rc = ioctl(fd, FPD_SEND_HOST_DATA, pkt);
            if( rc == 0 ) {
                if( pkt->actual_len == pkt->requested_len ) {
                    if( debug_on ) {
                        fprintf(stderr, "passed.");
                    }

                    /* reset time of last good send */
                    time(&start_time);
                }
                else {
                    time(&current_time);
                    if(( pkt->actual_len == 0 ) && ( difftime(current_time, start_time) < RETRY_TIMEOUT )) {
                        goto host_resend2;
                    }
                    else {
                        fprintf(stderr, "FAILED.\n");
                        if( !debug_on ) {
                            fprintf(stderr, "[%d]Sending (%d) %d bytes to Host buffer %d....FAILED\n",
                                    loop, i, pkt->requested_len, pkt->host_chan);
                        }
                        fprintf(stderr, "Actual bytes written %d\n", pkt->actual_len);
                        return -1;
                    }
                }
            }
            else {
                if( errno ) {
                    if( errno == ENOSPC ) {
                        time(&current_time);
                        if( difftime(current_time, start_time) < RETRY_TIMEOUT ) {
                            goto host_resend2;
                        }
                        else {
                            fprintf(stderr, "FAILED.\n");
                            if( !debug_on ) {
                                fprintf(stderr, "[%d]Sending (%d) %d bytes to Host buffer %d....FAILED\n",
                                        loop, i, pkt->requested_len, pkt->host_chan);
                            }
                            fprintf(stderr, "ERROR: %s\n", strerror(errno));
                            return -1;
                        }
                    }
                }
            }
        }
        else {
            time(&current_time);
            if( difftime(current_time, start_time) < RETRY_TIMEOUT ) {
                goto host_get_buffer2;
            }
            else {
                fprintf(stderr, "FAILED.\n");
                if( !debug_on ) {
                    fprintf(stderr, "[%d]Acquiring Host%d TX DMA Buffer (%d - %d bytes)............FAILED.\n",
                            loop, pkt->host_chan, i, pktlen);
                }
                return -1;
            }
        }
    }

    return result;
}

int send_host_data_continuously( int fd, unsigned int link_if, unsigned int host_chan )
{
    int size;
    unsigned int loop = 0;
    int just_do_it = 1;
    int rc = 0;

    fprintf(stderr, "\n");
    system("date");

    while( just_do_it ) {
        loop++;

        /* static size packet test using 3 different patterns */
        if( !debug_on ) {
            fprintf(stderr, "\n[%d]Sending static size packets to Host buffer %d.....",
                    loop, host_chan);
        }
        else {
            fprintf(stderr, "\n");
        }

        for( size = 256; size <= dma_page_size; size += 256 ) {
            rc = send_static_size_pkt(fd, link_if, host_chan, size, loop);
            if( rc != 0 ) {
                return rc;
            }
        }

        if( !debug_on ) {
            fprintf(stderr, "passed.");
        }

        /* incrementing packet test using 3 different patterns */
        if( !debug_on ) {
            fprintf(stderr, "\n[%d]Sending incrementing packets to Host buffer %d....",
                    loop, host_chan);
        }

        rc = send_incrementing_pkt(fd, link_if, host_chan, loop);
        if( rc == 0 ) {
            if( !debug_on ) {
                fprintf(stderr, "passed.");
            }
        }
        else {
            return rc;
        }

        if( one_pass_only ) {
            just_do_it = 0;
        }
    }

    fprintf(stderr, "\n\n");
    system("date");

    return rc;
}

static int receive_incrementing_pkt( int fd, unsigned int link_if, unsigned int host_chan, int loop )
{
    FPD_hostdata_t host_data, *pkt = &host_data;
    FPD_host_rxparam_t *prxinfo;
    unsigned char *bufaddr;
    int more_data_available;

    FPD_event_t event, *pevt = &event;
    unsigned int timeout = POLL_TIMEOUT;
    struct pollfd fds;
    int pktlen = 1;
    pkt_pattern_t test_pattern;
    int accepted_len = 0;
    int complete = 1;
    int offset = 0;
    int retry = 0;
    int rc;

    fds.fd = fd;
    fds.events = POLLIN;


    while(1) {
        more_data_available = 1;
        if( debug_on ) {
            fprintf(stderr, "waiting for events ... ");
        }

        rc = poll(&fds, 1, timeout);
        if ((rc == 0) || (rc < 0 && errno == EINTR)) {
            if( !debug_on ) {
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "[%d] (%d bytes) waiting for events ... timedout\n", loop, pktlen);
            }
            else {
                fprintf(stderr, "timedout\n");
            }
            return -1;
        }
        else if( rc < 0 ) {
            if( !debug_on ) {
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "waiting for events ... poll() returned ERROR %d\n", rc);
            }
            else {
                fprintf(stderr, "poll() returned %d\n", rc);
            }
            return -2;
        }
        
        if( debug_on ) {
            fprintf(stderr, "receive notifications\n");
            fprintf(stderr, "getting events ... ");
        }

        rc = ioctl(fd, FPD_CHECK_EVENTS, pevt);

        if( debug_on ) {
            fprintf(stderr, "done\n");
        }

        if (rc == 0) {
            /* error detected */
            if( pevt->pci_error ) {
                fprintf(stderr, "\n[%d] (%d bytes) Detected PCI errors", loop, pktlen);
                fpd_get_pci_errors( fd );
            }

            if( pevt->link_if[link_if].error ) {
                fprintf(stderr, "\n[%d] (%d bytes) Link IF %d receive errors", loop, pktlen, link_if);
                fpd_get_link_errors( fd );
            }

            /* check if FPGA receive host data */
            if( pevt->link_if[link_if].host[host_chan].has_data ) {
                pkt->link_if = link_if;
                pkt->host_chan = host_chan;
                do {
                    if (!complete) {
                        offset = accepted_len;
                        pkt->requested_len = pktlen - accepted_len;
                    }
                    else {
                        offset = 0;
                        pkt->requested_len = pktlen;

                        if( debug_on && retry < 2 ) {
                            fprintf(stderr, "[%d]Reading %d bytes of data from Host buffer %d....",
                                    loop, pkt->requested_len, host_chan);
                        }
                    }
                    rc = ioctl(fd, FPD_RECEIVE_HOST_DATA, pkt);
                    if( rc == 0 ) {
                        /* compute buffer address */
                        prxinfo = &pkt->info.rx;
                        bufaddr = (unsigned char *)host_buffer + (prxinfo->buf_id * prxinfo->buf_size) + prxinfo->offset;

                        /* check the data length received */
                        if( pkt->actual_len == pkt->requested_len ) {
                            accepted_len = 0;
                            complete = 1;
                            retry = 0;
                            if( debug_on ) {
                                fprintf(stderr, "passed.\n");
                            }
                        }
                        else if( pkt->actual_len == 0 ) {
                            if( debug_on ) {
                                fprintf(stderr, "No data.\n");
                            }
                            more_data_available = 0;
                            continue;
                        }
                        else {
                            if( debug_on && retry == 0 ) {
                                fprintf(stderr, "incomplete.\n");
                                fprintf(stderr, "Requested %d, Got %d\n",
                                        pkt->requested_len, pkt->actual_len);
                                fprintf(stderr, "BufID %d, startBD %d, endBD %d, offset %d, short_pkt %d, page_end %d\n",
                                        prxinfo->buf_id, prxinfo->startbd, prxinfo->endbd, prxinfo->offset,
                                        prxinfo->short_pkt, prxinfo->reached_page_end);
                            }

                            if( !prxinfo->short_pkt && !prxinfo->reached_page_end ) {
                                ++retry;
                                continue;
                            }
                            else if( prxinfo->reached_page_end ) {
                                accepted_len += pkt->actual_len;
                                complete = 0;
                            }
                            else {
                                fprintf(stderr, "FAILED.\n");
                                if( !debug_on ) {
                                    fprintf(stderr, "[%d]Reading %d bytes of data from Host buffer %d....FAILED\n",
                                            loop, pkt->requested_len, host_chan);
                                }
                                fprintf(stderr, "Requested %d, Got %d\n",
                                        pkt->requested_len, pkt->actual_len);
                                fprintf(stderr, "BufID %d, startBD %d, endBD %d, offset %d, short_pkt %d, page_end %d\n",
                                        prxinfo->buf_id, prxinfo->startbd, prxinfo->endbd, prxinfo->offset,
                                        prxinfo->short_pkt, prxinfo->reached_page_end);

                                /* dump data */
                                if( pkt->actual_len > 0 ) {
                                    dump_data32(bufaddr, pkt->actual_len);
                                }

                                /* release RX buffer */
                                release_rx_buffer(fd, link_if, host_chan, prxinfo);

                                return -3;
                            }
                        }

                        /* compare contents against expected data */
                        if( pkt->actual_len > 0 ) {
                            /* determine test pattern used */
                            switch( pktlen % 3 ) {
                                case 0:
                                    test_pattern = PKT_PATTERN_8BIT_REPEATING;
                                    break;
                                case 1:
                                    test_pattern = PKT_PATTERN_16BIT_ASCENDING;
                                    break;
                                default:
                                    test_pattern = PKT_PATTERN_16BIT_DESCENDING;
                                    break;
                            }

                            if( verify_data(pkt, pkt->actual_len, offset, test_pattern, loop) != 0 ) {
                                /* release RX buffer */
                                release_rx_buffer(fd, link_if, host_chan, prxinfo);

                                return -4;
                            }
                        }

                        /* increment packet length */
                        if( pkt->actual_len == pkt->requested_len ) {
                            /*
                            if( pktlen == 2048 ) {
                            */
                            if( pktlen == dma_page_size ) {
                                /* release buffer */
                                rc = release_rx_buffer(fd, link_if, host_chan, prxinfo);
                                return rc;
                            }
                            else {
                                ++pktlen;
                            }
                        }

                        /* release buffer */
                        if((rc = release_rx_buffer(fd, link_if, host_chan, prxinfo)) != 0) {
                            return rc;
                        }
                    }
                    else {
                        fprintf(stderr, "FAILED.\n");
                        if( !debug_on ) {
                            fprintf(stderr, "[%d]Reading %d bytes of data from Host buffer %d....FAILED\n",
                                    loop, pkt->requested_len, host_chan);
                        }
                        if(errno) {
                            fprintf(stderr, "ioctl failed : %s\n", strerror(errno));
                        }
                        return -6;
                    }
                } while( more_data_available );
            }
        }
    }

    return 0;
}

static int receive_static_size_pkt( int fd, unsigned int link_if, unsigned int host_chan, int pktlen, int loop )
{
    FPD_hostdata_t host_data, *pkt = &host_data;
    FPD_host_rxparam_t *prxinfo;
    unsigned char *bufaddr;
    int more_data_available;

    FPD_event_t event, *pevt = &event;
    unsigned int timeout = POLL_TIMEOUT;
    struct pollfd fds;
    pkt_pattern_t test_pattern;
    int accepted_len = 0;
    int complete = 1;
    int offset = 0;
    int retry = 0;
    int n_repetitions = 1;
    int rc;

    fds.fd = fd;
    fds.events = POLLIN;

    while(1) {
        more_data_available = 1;
        if( debug_on ) {
            fprintf(stderr, "waiting for events ... ");
        }

        rc = poll(&fds, 1, timeout);
        if ((rc == 0) || (rc < 0 && errno == EINTR)) {
            if( !debug_on ) {
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "[%d] (%d - %d bytes) waiting for events ... timedout\n", loop, n_repetitions+1, pktlen);
            }
            else {
                fprintf(stderr, "timedout\n");
            }
            return -1;
        }
        else if( rc < 0 ) {
            if( !debug_on ) {
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "waiting for events ... poll() returned ERROR %d\n", rc);
            }
            else {
                fprintf(stderr, "poll() returned %d\n", rc);
            }
            return -2;
        }
        
        if( debug_on ) {
            fprintf(stderr, "receive notifications\n");
            fprintf(stderr, "getting events ... ");
        }

        rc = ioctl(fd, FPD_CHECK_EVENTS, pevt);

        if( debug_on ) {
            fprintf(stderr, "done\n");
        }

        if (rc == 0) {
            /* error detected */
            if( pevt->pci_error ) {
                fprintf(stderr, "\n[%d] (%d bytes) Detected PCI errors", loop, pktlen);
                fpd_get_pci_errors( fd );
            }

            if( pevt->link_if[link_if].error ) {
                fprintf(stderr, "\n[%d] (%d bytes) Link IF %d receive errors", loop, pktlen, link_if);
                fpd_get_link_errors( fd );
            }

            /* check if FPGA receive host data */
            if( pevt->link_if[link_if].host[host_chan].has_data ) {
                pkt->link_if = link_if;
                pkt->host_chan = host_chan;
                do {
                    if (!complete) {
                        offset = accepted_len;
                        pkt->requested_len = pktlen - accepted_len;
                    }
                    else {
                        offset = 0;
                        pkt->requested_len = pktlen;

                        if( debug_on && retry < 2 ) {
                            fprintf(stderr, "[%d]Reading (%d) %d bytes of data from Host buffer %d....",
                                    loop, n_repetitions, pkt->requested_len, host_chan);
                        }
                    }
                    rc = ioctl(fd, FPD_RECEIVE_HOST_DATA, pkt);
                    if( rc == 0 ) {
                        /* compute buffer address */
                        prxinfo = &pkt->info.rx;
                        bufaddr = (unsigned char *)host_buffer + (prxinfo->buf_id * prxinfo->buf_size) + prxinfo->offset;

                        /* check the data length received */
                        if( pkt->actual_len == pkt->requested_len ) {
                            accepted_len = 0;
                            complete = 1;
                            retry = 0;
                            if( debug_on ) {
                                fprintf(stderr, "passed.\n");
                            }
                        }
                        else if( pkt->actual_len == 0 ) {
                            if( debug_on ) {
                                fprintf(stderr, "No data.\n");
                            }
                            more_data_available = 0;
                            continue;
                        }
                        else {
                            if( debug_on && retry == 0 ) {
                                fprintf(stderr, "incomplete.\n");
                                fprintf(stderr, "Requested %d, Got %d\n",
                                        pkt->requested_len, pkt->actual_len);
                                fprintf(stderr, "BufID %d, startBD %d, endBD %d, offset %d, short_pkt %d, page_end %d\n",
                                        prxinfo->buf_id, prxinfo->startbd, prxinfo->endbd, prxinfo->offset,
                                        prxinfo->short_pkt, prxinfo->reached_page_end);
                            }

                            if( !prxinfo->short_pkt && !prxinfo->reached_page_end ) {
                                ++retry;
                                continue;
                            }
                            else if( prxinfo->reached_page_end ) {
                                accepted_len += pkt->actual_len;
                                complete = 0;
                            }
                            else {
                                fprintf(stderr, "FAILED.\n");
                                if( !debug_on ) {
                                    fprintf(stderr, "[%d]Reading (%d) %d bytes of data from Host buffer %d....FAILED\n",
                                            loop, n_repetitions, pkt->requested_len, host_chan);
                                }
                                fprintf(stderr, "Requested %d, Got %d\n",
                                        pkt->requested_len, pkt->actual_len);
                                fprintf(stderr, "BufID %d, startBD %d, endBD %d, offset %d, short_pkt %d, page_end %d\n",
                                        prxinfo->buf_id, prxinfo->startbd, prxinfo->endbd, prxinfo->offset,
                                        prxinfo->short_pkt, prxinfo->reached_page_end);

                                /* dump data */
                                if( pkt->actual_len > 0 ) {
                                    dump_data32(bufaddr, pkt->actual_len);
                                }

                                /* release RX buffer */
                                release_rx_buffer(fd, link_if, host_chan, prxinfo);

                                return -3;
                            }
                        }

                        /* compare contents against expected data */
                        if( pkt->actual_len > 0 ) {
                            /* determine test pattern used */
                            switch( n_repetitions % 3 ) {
                                case 0:
                                    test_pattern = PKT_PATTERN_8BIT_REPEATING;
                                    break;
                                case 1:
                                    test_pattern = PKT_PATTERN_16BIT_ASCENDING;
                                    break;
                                default:
                                    test_pattern = PKT_PATTERN_16BIT_DESCENDING;
                                    break;
                            }

                            if( verify_data(pkt, pkt->actual_len, offset, test_pattern, loop) != 0 ) {
                                /* release RX buffer */
                                release_rx_buffer(fd, link_if, host_chan, prxinfo);

                                return -4;
                            }
                        }

                        /* increment number of repetitions */
                        if( pkt->actual_len == pkt->requested_len ) {
                            if( n_repetitions == STATIC_SIZE_TEST_LOOP ) {
                                /* release buffer */
                                rc = release_rx_buffer(fd, link_if, host_chan, prxinfo);
                                return rc;
                            }
                            else {
                                ++n_repetitions;
                            }
                        }

                        /* release buffer */
                        if((rc = release_rx_buffer(fd, link_if, host_chan, prxinfo)) != 0) {
                            return rc;
                        }
                    }
                    else {
                        fprintf(stderr, "FAILED.\n");
                        if( !debug_on ) {
                            fprintf(stderr, "[%d]Reading (%d) %d bytes of data from Host buffer %d....FAILED\n",
                                    loop, n_repetitions, pkt->requested_len, host_chan);
                        }
                        if(errno) {
                            fprintf(stderr, "ioctl failed : %s\n", strerror(errno));
                        }
                        return -6;
                    }
                } while( more_data_available );
            }
        }
    }

    return 0;
}

int receive_host_data_continuously( int fd, unsigned int link_if, unsigned int host_chan )
{
    int size;
    unsigned int loop = 0;
    int just_do_it = 1;
    int rc;

    fprintf(stderr, "\n");
    system("date");

    while( just_do_it ) {
        loop++;

        /* static size packet test */
        if( !debug_on ) {
            fprintf(stderr, "\n[%d]Receiving static size packets from Host %d....",
                    loop, host_chan);
        }
        else {
            fprintf(stderr, "\n");
        }

        for( size = 256; size <= dma_page_size; size += 256 ) {
            rc = receive_static_size_pkt(fd, link_if, host_chan, size, loop);
            if( rc != 0 ) {
                return rc;
            }
        }

        if( !debug_on ) {
            fprintf(stderr, "passed.");
        }

        /* incrementing packet test using 3 different patterns */
        if( !debug_on ) {
            fprintf(stderr, "\n[%d]Receiving incrementing packets from Host %d...",
                    loop, host_chan);
        }
        else {
            fprintf(stderr, "\n");
        }

        rc = receive_incrementing_pkt(fd, link_if, host_chan, loop);
        if( rc == 0 ) {
            if( !debug_on ) {
                fprintf(stderr, "passed.");
            }
        }
        else {
            return rc;
        }

        if( one_pass_only ) {
            just_do_it = 0;
        }
    }

    fprintf(stderr, "\n\n");
    system("date");

    return 0;
}
