/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Device Diagnostic Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>
#include <time.h>

#include "fpd_ioctl.h"
#include "fpd-diagnostics.h"
#include "legacy_cim.h"


static int pkt_buffer = FPD_CIM_BUFFER;
static int pkt_cnt_in_ram = 1;
static int pkt_w_chksum = 0;
static int table_setup = FILL_ALL_ADDR;
static int pkt_size = 32;
static pkt_pattern_t current_pattern;
static int one_pass_only = 1;


static int invalidate_all_cmds( int fd, int link_if )
{
    FPD_cmdinv_t cmd;
    int rc;

    cmd.link_if = link_if;
    rc = ioctl(fd, FPD_INVALIDATE_ALL_CMDS, &cmd);

    return rc;
}

static unsigned char compute_paragon_chksum( unsigned char *pbuf, int len )
{
    int chksum = 0, i;
    unsigned char result;

    for( i = 0; i < len; i++ ) {
        chksum += pbuf[i];
    }
    result = (0 - chksum) & 0xFF;

    return result;
}

static int configure_lookup_table( int fd, int link_if, int dest, int in_ram, int w_chksum, int fill )
{
    FPD_cmdprog_t iocmd;
    int i, rc;
    int start, end, skip;

    /* invalidate all commands in the lookup table */
    if( invalidate_all_cmds(fd, link_if) != 0 ) {
        fprintf(stderr, "ERROR: Failed to invalidate all cmds! (%s)\n", strerror(errno));
        return -1;
    }

    switch( fill ) {
        case FILL_ALL_ADDR:
            if( in_ram ) {
                if( w_chksum ) {
                    start = 2;
                }
                else {
                    start = 0;
                }
                end = FPD_MAX_PARAGON_PACKET_SIZE;
            }
            else {
                if( w_chksum ) {
                    start = 3;
                }
                else {
                    start = 2;
                }
                end = FPD_MAX_PARAGON_PACKET_SIZE + 1;
            }
            skip = 1;
            break;
        case FILL_ODD_ADDR:
            if( in_ram ) {
                if( w_chksum ) {
                    start = 3;
                }
                else {
                    start = 1;
                }
                end = FPD_MAX_PARAGON_PACKET_SIZE;
            }
            else {
                if( w_chksum ) {
                    start = 5;
                }
                else {
                    start = 3;
                }
                end = FPD_MAX_PARAGON_PACKET_SIZE + 1;
            }
            skip = 2;
            break;
        case FILL_EVEN_ADDR:
            if( in_ram ) {
                if( w_chksum ) {
                    start = 2;
                }
                else {
                    start = 0;
                }
            }
            else {
                if( w_chksum ) {
                    start = 4;
                }
                else {
                    start = 2;
                }
            }
            end = FPD_MAX_PARAGON_PACKET_SIZE + 1;
            skip = 2;
            break;
        default:
            fprintf(stderr, "ERROR: Invalid Fill Address!\n");
            return -2;
    }

    /* write commands */
    iocmd.link_if = link_if;
    iocmd.in_ram = in_ram;
    iocmd.chksum = w_chksum;
    iocmd.dest = dest;
    iocmd.valid = 1;
    for( i = start; i < end; i+=skip ) {
        iocmd.cmd = i;
        iocmd.byte_cnt = i + 1;
        rc = ioctl(fd, FPD_WRITE_ENTRY_CMD_TABLE, &iocmd);
        if( rc != 0 ) {
            fprintf(stderr, "ERROR: FPD_WRITE_ENTRY_CMD_TABLE ioctl failed (%s)\n", strerror(errno));
            if( invalidate_all_cmds(fd, link_if) != 0 ) {
                fprintf(stderr, "ERROR: Failed to invalidate all cmds! (%s)\n", strerror(errno));
                return -1;
            }
            return -3;
        }
    }

    /* everything's fine so store the new setup */
    pkt_buffer = dest;
    pkt_cnt_in_ram = in_ram;
    pkt_w_chksum = w_chksum;
    table_setup = fill;

    return 0;
}

static int fill_data( unsigned char *pbuf, int pktlen, pkt_pattern_t pattern )
{
    /* set command byte */
    pbuf[0] = pktlen - 1;

    if( pktlen > 3 ) {
        switch( pattern ) {
            case PKT_PATTERN_8BIT_ASCENDING:
                memcpy(pbuf+1, &ascending[0], pktlen);
                break;
            case PKT_PATTERN_8BIT_DESCENDING:
                memcpy(pbuf+1, &descending[0], pktlen);
                break;
            case PKT_PATTERN_8BIT_REPEATING:
                memcpy(pbuf+1, &repeating[0], pktlen);
                break;
            default:
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "ERROR: Unknown Packet Pattern!\n");
                return -1;
        }

        if( !pkt_cnt_in_ram ) {
            if( pkt_w_chksum ) {
                pbuf[1] = pktlen - 2;
            }
            else {
                pbuf[1] = pktlen - 1;
            }
        }
        if( pkt_w_chksum ) {
            pbuf[pktlen-1] = compute_paragon_chksum(pbuf, pktlen-1);
        }
    }
    else {
        switch( pktlen ) {
            case 1:
                if( pkt_w_chksum ) {
                    fprintf(stderr, "FAILED.\n");
                    fprintf(stderr, "Nothing to send (len=%d but pkt has chksum)\n", pktlen);
                    return -2;
                }
                if( !pkt_cnt_in_ram ) {
                    fprintf(stderr, "FAILED.\n");
                    fprintf(stderr, "Nothing to send (len=%d but it is inside pkt)\n", pktlen);
                    return -3;
                }
                break;
            case 2:
                if( pkt_w_chksum ) {
                    fprintf(stderr, "FAILED.\n");
                    fprintf(stderr, "Nothing to send (len=%d but pkt has chksum)\n", pktlen);
                    return -2;
                }
                if( !pkt_cnt_in_ram ) {
                    if( pkt_w_chksum ) {
                        pbuf[1] = pktlen - 2;
                    }
                    else {
                        pbuf[1] = pktlen - 1;
                    }
                }
                else {
                    switch( pattern ) {
                        case PKT_PATTERN_8BIT_ASCENDING:
                            pbuf[1] = ascending[0];
                            break;
                        case PKT_PATTERN_8BIT_DESCENDING:
                            pbuf[1] = descending[0];
                            break;
                        case PKT_PATTERN_8BIT_REPEATING:
                            pbuf[1] = repeating[0];
                            break;
                        default:
                            fprintf(stderr, "FAILED.\n");
                            fprintf(stderr, "ERROR: Unknown Packet Pattern!\n");
                            return -1;
                    }
                }
                break;
            case 3:
                if( pkt_w_chksum && !pkt_cnt_in_ram ) {
                    fprintf(stderr, "FAILED.\n");
                    fprintf(stderr, "Nothing to send (len=%d but pkt has chksum and cnt is inside pkt)\n", pktlen);
                    return -4;
                }

                switch( pattern ) {
                    case PKT_PATTERN_8BIT_ASCENDING:
                        pbuf[1] = ascending[0];
                        pbuf[2] = ascending[1];
                        break;
                    case PKT_PATTERN_8BIT_DESCENDING:
                        pbuf[1] = descending[0];
                        pbuf[2] = descending[1];
                        break;
                    case PKT_PATTERN_8BIT_REPEATING:
                        pbuf[1] = repeating[0];
                        pbuf[2] = repeating[1];
                        break;
                    default:
                        fprintf(stderr, "FAILED.\n");
                        fprintf(stderr, "ERROR: Unknown Packet Pattern!\n");
                        return -1;
                }

                if( !pkt_cnt_in_ram ) {
                    if( pkt_w_chksum ) {
                        pbuf[1] = pktlen - 2;
                    }
                    else {
                        pbuf[1] = pktlen - 1;
                    }
                }
                if( pkt_w_chksum ) {
                    pbuf[pktlen-1] = compute_paragon_chksum(pbuf, pktlen-1);
                }
                break;
            default:
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "ERROR: Packet size is %d\n", pktlen);
                return -5;
        }
    }

    return 0;
}

static void dump_data( unsigned char *bufaddr, int pktlen )
{
    int i;

    for( i = 0; i < pktlen; i++ ) {
        if((i % 0x10) == 0) {
            printf("\n%08x: ", i);
        }
        printf("%02x ", bufaddr[i]);
    }
    printf("\n");

    return;
}

static void display_verify_data_error( int len, int type, int round )
{
    fprintf(stderr, "FAILED\n");
    if( !debug_on ) {
        fprintf(stderr, "[%d]Comparing contents with expected data (%d bytes) from %s buffer....FAILED\n",
                round, len, type?"Priority":"CIM");
    }

    return;
}

static int verify_data( int type, unsigned char *pbuf, int len, pkt_pattern_t pattern, int round )
{
    int i, rc;
    int offset = 1, size = len - 1;
    int idx = 0;
    int paragon_len_field;
    int chksum;

    if( (len-1) != pbuf[0] ) {
        fprintf(stderr, "FAILED\n");
        fprintf(stderr, "ERROR: Expected CMD 0x%02x, Rcvd 0x%02x\n", len-1, pbuf[0]);
        return -1;
    }

    if( !pkt_cnt_in_ram ) {
        if( pkt_w_chksum ) {
            paragon_len_field = len - 2;
        }
        else {
            paragon_len_field = len - 1;
        }

        if( pbuf[1] != paragon_len_field ) {
            fprintf(stderr, "FAILED\n");
            fprintf(stderr, "ERROR: Expected Len 0x%02x, Rcvd 0x%02x\n", len, pbuf[1]);
            return -2;
        }
        offset += 1;
        idx += 1;
        size -= 1;
    }

    if( pkt_w_chksum ) {
        chksum = compute_paragon_chksum(pbuf, len-1);
        if( pbuf[len-1] != chksum ) {
            fprintf(stderr, "FAILED\n");
            fprintf(stderr, "ERROR: Expected Chksum 0x%02x, Rcvd 0x%02x", chksum, pbuf[len-1]);
            return -3;
        }
        size -= 1;
    }

    /* compare contents against expected data */
    switch( pattern ) {
        case PKT_PATTERN_8BIT_ASCENDING:
            rc = memcmp((const void *)pbuf+offset, (const void *)&ascending[idx], size);
            if( rc != 0 ) {
                display_verify_data_error(len, type, round);
                for( i = offset; i < size; i++ ) {
                    if( pbuf[i] != ascending[idx+i-offset] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, ascending[idx+i-offset], pbuf[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_8BIT_DESCENDING:
            rc = memcmp((const void *)pbuf+offset, (const void *)&descending[idx], size);
            if( rc != 0 ) {
                display_verify_data_error(len, type, round);
                for( i = offset; i < size; i++ ) {
                    if( pbuf[i] != descending[idx+i-offset] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, descending[idx+i-offset], pbuf[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_8BIT_REPEATING:
            rc = memcmp((const void *)pbuf+offset, (const void *)&repeating[idx], size);
            if( rc != 0 ) {
                display_verify_data_error(len, type, round);
                for( i = offset; i < size; i++ ) {
                    if( pbuf[i] != repeating[idx+i-offset] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, repeating[idx+i-offset], pbuf[i]);
                    }
                }
            }
            break;

        default:
            fprintf(stderr, "FAILED\n");
            fprintf(stderr, "ERROR: Unknown Packet Pattern!\n");
            return -1;
    }

    if( rc != 0 ) {
        fprintf(stderr, "Expected %d bytes of ", len);
        switch( current_pattern ) {
            case PKT_PATTERN_8BIT_ASCENDING:
                fprintf(stderr, "8-bit ascending ");
                break;
            case PKT_PATTERN_8BIT_DESCENDING:
                fprintf(stderr, "8-bit descending ");
                break;
            case PKT_PATTERN_8BIT_REPEATING:
                fprintf(stderr, "8-bit repeating ");
                break;
            default:
                break;
        }
        fprintf(stderr, "pattern but received different data\n");

        /* dump data */
        if( len > 0 ) {
            dump_data(pbuf, len);
        }
    }

    return rc;
}

void legacy_cim_init( int fd, int link_if, int connected )
{
    line_connection_t old_conn, *pconn = NULL;

    if( connected ) {
        /* retrieve current line connection to be used for re-connection */
        if( fpd_get_line_connection_info(link_if, &pconn) == 0 ) {
            old_conn = *pconn;
        }

        if( fpd_disconnect_link(fd, link_if) != 0 ) {
            fprintf(stderr, "\nERROR: Failed to disconnect Link %d\n", link_if);
            return;
        }
    }

    if( configure_lookup_table(fd, link_if, pkt_buffer, pkt_cnt_in_ram, pkt_w_chksum, table_setup) != 0 ) {
        return;
    }

    if( connected ) {
        /* re-connect link */
        pconn = &old_conn;
        if( fpd_switch_link(fd, link_if, pconn->target_port, pconn->protocol, pconn->tx_parity, pconn->rx_parity) != 0 ) {
            fprintf(stderr, "\nERROR: Failed to re-connect Link %d\n", link_if);
            return;
        }
        else {
            fprintf(stderr, "\nERROR: Failed to re-connect Link %d (2)\n", link_if);
        }
    }

    return;
}

void legacy_cim_display_setup( void )
{
    printf("\nPacket Destination Buffer      : ");
    switch( pkt_buffer ) {
        case FPD_CIM_BUFFER:
            printf("CIM");
            break;
        case FPD_PRIORITY_BUFFER:
            printf("Priority");
            break;
        default:
            printf("UNKNOWN");
            break;
    }
    printf("\nPacket Count Location          : ");
    switch( pkt_cnt_in_ram ) {
        case 0:
            printf("Inside packet (2nd byte)");
            break;
        case 1:
            printf("In Lookup Table");
            break;
        default:
            printf("UNKNOWN");
            break;
    }
    printf("\nPacket with Checksum           : %s",
           pkt_w_chksum?"Yes":"No");
    printf("\nCommand Lookup Table Setup     : ");
    switch( table_setup ) {
        case FILL_ALL_ADDR:
            printf("Fill all address");
            break;
        case FILL_ODD_ADDR:
            printf("Fill odd address (Even packet size)");
            break;
        case FILL_EVEN_ADDR:
            printf("Fill even address (Odd packet size)");
            break;
        default:
            printf("UNKNOWN");
            break;
    }

    printf("\n\nPacket Size                    : %d", pkt_size);
    printf("\nPacket Pattern                 : ");
    switch( current_pattern ) {
        case PKT_PATTERN_8BIT_ASCENDING:
            printf("8-bit ascending");
            break;
        case PKT_PATTERN_8BIT_DESCENDING:
            printf("8-bit descending");
            break;
        case PKT_PATTERN_8BIT_REPEATING:
            printf("8-bit repeating");
            break;
        default:
            printf("UNKNOWN");
    }
    printf("\n");

    return;
}

int legacy_cim_display_valid_cmds( int fd, int link_if )
{
    FPD_cmdprog_t cmd;
    line_connection_t *pconn = NULL;
    unsigned short addr;
    int rc;

    /* required to disconnect link before register is accessible */
    if( fpd_disconnect_link(fd, link_if) != 0 ) {
        fprintf(stderr, "\nERROR: Failed to disconnect Link %d\n", link_if);
        return -1;
    }

    printf("\nCommand\t\tCnt_in_Table\tLength\tHas_Chksum\tBuffer\n");
    for( addr = 0; addr <= FPD_MAX_PARAGON_PACKET_SIZE; addr++ ) {
        cmd.link_if = link_if;
        cmd.cmd = addr;
        rc = ioctl(fd, FPD_READ_ENTRY_CMD_TABLE, &cmd);
        if( rc == 0 ) {
            if( cmd.valid ) {
                printf("%02xh\t\t%s\t\t", cmd.cmd, cmd.in_ram?"y":"n");
                if( cmd.in_ram ) {
                    printf("%d\t", cmd.byte_cnt);
                }
                else {
                    printf("N/A\t");
                }
                printf("%s\t\t", cmd.chksum?"y":"n");
                printf("%s\n", cmd.dest?"Priority":"CIM");
            }
        }
        else {
            fprintf(stderr, "FPD_READ_ENTRY_CMD_TABLE ioctl failed : %s\n",
                    strerror(errno));
            return -2;
        }
    }

    /* retrieve prior line connection */
    if( fpd_get_line_connection_info(link_if, &pconn) == 0 ) {
        /* re-connect link */
        if( fpd_switch_link(fd, link_if, pconn->target_port, pconn->protocol, pconn->tx_parity, pconn->rx_parity) != 0 ) {
            fprintf(stderr, "\nERROR: Failed to re-connect Link %d\n", link_if);
            return -3;
        }
    }
    else {
        fprintf(stderr, "\nERROR: Failed to re-connect Link %d (2)\n", link_if);
        return -4;
    }

    return 0;
}

int legacy_cim_change_setup( int fd, int link_if )
{
    line_connection_t *pconn = NULL;
    int i;
    int buffer, cntloc, w_chksum, fill_type;

    printf("\nPacket Destination Buffer");
    printf("\n=========================");
    printf("\n\t %d: CIM Buffer", FPD_CIM_BUFFER);
    printf("\n\t %d: Priority Buffer", FPD_PRIORITY_BUFFER);
    printf("\n\nPlease choose: ");
    scanf("%d", &buffer);
    switch( buffer ) {
        case 0:
        case 1:
            break;
        default:
            printf("\nERROR: Invalid selection!\n");
            return -1;
    }

    i = 0;
    printf("\nPacket Count Location");
    printf("\n=====================");
    printf("\n\t %d: Inside packet (2nd byte)", i++);
    printf("\n\t %d: In Lookup Table", i++);
    printf("\n\nPlease choose: ");
    scanf("%d", &cntloc);
    switch( cntloc ) {
        case 0:
        case 1:
            break;
        default:
            printf("\nERROR: Invalid selection!\n");
            return -2;
    }

    i = 0;
    printf("\nPacket Contains Checksum");
    printf("\n========================");
    printf("\n\t %d: No", i++);
    printf("\n\t %d: Yes", i++);
    printf("\n\nPlease choose: ");
    scanf("%d", &w_chksum);
    switch( w_chksum ) {
        case 0:
        case 1:
            break;
        default:
            printf("\nERROR: Invalid selection!\n");
            return -3;
    }

    i = 0;
    printf("\nCommand Lookup Table Setup");
    printf("\n==========================");
    printf("\n\t %d: Fill all address", i++);
    printf("\n\t %d: Fill odd address (Even packet size)", i++);
    printf("\n\t %d: Fill even address (Odd packet size)", i++);
    printf("\n\nPlease choose: ");
    scanf("%d", &fill_type);
    switch( fill_type ) {
        case FILL_ALL_ADDR:
        case FILL_ODD_ADDR:
        case FILL_EVEN_ADDR:
            break;
        default:
            printf("\nERROR: Invalid selection!\n");
            return -4;
    }

    /* required to disconnect link before re-configuring */
    if( fpd_disconnect_link(fd, link_if) != 0 ) {
        fprintf(stderr, "\nERROR: Failed to disconnect Link %d\n", link_if);
        return -5;
    }

    if( configure_lookup_table(fd, link_if, buffer, cntloc, w_chksum, fill_type) != 0 ) {
        return -6;
    }

    /* retrieve prior line connection */
    if( fpd_get_line_connection_info(link_if, &pconn) == 0 ) {
        /* re-connect link */
        if( fpd_switch_link(fd, link_if, pconn->target_port, pconn->protocol, pconn->tx_parity, pconn->rx_parity) != 0 ) {
            fprintf(stderr, "\nERROR: Failed to re-connect Link %d\n", link_if);
            return -7;
        }
    }
    else {
        fprintf(stderr, "\nERROR: Failed to re-connect Link %d (2)\n", link_if);
        return -8;
    }

    return 0;
}

int legacy_cim_change_pkt( void )
{
    unsigned int val;
    int i = 0;
    int opt;
    int max_size = FPD_MAX_PARAGON_PACKET_SIZE;

    printf("\nEnter New Packet Size (%d max)  : ", max_size);
    scanf("%d", &val);
    if( val > max_size ) {
        printf("\nERROR: Invalid size\n");
        return -1;
    }
    pkt_size = val;

    i = 0;
    printf("\nPacket Pattern:");
    printf("\n===============");
    printf("\n\t %d: 8-bit Ascending", i++);
    printf("\n\t %d: 8-bit Descending", i++);
    printf("\n\t %d: 8-bit Repeating", i++);
    printf("\n\nPlease choose: ");
    scanf("%d", &opt);

    switch( opt ) {
        case PKT_PATTERN_8BIT_ASCENDING:
        case PKT_PATTERN_8BIT_DESCENDING:
        case PKT_PATTERN_8BIT_REPEATING:
            current_pattern = opt;
            break;
        default:
            printf("\nERROR: Invalid Pattern selection!\n");
            return -2;
    }

    return 0;
}

int legacy_cim_send_pkt( int fd, unsigned int link_if )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    int rc;

    pdata->link_if = link_if;
    pdata->requested_len = pkt_size;
    pdata->type = pkt_buffer;
    pdata->buf = protocol_buffer;

    fprintf(stderr, "\nSending %d bytes of data to %s buffer....",
            pdata->requested_len, pdata->type?"Priority":"CIM");

    if( fill_data(pdata->buf, pkt_size, current_pattern ) != 0 ) {
        return -1;
    }

    rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, pdata);
    if( rc == 0 ) {
        if( pdata->actual_len == pdata->requested_len )
            fprintf(stderr, "passed.\n");
        else {
            fprintf(stderr, "FAILED.\n");
            fprintf(stderr, "Actual bytes written %d\n", pdata->actual_len);
        }
    }
    else {
        fprintf(stderr, "FAILED.\n");
        if(errno) {
            fprintf(stderr, "FPD_SEND_PROTOCOL_DATA ioctl failed : %s\n", strerror(errno));
        }
        return -2;
    }

    return 0;
}

int legacy_cim_receive_pkt( int fd, unsigned int link_if )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    int rx_cim = 0, rx_priority = 0;
    int rc;

    if( fpd_poll_once(fd, pevt) == 0 ) {
        /* PCI error detected */
        if( pevt->pci_error ) {
            fprintf(stderr, "\nDetected PCI errors");
            fpd_get_pci_errors( fd );
        }

        /* check if FPGA receive protocol data */
        if( link_if == FPD_LINK_IF_ID_BGND ) {
            if( pkt_buffer == FPD_CIM_BUFFER ) {
                /* CIM Buffer */
                if( pevt->bgnd_link_if.rx_cim ) {
                    rx_cim = 1;
                }
            }

            /* Link error detected */
            if( pevt->bgnd_link_if.error ) {
                fprintf(stderr, "\nBackground Link IF receive errors");
                fpd_get_link_errors( fd );
            }
        }
        else {
            if( pkt_buffer == FPD_CIM_BUFFER ) {
                /* CIM Buffer */
                if( pevt->link_if[link_if].rx_cim ) {
                    rx_cim = 1;
                }
            }
            else {
                /* Priority buffer */
                if( pevt->link_if[link_if].rx_priority ) {
                    rx_priority = 1;
                }
            }

            /* error detected */
            if( pevt->link_if[link_if].error ) {
                fprintf(stderr, "\nLink IF %d receive errors", link_if);
                fpd_get_link_errors( fd );
            }
        }
    }

    if( rx_cim || rx_priority ) {
        if( (pkt_buffer == FPD_CIM_BUFFER) && rx_priority ) {
            fprintf(stderr, "ERROR: Expecting pkt from CIM buffer but rcvd from Priority\n");
        }
        if( (pkt_buffer == FPD_PRIORITY_BUFFER) && rx_cim ) {
            fprintf(stderr, "ERROR: Expecting pkt from Priority buffer but rcvd from CIM\n");
        }
        pdata->link_if = link_if;
        pdata->requested_len = pkt_size;
        pdata->type = pkt_buffer;
        pdata->buf = protocol_buffer;
        fprintf(stderr, "Reading %d bytes of data from %s buffer....",
                pdata->requested_len, pdata->type?"Priority":"CIM");
        rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
        if( rc == 0 ) {
            if( pdata->actual_len == pdata->requested_len ) {
                fprintf(stderr, "passed.\n");
            }
            else {
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "Requested %d, Got %d\n",
                        pdata->requested_len, pdata->actual_len);
            }

            /* compare contents against expected data */
            fprintf(stderr, "Comparing contents with expected data......");
            if( verify_data(pdata->type, pdata->buf, pdata->actual_len, current_pattern, 1) == 0 ) {
                fprintf(stderr, "passed.\n");
            }
            else {
                return -1;
            }
        }
        else {
            fprintf(stderr, "FAILED.\n");
            if(errno) {
                fprintf(stderr, "FPD_RECEIVE_PROTOCOL_DATA ioctl failed : %s\n", strerror(errno));
            }
        }
    }

    return 0;
}

int legacy_cim_send_pkts_continuously( int fd, unsigned int link_if )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    int rc;
    int pktlen;
    int result = 0;
    int keep_running = 1;
    time_t start_time, current_time;
    unsigned int loop = 0;
    pkt_pattern_t test_pattern;
    int start, skip;

    switch( table_setup ) {
        case FILL_ALL_ADDR:
            if( pkt_cnt_in_ram ) {
                if( pkt_w_chksum ) {
                    start = 3;
                }
                else {
                    start = 1;
                }
            }
            else {
                if( pkt_w_chksum ) {
                    start = 4;
                }
                else {
                    start = 3;
                }
            }
            skip = 1;
            break;
        case FILL_ODD_ADDR:
            if( pkt_cnt_in_ram ) {
                if( pkt_w_chksum ) {
                    start = 4;
                }
                else {
                    start = 2;
                }
            }
            else {
                if( pkt_w_chksum ) {
                    start = 6;
                }
                else {
                    start = 4;
                }
            }
            skip = 2;
            break;
        case FILL_EVEN_ADDR:
            if( pkt_cnt_in_ram ) {
                if( pkt_w_chksum ) {
                    start = 3;
                }
                else {
                    start = 1;
                }
            }
            else {
                if( pkt_w_chksum ) {
                    start = 5;
                }
                else {
                    start = 3;
                }
            }
            skip = 2;
            break;
        default:
            fprintf(stderr, "ERROR: Invalid Fill Address!\n");
            return -1;
    }

    pdata->link_if = link_if;
    pdata->buf = protocol_buffer;
    debug_on = 1;

    while( keep_running ) {
        loop++;

        if( !debug_on ) {
            fprintf(stderr, "\n[%d]Sending %d packets to %s buffer....",
                    loop, FPD_MAX_PARAGON_PACKET_SIZE,
                    pkt_buffer?"Priority":"CIM");
        }

        for( pktlen = start; pktlen <= FPD_MAX_PARAGON_PACKET_SIZE; pktlen+=skip ) {

            pdata->link_if = link_if;
            pdata->requested_len = pktlen;
            pdata->type = pkt_buffer;

            /* determine test pattern to use */
            switch( pktlen % 3 ) {
                case 0:
                    test_pattern = PKT_PATTERN_8BIT_ASCENDING;
                    break;
                case 1:
                    test_pattern = PKT_PATTERN_8BIT_DESCENDING;
                    break;
                default:
                    test_pattern = PKT_PATTERN_8BIT_REPEATING;
                    break;
            }

            if( debug_on ) {
                fprintf(stderr, "\n[%d]Sending %d bytes to %s buffer....",
                        loop, pdata->requested_len, pdata->type?"Priority":"CIM");
            }

            if( fill_data(pdata->buf, pktlen, test_pattern) != 0 ) {
                return -1;
            }

resend:
            rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, pdata);
            if( rc == 0 ) {
                if( pdata->actual_len == pdata->requested_len ) {
                    if( debug_on ) {
                        fprintf(stderr, "passed.");
                    }

                    /* reset time of last good send */
                    time(&start_time);
                }
                else {
                    time(&current_time);
                    if(( pdata->actual_len == 0 ) && ( difftime(current_time, start_time) < RETRY_TIMEOUT )) {
                        goto resend;
                    }
                    else {
                        fprintf(stderr, "FAILED.\n");
                        if( !debug_on ) {
                            fprintf(stderr, "\n[%d]Sending %d bytes to %s buffer....FAILED\n",
                                    loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                        }
                        fprintf(stderr, "Actual bytes written %d\n", pdata->actual_len);
                        result = -1;
                        goto send_finish;
                    }
                }
            }
            else {
                if( errno ) {
                    if( errno == ENOSPC ) {
                        time(&current_time);
                        if( difftime(current_time, start_time) < RETRY_TIMEOUT ) {
                            goto resend;
                        }
                        else {
                            fprintf(stderr, "FAILED.\n");
                            if( !debug_on ) {
                                fprintf(stderr, "\n[%d]Sending %d bytes to %s buffer....FAILED\n",
                                        loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                            }
                            fprintf(stderr, "ERROR: %s\n", strerror(errno));
                            result = -1;
                            goto send_finish;
                        }
                    }
                }
            }
        }

        if( !debug_on ) {
            fprintf(stderr, "passed.");
        }

        if( one_pass_only ) {
            keep_running = 0;
        }
    }

send_finish:
    printf("\n");
    debug_on = 0;

    return result;
}

int legacy_cim_receive_pkts_continuously( int fd, unsigned int link_if )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    unsigned int timeout = POLL_TIMEOUT;
    struct pollfd fds;
    pkt_pattern_t test_pattern;
    unsigned int loop = 1;
    int pktlen;
    int rx_cim, rx_priority;
    int more_data_available;
    int rc;
    int start, skip, max_pktlen;

    switch( table_setup ) {
        case FILL_ALL_ADDR:
            if( pkt_cnt_in_ram ) {
                if( pkt_w_chksum ) {
                    start = 3;
                }
                else {
                    start = 1;
                }
            }
            else {
                if( pkt_w_chksum ) {
                    start = 4;
                }
                else {
                    start = 3;
                }
            }
            skip = 1;
            max_pktlen = FPD_MAX_PARAGON_PACKET_SIZE;
            break;
        case FILL_ODD_ADDR:
            if( pkt_cnt_in_ram ) {
                if( pkt_w_chksum ) {
                    start = 4;
                }
                else {
                    start = 2;
                }
            }
            else {
                if( pkt_w_chksum ) {
                    start = 6;
                }
                else {
                    start = 4;
                }
            }
            skip = 2;
            max_pktlen = FPD_MAX_PARAGON_PACKET_SIZE - 1;
            break;
        case FILL_EVEN_ADDR:
            if( pkt_cnt_in_ram ) {
                if( pkt_w_chksum ) {
                    start = 3;
                }
                else {
                    start = 1;
                }
            }
            else {
                if( pkt_w_chksum ) {
                    start = 5;
                }
                else {
                    start = 3;
                }
            }
            skip = 2;
            max_pktlen = FPD_MAX_PARAGON_PACKET_SIZE;
            break;
        default:
            fprintf(stderr, "ERROR: Invalid Fill Address!\n");
            return -1;
    }

    fds.fd = fd;
    fds.events = POLLIN;

    pktlen = start;
    debug_on = 1;

    if( !debug_on ) {
        fprintf(stderr, "\n[%d]Reading %d packets from %s buffer....",
                loop, FPD_MAX_PARAGON_PACKET_SIZE, pkt_buffer?"Priority":"CIM");
    }
    else {
        printf("\n");
    }

    while( 1 ) {
        more_data_available = 1;
        rx_cim = rx_priority = 0;
        if( debug_on ) {
            fprintf(stderr, "waiting for events ... ");
        }

        rc = poll(&fds, 1, timeout);
        if ((rc == 0) || (rc < 0 && errno == EINTR)) {
            if( !debug_on ) {
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "[%d] (%d bytes) waiting for events ... timedout\n", loop, pktlen);
            }
            else {
                fprintf(stderr, "timedout\n");
            }
            return -1;
        }
        else if( rc < 0 ) {
            if( !debug_on ) {
                fprintf(stderr, "FAILED.\n");
                fprintf(stderr, "waiting for events ... poll() returned ERROR %d\n", rc);
            }
            else {
                fprintf(stderr, "poll() returned %d\n", rc);
            }
            return -2;
        }
        
        if( debug_on ) {
            fprintf(stderr, "receive notifications\n");
            fprintf(stderr, "getting events ... ");
        }

        rc = ioctl(fd, FPD_CHECK_EVENTS, pevt);

        if( debug_on ) {
            fprintf(stderr, "done\n");
        }

        if (rc == 0) {
            /* PCI error detected */
            if( pevt->pci_error ) {
                fprintf(stderr, "\n[%d] (%d bytes) Detected PCI errors", loop, pktlen);
                fpd_get_pci_errors( fd );
            }

            /* check if FPGA receive protocol data */
            if( link_if == FPD_LINK_IF_ID_BGND ) {
                if( pkt_buffer == FPD_CIM_BUFFER ) {
                    /* CIM Buffer */
                    if( pevt->bgnd_link_if.rx_cim ) {
                        rx_cim = 1;
                    }
                }

                /* error detected */
                if( pevt->bgnd_link_if.error ) {
                    fprintf(stderr, "\nBackground Link IF receive errors");
                    fpd_get_link_errors( fd );
                }
            }
            else {
                if( pkt_buffer == FPD_CIM_BUFFER ) {
                    /* CIM Buffer */
                    if( pevt->link_if[link_if].rx_cim ) {
                        rx_cim = 1;
                    }
                }
                else {
                    /* Priority buffer */
                    if( pevt->link_if[link_if].rx_priority ) {
                        rx_priority = 1;
                    }
                }

                /* error detected */
                if( pevt->link_if[link_if].error ) {
                    fprintf(stderr, "\n[%d] (%d bytes) Link IF %d receive errors", loop, pktlen, link_if);
                    fpd_get_link_errors( fd );
                }
            }

            if( rx_cim || rx_priority ) {
                if( (pkt_buffer == FPD_CIM_BUFFER) && rx_priority ) {
                    fprintf(stderr, "FAILED.\n");
                    fprintf(stderr, "ERROR: Expecting pkt from CIM buffer but rcvd from Priority\n");
                    return -3;
                }
                if( (pkt_buffer == FPD_PRIORITY_BUFFER) && rx_cim ) {
                    fprintf(stderr, "FAILED.\n");
                    fprintf(stderr, "ERROR: Expecting pkt from Priority buffer but rcvd from CIM\n");
                    return -4;
                }

                pdata->link_if = link_if;
                pdata->type = pkt_buffer;
                do {
                    pdata->requested_len = pktlen;
                    pdata->buf = protocol_buffer;
                    if( debug_on ) {
                        fprintf(stderr, "[%d]Reading %d bytes of data from %s buffer....",
                                loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                    }
                    rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
                    if( rc == 0 ) {
                        if( pdata->actual_len == pdata->requested_len ) {
                            if( debug_on ) {
                                fprintf(stderr, "passed.\n");
                            }
                        }
                        else if( pdata->actual_len == 0 ) {
                            if( debug_on ) {
                                fprintf(stderr, "No data.\n");
                            }
                            more_data_available = 0;
                        }
                        else {
                            fprintf(stderr, "FAILED.\n");
                            if( !debug_on ) {
                                fprintf(stderr, "[%d]Reading %d bytes of data from %s buffer....FAILED\n",
                                        loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                            }
                            fprintf(stderr, "Requested %d, Got %d\n",
                                    pdata->requested_len, pdata->actual_len);
                            return -5;
                        }

                        if( pdata->actual_len > 0 ) {
                            if( debug_on ) {
                                fprintf(stderr, "Comparing contents with expected data......");
                            }

                            /* determine test pattern used */
                            switch( pktlen % 3 ) {
                                case 0:
                                    test_pattern = PKT_PATTERN_8BIT_ASCENDING;
                                    break;
                                case 1:
                                    test_pattern = PKT_PATTERN_8BIT_DESCENDING;
                                    break;
                                default:
                                    test_pattern = PKT_PATTERN_8BIT_REPEATING;
                                    break;
                            }

                            /* compare contents against expected data */
                            if( verify_data(pdata->type, pdata->buf, pdata->actual_len, test_pattern, loop) == 0 ) {
                                if( debug_on ) {
                                    fprintf(stderr, "passed.\n");
                                }
                            }
                            else {
                                return -6;
                            }
                        }

                        /* increment packet length */
                        if( pdata->actual_len == pdata->requested_len ) {
                            if( pktlen == max_pktlen ) {
                                pktlen = start;
                                loop++;
                                if( !debug_on ) {
                                    fprintf(stderr, "passed.\n");
                                    if( !one_pass_only ) {
                                        fprintf(stderr, "[%d]Reading %d packets from %s buffer....",
                                                loop, FPD_MAX_PARAGON_PACKET_SIZE, pdata->type?"Priority":"CIM");
                                    }
                                }

                                if( one_pass_only ) {
                                    debug_on = 0;
                                    return 0;
                                } 
                            }
                            else {
                                pktlen += skip;
                            }
                        }
                    }
                    else {
                        fprintf(stderr, "FAILED.\n");
                        if( !debug_on ) {
                            fprintf(stderr, "[%d]Reading %d bytes of data from %s buffer....FAILED\n",
                                    loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                        }
                        if(errno) {
                            fprintf(stderr, "FPD_RECEIVE_PROTOCOL_DATA ioctl failed : %s\n", strerror(errno));
                        }
                        return -7;
                    }
                } while( more_data_available );
            }
        }
    }

    debug_on = 0;
    return 0;
}
