/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Application to test all the IOCTLs of the FPGA Protocol Device Driver.
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"

#define FPD_TEST_REV        "$Revision: 1.6 $"
#define FPD_APPS_PASS       0
#define FPD_APPS_FAIL      -1

#define LOCAL_VIDEO_DISABLE 0
#define LOCAL_VIDEO_PCIM    1
#define LOCAL_VIDEO_VMCIM   2
#define LOCAL_VIDEO_GUI     3

/* commands */
typedef enum
{
    CMD_GET_INFO = 1,
    CMD_GET_PCI_CFG,
    CMD_READ_REG,
    CMD_WRITE_REG,
    CMD_CHECK_EVENTS,
    CMD_SEND_PROTOCOL_DATA,
    CMD_RECEIVE_PROTOCOL_DATA,
    CMD_SWITCH_CHANNEL,
    CMD_DISCONNECT_CHANNEL,
    CMD_GET_SWITCH_INFO,
    CMD_GET_LINE_STATUS,
    CMD_GET_ERROR,
    CMD_GET_STATISTICS,
    CMD_RUN_DIAGNOSTICS,
    CMD_SET_HOST_BUFFER_CFG,
    CMD_GET_HOST_BUFFER_CFG,
    CMD_GET_HOST_BUFFER,
    CMD_SEND_HOST_DATA,
    CMD_RECEIVE_HOST_DATA,

    CMD_SET_DMA_BURST_SIZE = 21,
    CMD_GET_DMA_BURST_SIZE,
    CMD_SET_TX_TIMEOUT,
    CMD_GET_TX_TIMEOUT,
    CMD_INVALIDATE_ALL_CMDS,
    CMD_GET_RX_INVALID_CMD = 28,

    CMD_SET_LINK0_MODE = 29,
    CMD_GET_LINK0_MODE,

    CMD_SWITCH_REMOTE_VIDEO = 34,
    CMD_DISABLE_REMOTE_VIDEO,
    CMD_SWITCH_LOCAL_VIDEO,
    CMD_WRITE_VSCHIP_REG,
    CMD_READ_ALL_VSCHIP_REG,
    CMD_RESET_LINK_INTERFACE,

    CMD_TRACE_ON = 50,
    CMD_TRACE_OFF,
    CMD_SET_LOG_LVL,
    CMD_GET_LOG_LVL,
    CMD_GET_BUFFER_INFO,
    CMD_GET_HOST_RXBDTBL,
    CMD_GET_HOST_TXBUF_INFO,
} FPD_TEST_CMDS;

static int fpd_verbose = 0;
static void *host_buffer;
static int host_buffer_mmap_len = 0;
static int max_link_if = 0;
static int max_line_cnt = 0;
static char bgnd_link_if_present = 0;

static void display_pci_cfg( FPD_pcicfg_t *pci );
static void display_events( FPD_event_t *pcond );
static int  get_switch_params( FPD_switch_t *pswitch );
static void display_switch_info( FPD_switchinfo_t *pinfo );
static void display_line_status( FPD_linestat_t *pinfo );
static void display_error( FPD_error_t *pinfo );
static void display_stats( FPD_stats_t *pinfo );
static int  get_txpdata_params( FPD_data_t *pdata );
static int  get_rxpdata_params( FPD_data_t *pdata );
static int  get_host_cfg_params( FPD_hostcfg_t *pcfg );
static int  display_host_cfg_params( FPD_hostcfg_t *pcfg );
static void display_buffer_info( FPD_buffer_t *pbinfo );
static void display_host_rxbdtbl( FPD_hostrxbdtbl_t *pbdtbl );
static void display_host_txbuf_info( FPD_host_txbuf_info_t *pinfo );
static int  display_invalid_paragon_cmd( FPD_rxinvcmd_t *pcmd );
static int  get_remote_video_params( FPD_video_switch_t *pvideo );
static int  get_local_video_params( FPD_local_video_switch_t *pvideo );
static void display_vschip_contents( FPD_vschip_reg_t *pchip );
static void usage( void );
static void log( char *fmt, ... );
void msg( char *fmt, ... );
void err( char *fmt, ... );


static int init_host_module( int fd, int overall_buf_len )
{
    host_buffer_mmap_len = overall_buf_len;

    host_buffer = (void *) mmap(NULL,
                                host_buffer_mmap_len,
                                PROT_READ | PROT_WRITE,
                                MAP_SHARED,
                                fd,
                                0 );

    if( host_buffer == (void *) -1 ) {
        return -1;
    }

    return 0;
}

static void cleanup_host_module( void )
{
    munmap(host_buffer, host_buffer_mmap_len);
    return;
}

int main( int argc, char * const argv[] )
{
    int ch;
    int fd;
    char dev[128] = "/dev/fpd";
    unsigned int io = 0;
    int param = -1, param2 = -1, param3 = -1;
    FPD_msg_t gmsg;
    int rc;

    while( (ch = getopt(argc, argv, "hvi:a:b:c:")) != EOF ) {
        switch( ch ) {
            case 'v':
                fpd_verbose = 1;
                break;

            case 'i':
                io = atoi(optarg);
                break;

            case 'a':
                param = strtoul(optarg, NULL, 16);
                break;

            case 'b':
                param2 = strtoul(optarg, NULL, 16);
                break;

            case 'c':
                param3 = strtoul(optarg, NULL, 16);
                break;

            case 'h':
            default:
                usage();
                return 1;
        }
    }

    if( optind == argc-1 ) {
        strcpy(dev, argv[optind++]);
        log( "using %s as device node ...\n", dev );
    }

    if( (fd = open(dev, O_RDWR)) < 0 ) {
        err( "Cannot open %s : %s\n", dev, strerror(errno) );
        return -1;
    }
    log( "File Descriptor %d (%s)\n", fd, dev);

    log( "testing cntl %d...\n", io);

    /* determine Link and Line Interface count */
    rc = ioctl(fd, FPD_GET_INFO, &gmsg);
    if( rc == 0 ) {
        max_link_if = gmsg.info.link_if_cnt;
        max_line_cnt = gmsg.info.line_cnt;
        host_buffer_mmap_len = gmsg.info.host_buffer_memsize;
        bgnd_link_if_present = gmsg.info.bgnd_link_if_present;

        /* initialize host module */
        if( init_host_module(fd, gmsg.info.host_buffer_memsize) < 0 ) {
            fprintf(stderr, "Host Module init failed\n");
            return -2;
        }
    }

    switch(io) {
        case CMD_GET_INFO:
            rc = ioctl(fd, FPD_GET_INFO, &gmsg);
            if (rc == 0) {
                printf("Protocol Version                    : %d.%d.%d\n"
                       "FPGA Version                        : %d.%d.%d\n"
                       "Driver Version                      : %d.%d.%d\n"
                       "Number of Link Interfaces           : %d\n"
                       "Background Link Interface Present   : %s\n"
                       "Number of Lines                     : %d\n"
                       "Number of Host Buffers per Link IF  : %d\n"
                       "Host Buffers Memory Size            : %d\n"
                       "Number of Pages used in Host Buffer : %d\n"
                       "Page size used in Host Buffer       : %d\n",
                       gmsg.info.protocol_version.major,
                       gmsg.info.protocol_version.minor,
                       gmsg.info.protocol_version.patch,
                       gmsg.info.fpga_version.major,
                       gmsg.info.fpga_version.minor,
                       gmsg.info.fpga_version.patch,
                       gmsg.info.driver_version.major,
                       gmsg.info.driver_version.minor,
                       gmsg.info.driver_version.patch,
                       gmsg.info.link_if_cnt,
                       (gmsg.info.bgnd_link_if_present ? "yes" : "no"),
                       gmsg.info.line_cnt,
                       gmsg.info.host_buffer_cnt,
                       gmsg.info.host_buffer_memsize,
                       gmsg.info.host_buffer_pagecnt,
                       gmsg.info.host_buffer_pagesize);
            }
            break;
        case CMD_GET_PCI_CFG:
            rc = ioctl(fd, FPD_GET_PCI_CFG, &gmsg);
            if (rc == 0) {
                /* display info */
                display_pci_cfg(&gmsg.pcicfg);
            }
            break;
        case CMD_READ_REG:
            if(param >= 0) {
                gmsg.rwreg.reg = param;
                rc = ioctl(fd, FPD_READ_REG, &gmsg);
                if (rc == 0) {
                    printf("Reg %x: %08x\n", param, gmsg.rwreg.value);
                }
            }
            else {
                printf("specify register offset in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_WRITE_REG:
            if(param >= 0 && param2 >= 0) {
                gmsg.rwreg.reg = param;
                gmsg.rwreg.value = param2;
                rc = ioctl(fd, FPD_WRITE_REG, &gmsg);
            }
            else {
                printf("specify register offset in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_CHECK_EVENTS:
            rc = ioctl(fd, FPD_CHECK_EVENTS, &gmsg);
            if (rc == 0) {
                /* display info */
                display_events(&gmsg.event);
            }
            break;
        case CMD_SEND_PROTOCOL_DATA:
            if( get_txpdata_params(&gmsg.pdata) == 0 ) {
                printf("Sending %d bytes of data to %s buffer....",
                       gmsg.pdata.requested_len,
                       gmsg.pdata.type?"priority":"CIM");
                rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, &gmsg);
                if( rc == 0 ) {
                    if( gmsg.pdata.actual_len == gmsg.pdata.requested_len )
                        printf("passed.\n");
                    else {
                        printf("FAILED.\n");
                        printf("Actual bytes written %d\n", gmsg.pdata.actual_len);
                    }
                }
                else {
                    printf("FAILED.\n");
                }
                free(gmsg.pdata.buf);
            }
            else {
                rc = -EINVAL;
            }
            break;
        case CMD_RECEIVE_PROTOCOL_DATA:
            if( get_rxpdata_params(&gmsg.pdata) == 0 ) {
                printf("Reading %d bytes of data from %s buffer....",
                       gmsg.pdata.requested_len,
                       gmsg.pdata.type?"priority":"CIM");
                rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, &gmsg);
                if( rc == 0 ) {
                    int i;

                    if( gmsg.pdata.actual_len == gmsg.pdata.requested_len ) {
                        printf("passed.\n");
                    }
                    else {
                        printf("FAILED.\n");
                        printf("Requested %d, Got %d\n",
                               gmsg.pdata.requested_len,
                               gmsg.pdata.actual_len);
                    }
                    for( i = 0; i < gmsg.pdata.actual_len; i++ ) {
                        printf("%02x ", gmsg.pdata.buf[i]);
                    }
                    printf("\n");
                    free(gmsg.pdata.buf);
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                rc = -EINVAL;
            }
            break;
        case CMD_SWITCH_CHANNEL:
            if( get_switch_params(&gmsg.conn) == 0 ) {
                printf("Switching Link IF %d to Target %d....",
                       gmsg.conn.link_if, gmsg.conn.target_port);
                rc = ioctl(fd, FPD_SWITCH_CHANNEL, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                rc = -EINVAL;
            }
            break;
        case CMD_DISCONNECT_CHANNEL:
            if(param >= 0) {
                printf("Disconnect Link IF %d....", param);
                gmsg.disc.link_if = param;

                if( param2 >= 0 ) {
                    gmsg.disc.driver_txbuf_wait_time = param2;
                }
                else {
                    gmsg.disc.driver_txbuf_wait_time = 0;
                }

                if( param3 >= 0 ) {
                    gmsg.disc.fpga_txbuf_wait_time = param3;
                }
                else {
                    gmsg.disc.fpga_txbuf_wait_time = 0;
                }

                rc = ioctl(fd, FPD_DISCONNECT_CHANNEL, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                printf("specify Link IF in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_SWITCH_INFO:
            rc = ioctl(fd, FPD_GET_SWITCH_INFO, &gmsg);
            if (rc == 0) {
                /* display info */
                display_switch_info(&gmsg.switch_info);
            }
            break;
        case CMD_GET_LINE_STATUS:
            rc = ioctl(fd, FPD_GET_LINE_STATUS, &gmsg);
            if (rc == 0) {
                /* display info */
                display_line_status(&gmsg.linestat);
            }
            break;
        case CMD_GET_ERROR:
            if(param >= 0) {
                gmsg.error.type = param;
                if(param == FPD_ERR_TYPE_PCI) {
                    rc = ioctl(fd, FPD_GET_ERROR, &gmsg);
                }
                else if(param == FPD_ERR_TYPE_LINK_IF) {
                    if(param2 >= 0) {
                        gmsg.error.link_if = param2;
                        rc = ioctl(fd, FPD_GET_ERROR, &gmsg);
                    }
                    else {
                        printf("specify valid Link IF in -b option\n");
                        rc = -EINVAL;
                    }
                }
                else {
                    printf("specify correct Error type in -a option\n");
                    rc = -EINVAL;
                }

                if (rc == 0) {
                    /* display info */
                    display_error(&gmsg.error);
                }
            }
            else {
                printf("specify Error type in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_STATISTICS:
            if(param >= 0) {
                gmsg.stats.link_if = param;
                rc = ioctl(fd, FPD_GET_STATISTICS, &gmsg);
                if (rc == 0) {
                    /* display info */
                    display_stats(&gmsg.stats);
                }
            }
            else {
                printf("specify Link IF in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_SET_HOST_BUFFER_CFG:
            if( get_host_cfg_params(&gmsg.host_cfg) == 0 ) {
                printf("Configuring Link IF %d Host Buffer %d....",
                       gmsg.host_cfg.link_if, gmsg.host_cfg.host_chan);
                rc = ioctl(fd, FPD_SET_HOST_BUFFER_CFG, &gmsg);
                if( rc == 0 ) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                rc = -EINVAL;
            }
            break;
        case CMD_GET_HOST_BUFFER_CFG:
            if(param >= 0 && param2 >= 0) {
                gmsg.host_cfg.link_if = param;
                gmsg.host_cfg.host_chan = param2;
                rc = ioctl(fd, FPD_GET_HOST_BUFFER_CFG, &gmsg);
                if( rc == 0 ) {
                    display_host_cfg_params(&gmsg.host_cfg);
                }
            }
            else {
                printf("specify link IF in -a and Host Channel in -b option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_SEND_HOST_DATA:
            rc = 0;
            break;
        case CMD_RECEIVE_HOST_DATA:
            rc = 0;
            break;
        case CMD_SET_DMA_BURST_SIZE:
            if(param >= 0 ) {
                printf("Setting DMA burst size to %d....", param);
                gmsg.dma_burst.size = param;
                rc = ioctl(fd, FPD_SET_DMA_BURST_SIZE, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                printf("specify DMA burst size in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_DMA_BURST_SIZE:
            rc = ioctl(fd, FPD_GET_DMA_BURST_SIZE, &gmsg);
            if (rc == 0) {
                printf("\nDMA burst size : %d\n", gmsg.dma_burst.size);
            }
            break;
        case CMD_SET_TX_TIMEOUT:
            if(param >= 0 ) {
                printf("Setting TX timeout to %d....", param);
                gmsg.tx_timeout.timeout = param;
                rc = ioctl(fd, FPD_SET_TX_TIMEOUT, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                printf("specify TX timeout in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_TX_TIMEOUT:
            rc = ioctl(fd, FPD_GET_TX_TIMEOUT, &gmsg);
            if (rc == 0) {
                printf("\nTransmit timeout : %d\n", gmsg.tx_timeout.timeout);
            }
            break;
        case CMD_INVALIDATE_ALL_CMDS:
            if(param >= 0 ) {
                printf("Invalidating Paragon Cmd Lookup Table for Link %d....", param);
                gmsg.invalidate_cmd.link_if = param;
                rc = ioctl(fd, FPD_INVALIDATE_ALL_CMDS, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                printf("specify Link ID in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_RX_INVALID_CMD:
            if(param >= 0 ) {
                gmsg.invalid_cmd.link_if = param;
                rc = ioctl(fd, FPD_GET_RX_INVALID_CMD, &gmsg);
                if (rc == 0) {
                    display_invalid_paragon_cmd(&gmsg.invalid_cmd);
                }
            }
            else {
                printf("specify Link ID in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_SET_LINK0_MODE:
            if(( param == 0 ) || ( param == 1 ) ) {
                printf("Setting Link 0 Mode to %s....", param?"Master":"Slave");
                if( param == 0 )
                    gmsg.link0mode.mode = FPD_LINK0_MODE_SLAVE;
                else
                    gmsg.link0mode.mode = FPD_LINK0_MODE_MASTER;
                rc = ioctl(fd, FPD_SET_LINK0_MODE, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                printf("specify mode (1=Master, 0=Slave) in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_LINK0_MODE:
            rc = ioctl(fd, FPD_GET_LINK0_MODE, &gmsg);
            if( rc == 0 ) {
                printf("Current setting                      : ");
                if( gmsg.link0mode.mode == FPD_LINK0_MODE_MASTER )
                    printf("Master\n");
                else
                    printf("Slave\n");
            }
            break;
        case CMD_SWITCH_REMOTE_VIDEO:
            get_remote_video_params(&gmsg.video_switch);
            printf("Switching Link %d, Remote %d Video to Target %d....",
                   gmsg.video_switch.link_if,
                   gmsg.video_switch.kvm_chan,
                   gmsg.video_switch.target_port);
            rc = ioctl(fd, FPD_SWITCH_REMOTE_VIDEO, &gmsg);
            if (rc == 0) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
            }
            break;
        case CMD_DISABLE_REMOTE_VIDEO:
            if( param >= 0 && param2 >= 0 ) {
                printf("Disconnecting Link %d, Remote %d Video....",
                       param, param2);
                gmsg.video_switch.link_if = param;
                gmsg.video_switch.kvm_chan = param2;
                rc = ioctl(fd, FPD_DISABLE_REMOTE_VIDEO, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                printf("specify Link IF in -a and KVM Channel in -b option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_SWITCH_LOCAL_VIDEO:
            get_local_video_params( &gmsg.local_video_switch );
            rc = 0;
            switch( gmsg.local_video_switch.local_video_src ) {
                case LOCAL_VIDEO_DISABLE:
                    printf("Disabling Local Video....");
                    break;
                case LOCAL_VIDEO_GUI:
                    printf("Switching Local Video to Local GUI....");
                    break;
                case LOCAL_VIDEO_PCIM:
                    printf("Switching Local Video to PCIM Target %d....",
                           gmsg.local_video_switch.target_port);
                    break;
                case LOCAL_VIDEO_VMCIM:
                    printf("Switching Local Video to VMCIM Target %d....",
                           gmsg.local_video_switch.target_port);
                    break;
                default:
                    rc = -EINVAL;
            }

            if( rc == 0 ) {
                rc = ioctl(fd, FPD_SWITCH_LOCAL_VIDEO, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            break;
        case CMD_WRITE_VSCHIP_REG:
            if((param >= 0) && (param2 >= 0) && (param3 >= 0)) {
                gmsg.vschip_write.chip_id = param;
                gmsg.vschip_write.reg = param2;
                gmsg.vschip_write.data = param3;
                printf("Writing to vschip%d reg %08x with data %08x....",
                       param, param2, param3);
                rc = ioctl(fd, FPD_WRITE_VSCHIP_REG, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                printf("specify chip ID in -a, reg in -b, and data in -c option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_READ_ALL_VSCHIP_REG:
            if(param >= 0) {
                gmsg.vschip_contents.dev = param;
                rc = ioctl(fd, FPD_READ_ALL_VSCHIP_REG, &gmsg);
                if( rc == 0 ) {
                    display_vschip_contents(&gmsg.vschip_contents.reg);
                }
            }
            else {
                printf("specify chip ID in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_RESET_LINK_INTERFACE:
            if(param >= 0) {
                printf("Resetting Link IF %d...........", param);
                gmsg.reset_link.link_if = param;
                rc = ioctl(fd, FPD_RESET_LINK_INTERFACE, &gmsg);
                if (rc == 0) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                }
            }
            else {
                printf("specify Link IF in -a option\n");
                rc = -EINVAL;
            }
            break;

        /* DEBUGGING PURPOSES */
        case CMD_TRACE_ON:
            rc = ioctl(fd, FPD_TRACE_ON);
            break;
        case CMD_TRACE_OFF:
            rc = ioctl(fd, FPD_TRACE_OFF);
            break;
        case CMD_SET_LOG_LVL:
            if(param >= 0) {
                gmsg.set_loglvl.level = param;
                printf("setting log level to %d\n", gmsg.set_loglvl.level);
                rc = ioctl(fd, FPD_SET_LOG_LVL, &gmsg);
            }
            else {
                printf("specify log level in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_LOG_LVL:
            rc = ioctl(fd, FPD_GET_LOG_LVL, &gmsg);
            if (rc == 0) {
                printf("Current LOG level: %d\n", gmsg.get_loglvl.level);
            }
            break;
        case CMD_GET_BUFFER_INFO:
            if(param >= 0 && param2 >= 0) {
                gmsg.buffer.link_if = param;
                gmsg.buffer.type = param2;
                rc = ioctl(fd, FPD_GET_BUFFER_INFO, &gmsg);
                if( rc == 0 ) {
                    display_buffer_info(&gmsg.buffer);
                }
            }
            else {
                printf("specify link IF in -a, Buffer type in -b option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_HOST_RXBDTBL:
            if(param >= 0 && param2 >= 0 && param3 >= 0) {
                gmsg.host_rxbdtbl.link_if = param;
                gmsg.host_rxbdtbl.host_chan = param2;
                gmsg.host_rxbdtbl.buf_index = param3;
                rc = ioctl(fd, FPD_GET_HOST_RXBDTBL, &gmsg);
                if( rc == 0 ) {
                    display_host_rxbdtbl(&gmsg.host_rxbdtbl);
                }
            }
            else {
                printf("specify link IF in -a, Host Channel in -b and Buffer Index in -c option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_GET_HOST_TXBUF_INFO:
            if(param >= 0 && param2 >= 0) {
                gmsg.host_rxbdtbl.link_if = param;
                gmsg.host_rxbdtbl.host_chan = param2;
                rc = ioctl(fd, FPD_GET_HOST_TXBUF_INFO, &gmsg);
                if( rc == 0 ) {
                    display_host_txbuf_info(&gmsg.host_txinfo);
                }
            }
            else {
                printf("specify link IF in -a, Host Channel in -b option\n");
                rc = -EINVAL;
            }
            break;
        case 255:
            rc = ioctl(fd, FPD_DRIVER_RESET);
            break;
        default:
            rc = -1;
    }

    if (rc < 0) {
        if(errno) {
            err( "ioctl failed : %s\n", strerror(errno) );
        }
        else {
            err( "ioctl failed : Invalid argument\n" );
        }
    }

    cleanup_host_module();
    close(fd);
	
    return 0;
}

static void display_pci_cfg( FPD_pcicfg_t *pci )
{
    printf("PCI_VENDOR_ID          : %04x\n", pci->vendor_id);
    printf("PCI_DEVICE_ID          : %04x\n", pci->device_id);
    printf("PCI_COMMAND            : %04x\n", pci->command);
    printf("PCI_STATUS             : %04x\n", pci->status);
    printf("PCI_REVISION_ID        : %02x\n", pci->revision_id);
    printf("PCI_CLASS_PROG         : %02x\n", pci->prog_if);
    printf("PCI_CLASS_DEVICE       : %02x\n", pci->sub_class_code);
    printf("PCI_CLASS_CODE         : %02x\n", pci->base_class_code);
    printf("PCI_CACHE_LINE_SIZE    : %02x\n", pci->cache_line_size);
    printf("PCI_LATENCY_TIMER      : %02x\n", pci->latency_timer);
    printf("PCI_HEADER_TYPE        : %02x\n", pci->header_type);
    printf("PCI_BIST               : %02x\n", pci->bist);
    printf("PCI_BASE_ADDRESS_0     : %08x\n", pci->bar0);
    printf("PCI_BASE_ADDRESS_1     : %08x\n", pci->bar1);
    printf("PCI_BASE_ADDRESS_2     : %08x\n", pci->bar2);
    printf("PCI_BASE_ADDRESS_3     : %08x\n", pci->bar3);
    printf("PCI_BASE_ADDRESS_4     : %08x\n", pci->bar4);
    printf("PCI_BASE_ADDRESS_5     : %08x\n", pci->bar5);
    printf("PCI_CARDBUS_CIS        : %08x\n", pci->cardbus_cis_ptr);
    printf("PCI_SUBSYSTEM_VENDOR_ID: %04x\n", pci->subsystem_vendor_id);
    printf("PCI_SUBSYSTEM_ID       : %04x\n", pci->subsystem_id);
    printf("PCI_ROM_ADDRESS        : %08x\n", pci->rom_bar);
    printf("PCI_CAPABILITY_LIST    : %02x\n", pci->cap_ptr);
    printf("PCI_INTERRUPT_LINE     : %02x\n", pci->interrupt_line);
    printf("PCI_INTERRUPT_PIN      : %02x\n", pci->interrupt_pin);
    printf("PCI_MIN_GNT            : %02x\n", pci->min_grant);
    printf("PCI_MAX_LAT            : %02x\n", pci->max_latency);

    return;
}

static void display_events( FPD_event_t *pevt )
{
    int link;
    int id;

    printf("Events:\n");

    /* Link Interfaces */
    for( link = 0; link < max_link_if; link++ ) {
        if( !(pevt->event_mask & (FPD_EVENT_MASK_LINK0 << link)) )
            continue;

        /* Host buffer */
        for( id = 0; id < FPD_HOST_CNT; id++ ) {
            if( pevt->link_if[link].host[id].has_data ) {
                printf("Link IF %d require Host Buffer %d processing\n",
                       link, id);
            }
        }

        /* Priority buffer */
        if( pevt->link_if[link].rx_priority ) {
            printf("Link IF %d receive Priority Buffer data\n", link);
        }

        /* CIM buffer */
        if( pevt->link_if[link].rx_cim ) {
            printf("Link IF %d receive CIM Buffer data\n", link);
        }

        /* error detected */
        if( pevt->link_if[link].error ) {
            printf("Link IF %d receive errors\n", link);
        }

        /* echo response detected */
        if( pevt->link_if[link].echo_rsp ) {
            printf("\tLink IF %d receive echo response\n", link);
        }
    }

    /* Background Link Interface events */
    if( pevt->event_mask & (FPD_EVENT_MASK_BGND_LINK) ) {
        if( pevt->bgnd_link_if.rx_cim ) {
            printf("Bacground Link IF receive CIM Buffer data\n");
        }

        if( pevt->bgnd_link_if.error ) {
            printf("Background Link IF receive errors\n");
        }

        if( pevt->bgnd_link_if.echo_rsp ) {
            printf("\tBackground Link IF receive echo response\n");
        }
    }

    if( pevt->pci_error ) {
        printf("Detected PCI error\n");
    }

    if( pevt->cim_detect ) {
        printf("Detected changes to CIM topology\n");
    }

    return;
}

static int get_switch_params( FPD_switch_t *pswitch )
{
    unsigned int val;
    unsigned int cim_protocol, tx_parity, rx_parity;

    printf("\nEnter Link Interface [0-%d] or Background Link [%d]   : ",
           max_link_if - 1, FPD_LINK_IF_ID_BGND);
    scanf("%d", &val);
    if( val < max_link_if ) {
        pswitch->link_if = val;
    }
    else if ( val == FPD_LINK_IF_ID_BGND ) {
        if( bgnd_link_if_present ) {
            pswitch->link_if = val;
        }
        else {
            printf("\nERROR: Background Link Interface is not present\n");
            printf("Please re-do!!!\n");
            return -1;
        }
    }
    else {
        printf("\nERROR: Invalid Link Interface ID %d\n", val);
        printf("Please re-do!!!\n");
        return -1;
    }

    printf("Enter Protocol Type [0=VM, 1=Paragon]                  : ");
    scanf("%d", &val);
    switch( val ) {
        case FPD_PROTOCOL_VM:
            cim_protocol = FPD_PROTOCOL_VM;
            break;
        case FPD_PROTOCOL_PARAGON:
            cim_protocol = FPD_PROTOCOL_PARAGON;
            printf("Enter TX Parity [0=Even, 1=Odd]                        : ");
            scanf("%d", &tx_parity);
            switch( tx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                case FPD_PARAGON_PARITY_ODD:
                    break;
                default:
                    printf("\nERROR: Invalid Parity\n");
                    printf("Please re-do!!!\n");
                    return -3;
            }

            printf("Enter RX Parity [0=Even, 1=Odd, 2=Ignore]              : ");
            scanf("%d", &rx_parity);
            switch( rx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                case FPD_PARAGON_PARITY_ODD:
                case FPD_PARAGON_PARITY_IGNORE:
                    break;
                default:
                    printf("\nERROR: Invalid Parity\n");
                    printf("Please re-do!!!\n");
                    return -4;
            }
            break;
        default:
            printf("\nERROR: Invalid Protocol Type\n");
            printf("Please re-do!!!\n");
            return -2;
    }

    pswitch->protocol = cim_protocol;
    pswitch->tx_parity = tx_parity;
    pswitch->rx_parity = rx_parity;

    printf("Enter Line for connection [0-%d]                        : ",
           max_line_cnt - 1);
    scanf("%d", &val);

    if( val < max_line_cnt ) {
        pswitch->target_port = val;
    }
    else {
        printf("\nERROR: Invalid Line ID %d\n", val);
        printf("Please re-do!!!\n");
        return -3;
    }

    printf("Enter Wait Time to empty Driver's Tx Buffer in msec    : ");
    scanf("%d", &val);
    if( val > 0 && val <= 60000 ) {
        pswitch->driver_txbuf_wait_time = val;
    }
    else {
        printf("\nERROR: Invalid Timeout value %d\n", val);
        printf("Please re-do!!!\n");
        return -4;
    }

    printf("Enter Wait Time to empty FPGA's Tx Buffer in msec      : ");
    scanf("%d", &val);
    if( val > 0 && val <= 60000 ) {
        pswitch->fpga_txbuf_wait_time = val;
    }
    else {
        printf("\nERROR: Invalid Timeout value %d\n", val);
        printf("Please re-do!!!\n");
        return -5;
    }

    return 0;
}

static void display_switch_info( FPD_switchinfo_t *pinfo )
{
    int id;

    printf("Switch Information:\n");
    for( id = 0; id < pinfo->link_if_cnt; id++ ) {
        printf("Link IF %x    <--> Target %d ",
               pinfo->link[id].link_if,
               pinfo->link[id].target_port);
        if( pinfo->link[id].enabled ) {
            printf("[");
            switch( pinfo->link[id].protocol ) {
                case FPD_PROTOCOL_VM:
                    printf("VM protocol");
                    break;
                case FPD_PROTOCOL_PARAGON:
                    printf("Paragon protocol, ");
                    switch( pinfo->link[id].tx_parity ) {
                        case FPD_PARAGON_PARITY_EVEN:
                            printf("Even TX Parity, ");
                            break;
                        case FPD_PARAGON_PARITY_ODD:
                            printf("Odd TX Parity, ");
                            break;
                        default:
                            printf("Unknown TX Parity, ");
                            break;
                    }
                    switch( pinfo->link[id].rx_parity ) {
                        case FPD_PARAGON_PARITY_EVEN:
                            printf("Even RX Parity");
                            break;
                        case FPD_PARAGON_PARITY_ODD:
                            printf("Odd RX Parity");
                            break;
                        default:
                            printf("Unknown RX Parity");
                            break;
                    }
                    break;
                default:
                    printf("Unspecified protocol");
                    break;
            }
            printf("]\n");
        }
        else {
            printf("(disabled)\n");
        }
    }

    if( pinfo->bgnd_link_if_present ) {
        printf("BGND Link IF <--> Target %d ",
               pinfo->bgnd_link.target_port);
        if( pinfo->bgnd_link.enabled ) {
            printf("[");
            switch( pinfo->bgnd_link.protocol ) {
                case FPD_PROTOCOL_VM:
                    printf("VM protocol");
                    break;
                case FPD_PROTOCOL_PARAGON:
                    printf("Paragon protocol, ");
                    switch( pinfo->bgnd_link.tx_parity ) {
                        case FPD_PARAGON_PARITY_EVEN:
                            printf("Even TX Parity, ");
                            break;
                        case FPD_PARAGON_PARITY_ODD:
                            printf("Odd TX Parity, ");
                            break;
                        default:
                            printf("Unknown TX Parity, ");
                            break;
                    }
                    switch( pinfo->bgnd_link.rx_parity ) {
                        case FPD_PARAGON_PARITY_EVEN:
                            printf("Even RX Parity");
                            break;
                        case FPD_PARAGON_PARITY_ODD:
                            printf("Odd RX Parity");
                            break;
                        default:
                            printf("Unknown RX Parity");
                            break;
                    }
                    break;
                default:
                    printf("Unspecified protocol");
                    break;
            }
            printf("]\n");
        }
        else {
            printf("(disabled)\n");
        }
    }

    return;
}

static void display_line_status( FPD_linestat_t *pinfo )
{
    int i, id, line;
    unsigned int status;
    unsigned char linestat;

    i = 0;
    line = 0;
    do {
        status = pinfo->line_status[i];
        for( id = 0; id < 16; id++ ) {
            if( (line + id) >= max_line_cnt ) {
                return;
            }

            printf("\t\tLine %02d: ", line + id);
            linestat = status & 0x3;
            switch( linestat ) {
                case 0:
                    printf("-");
                    break;
                case 1:
                    printf("Paragon CIM");
                    break;
                case 2:
                    printf("KX2.0 CIM");
                    break;
                default:
                    printf("Unknown!");
                    break;
            }

            if( line < 32 ) {
                if( pinfo->status_change[0] & (1 << (line+id)) ) {
                    printf(" (Status changed)\n");
                }
                else {
                    printf("\n");
                }
            }
            else {
                if( pinfo->status_change[1] & (1 << ((line - 32) + id)) ) {
                    printf(" (Status changed)\n");
                }
                else {
                    printf("\n");
                }
            }

            status >>= 2;
        }
        i++;
        line += 16;
    } while( line < max_line_cnt );

    return;
}

static void display_error( FPD_error_t *pinfo )
{
    switch( pinfo->type ) {
        case FPD_ERR_TYPE_PCI:
            printf("PCI Error: %08x\n", pinfo->error);
            break;
        case FPD_ERR_TYPE_LINK_IF:
            if( pinfo->link_if == FPD_LINK_IF_ID_BGND ) {
                printf("Background Link IF Error: %08x", pinfo->error);
            }
            else {
                printf("Link IF %d Error: %08x", pinfo->link_if, pinfo->error);
            }

            if( pinfo->error ) {
                if( pinfo->error & FPD_ERR_RX_INV_CMD ) {
                    printf("       FPD_ERR_RX_INV_CMD\n");
                }
                if( pinfo->error & FPD_ERR_UPDATE_INV_PKT ) {
                    printf("       FPD_ERR_UPDATE_INV_PKT\n");
                }
                if( pinfo->error & FPD_ERR_RX_PRI_RD_EMPTY_BUF ) {
                    printf("       FPD_ERR_RX_PRI_RD_EMPTY_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_SYNC ) {
                    printf("       FPD_ERR_TX_PRI_SYNC\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_NO_EOP ) {
                    printf("       FPD_ERR_TX_PRI_NO_EOP\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_WR_FULL_BUF ) {
                    printf("       FPD_ERR_TX_PRI_WR_FULL_BUF\n");
                }
                if( pinfo->error & FPD_ERR_RX_CIM_RD_EMPTY_BUF ) {
                    printf("       FPD_ERR_RX_CIM_RD_EMPTY_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_SYNC ) {
                    printf("       FPD_ERR_TX_CIM_SYNC\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_NO_EOP ) {
                    printf("       FPD_ERR_TX_CIM_NO_EOP\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_WR_FULL_BUF ) {
                    printf("       FPD_ERR_TX_CIM_WR_FULL_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_TIMEOUT ) {
                    printf("       FPD_ERR_TX_PRI_TIMEOUT\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_TIMEOUT ) {
                    printf("       FPD_ERR_TX_CIM_TIMEOUT\n");
                }
                if( pinfo->error & FPD_ERR_TX_HOST_TIMEOUT_0 ) {
                    printf("       FPD_ERR_TX_HOST_TIMEOUT_0\n");
                }
                if( pinfo->error & FPD_ERR_TX_HOST_TIMEOUT_1 ) {
                    printf("       FPD_ERR_TX_HOST_TIMEOUT_1\n");
                }
                if( pinfo->error & FPD_ERR_RX_DMA_NO_BUF_0 ) {
                    printf("       FPD_ERR_RX_DMA_NO_BUF_0\n");
                }
                if( pinfo->error & FPD_ERR_RX_DMA_NO_BUF_1 ) {
                    printf("       FPD_ERR_RX_DMA_NO_BUF_1\n");
                }
            }
            else {
                printf("       NONE\n");
            }
            break;
        default:
            break;
    }

    return;
}

static void display_stats( FPD_stats_t *pinfo )
{
    if( pinfo->link_if == FPD_LINK_IF_ID_BGND ) {
        printf("Background Link IF:\n");
    }
    else {
        printf("Link IF %d:\n", pinfo->link_if);
    }

    printf("  RX Statistics\n");
    printf("\tBuffer Full   : %d\n", pinfo->rx.buf_full_cnt);
    printf("\tProtocol Error: %d\n", pinfo->rx.protocol_err_cnt);
    printf("\tCRC Error     : %d\n", pinfo->rx.crc_err_cnt);
    printf("\tInvalid Packet: %d\n", pinfo->rx.inv_pkt_cnt);
    printf("\tParity Error  : %d\n", pinfo->rx.parity_err_cnt);
    printf("\tSequence Error: %d\n", pinfo->rx.seq_err_cnt);
    printf("\tTimeout       : %d\n", pinfo->rx.timeout_cnt);
    printf("\tNoise         : %d\n", pinfo->rx.noise_cnt);

    printf("  TX Statistics\n");
    printf("\tMax Retry     : %d\n", pinfo->tx.max_retry);
    printf("\tBuffer Full   : %d\n", pinfo->tx.buf_full_cnt);
    printf("\tProtocol Error: %d\n", pinfo->tx.protocol_err_cnt);
    printf("\tCRC Error     : %d\n", pinfo->tx.crc_err_cnt);

    return;
}

static int get_txpdata_params( FPD_data_t *pdata )
{
    unsigned char *buf;
    int i;

    printf("Enter Parameters\n");
    printf("Link IF                     : ");
    scanf("%d", &pdata->link_if);
    printf("Type (0 = CIM, 1 = Priority): ");
    scanf("%d", &pdata->type);
    printf("Data Length                 : ");
    scanf("%d", &pdata->requested_len);
    buf = (unsigned char *)malloc(pdata->requested_len);
    if( !buf )
        return -1;

    pdata->buf = buf;
    for( i = 0; i < pdata->requested_len; i++ ) {
        printf("Data byte %03d               : 0x", i+1);
        scanf("%hhx", &buf[i]);
    }
    return 0;
}

static int get_rxpdata_params( FPD_data_t *pdata )
{
    unsigned char *buf;

    printf("Enter Parameters\n");
    printf("Link IF                     : ");
    scanf("%d", &pdata->link_if);
    printf("Type (0 = CIM, 1 = Priority): ");
    scanf("%d", &pdata->type);
    printf("Data Length                 : ");
    scanf("%d", &pdata->requested_len);
    buf = (unsigned char *)malloc(pdata->requested_len);
    if( !buf )
        return -1;

    pdata->buf = buf;
    return 0;
}

static int get_host_cfg_params( FPD_hostcfg_t *pcfg )
{
    int i = 0;

    printf("Enter Parameters\n");
    printf("Link IF                     : ");
    scanf("%d", &pcfg->link_if);
    printf("Host Buffer Channel         : ");
    scanf("%d", &pcfg->host_chan);
    printf("DMA Size (64 or 512 bytes)  : ");
    scanf("%d", &pcfg->dma_size);
    printf("Rx Pkt Size Notification    : ");
    scanf("%d", &pcfg->rx_pkt_size_notification);
    printf("Host Function");
    printf("\n\t %d: None", i++);
    printf("\n\t %d: Transparent", i++);
    printf("\n\t %d: Virtual Media\n", i++);
    printf("Host Function               : ");
    scanf("%d", &pcfg->function);

    return 0;
}

static int display_host_cfg_params( FPD_hostcfg_t *pcfg )
{
    printf("Host Buffer Configuration\n");
    printf("Link IF                  : %d\n", pcfg->link_if);
    printf("Host Buffer Channel      : %d\n", pcfg->host_chan);
    printf("DMA Size                 : %d\n", pcfg->dma_size);
    printf("Rx Pkt Size Notification : %d\n", pcfg->rx_pkt_size_notification);
    printf("Host Function            : ");
    switch(pcfg->function) {
        case FPD_HOST_FUNCTION_NONE:
            printf("Disabled\n");
            break;
        case FPD_HOST_FUNCTION_TRANSPARENT:
            printf("Transparent\n");
            break;
        case FPD_HOST_FUNCTION_VIRTUAL_MEDIA:
            printf("Virtual Media\n");
            break;
    }

    return 0;
}

static void display_buffer_info( FPD_buffer_t *pbinfo )
{
    printf("\nBuffer INFO\n");
    printf("Link Interface        : %d\n", pbinfo->link_if);
    printf("Buffer Type           : ");
    if( pbinfo->type == 0 )
        printf("CIM\n");
    else
        printf("Priority\n");
    printf("TX Buffer Start       : 0x%08x\n", pbinfo->txbuf_start);
    printf("TX Buffer End         : 0x%08x\n", pbinfo->txbuf_end);
    printf("TX Buffer RDptr       : 0x%08x\n", pbinfo->txbuf_rdptr);
    printf("TX Buffer WRptr       : 0x%08x\n", pbinfo->txbuf_wrptr);
    printf("TX Buffer Bytes Left  : %d\n", pbinfo->txbuf_bytes_left);
    printf("RX Buffer Start       : 0x%08x\n", pbinfo->rxbuf_start);
    printf("RX Buffer End         : 0x%08x\n", pbinfo->rxbuf_end);
    printf("RX Buffer RDptr       : 0x%08x\n", pbinfo->rxbuf_rdptr);
    printf("RX Buffer WRptr       : 0x%08x\n", pbinfo->rxbuf_wrptr);
    printf("RX Buffer Bytes Unused: %d\n", pbinfo->rxbuf_bytes_unused);
    printf("RX Buffer No Space    : %d\n", pbinfo->rxbuf_no_space);

    return;
}

static void display_host_rxbdtbl( FPD_hostrxbdtbl_t *pbdtbl )
{
    u_int32_t *ptr = (u_int32_t *)&pbdtbl->bdtbl[0];
    int i;

    printf("Number of BD  : %d\n", pbdtbl->bdnum);
    printf("Physical Addr : %08x\n", pbdtbl->phys_addr);
    printf("Bytes Unread  : %d\n", pbdtbl->overall_len);
    printf("State         : ");
    switch( pbdtbl->state ) {
        case 0:
            printf("Free\n");
            break;
        case 1:
            printf("User Busy\n");
            break;
        case 2:
            printf("Processing\n");
            break;
        default:
            printf("Limbo\n");
    }
    printf("BD Start      : %d\n", pbdtbl->startbd);
    printf("BD End        : %d\n", pbdtbl->endbd);
    printf("Current Buffer: %s\n", pbdtbl->is_current_buffer?"YES":"NO");
    for( i = 0; i < pbdtbl->bdnum; i++, ptr += 2 ) {
        printf("[%d] bufPtr = %08x, bufStatus = %08x\n",
               i, *(u_int32_t *)ptr, *(u_int32_t *)(ptr + 1));
    }

    return;
}

static int display_invalid_paragon_cmd( FPD_rxinvcmd_t *pcmd )
{
    if( pcmd->n_entries > 0 ) {
        int i;

        printf("Received Invalid Paragon Commands\n");
        for( i = 0; i < pcmd->n_entries; i++ ) {
            printf("CMD 0x%02x\n", pcmd->inv_cmd[i]);
        }
    }
    else {
        printf("No entries found.\n");
    }

    return 0;
}

static void display_host_txbuf_info( FPD_host_txbuf_info_t *pinfo )
{
    int i;

    printf("Number of TX Buffers : %d\n", pinfo->buf_cnt);
    for( i = 0; i < pinfo->buf_cnt; i++ ) {
        printf("buf%d: len = %d, state = ", i, pinfo->info[i].len);
        switch( pinfo->info[i].state ) {
            case 0:
                printf("Free\n");
                break;
            case 1:
                printf("User Busy\n");
                break;
            case 2:
                printf("Processing\n");
                break;
            default:
                printf("Limbo\n");
        }
    }

    return;
}

static int get_remote_video_params( FPD_video_switch_t *pvideo )
{
    printf("Enter Parameters\n");
    printf("Link IF                   : ");
    scanf("%d", &pvideo->link_if);
    printf("KVM Channel               : ");
    scanf("%d", &pvideo->kvm_chan);
    printf("Target Port               : ");
    scanf("%d", &pvideo->target_port);
    printf("Protocol (0=VM, 1=Paragon): ");
    scanf("%hhd", &pvideo->protocol);
    if( pvideo->protocol == FPD_PROTOCOL_PARAGON ) {
        printf("Hsync Pol (0=pos, 1=neg)  : ");
        scanf("%hhd", &pvideo->hsync_pol);
        printf("Vsync Pol (0=pos, 1=neg)  : ");
        scanf("%hhd", &pvideo->vsync_pol);
    }

    return 0;
}

static int get_local_video_params( FPD_local_video_switch_t *pvideo )
{
    printf("Enter Parameters\n");
    printf("Video Source [0=disable, 1=PCIM, 2=VMCIM, 3=LocalGUI]\n");
    printf("Local Video Source        : ");
    scanf("%d", &pvideo->local_video_src);
    if( pvideo->local_video_src != FPD_LOCAL_VIDEO_SRC_DISABLE ) {
        printf("Link IF                   : ");
        scanf("%d", &pvideo->link_if);
        if( pvideo->local_video_src != FPD_LOCAL_VIDEO_SRC_GRAPHIC_ENGINE ) {
            printf("Target Port               : ");
            scanf("%d", &pvideo->target_port);
            if( pvideo->local_video_src == FPD_LOCAL_VIDEO_SRC_PARAGON_CIM ) {
                printf("Hsync Pol (0=pos, 1=neg)  : ");
                scanf("%hhd", &pvideo->hsync_pol);
                printf("Vsync Pol (0=pos, 1=neg)  : ");
                scanf("%hhd", &pvideo->vsync_pol);
            }
        }
    }

    return 0;
}

static void display_vschip_contents( FPD_vschip_reg_t *pchip )
{
    printf("chip ID         : %02x\n", pchip->chip_id);
    printf("Ai              : %02x\n", pchip->Ai);
    printf("Bi              : %02x\n", pchip->Bi);
    printf("Ci              : %02x\n", pchip->Ci);
    printf("Di              : %02x\n", pchip->Di);
    printf("Enable Input Mux: %02x\n", pchip->enable_input_mux);
    printf("Ti              : %02x\n", pchip->Ti);
    printf("Test Mux        : %02x\n", pchip->test_mux);
    printf("Sync Detectors  : %02x\n", pchip->sync_detectors);
    printf("Ax              : %02x\n", pchip->Ax);
    printf("Bx              : %02x\n", pchip->Bx);
    printf("Cx              : %02x\n", pchip->Cx);
    printf("Dx              : %02x\n", pchip->Dx);
    printf("Sx              : %02x\n", pchip->Sx);
    printf("SOC             : %02x\n", pchip->soc);
    printf("Gain Input Mux  : %02x\n", pchip->gain_input_mux);

    return;
}

static void usage( void )
{
    printf( "Use (" FPD_TEST_REV "):\n"
            "fpd-test [options] [devname]\n\n"
            "    -h         help\n"
            "    -i <cmd>   ioctl cmd\n"
            "    -a <arg>   argument used by ioctl\n"
            "    -b <arg>   2nd argument used by ioctl\n"
            "    -v         verbose\n"
            " [devname] is /dev/fpd\n\n");
    printf( " Cmd  Arguments     Definitions\n");
    printf( " ------------------------------------------------------------\n");
    printf( " 1    none              get general info\n");
    printf( " 2    none              get PCI Configuration Space\n");
    printf( " 3    reg               read FPGA Protocol register\n");
    printf( " 4    reg  val          write to FPGA Protocol register\n");
    printf( " 5    none              check events\n");
    printf( " 6    none              transmit data to CIM/Priority buffer\n");
    printf( " 7    none              receive data from CIM/Priority buffer\n");
    printf( " 8    none              switch link IF to target specified\n");
    printf( " 9    link[0-4]         disconnect link IF from target\n");
    printf( "      driver_wait_time    Link ID %d is the background Link\n", FPD_LINK_IF_ID_BGND);
    printf( "      fpga_wait_time      Timeouts in msec (100ms interval)\n");
    printf( " 10   none              get switch info\n");
    printf( " 11   none              get line status\n");
    printf( " 12   type link[0-4]    get error condition\n");
    printf( "                          type 0=PCI error, 1=Link IF error\n");
    printf( "                          Link ID %d is the background Link\n", FPD_LINK_IF_ID_BGND);
    printf( " 13   link[0-4]         get statistics\n");
    printf( "                          Link ID %d is the background Link\n", FPD_LINK_IF_ID_BGND);
    printf( " 15   none              set Host buffer configuration\n");
    printf( " 16   link[0-4]         get Host buffer configuration\n");
    printf( "      host[0-1]\n");
    printf( " 17   none              get transmit Host buffer\n");
    printf( " 18   none              transmit data to Host buffer\n");
    printf( " 19   none              receive data from Host buffer\n");
    printf( " 21   DMA burst size    set burst size to 8, 16, 32, or 64\n");
    printf( " 22   none              get DMA burst size\n");
    printf( " 23   TX timeout[1-15]  set TX timeout\n");
    printf( " 24   none              get TX timeout\n");
    printf( " 25   linkID[0-4,%d]  invalidate Paragon Cmd Lookup Table\n", FPD_LINK_IF_ID_BGND);
    printf( " 28   linkID[0-4,%d]  get RX Invalid Paragon Cmd\n", FPD_LINK_IF_ID_BGND);
    printf( " 29   Link 0 mode       set Link IF 0 as master(1) or slave(0)\n");
    printf( " 30   none              get Link IF 0 mode\n");
    printf( " 34   none              switch remote video\n");
    printf( " 35   linkID kvmChan    disconnect remote video\n");
    printf( " 36   none              switch local video\n");
    printf( " 37   chip[0-3]         write to EL4544 register\n");
    printf( "      reg\n");
    printf( "      data\n");
    printf( " 38   chip[0-3]         read all EL4544 registers\n");
    printf( " 39   link[0-4,1010]    reset Link Interface\n");
    printf( " 50   none              turn on trace\n");
    printf( " 51   none              turn off trace\n");
    printf( " 52   level [0-4]       set debug level\n");
    printf( " 53   none              get debug level\n");
    printf( " 54   link[0=4]         get CIM buffer info\n");
    printf( "      bufferType[0/1]   Buffer type CIM=0, Priority=1\n");
    printf( " 55   link[0-4]         get Host buffer RX BD Table\n");
    printf( "      host[0-1]\n");
    printf( "      BufID[0-7]\n");
    printf( " 56   link[0-4]         get Host buffer TX Buffer info\n");
    printf( "      host[0-1]\n\n");
}

static void log( char *fmt, ... )
{
    va_list ap;
    if( fpd_verbose ) {
        printf( "[fpd-test] " );
        va_start(ap,fmt);
        vprintf( fmt, ap );
        va_end(ap);
    }	
}

void msg( char *fmt, ... )
{
   va_list ap;
   printf( "[fpd-test] " );
   va_start(ap,fmt);
   vprintf( fmt, ap );
   va_end(ap);
}

void err( char *fmt, ... )
{
   va_list ap;
   fprintf( stderr, "[fpd-test] " );
   va_start(ap,fmt);
   vfprintf( stderr, fmt, ap );
   va_end(ap);	
}
