/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver Interrupt and Event handler
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>

#include "fpd.h"
#include "fpd_ioctl.h"
#include "fpd_reg.h"
#include "fpd_pdata.h"
#include "fpd_host.h"
#include "fpd_paragon.h"
#include "txcim_thread.h"
#include "debug.h"


/******************************************************************************
 *  ROUTINE:        fpd_handler_linkif()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Processes all Link Interface interrupts triggered by the FPGA Protocol
 *      device that requires attention.
 *
 *  PARAMETERS:
 *
 *      plink   Pointer to the FPGA protocol Link Interface structure.
 *      lifisr  Current value of Link Interface Interrupt Status Register.
 *      pevent  Pointer to the FPGA protocol event structure.
 *
 *  RETURNS:
 *
 *     	Number of events handled.
 *
 *****************************************************************************/
int
fpd_handler_linkif( fpd_linkif_t *plink, u32 lifisr, FPD_event_t *pevent )
{
    fpd_device_t *pfpd;
    fpd_pdata_t *pcim;
    fpd_host_t *phost;
    fpd_host_chan_t *pchan;
    u32 link_error = 0;
    u32 cur_error;
    int event_cnt = 0;
    u32 reg;

    /* handle priority buffer */
    if( lifisr & LIFISR_RX_PRI_BUF ) {
        FPD_DDEBUG(plink->link_id, 2, "Processing RX Priority data\n");
        pdata_receive(&plink->priority);
        ++event_cnt;

        /* set the event */
        if( plink->link_id != FPD_LINK_IF_ID_BGND )
            pevent->link_if[plink->link_id].rx_priority = 1;
        else
            FPD_ERROR("BGND Link should not have priority buffer\n");
    }

    /* handle CIM buffer */
    if( lifisr & LIFISR_RX_CIM_BUF ) {
        FPD_DDEBUG(plink->link_id, 2, "Processing RX CIM data\n");
        pdata_receive(&plink->cim);
        ++event_cnt;

        /* set the event */
        if( plink->link_id == FPD_LINK_IF_ID_BGND )
            pevent->bgnd_link_if.rx_cim = 1;
        else
            pevent->link_if[plink->link_id].rx_cim = 1;
    }

    /* handle host buffer */
    if( lifisr & LIFISR_TX_DMA_DONE_0 ) {
        FPD_DDEBUG(plink->link_id, 2, "TX DMA DONE0\n");
        fpd_host_handle_txdone(&plink->host, 0);
    }

    if( lifisr & LIFISR_TX_DMA_DONE_1 ) {
        FPD_DDEBUG(plink->link_id, 2, "TX DMA DONE1\n");
        fpd_host_handle_txdone(&plink->host, 1);
    }

    if( lifisr & LIFISR_RX_DMA_BUF_FULL_0 ) {
        FPD_DDEBUG(plink->link_id, 2, "RX DMA0 BUF FULL\n");
        phost = &plink->host;
        pchan = phost->channel[0];

        fpd_host_handle_rx(phost, 0);

        /* set the event */
        if( pchan->function == FPD_HOST_FUNCTION_TRANSPARENT ) {
            ++event_cnt;
            pevent->link_if[plink->link_id].host[0].function = pchan->function;
            pevent->link_if[plink->link_id].host[0].has_data = 1;
        }
    }

    if( lifisr & LIFISR_RX_DMA_BUF_FULL_1 ) {
        FPD_DDEBUG(plink->link_id, 2, "RX DMA1 BUF FULL\n");
        phost = &plink->host;
        pchan = phost->channel[1];

        fpd_host_handle_rx(phost, 1);

        /* set the event */
        if( pchan->function == FPD_HOST_FUNCTION_TRANSPARENT ) {
            ++event_cnt;
            pevent->link_if[plink->link_id].host[1].function = pchan->function;
            pevent->link_if[plink->link_id].host[1].has_data = 1;
        }
    }

    if( lifisr & LIFISR_RX_DMA_NO_BUF_0 ) {
        FPD_DDEBUG(plink->link_id, 2, "RX DMA0 NO BUF\n");
        fpd_host_handle_rx_nobuf(&plink->host, 0);
    }

    if( lifisr & LIFISR_RX_DMA_NO_BUF_1 ) {
        FPD_DDEBUG(plink->link_id, 2, "RX DMA1 NO BUF\n");
        fpd_host_handle_rx_nobuf(&plink->host, 1);
    }

    /* handle Priority buffer transitioning to empty state */
    if( lifisr & LIFISR_TX_PRI_BUF ) {
        FPD_DDEBUG(plink->link_id, 2, "TX Priority Buffer empty\n");
        pcim = &plink->priority;
        pfpd = (fpd_device_t *)pcim->private;

        /* disable this interrupt */
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
        reg &= ~LIFCR_TX_PRI_BUF_INT_EN;
        FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

        atomic_inc(&pcim->req_tx);
        txcim_thread_wakeup(plink);
    }

    /* handle CIM buffer transitioning to empty state */
    if( lifisr & LIFISR_TX_CIM_BUF ) {
        FPD_DDEBUG(plink->link_id, 2, "TX CIM Buffer empty\n");
        pcim = &plink->cim;
        pfpd = (fpd_device_t *)pcim->private;

        /* disable this interrupt */
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
        reg &= ~LIFCR_TX_CIM_BUF_INT_EN;
        FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

        atomic_inc(&pcim->req_tx);
        txcim_thread_wakeup(plink);
    }

    /* handle any errors */
    if( lifisr & LIFISR_RX_INV_CMD ) {
        link_error |= FPD_ERR_RX_INV_CMD;
        fpd_paragon_handle_invalid_cmd(plink);
    }
    if( lifisr & LIFISR_UPDATE_INV_PKT ) {
        link_error |= FPD_ERR_UPDATE_INV_PKT;
    }
    if( lifisr & LIFISR_RX_PRI_RD_EMPTY_BUF ) {
        link_error |= FPD_ERR_RX_PRI_RD_EMPTY_BUF;
    }
    if( lifisr & LIFISR_TX_PRI_SYNC_ERR ) {
        link_error |= FPD_ERR_TX_PRI_SYNC;
    }
    if( lifisr & LIFISR_TX_PRI_NO_EOP ) {
        link_error |= FPD_ERR_TX_PRI_NO_EOP;
    }
    if( lifisr & LIFISR_TX_PRI_WR_FULL_BUF ) {
        link_error |= FPD_ERR_TX_PRI_WR_FULL_BUF;
    }
    if( lifisr & LIFISR_RX_CIM_RD_EMPTY_BUF ) {
        link_error |= FPD_ERR_RX_CIM_RD_EMPTY_BUF;
    }
    if( lifisr & LIFISR_TX_CIM_SYNC_ERR ) {
        link_error |= FPD_ERR_TX_CIM_SYNC;
    }
    if( lifisr & LIFISR_TX_CIM_NO_EOP ) {
        link_error |= FPD_ERR_TX_CIM_NO_EOP;
    }
    if( lifisr & LIFISR_TX_CIM_WR_FULL_BUF ) {
        link_error |= FPD_ERR_TX_CIM_WR_FULL_BUF;
    }
    if( lifisr & LIFISR_TX_PRI_TIMEOUT ) {
        link_error |= FPD_ERR_TX_PRI_TIMEOUT;
    }
    if( lifisr & LIFISR_TX_CIM_TIMEOUT ) {
        link_error |= FPD_ERR_TX_CIM_TIMEOUT;
    }
    if( lifisr & LIFISR_TX_HOST_TIMEOUT_0 ) {
        link_error |= FPD_ERR_TX_HOST_TIMEOUT_0;
    }
    if( lifisr & LIFISR_TX_HOST_TIMEOUT_1 ) {
        link_error |= FPD_ERR_TX_HOST_TIMEOUT_1;
    }

    if( link_error ) {
        ++event_cnt;

        /* update link error */
        fpd_lock_acquire(&plink->error);
        cur_error = fpd_lock_read_resource(&plink->error);
        cur_error |= link_error;
        fpd_lock_write_resource(&plink->error, cur_error);
        fpd_lock_release(&plink->error);

        /* set the event */
        if( plink->link_id == FPD_LINK_IF_ID_BGND )
            pevent->bgnd_link_if.error = 1;
        else
            pevent->link_if[plink->link_id].error = 1;
    }

    if( lifisr & LIFISR_ECHO_RESP_INT ) {
        FPD_DDEBUG(plink->link_id, 2, "RX ECHO RSP\n");
        fpd_paragon_handle_echo_rsp(plink);
        ++event_cnt;

        /* set the event */
        if( plink->link_id == FPD_LINK_IF_ID_BGND )
            pevent->bgnd_link_if.echo_rsp = 1;
        else
            pevent->link_if[plink->link_id].echo_rsp = 1;
    }

    return event_cnt;
}

/******************************************************************************
 *  ROUTINE:        fpd_handler()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Processes any interrupts triggered by the FPGA Protocol device or
 *      any events that needs attention.
 *
 *  PARAMETERS:
 *
 *      pfpd    Pointer to the FPGA protocol device structure.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
void
fpd_handler( fpd_device_t *pfpd )
{
    fpd_linkif_t *plink;
    FPD_event_t *pevent = &pfpd->event;
    int link_id;
    u32 lifisr;
    u32 gisr;
    u32 events = 0;

    /* read Global Interrupt Status for CIM detection and PCI error */
    gisr = FPD_READ(pfpd->remap, FPD_GLOBAL_INTERRUPT_STATUS);
    FPD_DDEBUG(pfpd->id, 2, "Global ISR = 0x%08x\n", gisr);

    fpd_lock_acquire(&pfpd->event_lock);

    /* check each Link Interface */
    for( link_id = 0; link_id < pfpd->info.link_if_cnt; link_id++ ) {
        if( !(gisr & (GISR_LINK_IF_0 << link_id)) )
            continue;

        plink = pfpd->link[link_id];
        lifisr = FPD_READ(pfpd->remap, plink->base+LIF_LINK_INTERRUPT_STATUS);
        FPD_DDEBUG(pfpd->id, 2, "LinkIF %d ISR = 0x%08x\n", link_id, lifisr);

        /* clear all Link interrupt */
        FPD_DDEBUG(pfpd->id, 2, "Clearing LinkIF %d ISR = 0x%08x\n", link_id, lifisr);
        FPD_WRITE(pfpd->remap, plink->base+LIF_LINK_INTERRUPT_STATUS, lifisr);

        if( fpd_handler_linkif(plink, lifisr, pevent) > 0 ) {
            pevent->event_mask |= (FPD_EVENT_MASK_LINK0 << link_id);
        }
    }

    /* handle Background Link Interface */
    if( gisr & GISR_BG_LINK_IF ) {
        plink = pfpd->bgnd_link;

        lifisr = FPD_READ(pfpd->remap, plink->base+LIF_LINK_INTERRUPT_STATUS);

        /* clear all Link interrupt */
        FPD_DDEBUG(pfpd->id, 2, "Clearing Bgnd LinkIF ISR = 0x%08x\n", lifisr);
        FPD_WRITE(pfpd->remap, plink->base+LIF_LINK_INTERRUPT_STATUS, lifisr);

        if( fpd_handler_linkif(plink, lifisr, pevent) > 0 ) {
            pevent->event_mask |= FPD_EVENT_MASK_BGND_LINK;
        }
    }

    /* handle CIM detection */
    if( gisr & GISR_LINE_STATUS_CHANGE ) {
        u32 status;

        if( gisr & GISR_LINE_STATUS_INT01 ) {
            status = FPD_READ(pfpd->remap, FPD_LINE_STATUS_INTERRUPT_01);
            pfpd->line_status_chg[0] = status;

            /* clear interrupt */
            FPD_WRITE(pfpd->remap, FPD_LINE_STATUS_INTERRUPT_01, status);
        }

        if( gisr & GISR_LINE_STATUS_INT23 ) {
            status = FPD_READ(pfpd->remap, FPD_LINE_STATUS_INTERRUPT_23);
            pfpd->line_status_chg[1] = status;

            /* clear interrupt */
            FPD_WRITE(pfpd->remap, FPD_LINE_STATUS_INTERRUPT_23, status);
        }

        /* no need to read current Line Status since driver don't use it */

        /* wake up cim detect polling thread */
        atomic_set(&pfpd->cim_detect_event, 1);
        wake_up_interruptible(&pfpd->cim_detect_queue);

        /* set the event */
        pevent->cim_detect = 1;
        pevent->event_mask |= FPD_EVENT_MASK_CIM_DETECT;
    }

    /* handle PCI error */
    if( gisr & GISR_PCI_ERR ) {
        u16 status;
        /*
         * to clear interrupt, error bits in the PCI Configuration
         * Space Status Register must be cleared
         */
        pci_read_config_word(pfpd->pcidev, PCI_STATUS, &status);
        FPD_ERROR("Received PCI Error interrupt! PCI_STATUS=%04x\n", status);
        pci_write_config_word(pfpd->pcidev, PCI_STATUS, status);

        status &= (PCI_STATUS_DETECTED_PARITY | PCI_STATUS_SIG_SYSTEM_ERROR |
                   PCI_STATUS_REC_MASTER_ABORT | PCI_STATUS_REC_TARGET_ABORT |
                   PCI_STATUS_SIG_TARGET_ABORT | PCI_STATUS_PARITY);
        fpd_lock_acquire(&pfpd->pci_error);
        fpd_lock_write_resource(&pfpd->pci_error, status);
        fpd_lock_release(&pfpd->pci_error);

        /* set the event */
        pevent->pci_error = 1;
        pevent->event_mask |= FPD_EVENT_MASK_PCI_ERROR;
    }

    events = pevent->event_mask;

    /* copy current events to event lock resource */
    fpd_lock_write_resource(&pfpd->event_lock, pevent->event_mask);
    fpd_lock_release(&pfpd->event_lock);

    /* wake user-app if any event occur */
    if( events ) {
        fpd_lock_acquire(&pfpd->inform_user);
        fpd_lock_write_resource(&pfpd->inform_user, 1);
        fpd_lock_release(&pfpd->inform_user);
        wake_up_interruptible(&pfpd->user_wq);
    }

    return;
}
