/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver Link Interface
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>

#include "fpd.h"
#include "fpd_reg.h"
#include "fpd_link.h"
#include "fpd_pdata.h"
#include "fpd_host.h"
#include "txcim_thread.h"
#include "debug.h"


/******************************************************************************
 *  ROUTINE:        fpd_link_setup_interrupts()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Setup the Link Interface interrupts.
 *
 *  PARAMETERS:
 *
 *      pfpd      Pointer to the FPGA Protocol device structure.
 *      plink     Pointer to the Link Interface structure.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
void
fpd_link_setup_interrupt( fpd_device_t *pfpd, fpd_linkif_t *plink )
{
    u32 reg;

    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg &= ~(LIFCR_TX_DMA_DONE_INT_EN_0 | LIFCR_TX_DMA_DONE_INT_EN_1 |
             LIFCR_RX_DMA_INT_EN_0 | LIFCR_RX_DMA_INT_EN_1 |
             LIFCR_TX_CIM_BUF_INT_EN | LIFCR_RX_CIM_BUF_INT_EN | 
             LIFCR_TX_PRI_BUF_INT_EN | LIFCR_RX_PRI_BUF_INT_EN |
             LIFCR_TX_TIMEOUT_INT_EN | LIFCR_BUF_SYNC_INT_EN |
             LIFCR_RX_INV_CMD_INT_EN | LIFCR_ECHO_RESP_INT_EN);
    if( plink->protocol == FPD_PROTOCOL_VM ) {
        reg |= (LIFCR_RX_CIM_BUF_INT_EN | LIFCR_TX_TIMEOUT_INT_EN |
                LIFCR_BUF_SYNC_INT_EN);
        if( plink->link_id != FPD_LINK_IF_ID_BGND ) {
            reg |= (LIFCR_TX_DMA_DONE_INT_EN_0 | LIFCR_TX_DMA_DONE_INT_EN_1 |
                    LIFCR_RX_DMA_INT_EN_0 | LIFCR_RX_DMA_INT_EN_1 |
                    LIFCR_RX_PRI_BUF_INT_EN);
        }
    }
    else {
        /* Legacy CIM */
        reg |= (LIFCR_RX_CIM_BUF_INT_EN | LIFCR_TX_TIMEOUT_INT_EN |
                LIFCR_BUF_SYNC_INT_EN | LIFCR_RX_INV_CMD_INT_EN |
                LIFCR_ECHO_RESP_INT_EN);
        if( plink->link_id != FPD_LINK_IF_ID_BGND ) {
            reg |= (LIFCR_RX_PRI_BUF_INT_EN);
        }
    }
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    return;
}

/******************************************************************************
 *  ROUTINE:        fpd_link_disable_interrupts()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Setup the Link Interface interrupts.
 *
 *  PARAMETERS:
 *
 *      pfpd      Pointer to the FPGA Protocol device structure.
 *      plink     Pointer to the Link Interface structure.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
static void
fpd_link_disable_interrupts( fpd_device_t *pfpd, fpd_linkif_t *plink )
{
    u32 reg;

    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg &= ~(LIFCR_TX_DMA_DONE_INT_EN_0 | LIFCR_TX_DMA_DONE_INT_EN_1 |
             LIFCR_RX_DMA_INT_EN_0 | LIFCR_RX_DMA_INT_EN_1 |
             LIFCR_TX_CIM_BUF_INT_EN | LIFCR_RX_CIM_BUF_INT_EN | 
             LIFCR_TX_PRI_BUF_INT_EN | LIFCR_RX_PRI_BUF_INT_EN |
             LIFCR_TX_TIMEOUT_INT_EN | LIFCR_BUF_SYNC_INT_EN |
             LIFCR_RX_INV_CMD_INT_EN | LIFCR_ECHO_RESP_INT_EN);
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    return;
}

/******************************************************************************
 *  ROUTINE:        fpd_link_init()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Initializes the Link Interface
 *
 *  PARAMETERS:
 *
 *      pfpd      Pointer to the FPGA Protocol device structure.
 *      plink     Pointer to the Link Interface structure.
 *      link_id   Link Interface ID.
 *
 *  RETURNS:
 *
 *     	0       Initialization is successful.
 *     -errno   Initialization failed.
 *
 *****************************************************************************/
int
fpd_link_init( fpd_device_t *pfpd, fpd_linkif_t *plink, int link_id )
{
    int rc;

    FPD_INFO("plink %p, ID %d\n", plink, link_id);

    /* initialize cim data buffer */
    if( (rc = pdata_init(&plink->cim, link_id, FPD_CIM_BUFFER, pfpd)) != 0 ) {
        FPD_ERROR("fpd%d LinkIF%d failed to init CIM Data Buffer",
                  pfpd->id, link_id);
        return rc;
    }

    /* initialize priority data buffer */
    if( link_id != FPD_LINK_IF_ID_BGND ) {
        if( (rc = pdata_init(&plink->priority, link_id, FPD_PRIORITY_BUFFER, pfpd)) != 0 ) {
            FPD_ERROR("fpd%d LinkIF%d failed to init Priority Data Buffer",
                      pfpd->id, link_id);
            pdata_cleanup(&plink->cim);
            return rc;
        }
    }

    if( link_id == FPD_LINK_IF_ID_BGND ) {
        plink->base = FPD_BGND_LINK_IF_BASE;
    }
    else {
        plink->base = FPD_LINK_IF_BASE0 + (link_id * FPD_LINK_IF_SET_SIZE);

	/* initialize Host Buffer */
	fpd_host_init(&plink->host, link_id, pfpd->info.host_buffer_cnt, pfpd);
    }
    plink->link_id = link_id;
    plink->target_port = 0;
    plink->enabled = 0;
    plink->private = pfpd;
    fpd_lock_init(&plink->error, 0);
    memset(plink->inv_cmd, 0, FPD_MAX_INVALID_PARAGON_CMD);
    plink->inv_cmd_index = 0;
    fpd_lock_init(&plink->invcmd_lock, 0);
    fpd_lock_init(&plink->echo_rsp, 0);
    atomic_set(&plink->close_comm, 0);

    /* initialize timer */
    plink->timer.timeout_value = 0;
    atomic_set(&plink->timer.running, 0);
    plink->timer.private = plink;

    /* start Link Interface thread that deals with data transmission */
    txcim_thread_init(plink);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_link_cleanup()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Cleanup the Link Interface
 *
 *  PARAMETERS:
 *
 *      plink     Pointer to the Link Interface structure.
 *
 *  RETURNS:
 *
 *     	0       Initialization is successful.
 *     -errno   Initialization failed.
 *
 *****************************************************************************/
int
fpd_link_cleanup( fpd_linkif_t *plink )
{
    /* stop Link IF thread */
    txcim_thread_cleanup(plink);

    /* cleanup CIM data buffer */
    pdata_cleanup(&plink->cim);

    if( plink->link_id != FPD_LINK_IF_ID_BGND ) {
        /* cleanup priority data buffer */
        pdata_cleanup(&plink->priority);

        /* cleanup host buffer */
        fpd_host_cleanup(&plink->host);
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_link_reset()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Resets the Link Interface.  The caller has to obtain the switch lock
 *      prior to calling this function.
 *
 *  PARAMETERS:
 *
 *      pfpd   -  Pointer to the FPGA Protocol device operated on.
 *      plink  -  Pointer to the specific Link Interface
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
fpd_link_reset( fpd_device_t *pfpd, fpd_linkif_t *plink )
{
    FPD_event_t *pevent = &pfpd->event;
    u32 reg;
    u32 link_base = plink->base;
    int i;

    /* reset the Link IF */
    reg = FPD_READ(pfpd->remap, link_base + LIF_LINK_CONTROL);
    reg |= LIFCR_LINK_IF_RST;
    FPD_WRITE(pfpd->remap, link_base + LIF_LINK_CONTROL, reg);

    /* clear the events for this Link IF */
    fpd_lock_acquire(&pfpd->event_lock);
    if( plink->link_id == FPD_LINK_IF_ID_BGND ) {
        pevent->bgnd_link_if.rx_cim = 0;
        pevent->bgnd_link_if.error = 0;
        pevent->bgnd_link_if.echo_rsp = 0;
    }
    else {
        pevent->link_if[plink->link_id].rx_priority = 0;
        pevent->link_if[plink->link_id].rx_cim = 0;
        pevent->link_if[plink->link_id].error = 0;
        pevent->link_if[plink->link_id].echo_rsp = 0;
        for( i = 0; i < FPD_HOST_CNT; i++ ) {
            pevent->link_if[plink->link_id].host[i].has_data = 0;
        }
    }
    fpd_lock_release(&pfpd->event_lock);

    /* reset CIM Buffer IF */
    pdata_reset(&plink->cim);

    if( plink->link_id != FPD_LINK_IF_ID_BGND ) {
        /* reset Priority Buffer IF */
        pdata_reset(&plink->priority);

        /* reset Host Buffer IF */
        for( i = 0; i < pfpd->info.host_buffer_cnt; i++ ) {
            fpd_host_reset(&plink->host, i);
        }
    }

    /* clear Paragon CIM Update mode bits */
    reg = FPD_READ(pfpd->remap, link_base + LIF_LINK_CONTROL);
    reg &= ~( LIFCR_UPDATE_MODE | LIFCR_CHG_MODE_NOW );
    FPD_WRITE(pfpd->remap, link_base + LIF_LINK_CONTROL, reg);

    /* disable Link Interface interrupts */
    fpd_link_disable_interrupts(pfpd, plink);

    plink->enabled = 0;

    return 0;
}
