/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Device Thread
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>	/* For (un)lock_kernel */
#include <linux/poll.h>
#include <linux/wait.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <linux/wrapper.h> /* wrapper for compatibility with future versions */
#endif
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/semaphore.h>

#include "fpd.h"
#include "fpd_handler.h"
#include "fpd_intr.h"
#include "debug.h"

#undef	DEBUG

#ifdef	DEBUG
# define debugk(fmt,args...)	printk(fmt ,##args)
#else
# define debugk(fmt,args...)
#endif

/* ==================================================================== */

/* ==================================================================== */

static irqreturn_t fpd_intr_handler(int irq, void *dev_arg, struct pt_regs *rgs)
{
    fpd_device_t *pfpd = dev_arg;

    debugk("%s: called %02x\n", __FUNCTION__, pfpd->id);

    /* disable interrupt for this device */
    fpd_disable_interrupt(pfpd);

    atomic_inc(&pfpd->fpd_req);
    wake_up_interruptible(&pfpd->fpd_queue);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    return IRQ_HANDLED;
#endif
}

static int fpd_set_interrupt(fpd_device_t *pfpd)
{
    int ret;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    /* This should really be initialized by either U-Boot or Linux 
     * Kernel and not hardcoded.
     *
     * For now use 0x1B for KIM on Wingle and 0x19 for KX2.0
     */
    pfpd->irq = 0x19;
#endif
    /*
    ret = request_irq(pfpd->irq, &fpd_intr_handler, SA_SHIRQ, pfpd->pcidev->name, pfpd);
    */
    ret = request_irq(pfpd->irq, &fpd_intr_handler, SA_SHIRQ, "fpd", pfpd);
    if(ret < 0) {
        FPD_ERROR("Can't install fpd%d IRQ %d (ret=%d)\n", pfpd->id, pfpd->irq, ret);
        return ret;
    }

    debugk("Installed fpd%d interrupt irq=%d\n", pfpd->id, pfpd->irq);

    return 0;
}

/* ==================================================================== */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void fpd_thread_setup(void *dev_arg)
{
    fpd_device_t *pfpd = (fpd_device_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    /* lock the kernel */
    lock_kernel();

    /* release insmod's memory map, use the kernel one's */
    exit_mm(current);

    /* set session leader and process group to init process */
    current->session = 1;
    current->pgrp = 1;

    /* set name of this process (max 15 chars + 0 !) */
    switch( pfpd->id ) {
        case 0:
            strcpy(current->comm, "fpd-thread");
            break;
        default:
            strcpy(current->comm, "CRAPPY-fpd");
            break;
    }

    /* disconnect from the fs usage */
    exit_files(current);
    exit_fs(current);

    /* fill in thread structure */
    pfpd->fpd_thread = current;

    /* set signal mask to what we want to respond */
    siginitsetinv(&current->blocked, sigmask(SIGKILL));

    /* let others run */
    unlock_kernel();

    /* tell the creator that we are ready and let him run */
    up(&pfpd->fpd_sem);
}
#endif

static void fpd_thread_leave(void *dev_arg)
{
    fpd_device_t *pfpd = (fpd_device_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    /* lock the kernel, the exit will unlock it */
    pfpd->fpd_thread = NULL;

    /* notify the kill_thread() routine that we are terminating. */
    up(&pfpd->fpd_sem);
}

static int fpd_thread(void *dev_arg)
{
    fpd_device_t *pfpd = (fpd_device_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    init_waitqueue_head(&pfpd->fpd_queue);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    /* Setup the thread environment */
    fpd_thread_setup(dev_arg);
#else
    lock_kernel();
    daemonize("fpd-thread");
    unlock_kernel();

    /* fill in thread structure */
    pfpd->fpd_thread = current;

    allow_signal(SIGKILL);

    /* tell the creator that we are ready and let him run */
    up(&pfpd->fpd_sem);
#endif

    FPD_INFO("FPGA Protocol Thread started\n");

    fpd_set_interrupt(pfpd);

    while (1) {
        /*
         * Wait for interrupt
         */
        wait_event_interruptible(pfpd->fpd_queue,
                                 (atomic_read(&pfpd->fpd_req) ||
                                  signal_pending(current)));

        debugk("%s%d: awakened\n", __FUNCTION__, pfpd->id);

        if (signal_pending(current)) {
            debugk("%s: Got a signal\n", __FUNCTION__);
            break;
        }

        /* clear the request */
        atomic_set(&pfpd->fpd_req, 0);

        /* process any interrupt or event */
        fpd_handler(pfpd);

        /* enable back interrupt */
        fpd_enable_interrupt(pfpd);
    }

    fpd_thread_leave(pfpd);

    return 0;
}

static void fpd_thread_start(void *dev_arg)
{
    fpd_device_t *pfpd = (fpd_device_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    init_MUTEX_LOCKED(&pfpd->fpd_sem);

    kernel_thread(fpd_thread, dev_arg, 0);

    /* Wait till it has reached the setup_thread routine */
    down(&pfpd->fpd_sem);
}

static void fpd_thread_stop(void *dev_arg)
{
    fpd_device_t *pfpd = (fpd_device_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    if (!pfpd->fpd_thread) {
        printk ("Can't terminate non-existant fpd_thread\n");
        return;
    }

    init_MUTEX_LOCKED(&pfpd->fpd_sem);

    kill_proc(pfpd->fpd_thread->pid, SIGKILL, 1);

    /* block till thread terminated */
    down(&pfpd->fpd_sem);
}

/* ==================================================================== */

int fpd_thread_init(void *dev_arg)
{
    fpd_device_t *pfpd = (fpd_device_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);
    atomic_set(&pfpd->fpd_req, 0);
    fpd_thread_start(dev_arg);

    return 0;
}

void fpd_thread_cleanup(void *dev_arg)
{
    fpd_device_t *pfpd = (fpd_device_t *)dev_arg;

    free_irq(pfpd->irq, pfpd);
    debugk("fpd%d IRQ %d uninstalled\n", pfpd->id, pfpd->irq);

    fpd_thread_stop(dev_arg);
}

void fpd_thread_wakeup(void *dev_arg)
{
    fpd_device_t *pfpd = (fpd_device_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);
    atomic_inc(&pfpd->fpd_req);
    wake_up_interruptible(&pfpd->fpd_queue);
}
