/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver Timer Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/timer.h>

#include "fpd.h"
#include "fpd_timer.h"
#include "debug.h"


/******************************************************************************
 *  ROUTINE:        timeout_handler()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Timeout handler.
 *
 *  PARAMETERS:
 *
 *     	ptr      Private data to be used if needed.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
static void
timeout_handler( unsigned long ptr )
{
    fpd_timer_t *ptimer = (fpd_timer_t *)ptr;

    atomic_set(&ptimer->running, 0);
    wake_up_interruptible(&ptimer->wait_queue);

    return;
}

/******************************************************************************
 *  ROUTINE:        fpd_setup_timer()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Sets up a timer.
 *
 *  PARAMETERS:
 *
 *     	ptimer    Pointer to a timer structure containing the required
 *                information.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
void
fpd_setup_timer( fpd_timer_t *ptimer )
{
    fpd_linkif_t *plink = (fpd_linkif_t *)ptimer->private;

    FPD_DEBUG(2, "timeout Link %d setup %d ms\n",
              plink->link_id, ptimer->timeout_value * 10);
    atomic_set(&ptimer->running, 1);
    init_timer(&ptimer->tlist);
    ptimer->tlist.function = timeout_handler;
    ptimer->tlist.data = (unsigned long)ptimer;
    /* 10ms interval */
    ptimer->tlist.expires = jiffies + ptimer->timeout_value;
    add_timer(&ptimer->tlist);

    return;
}

/******************************************************************************
 *  ROUTINE:        fpd_modify_timer()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Re-initializes the timeout value used in the specified timer.
 *
 *  PARAMETERS:
 *
 *     	ptimer    Pointer to a timer structure containing the required
 *                information.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
void
fpd_modify_timer( fpd_timer_t *ptimer )
{
    fpd_linkif_t *plink = (fpd_linkif_t *)ptimer->private;
    unsigned long expires;

    FPD_DEBUG(2, "timeout ID %d modified %d\n", plink->link_id,
              ptimer->timeout_value);

    expires = jiffies + (ptimer->timeout_value * HZ);
    mod_timer(&ptimer->tlist, expires);

    return;
}

/******************************************************************************
 *  ROUTINE:        fpd_cleanup_timer()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Cleans up the timer.
 *
 *  PARAMETERS:
 *
 *     	ptimer    Pointer to a timer structure containing the required
 *                information.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
void
fpd_cleanup_timer( fpd_timer_t *ptimer )
{
    fpd_linkif_t *plink = (fpd_linkif_t *)ptimer->private;

    FPD_DEBUG(2, "timeout Link %d cleanup\n", plink->link_id);

    del_timer_sync(&ptimer->tlist);
    atomic_set(&ptimer->running, 0);

    return;
}
