/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver Video Switch Control Interface
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>

#include "fpd.h"
#include "fpd_reg.h"
#include "fpd_ioctl.h"
#include "fpd_video_switch.h"
#include "el4544.h"
#include "debug.h"


/******************************************************************************
 *  ROUTINE:        fpd_vs_reset()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Resets video switch interface.
 *
 *  PARAMETERS:
 *
 *      pvs      Pointer to the video switch data structure.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
static int
fpd_vs_reset( fpd_vs_t *pvs )
{
    fpd_device_t *pfpd = (fpd_device_t *)pvs->private;
    u32 vsreg = ( VSR_VS_RST_MAIN | VSR_VS_RST_EXP );
    int chip;

    /* reset main and expander board video switches */
    FPD_WRITE(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL, vsreg);

    vsreg = 0;

    /* enable video output from all video switches */
    switch( pvs->vschip_cnt ) {
        case 4:
            vsreg |= VSR_VS_CHIPSEL3;
        case 3:
            vsreg |= VSR_VS_CHIPSEL2;
        case 2:
            vsreg |= VSR_VS_CHIPSEL1;
        case 1:
            vsreg |= VSR_VS_CHIPSEL0;
            break;
        default:
            FPD_WARNING("Detected incorrect video switch count\n");
            return -1;
    }
    FPD_WRITE(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL, vsreg);

    /* write default values for each video switch */
    for( chip = 0; chip < pvs->vschip_cnt; chip++ ) {
        /* EL4544 gain settings */
        fpd_vs_write_chip(pvs, chip, EL4544_GAIN_INPUT_MUX, EL4544_GAIN_1);

        /* disable all 4 crosspoint muxes */
        fpd_vs_write_chip(pvs, chip, EL4544_Ax_CROSSPOINT_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_Bx_CROSSPOINT_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_Cx_CROSSPOINT_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_Dx_CROSSPOINT_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_Sx_CROSSPOINT_MUX, 0);

        /* enable EL4544 sync detectors */
        fpd_vs_write_chip(pvs, chip, EL4544_SYNC_DETECTORS, EL4544_ENABLE_ALL_SYNC_DETECTORS);

        /* disable all 4 input muxes */
        fpd_vs_write_chip(pvs, chip, EL4544_ENABLE_INPUT_MUX, 0);

        /* zero out all 4 input mux */
        fpd_vs_write_chip(pvs, chip, EL4544_Ai_INPUT_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_Bi_INPUT_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_Ci_INPUT_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_Di_INPUT_MUX, 0);

        /* miscellaneous registers */
        fpd_vs_write_chip(pvs, chip, EL4544_Ti_INPUT_TEST_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_TEST_MUX, 0);
        fpd_vs_write_chip(pvs, chip, EL4544_SYNC_OVERLAY_CALIBRATION, 0);
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_vs_init()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Initializes the Video Switch Control Interface of the FPGA Protocol.
 *
 *  PARAMETERS:
 *
 *      pvs     Pointer to the Video Switch data structure.
 *      parg    Pointer to data to be stored in private member for later
 *              access 
 *
 *  RETURNS:
 *
 *     	0       Initialization is successful.
 *     -errno   Initialization failed.
 *
 *****************************************************************************/
int
fpd_vs_init( fpd_vs_t *pvs, void *parg )
{
    int remote_chan;
    int chip_cnt;
    int i;
    fpd_device_t *pfpd;
    u32 vsreg;

    /* determine model type to get the number of video switch installed */
    remote_chan = 4;
    chip_cnt = 2;

    /* init local video switch */
    pvs->local_video = kmalloc(sizeof(fpd_lvs_t), GFP_KERNEL);
    if( pvs->local_video ) {
        pvs->local_video->video_source = FPD_LOCAL_VIDEO_SRC_PARAGON_CIM;
        pvs->local_video->target_port = 0;
        pvs->local_video->vschip_id = -1;
        fpd_lock_init(&pvs->local_video->lock, 0);
    }
    else {
        return -ENOMEM;
    }

    /* init video switch shadow register */
    pvs->vschip = kmalloc(chip_cnt * sizeof(FPD_vschip_reg_t *), GFP_KERNEL);
    if( pvs->vschip ) {
        for( i = 0; i < chip_cnt; i++ ) {
            pvs->vschip[i] = kmalloc(sizeof(FPD_vschip_reg_t), GFP_KERNEL);
            if( pvs->vschip[i] ) {
                memset(pvs->vschip[i], 0, sizeof(FPD_vschip_reg_t));
            }
            else {
                kfree(pvs->vschip);
                kfree(pvs->local_video);
                return -ENOMEM;
            }
        }
    }
    else {
        kfree(pvs->local_video);
        return -ENOMEM;
    }

    /* init remote video switch */
    pvs->remote_video = kmalloc(remote_chan * sizeof(fpd_rvs_t *), GFP_KERNEL);
    if( pvs->remote_video ) {
        for( i = 0; i < remote_chan; i++ ) {
            pvs->remote_video[i] = kmalloc(sizeof(fpd_rvs_t), GFP_KERNEL);
            if( pvs->remote_video[i] ) {
                fpd_rvs_t *prvs = pvs->remote_video[i];
                prvs->remote_chan = i;
                prvs->target_port = 0;
                prvs->enabled = 0;
                fpd_lock_init(&prvs->lock, 0);
            }
            else {
                for( i = 0; i < chip_cnt; i++ ) {
                    kfree(pvs->vschip[i]);
                }
                kfree(pvs->vschip);
                kfree(pvs->remote_video);
                kfree(pvs->local_video);
                return -ENOMEM;
            }
        }
    }
    else {
        for( i = 0; i < chip_cnt; i++ ) {
            kfree(pvs->vschip[i]);
        }
        kfree(pvs->vschip);
        kfree(pvs->local_video);
        return -ENOMEM;
    }

    pvs->private = parg;
    pvs->remote_cnt = remote_chan;
    pvs->vschip_cnt = chip_cnt;

    /* reset all video switches */
    if( fpd_vs_reset(pvs) < 0 ) {
        fpd_vs_cleanup(pvs);
        return -EINVAL;
    }

    /* local video should point to on-board Graphic video */
    pfpd = (fpd_device_t *)pvs->private;
    vsreg = FPD_READ(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL);
    vsreg &= ~VSR_LV_SELECT;
    vsreg |= (VSR_LV_ENABLE| LV_GRAPHIC_ENGINE_VIDEO);
    FPD_WRITE(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL, vsreg);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_vs_cleanup()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Cleans up the Video Switch Control Interface of the FPGA Protocol.
 *
 *  PARAMETERS:
 *
 *      pvs     Pointer to the Video Switch data structure.
 *
 *  RETURNS:
 *
 *     	0       Action is successful.
 *
 *****************************************************************************/
int
fpd_vs_cleanup( fpd_vs_t *pvs )
{
    int i;

    /* free video switch shadow register */
    for( i = 0; i < pvs->vschip_cnt; i++ ) {
        kfree(pvs->vschip[i]);
    }
    kfree(pvs->vschip);

    /* free remote video data */
    for( i = 0; i < pvs->remote_cnt; i++ ) {
        kfree(pvs->remote_video[i]);
    }
    kfree(pvs->remote_video);

    /* free local video data */
    kfree(pvs->local_video);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_vs_write_chip()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Write access to the video switch chips through the Video Switch
 *      Control Register of FPGA.
 *
 *  PARAMETERS:
 *
 *      pvs     Pointer to the Video Switch data structure.
 *      chip    Video switch id to access.
 *      reg     Video switch register.
 *      data    Data in nibble to write.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
int
fpd_vs_write_chip( fpd_vs_t *pvs, u8 chip, u8 reg, u8 data )
{
    fpd_device_t *pfpd = (fpd_device_t *)pvs->private;
    FPD_vschip_reg_t *pvschip = pvs->vschip[chip];
    u32 vsreg;

    vsreg = FPD_READ(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL);
    FPD_DEBUG(2, "chip=%x, reg=%x, data=%x, vsreg=%08x\n", chip, reg, data, vsreg);

    /* select chip where the data should be written */
    vsreg &= ~(VSR_VS_WRITESEL_ALL | VSR_VS_DATA);
    vsreg |= ((VSR_VS_WRITESEL0 << chip) | reg | (data & 0xF) | VSR_VS_WRITE);
    FPD_DEBUG(2, "WRITE vsreg=%08x\n", vsreg);
    FPD_WRITE(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL, vsreg);

#ifndef USING_PCI_TEST_BOARD
    /* poll until write is complete
     * 
     * NOTE: Only applies for the real KX2.0 HW.
     *       FPGA Receiver Test board will be stuck here forever.
     */
    vsreg = FPD_READ(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL);
    while( vsreg & VSR_VS_WRITE ) {
        schedule();
        vsreg = FPD_READ(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL);
    };
#endif

    switch( reg ) {
        case EL4544_Ai_INPUT_MUX:
            pvschip->Ai = data;
            break;
        case EL4544_Bi_INPUT_MUX:
            pvschip->Bi = data;
            break;
        case EL4544_Ci_INPUT_MUX:
            pvschip->Ci = data;
            break;
        case EL4544_Di_INPUT_MUX:
            pvschip->Di = data;
            break;
        case EL4544_ENABLE_INPUT_MUX:
            pvschip->enable_input_mux = data;
            break;
        case EL4544_Ti_INPUT_TEST_MUX:
            pvschip->Ti = data;
            break;
        case EL4544_TEST_MUX:
            pvschip->test_mux = data;
            break;
        case EL4544_SYNC_DETECTORS:
            pvschip->sync_detectors = data;
            break;
        case EL4544_Ax_CROSSPOINT_MUX:
            pvschip->Ax = data;
            break;
        case EL4544_Bx_CROSSPOINT_MUX:
            pvschip->Bx = data;
            break;
        case EL4544_Cx_CROSSPOINT_MUX:
            pvschip->Cx = data;
            break;
        case EL4544_Dx_CROSSPOINT_MUX:
            pvschip->Dx = data;
            break;
        case EL4544_Sx_CROSSPOINT_MUX:
            pvschip->Sx = data;
            break;
        case EL4544_SYNC_OVERLAY_CALIBRATION:
            pvschip->soc = data;
            break;
        case EL4544_GAIN_INPUT_MUX:
            pvschip->gain_input_mux = data;
            break;
        default:
            break;
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_vs_disable()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Disables the cross-point switch.
 *
 *        Input Mux <-------> Crosspoint Mux
 *         (input)              (output)
 *           Ai     <------->      Ax
 *           Bi     <------->      Bx
 *           Ci     <------->      Cx
 *           Di     <------->      Dx
 *
 *  PARAMETERS:
 *
 *      pvs          Pointer to the Video Switch data structure.
 *      remote_chan  Remote KVM channel used.
 *      locked       Lock state. 0 means it does not have the lock yet.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
int
fpd_vs_disable( fpd_vs_t *pvs, u32 link_if, u32 remote_chan, u8 locked )
{
    fpd_device_t *pfpd = (fpd_device_t *)pvs->private;
    fpd_linkif_t *plink =  pfpd->link[link_if];
    fpd_rvs_t *prvs = pvs->remote_video[remote_chan];
    FPD_vschip_reg_t *pvschip;
    u32 vsc_control;
    u8 chip;
    u8 output_mux;
    u8 disable_input;
    u8 enable_input;

    switch( remote_chan ) {
        case 0:
            output_mux    = EL4544_Ax_CROSSPOINT_MUX;
            disable_input = EL4544_ENABLE_INPUT_Ai;
            break;
        case 1:
            output_mux    = EL4544_Bx_CROSSPOINT_MUX;
            disable_input = EL4544_ENABLE_INPUT_Bi;
            break;
        case 2:
            output_mux    = EL4544_Cx_CROSSPOINT_MUX;
            disable_input = EL4544_ENABLE_INPUT_Ci;
            break;
        case 3:
            output_mux    = EL4544_Dx_CROSSPOINT_MUX;
            disable_input = EL4544_ENABLE_INPUT_Di;
            break;
        default:
            return -EINVAL;
    }

    /* determine EL4544 chip to enable */
    if( prvs->target_port < 16 ) {
        chip = 0;
    }
    else if( prvs->target_port < 32 ) {
        chip = 1;
    }
    else if( prvs->target_port < 48 ) {
        chip = 2;
    }
    else if( prvs->target_port < 64 ) {
        chip = 3;
    }
    else {
        return -EINVAL;
    }

    if( chip >= pvs->vschip_cnt ) {
        return -EINVAL;
    }

    if( !locked ) {
        fpd_lock_acquire(&prvs->lock);
    }

    /* disable video output thru EL4544 Ax/Bx/Cx/Dx crosspoint switch */
    fpd_vs_write_chip(pvs, chip, output_mux, 0);

    /* disable specific input mux */
    pvschip = pvs->vschip[chip];
    enable_input = (pvschip->enable_input_mux & ~disable_input);
    fpd_vs_write_chip(pvs, chip, EL4544_ENABLE_INPUT_MUX, enable_input);

    /* VSC Control */
    vsc_control = FPD_READ(pfpd->remap, plink->base + LIF_VSC_CONTROL);
    vsc_control &= ~LIFVCR_VSC_ENABLE;
    FPD_WRITE(pfpd->remap, plink->base + LIF_VSC_CONTROL, vsc_control);

    prvs->enabled = 0;

    if( !locked ) {
        fpd_lock_release(&prvs->lock);
    }

    return 0;

}

/******************************************************************************
 *  ROUTINE:        fpd_vs_select()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Configures the cross-point switch to connect/disconnect the KVM video
 *      to/from the target port.
 *
 *        Input Mux <-------> Crosspoint Mux
 *         (input)              (output)
 *           Ai     <------->      Ax
 *           Bi     <------->      Bx
 *           Ci     <------->      Cx
 *           Di     <------->      Dx
 *
 *  PARAMETERS:
 *
 *      pvs          Pointer to the Video Switch data structure.
 *      input_chan   Input video channel that corresponds to the target port
 *                   channel
 *      output_chan  Output video channel Ax/Bx/Cx/Dx of the Intersil EL4544
 *                   chip.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
int
fpd_vs_select( fpd_vs_t *pvs, FPD_video_switch_t *premote )
{
    fpd_device_t *pfpd = (fpd_device_t *)pvs->private;
    fpd_linkif_t *plink =  pfpd->link[premote->link_if];
    fpd_rvs_t *prvs;
    FPD_vschip_reg_t *pvschip;
    u32 input_chan = premote->target_port;
    u32 output_chan = premote->kvm_chan;
    u32 vsc_control;
    u8 chip;
    u8 input_mux;
    u8 output_mux;
    u8 enable_input;

    /* determine EL4544 chip to enable */
    if( input_chan < 16 ) {
        chip = 0;
    }
    else if( input_chan < 32 ) {
        chip = 1;
    }
    else if( input_chan < 48 ) {
        chip = 2;
    }
    else if( input_chan < 64 ) {
        chip = 3;
    }
    else {
        return -EINVAL;
    }

    if( chip >= pvs->vschip_cnt ) {
        return -EINVAL;
    }

    switch( output_chan ) {
        case 0:
            input_mux    = EL4544_Ai_INPUT_MUX;
            output_mux   = EL4544_Ax_CROSSPOINT_MUX;
            enable_input = EL4544_ENABLE_INPUT_Ai;
            break;
        case 1:
            input_mux    = EL4544_Bi_INPUT_MUX;
            output_mux   = EL4544_Bx_CROSSPOINT_MUX;
            enable_input = EL4544_ENABLE_INPUT_Bi;
            break;
        case 2:
            input_mux    = EL4544_Ci_INPUT_MUX;
            output_mux   = EL4544_Cx_CROSSPOINT_MUX;
            enable_input = EL4544_ENABLE_INPUT_Ci;
            break;
        case 3:
            input_mux    = EL4544_Di_INPUT_MUX;
            output_mux   = EL4544_Dx_CROSSPOINT_MUX;
            enable_input = EL4544_ENABLE_INPUT_Di;
            break;
        default:
            return -EINVAL;
    }

    prvs = pvs->remote_video[output_chan];

    fpd_lock_acquire(&prvs->lock);

    /* check if this remote is currently connected */
    if( prvs->enabled ) {
        /* disconnect first */
        if( fpd_vs_disable(pvs, premote->link_if, output_chan, 1) < 0 ) {
            fpd_lock_release(&prvs->lock);
            return -EINVAL;
        }
    }

    /* write to EL4544 Ai/Bi/Ci/Di input mux */
    fpd_vs_write_chip(pvs, chip, input_mux, (input_chan % 16));
        
    /* write to EL4544 enable input mux */
    pvschip = pvs->vschip[chip];
    fpd_vs_write_chip(pvs, chip, EL4544_ENABLE_INPUT_MUX, (pvschip->enable_input_mux | enable_input));

    /* write to EL4544 Ax/Bx/Cx/Dx crosspoint switch */
    if( premote->protocol == FPD_PROTOCOL_VM ) {
        fpd_vs_write_chip(pvs, chip, output_mux, (output_chan | EL4544_CPMUX_ENABLE | EL4544_CPMUX_GAIN));
    }
    else {
        fpd_vs_write_chip(pvs, chip, output_mux, (output_chan | EL4544_CPMUX_ENABLE));
    }

    /* VSC Control */
    vsc_control = FPD_READ(pfpd->remap, plink->base + LIF_VSC_CONTROL);
    vsc_control |= LIFVCR_VSC_ENABLE;
    vsc_control &= ~LIFVCR_VSC_SELECT;
    vsc_control |= (output_chan << 28);
    if( premote->protocol == FPD_PROTOCOL_PARAGON ) {
        if( premote->hsync_pol ) {
            vsc_control |= LIFVCR_HSYNC_INV;
        }
        else {
            vsc_control &= ~LIFVCR_HSYNC_INV;
        }

        if( premote->vsync_pol ) {
            vsc_control |= LIFVCR_VSYNC_INV;
        }
        else {
            vsc_control &= ~LIFVCR_VSYNC_INV;
        }
    }
    FPD_WRITE(pfpd->remap, plink->base + LIF_VSC_CONTROL, vsc_control);

    prvs->target_port = input_chan;
    prvs->enabled = 1;

    fpd_lock_release(&prvs->lock);

    return 0;

}

/******************************************************************************
 *  ROUTINE:        fpd_vs_local_select()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Configures the Local Video to connect to either one of the following
 *      sources:
 *          - Paragon CIM
 *          - Virtual Media CIM
 *          - XGI Z7 Local Graphic Engine
 *      It is also possible to disable the Local Video (Green Mode).
 *
 *      For connection to the target ports, it is expected that a connection 
 *      between Link Interface and Line Interface has already been made prior
 *      to calling this function.
 *
 *  PARAMETERS:
 *
 *      pvs           Pointer to the Video Switch data structure.
 *      plocal        Pointer to the Local Video Switch data structure that
 *                    contains:
 *                      1. Local video source
 *                      2. Link Interface
 *                      3. Remote target channel. Only applicable if source is
 *                         Paragon or VM CIM
 *                      4. Horizontal sync polarity
 *                      5. Vertical sync polarity
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
int
fpd_vs_local_select( fpd_vs_t *pvs, FPD_local_video_switch_t *plocal )
{
    fpd_device_t *pfpd = (fpd_device_t *)pvs->private;
    fpd_linkif_t *plink;
    u32 vsreg, vsc_control;
#if 0
    int result;
#endif

    fpd_lock_acquire(&pvs->local_video->lock);

    switch( plocal->local_video_src ) {
        case FPD_LOCAL_VIDEO_SRC_DISABLE:
            vsreg = FPD_READ(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL);
            vsreg &= ~VSR_LV_ENABLE;
            break;

        case FPD_LOCAL_VIDEO_SRC_PARAGON_CIM:
            /* check for invalid parameters */
            if( plocal->link_if >= pfpd->info.link_if_cnt ) {
                fpd_lock_release(&pvs->local_video->lock);
                return -EINVAL;
            }

            plink = pfpd->link[plocal->link_if];

            /* VSC Control */
            vsc_control = FPD_READ(pfpd->remap, plink->base + LIF_VSC_CONTROL);
            FPD_DEBUG(2, "Read VSC Ctl = %08x\n", vsc_control);
            vsc_control |= LIFVCR_LOCAL_ENABLE;
            if( plocal->hsync_pol ) {
                vsc_control |= LIFVCR_HSYNC_INV;
            }
            else {
                vsc_control &= ~LIFVCR_HSYNC_INV;
            }

            if( plocal->vsync_pol ) {
                vsc_control |= LIFVCR_VSYNC_INV;
            }
            else {
                vsc_control &= ~LIFVCR_VSYNC_INV;
            }
            FPD_DEBUG(2, "Write VSC Ctl = %08x\n", vsc_control);
            FPD_WRITE(pfpd->remap, plink->base + LIF_VSC_CONTROL, vsc_control);

#if 0
            /* video switch goes thru EL4544 */
            if( pvs->local_video->vschip_id >= 0 ) {
                result = fpd_vs_select(pvs, plocal->link_if, plocal->target_port, pvs->local_video->vschip_id);
                if( result < 0 ) {
                    fpd_lock_release(&pvs->local_video->lock);
                    return result;
                }
            }
#endif
            pvs->local_video->target_port = plocal->target_port;

            vsreg = FPD_READ(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL);
            FPD_DEBUG(2, "Read Video Switch= %08x\n", vsreg);
            vsreg &= ~VSR_LV_SELECT;
            vsreg |= (VSR_LV_ENABLE| LV_PARAGON_CIM_VIDEO);
            /* done only if board is KX4xx */
            vsreg &= ~VSR_LV_LINE;
            vsreg |= (plocal->target_port << 26);
            break;

        case FPD_LOCAL_VIDEO_SRC_VM_CIM:
            /* check for invalid parameters */
            if( plocal->link_if >= pfpd->info.link_if_cnt ) {
                fpd_lock_release(&pvs->local_video->lock);
                return -EINVAL;
            }

            plink = pfpd->link[plocal->link_if];

            /* VSC Control */
            vsc_control = FPD_READ(pfpd->remap, plink->base + LIF_VSC_CONTROL);
            vsc_control |= LIFVCR_LOCAL_ENABLE;
            FPD_WRITE(pfpd->remap, plink->base + LIF_VSC_CONTROL, vsc_control);

#if 0
            /* video switch goes thru EL4544 */
            if( pvs->local_video->vschip_id >= 0 ) {
                result = fpd_vs_select(pvs, plocal->link_if, plocal->target_port, pvs->local_video->vschip_id);
                if( result < 0 ) {
                    fpd_lock_release(&pvs->local_video->lock);
                    return result;
                }
            }
#endif
            pvs->local_video->target_port = plocal->target_port;

            vsreg = FPD_READ(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL);
            vsreg &= ~VSR_LV_SELECT;
            vsreg |= (VSR_LV_ENABLE| LV_VM_CIM_VIDEO);
            /* done only if board is KX4xx */
            vsreg &= ~VSR_LV_LINE;
            vsreg |= (plocal->target_port << 26);
            break;

        case FPD_LOCAL_VIDEO_SRC_GRAPHIC_ENGINE:
            if( plocal->link_if >= pfpd->info.link_if_cnt ) {
                fpd_lock_release(&pvs->local_video->lock);
                return -EINVAL;
            }

            plink = pfpd->link[plocal->link_if];

            /* VSC Control */
            vsc_control = FPD_READ(pfpd->remap, plink->base + LIF_VSC_CONTROL);
            vsc_control &= ~LIFVCR_LOCAL_ENABLE;
            FPD_WRITE(pfpd->remap, plink->base + LIF_VSC_CONTROL, vsc_control);

            vsreg = FPD_READ(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL);
            vsreg &= ~VSR_LV_SELECT;
            vsreg |= (VSR_LV_ENABLE| LV_GRAPHIC_ENGINE_VIDEO);
            break;

        default:
            fpd_lock_release(&pvs->local_video->lock);
            return -EINVAL;
    }

    FPD_DEBUG(2, "Write Video Switch= %08x\n", vsreg);
    FPD_WRITE(pfpd->remap, FPD_VIDEO_SWITCH_CONTROL, vsreg);

    pvs->local_video->video_source = plocal->local_video_src;

    fpd_lock_release(&pvs->local_video->lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_vs_get_vschip_contents()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Get all the register contents of the video switch chips.
 *
 *  PARAMETERS:
 *
 *      pvs     Pointer to the Video Switch data structure.
 *      pchip   Pointer to the Video Switch Contents structure that will be
 *              used to store the register contents.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
int
fpd_vs_get_vschip_contents( fpd_vs_t *pvs, FPD_vschip_contents_t *pchip )
{
    if( pchip->dev >= pvs->vschip_cnt ) {
        return -EINVAL;
    }
    
    memcpy(&pchip->reg, pvs->vschip[pchip->dev], sizeof(FPD_vschip_reg_t));

    return 0;
}
