/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Module Initialization, Cleanup, and File Operations
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <linux/pci.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include <linux/cdev.h>
#endif

#if 0
#include <asm/ibm_ocp_pci.h>
#include <asm/page.h>
#endif

#include <fpd.h>
#include <fpd_ioctl.h>
#include <fpd_reg.h>
#include <fpd_intr.h>
#include <fpd_link.h>
#include <fpd_pdata.h>
#include <fpd_lock.h>
#include <fpd_thread.h>
#include <fpd_video_switch.h>
#include <txcim_thread.h>
#include <debug.h>
#include <version.h>
#include <dmabuf.h>


/*----------------------*
 * Macros and Constants *
 *----------------------*/

#define FPD_DRIVER_NAME                 "fpga-protocol"
#define FPD_DEVICE_NAME                 "fpd"
#define FPD_MAX_DEVICES                 1

/* this should be in pci_ids.h */
#define PCI_VENDOR_ID_ALTERA            0x1172
#define PCI_DEVICE_ID_ALTERA            0x0004

/*---------------------*
 * function prototypes *
 *---------------------*/

static int  fpd_init(void);
static void fpd_cleanup(void);

static int  fpd_open(struct inode *inode, struct file *file);
static int  fpd_release(struct inode *inode, struct file *file);
static int  fpd_ioctl(struct inode *inode, struct file *file,
                      unsigned int cmd, unsigned long arg);
static unsigned int fpd_poll(struct file *file, poll_table *wait);
static int  fpd_mmap(struct file *file, struct vm_area_struct *vma);

/*------------*
 * structures *
 *------------*/
static struct file_operations fpd_fops =
{
    ioctl:   fpd_ioctl,
    open:    fpd_open,
    release: fpd_release,
    poll:    fpd_poll,
    mmap:    fpd_mmap,
};

static struct pci_device_id fpd_pci_tbl[] = {
    { PCI_VENDOR_ID_ALTERA, PCI_DEVICE_ID_ALTERA, PCI_ANY_ID, PCI_ANY_ID, },
    { 0, },
};
MODULE_DEVICE_TABLE(pci, fpd_pci_tbl);

/*-----------*
 * variables *
 *-----------*/
const char      *fpd_module_name = FPD_DEVICE_NAME;
int              fpd_trace_on = 0;
int              fpd_debug_level = 1;
int              fpd_instance = 0;
int              fpd_major_number = 0;
fpd_device_t    *FPDdev[FPD_MAX_DEVICES];

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
static struct cdev *fpd_cdev = NULL;
#endif


/******************************************************************************
 *  ROUTINE:        fpd_probe()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Driver initialization routine : Calls the generic driver to 
 *      find all devices of type 'fpd', registers chardev and installs 
 *      /proc/dummy node.
 *
 *  PARAMETERS:
 *
 *      none.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int __devinit
fpd_probe(struct pci_dev *ppcidev, const struct pci_device_id *ent)
{
    fpd_device_t *pfpd;
    fpd_linkif_t *plink;
    fpd_host_t *phost;
    fpd_host_chan_t *pchan;
    u32 reg;
    int rc;
    int dev, link, host;
    int successful_dev_init = 0;
    int successful_linkif_init[FPD_MAX_DEVICES];
    int init_dma = 0, dma_size = 0, dma_rxtbl_size;
    enum fpd_initstage {
        STAGE_DEV_ALLOC = 0,
        STAGE_REQ_MEM_REGION,
        STAGE_IOREMAP,
        STAGE_BGND_LINKIF_ALLOC,
        STAGE_INIT_BGND_LINKIF,
        STAGE_LINKIF_ALLOC,
        STAGE_INIT_LINKIF,
        STAGE_INIT_DMA_RXBDTBL,
        STAGE_INIT_VIDEO_SWITCH,
        STAGE_COMPLETE,
    } init_stage[FPD_MAX_DEVICES];
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    int minor = 0;
    dev_t devno;
#endif

    FPD_TRACE_ENTER(__FUNCTION__);

    if(pci_enable_device(ppcidev) < 0)
        return -ENODEV;

    /* detect number of devices */
    fpd_instance = FPD_MAX_DEVICES;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    fpd_cdev = cdev_alloc();
    if( fpd_cdev == NULL ) {
        return -ENOMEM;
    }
#endif

    /* allocate memory for private data and initialize it */
    for(dev = 0; dev < fpd_instance; dev++) {
        FPD_INFO("STAGE_DEV_ALLOC\n");
        init_stage[dev] = STAGE_DEV_ALLOC;
        pfpd = kmalloc(sizeof(fpd_device_t), GFP_KERNEL);
        if( !pfpd ) {
            FPD_ERROR("(%s) FPDdev[%d] not allocated\n", __FUNCTION__, dev);
            rc = -ENOMEM;
            goto fpd_probe_error;
        }
        memset(pfpd, 0, sizeof(fpd_device_t));

        pfpd->id     = dev;
        pfpd->pcidev = ppcidev;
        pfpd->irq    = ppcidev->irq;
        pfpd->ioaddr = pci_resource_start(ppcidev, 0);
        pfpd->iolen  = pci_resource_len(ppcidev, 0);

        FPD_INFO("FPGA Protocol device #%d found at %02x:%02x\n",
                 pfpd->id, ppcidev->bus->number, ppcidev->devfn);

        FPD_INFO("STAGE_MEM_REGION\n");
        init_stage[dev] = STAGE_REQ_MEM_REGION;
        if(!request_mem_region(pfpd->ioaddr, pfpd->iolen, "FPGA Protocol")) {
            FPD_ERROR("fpd%d: Unable to request memory region\n", pfpd->id);
            rc = -ENOMEM;
            goto fpd_probe_error;
        }

        FPD_INFO("STAGE_IOREMAP\n");
        init_stage[dev] = STAGE_IOREMAP;
        pfpd->remap = ioremap_nocache(pfpd->ioaddr, pfpd->iolen);
        if(!pfpd->remap) {
            FPD_ERROR("fpd%d: Unable to remap memory region\n", pfpd->id);
            rc = -ENOMEM;
            goto fpd_probe_error;
        }

        FPD_DINFO(dev, "0x%08lx -> 0x%p (0x%lx)\n",
                  pfpd->ioaddr, pfpd->remap, pfpd->iolen);

        fpd_lock_init(&pfpd->event_lock, 0);
        fpd_lock_init(&pfpd->pci_error, 0);
        fpd_lock_init(&pfpd->switch_lock, 0);
        fpd_lock_init(&pfpd->inform_user, 0);
        fpd_lock_init(&pfpd->linestat_lock, 0);

        /* initialize user space wait queue */
        init_waitqueue_head(&pfpd->user_wq);

        /* get FPGA capabilities based on FPGA Version 0.24 and above */
        reg = FPD_READ(pfpd->remap, FPD_VERSION);
        pfpd->info.fpga_version.major = (reg & FVR_FPGA_VER) >> 16;
        pfpd->info.fpga_version.minor = (reg & FVR_FPGA_SUB_VER) >> 8;
        pfpd->info.fpga_version.patch = 0;
        pfpd->info.protocol_version.major = (reg & FVR_PROTOCOL_VER) >> 4;
        pfpd->info.protocol_version.minor = (reg & FVR_PROTOCOL_SUB_VER);
        pfpd->info.protocol_version.patch = 0;

        reg = FPD_READ(pfpd->remap, FPD_INTERFACE_COUNT);
        pfpd->info.link_if_cnt = (reg & ICR_LINK_IF_CNT) >> 7;
        pfpd->info.host_buffer_cnt = (reg & ICR_HOST_CHAN_CNT) >> 12;
        pfpd->info.line_cnt = (reg & ICR_LINE_CNT);
        if( reg & ICR_BGND_LINK_IF ) {
            pfpd->info.bgnd_link_if_present = 1;
        }
        else {
            pfpd->info.bgnd_link_if_present = 0;
        }

        pfpd->info.driver_version.major = FPD_MAJOR_NUM;
        pfpd->info.driver_version.minor = FPD_MINOR_NUM;
        pfpd->info.driver_version.patch = FPD_PATCH_NUM;

        /* initialize wait time (in seconds) for Link disconnection */
        atomic_set(&pfpd->driver_txbuf_wait_time, DRIVER_TXBUF_DEFAULT_WAIT_TIME);
        atomic_set(&pfpd->fpga_txbuf_wait_time, FPGA_TXBUF_DEFAULT_WAIT_TIME);

        /* initialize cim detection polling */
        atomic_set(&pfpd->cim_detect_event, 0);
        init_waitqueue_head(&pfpd->cim_detect_queue);

        /* Grab memory for host buffers */
        FPD_INFO("DMA init\n");
        if( !init_dma ) {
            /* Calculate required DMA buffers for each Host Buffer channel's 
             * Tx and Rx.
             *
             * dma_size = (# of Link IF cnt) * (# of Host Buffer per Link IF) *
             *            (2) * FPD_HOST_DMA_BUFFER_SIZE
             *   where 2 is derived from (TX side + RX side).
             */
            dma_size = (pfpd->info.link_if_cnt * 2 * 2) * FPD_HOST_DMA_BUFFER_SIZE;

            /* Calculate DMA RX buffer descriptor table */
            dma_rxtbl_size = (pfpd->info.link_if_cnt * 
                              pfpd->info.host_buffer_cnt * 
                              FPD_HOST_BUFFER_CNT * 
                              FPD_HOST_MAX_RX_BD_TABLE_SIZE);

            FPD_DDEBUG(dev, 1, "DMA size %d, rxtbl %d\n", dma_size, dma_rxtbl_size);
            rc = dma_buffer_init((dma_size + dma_rxtbl_size) * fpd_instance);
            if( rc < 0 ) {
                goto fpd_probe_error;
            }
            init_dma = 1;
        }

        /* initialize background link interface */
        FPD_INFO("STAGE_BGND_LINKIF_ALLOC\n");
        init_stage[dev] = STAGE_BGND_LINKIF_ALLOC;
        if( pfpd->info.bgnd_link_if_present ) {
            pfpd->bgnd_link = kmalloc(sizeof(fpd_linkif_t), GFP_KERNEL);
            if( pfpd->bgnd_link != NULL ) {
                init_stage[dev] = STAGE_INIT_BGND_LINKIF;
                if(( rc = fpd_link_init(pfpd, pfpd->bgnd_link, FPD_LINK_IF_ID_BGND) ) != 0 ) {
                    goto fpd_probe_error;
                }
            }
            else {
                rc = -ENOMEM;
                goto fpd_probe_error;
            }
        }
        else {
            pfpd->bgnd_link = NULL;
        }

        /* initialize each link interface */
        FPD_INFO("STAGE_LINKIF_ALLOC\n");
        init_stage[dev] = STAGE_LINKIF_ALLOC;
        successful_linkif_init[dev] = 0;
        pfpd->link = kmalloc(pfpd->info.link_if_cnt * sizeof(fpd_linkif_t *), GFP_KERNEL);
        if( pfpd->link ) {
            init_stage[dev] = STAGE_INIT_LINKIF;
            for(link = 0; link < pfpd->info.link_if_cnt; link++) {
                pfpd->link[link] = kmalloc(sizeof(fpd_linkif_t), GFP_KERNEL);
                if( pfpd->link[link] == NULL ) {
                    rc = -ENOMEM;
                    goto fpd_probe_error;
                }
                if(( rc = fpd_link_init(pfpd, pfpd->link[link], link) ) != 0 ) {
                    goto fpd_probe_error;
                }
                ++successful_linkif_init[dev];
            }
        }
        else {
            rc = -ENOMEM;
            goto fpd_probe_error;
        }

        /* initialize host buffer RX Buffer Descriptor Tables */
        FPD_INFO("STAGE_INIT_DMA_RXBDTBL\n");
        init_stage[dev] = STAGE_INIT_DMA_RXBDTBL;
        for(link = 0; link < pfpd->info.link_if_cnt; link++) {
            FPD_DEBUG(2, "Link %d Rx BD table\n", link);
            for( host = 0; host < pfpd->info.host_buffer_cnt; host++ ) {
                plink = pfpd->link[link];
                phost = &plink->host;
                if(( rc = fpd_host_init_rxbdtbl(phost, host) ) != 0 ) {
                    goto fpd_probe_error;
                }
            }
        }

        /* set the DMA memory that will be mmap by user-app */
        plink = pfpd->link[0];
        phost = &plink->host;
        pchan = phost->channel[0];
        pfpd->dmabuf.physicalAddress = pchan->tx.dmabuf.physicalAddress;
        pfpd->dmabuf.busAddress = pchan->tx.dmabuf.busAddress;
        pfpd->dmabuf.virtualAddress = pchan->tx.dmabuf.virtualAddress;
        pfpd->dmabuf.size = dma_size;
        FPD_DDEBUG(dev, 2, "dmabuf.physicalAddress 0x%p\n", pfpd->dmabuf.physicalAddress);
        FPD_DDEBUG(dev, 2, "dmabuf.busAddress 0x%p\n", pfpd->dmabuf.busAddress);
        FPD_DDEBUG(dev, 2, "dmabuf.virtualAddress 0x%p\n", pfpd->dmabuf.virtualAddress);
        FPD_DDEBUG(dev, 2, "dmabuf.size 0x%x\n", pfpd->dmabuf.size);
        pfpd->info.host_buffer_memsize = pfpd->dmabuf.size;
        pfpd->info.host_buffer_pagecnt = FPD_HOST_BUFFER_CNT;
        pfpd->info.host_buffer_pagesize = FPD_HOST_BUFFER_SIZE;

        /* initialize video switch interface */
        FPD_INFO("STAGE_INIT_VIDEO_SWITCH\n");
        init_stage[dev] = STAGE_INIT_VIDEO_SWITCH;
        if(( rc = fpd_vs_init(&pfpd->vs, pfpd) ) < 0 ) {
            goto fpd_probe_error;
        }

        /* store the current line status */
        switch( pfpd->info.line_cnt ) {
            case 64:
            case 48:
                pfpd->line_status_chg[1] = FPD_READ(pfpd->remap, FPD_LINE_STATUS_INTERRUPT_23);
                FPD_WRITE(pfpd->remap, FPD_LINE_STATUS_INTERRUPT_23, pfpd->line_status_chg[1]);
            case 32:
            case 16:
            case 1:
                pfpd->line_status_chg[0] = FPD_READ(pfpd->remap, FPD_LINE_STATUS_INTERRUPT_01);
                FPD_WRITE(pfpd->remap, FPD_LINE_STATUS_INTERRUPT_01, pfpd->line_status_chg[0]);
                break;
            default:
                FPD_WARNING("Invalid # of Lines\n");
                break;
        }
        pfpd->line_status_chg[0] = 0;
        pfpd->line_status_chg[1] = 0;

        FPDdev[dev] = pfpd;
        pci_set_drvdata(ppcidev, pfpd);

        /* start thread, if any */
        FPD_INFO("STAGE_THREAD_INIT\n");
        fpd_thread_init(pfpd);

        FPD_DINFO(dev, "+++ Protocol Ver %d.%d.%d FPGA Ver %d.%d.%d "
                  "Driver Ver %d.%d.%d (%d Link IF, %d lines, %d Host buffer)\n",
                  pfpd->info.protocol_version.major,
                  pfpd->info.protocol_version.minor,
                  pfpd->info.protocol_version.patch,
                  pfpd->info.fpga_version.major,
                  pfpd->info.fpga_version.minor,
                  pfpd->info.fpga_version.patch,
                  pfpd->info.driver_version.major,
                  pfpd->info.driver_version.minor,
                  pfpd->info.driver_version.patch,
                  pfpd->info.link_if_cnt,
                  pfpd->info.line_cnt,
                  pfpd->info.host_buffer_cnt);

        /* enable PCI error and Line status change interrupt */
        reg = FPD_READ(pfpd->remap, FPD_GLOBAL_CONTROL);
        reg |= (GCR_LINE_STATUS_CHG_INT_EN | GCR_PCI_ERR_INT_EN);
        FPD_WRITE(pfpd->remap, FPD_GLOBAL_CONTROL, reg);

        /* enable interrupt for this device */
        fpd_enable_interrupt(pfpd);

        init_stage[dev] = STAGE_COMPLETE;
        ++successful_dev_init;
    }

    pci_write_config_word(ppcidev, PCI_COMMAND, 0x0146);

#if 0
    {
        struct pcil0_regs *pcip;
        int reg_index;

        pcip = ioremap((uint)0xef400000, PAGE_SIZE);
        if (pcip != NULL) {
            for (reg_index = 0; reg_index < 3; reg_index++) {
                printk("PMM%dLA    %08x\n", reg_index, in_le32(&pcip->pmm[reg_index].la));
                printk("PMM%dMA    %08x\n", reg_index, in_le32(&pcip->pmm[reg_index].ma));
                printk("PMM%dPCILA %08x\n", reg_index, in_le32(&pcip->pmm[reg_index].pcila));
                printk("PMM%dPCIHA %08x\n", reg_index, in_le32(&pcip->pmm[reg_index].pciha));
            }
            printk("PTM1MS    %08x\n", in_le32(&pcip->ptm1ms));
            printk("PTM1LA    %08x\n", in_le32(&pcip->ptm1la));
            printk("PTM2MS    %08x\n", in_le32(&pcip->ptm2ms));
            printk("PTM2LA    %08x\n", in_le32(&pcip->ptm2la));
        }        

        iounmap(pcip);
    }
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#ifndef PRODUCT_KX2
    /* ask for a major number to be assigned for this driver */
    if((rc = alloc_chrdev_region(&devno, minor, fpd_instance, FPD_DEVICE_NAME)) < 0) {
	goto fpd_probe_error;
    }
#else /* KX2 case */
    /* in KX2 case we always use fixed major device numbers:
     * fs-skeleton/common/etc/devices 
     */

    /* FIXME: use a global include file instead of hard coded numbers */
    devno = MKDEV(254, minor);
    if((rc = register_chrdev_region(devno, fpd_instance, FPD_DEVICE_NAME)) < 0) {
	goto fpd_probe_error;
    }
#endif
    /* assign the dynamic major number */
    fpd_major_number = MAJOR(devno);

    cdev_init(fpd_cdev, &fpd_fops);
    fpd_cdev->owner = THIS_MODULE;
    fpd_cdev->ops = &fpd_fops;
    if(( rc = cdev_add(fpd_cdev, devno, 1) ) < 0) {
        unregister_chrdev_region(fpd_cdev->dev, fpd_instance);
        goto fpd_probe_error;
    }
#else /* linux 2.4 */
    if( (rc = register_chrdev(0, FPD_DEVICE_NAME, &fpd_fops)) < 0 ) {
        FPD_ERROR("(%s) unable to get major number\n", __FUNCTION__);
        goto fpd_probe_error;
    }

    /* assign the dynamic major number */
    fpd_major_number = rc;
#endif /* linux 2.4 */

    FPD_INFO("Registered FPGA Protocol device major=%d\n", fpd_major_number);

    FPD_TRACE_EXIT(__FUNCTION__);

    return(0);

 fpd_probe_error:
    for(dev = (successful_dev_init - 1); dev >= 0; dev--) {
        pfpd = FPDdev[dev];

        switch( init_stage[dev] ) {
            case STAGE_COMPLETE:
                /* stop thread, if any */
                fpd_thread_cleanup(pfpd);

                /* free allocated memory for video switch interface */
                fpd_vs_cleanup(&pfpd->vs);

            case STAGE_INIT_VIDEO_SWITCH:
                for(link = 0; link < pfpd->info.link_if_cnt; link++) {
                    for( host = 0; host < pfpd->info.host_buffer_cnt; host++ ) {
                        plink = pfpd->link[link];
                        phost = &plink->host;
                        fpd_host_cleanup_rxbdtbl(&pfpd->link[link]->host, host);
                    }
                }

            case STAGE_INIT_DMA_RXBDTBL:
            case STAGE_INIT_LINKIF:
                /* free buffers allocated for each link interface */
                for(link=(successful_linkif_init[dev] - 1); link >= 0; link--){
                    fpd_link_cleanup(pfpd->link[link]);
                    kfree(pfpd->link[link]);
                }

                kfree(pfpd->link);

            case STAGE_LINKIF_ALLOC:
                /* do the same if there is a background link interface */
                if( pfpd->info.bgnd_link_if_present ) {
                    fpd_link_cleanup(pfpd->bgnd_link);
                }

            case STAGE_INIT_BGND_LINKIF:
                if( pfpd->info.bgnd_link_if_present ) {
                    kfree(pfpd->bgnd_link);
                }

            case STAGE_BGND_LINKIF_ALLOC:
                iounmap(pfpd->remap);

            case STAGE_IOREMAP:
                release_mem_region(pfpd->ioaddr, pfpd->iolen);
                kfree(pfpd);
                break;

            default:
                break;
        }
    }

    /* release the host buffers */
    if( init_dma ) {
        dma_buffer_cleanup();
    }

    return rc;
}

/******************************************************************************
 *  ROUTINE:        fpd_remove()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Driver cleanup routine -- unregister /proc/g node, unregister 
 *      chardev and call generic driver cleanup.
 *
 *  PARAMETERS:
 *
 *      none.
 *
 *  RETURNS:
 *
 *     	nothing.
 *
 *****************************************************************************/
static void __devexit
fpd_remove(struct pci_dev *pdev)
{
    int dev, link, host;
    fpd_device_t *pfpd;
    fpd_linkif_t *plink;
    fpd_host_t *phost;

    FPD_TRACE_ENTER(__FUNCTION__);

    /* deallocate all private data */
    for( dev = 0; dev < fpd_instance; dev++ ) {
        pfpd = FPDdev[dev];

        /* stop thread, if any */
        fpd_thread_cleanup(pfpd);

        /* free allocated memory for video switch interface */
        fpd_vs_cleanup(&pfpd->vs);

        /* free buffers allocated for each link interface */
        for(link = 0; link < pfpd->info.link_if_cnt; link++) {
            fpd_link_cleanup(pfpd->link[link]);
            for( host = 0; host < pfpd->info.host_buffer_cnt; host++ ) {
                plink = pfpd->link[link];
                phost = &plink->host;
                fpd_host_cleanup_rxbdtbl(phost, host);
            }
            kfree(pfpd->link[link]);
        }

        kfree(pfpd->link);

        /* do the same if there is a background link interface */
        if( pfpd->info.bgnd_link_if_present ) {
            fpd_link_cleanup(pfpd->bgnd_link);
            kfree(pfpd->bgnd_link);
        }

        iounmap(pfpd->remap);
        release_mem_region(pfpd->ioaddr, pfpd->iolen);
        kfree(pfpd);
    }

    /* release the host buffers */
    dma_buffer_cleanup();

    /* unregister module (release the major number) */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    unregister_chrdev_region(fpd_cdev->dev, fpd_instance);
    cdev_del(fpd_cdev);
#else
    if( unregister_chrdev(fpd_major_number, FPD_DEVICE_NAME) < 0 ) {
        FPD_ERROR("(%s) unregister_chrdev() failed!\n", __FUNCTION__);
    }
#endif
    FPD_DEBUG(3, "Unregistered FPGA Protocol device major=%d\n",
              fpd_major_number);

    FPD_INFO("--- fpd driver unloaded\n");

    FPD_TRACE_EXIT(__FUNCTION__);
}

/******************************************************************************
 *  ROUTINE:        fpd_open()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      This routined gets called evey time user performs 'open' on the device.
 *	Find out correspondent driver structure and set file private data
 *	point to it.
 *
 *  PARAMETERS:
 *
 *      inode     	- inode # for the device to be opened.
 *	file		- file structure for that device
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int 
fpd_open(struct inode *inode, struct file *file)
{	
    unsigned int dev = MINOR(inode->i_rdev);

    FPD_DEBUG(5, "OPEN: file %p, minor %u, flags %08x\n", 
              file, (u32)inode->i_rdev, file->f_flags );

    file->f_op = &fpd_fops;
    file->private_data = FPDdev[dev];

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    MOD_INC_USE_COUNT;
#else
    try_module_get(THIS_MODULE);
#endif

    return(0);
}

/******************************************************************************
 *  ROUTINE:        fpd_release()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Just decrement usage count.
 *
 *  PARAMETERS:
 *
 *      inode     	- inode # for the device to be opened.
 *	file		- file structure for that device
 *
 *  RETURNS:
 *
 *     	always 0 (success).
 *
 *****************************************************************************/
static int
fpd_release(struct inode *inode, struct file *file)
{
    FPD_DEBUG(5, "CLOSE: file %08X\n", (u32) file);	

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0) 
    MOD_DEC_USE_COUNT;
#else
    module_put(THIS_MODULE);
#endif

    return(0);
}

/******************************************************************************
 *  ROUTINE:        fpd_ioctl()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Handles IOCTL request -- for now just passes to common driver IOCTL
 *	handler routine.
 *
 *  PARAMETERS:
 *
 *	inode   - inode for the device
 *      file   	- file structure that open device file info.
 *	cmd	- ioctl command
 *	arg	- ioctl argument
 *
 *  RETURNS:
 *
 *      0 on success, negative error code otherwise.
 *
 *****************************************************************************/
static int
fpd_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
         unsigned long arg)
{
    unsigned int dev = MINOR(inode->i_rdev);
    fpd_device_t *pdev = (fpd_device_t *)file->private_data;
    
    FPD_DEBUG(5, "IOCTL: cmd = %08X (%d), arg = %08lX\n", 
             cmd, _IOC_NR(cmd), arg );
    
    if(_IOC_TYPE(cmd) != FPD_MAGIC) return -ENOTTY;
    if( pdev == NULL ) {
	FPD_ERROR("(%s) NULL pdev.\n", __FUNCTION__);
	return -ENOTTY;
    }

    switch ( cmd ) {
        case 0:
            break;

        default:
            if(dev > FPD_MAX_DEVICES)
                return -ENOTTY;

            return( fpd_driver_ioctl(pdev, cmd, arg) );
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_poll()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Allow a process to determine whether it can read from the device.
 *
 *  PARAMETERS:
 *
 *      file   	- file structure that open device file info.
 *	wait	- 
 *
 *  RETURNS:
 *
 *      0 on success, negative error code otherwise.
 *
 *****************************************************************************/
static unsigned int
fpd_poll(struct file * file, poll_table * wait)
{
    fpd_device_t *pdev = (fpd_device_t *)file->private_data;
    fpd_linkif_t *plink;
    unsigned int mask = 0;
    int link, host;

    fpd_lock_acquire(&pdev->event_lock);

    for( link = 0; link < pdev->info.link_if_cnt; link++ ) {
        plink = pdev->link[link];

        /* check if there is Priority data available */
        if( pdata_check_events(&plink->priority) ){
            mask = POLLIN | POLLRDNORM;
        }

        /* check if there is CIM data available */
        if( pdata_check_events(&plink->cim) ){
            mask = POLLIN | POLLRDNORM;
        }

        /* check if there is host data available */
        for( host = 0; host < pdev->info.host_buffer_cnt; host++ ) {
            if( fpd_host_check_events(&plink->host, host) ){
                mask = POLLIN | POLLRDNORM;
            }
        }
    }

    /* check if there is Background Link CIM data available */
    if( pdev->info.bgnd_link_if_present ) {
        if( pdata_check_events(&pdev->bgnd_link->cim) ){
            mask = POLLIN | POLLRDNORM;
        }
    }

    /* check if there is an event pending already */
    if( fpd_lock_read_resource(&pdev->event_lock) ) {
        fpd_lock_release(&pdev->event_lock);

        /* clear in case of race condition behavior */
        fpd_lock_acquire(&pdev->inform_user);
        fpd_lock_write_resource(&pdev->inform_user, 0);
        fpd_lock_release(&pdev->inform_user);

        return( POLLIN | POLLRDNORM );
    }

    fpd_lock_release(&pdev->event_lock);

    if( mask ) {
        return mask;
    }

    /* wait for events to occur */
    poll_wait(file, &pdev->user_wq, wait);

    fpd_lock_acquire(&pdev->inform_user);
    if( fpd_lock_read_resource(&pdev->inform_user) ) {
        fpd_lock_write_resource(&pdev->inform_user, 0);
	mask = POLLIN | POLLRDNORM;
    }
    fpd_lock_release(&pdev->inform_user);

    return mask;
}

/******************************************************************************
 *  ROUTINE:        fpd_mmap()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Maps the FPGA Protocol device's DMA buffer to user space.
 *
 *  PARAMETERS:
 *
 *      file   	- file structure that open device file info.
 *	wait	- 
 *
 *  RETURNS:
 *
 *      0 on success, negative error code otherwise.
 *
 *****************************************************************************/
static int
fpd_mmap(struct file *file, struct vm_area_struct *vma)
{
    fpd_device_t *pdev = (fpd_device_t *)file->private_data;
    struct inode *inode = file->f_dentry->d_inode;
    unsigned int dev = MINOR(inode->i_rdev);
    int	result;
    unsigned long start;
    unsigned long size;
    unsigned long pos;
    DMA_MEM *pDmaMem;

    if(dev >= fpd_instance)
        return -ENODEV;

    pDmaMem = &pdev->dmabuf;

    FPD_DEBUG(3, "mmap start %08x end %08x\n", (int) vma->vm_start, (int) vma->vm_end );
    FPD_DEBUG(3, "mmap alloc %08x\n",(int) pDmaMem->virtualAddress);

    vma->vm_flags |= VM_RESERVED | VM_IO;
    start = vma->vm_start;
    size = vma->vm_end - vma->vm_start;
    pos = (unsigned long) pDmaMem->physicalAddress;

    /*
     * Configure this user-space memory map non-cacheable.  Without this, all
     * sorts of data integrity issue arises.  It does not matter if the memory
     * has been ioremap (_PAGE_NO_CACHE | _PAGE_GUARDED) before.  This seems
     * to be a new TLB entry.
     */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    pgprot_val(vma->vm_page_prot) |= _PAGE_NO_CACHE|_PAGE_GUARDED;
#else
    vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    result = remap_page_range(start, pos, size, vma->vm_page_prot);
#else
    result = remap_pfn_range(vma, start, pos >> PAGE_SHIFT, size, vma->vm_page_prot);
#endif
    if(result != 0) {
        FPD_ERROR("remap_page_range returned %d\n",result);
        return result;
    }

    return 0;
}

static struct pci_driver fpd_driver = {
    .name           = FPD_DRIVER_NAME,
    .id_table       = fpd_pci_tbl,
    .probe          = fpd_probe,
    .remove         = __devexit_p(fpd_remove),
};

static int __init fpd_init(void)
{
    return pci_module_init(&fpd_driver);
}

static void __exit fpd_cleanup(void)
{
    pci_unregister_driver(&fpd_driver);
}


#ifdef MODULE
/******************************************************************************
 *  ROUTINE:        module_init()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Mandatory entry point to the driver. First routine that gets called.
 *
 *****************************************************************************/
module_init(fpd_init);

/******************************************************************************
 *  ROUTINE:        module_exit()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Mandatory exit point to the driver. 
 *
 *****************************************************************************/
module_exit(fpd_cleanup);

MODULE_AUTHOR("Maynard Cabiente");
MODULE_DESCRIPTION("FPGA Protocol Driver");
#endif
