/**
 * gpio_core_xr17d158.c
 *
 * Low-level driver for the Exar XR17D158 Serial IO GPIO part
 *
 * (c) 2005 Peppercon AG, Ronald Wahl <rwa@peppercon.de>
 */

#include <linux/pci.h>
#include <linux/interrupt.h>

#include "gpio_common.h"
#include "gpio_core_opencores.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define XR_PCI_VENDOR_ID	0x0000 /* no serial eeprom attached - so these IDs are 0x0000 */
#define XR_PCI_DEVICE_ID	0x0000
#define XR_PCI_BAR_NO		0
#define XR_PCI_BAR_SIZE		4096
#define XR_NR_BITS		8
#define XR_NAME			"XR17D158"

/* GPIO register offsets */
#define XR_REG_INT1		0x81
#define XR_REG_MPIOINT		0x8F
#define XR_REG_MPIOLVL		0x90
#define XR_REG_MPIO3T		0x91
#define XR_REG_MPIOINV		0x92
#define XR_REG_MPIOSEL		0x93

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int		pci_bus;
    u_int		pci_slot;
    u_int		pci_func;
    struct pci_dev *	pci_dev;
    volatile u8 *	regs;
    int			isr_registered;
    int			mem_region_requested;
    u8			invert_input;
} xr_data_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

xr_data_t xr_data[] = {
    { .pci_bus			= 0,
      .pci_slot			= 8,
      .pci_func			= 0,
      .pci_dev			= NULL,
      .regs			= NULL,
      .isr_registered		= 0,
      .mem_region_requested	= 0,
      .invert_input             = 0
    },
    { .pci_bus			= 0,
      .pci_slot			= 9,
      .pci_func			= 0,
      .pci_dev			= NULL,
      .regs			= NULL,
      .isr_registered		= 0,
      .mem_region_requested	= 0,
      .invert_input             = 0
    }
};

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static void xr_cleanup(gpio_core_t *core);

static int xr_get_input(gpio_core_t *core, u_long *value);
static int xr_get_output(gpio_core_t *core, u_long *value);
static int xr_get_enable(gpio_core_t *core, u_long *value);
static int xr_set_output(gpio_core_t *core, u_long set, u_long clr);
static int xr_set_enable(gpio_core_t *core, u_long set, u_long clr);
static int xr_irq_enable(gpio_core_t *core, u_long set, u_long clr);
static int xr_irq_edge(gpio_core_t *core, u_long level, u_long edge, u_long pos, u_long neg);

static irqreturn_t xr_isr(int irq, void *dev_id, struct pt_regs *regs);

/* ------------------------------------------------------------------------- *
 * register access inline functions
 * ------------------------------------------------------------------------- */

static __inline u8
gpio_get_reg(xr_data_t *gpio, int reg)
{
    return readb(&gpio->regs[reg]);
}

static __inline void
gpio_set_reg(xr_data_t *gpio, int reg, u8 val)
{
    writeb(val, &gpio->regs[reg]);
}

static __inline u_char
gpio_get_reg_bits(xr_data_t *gpio, int reg, u8 mask)
{
    return gpio_get_reg(gpio, reg) & mask;
}

static __inline void
gpio_set_reg_bits(xr_data_t *gpio, int reg, u8 set_mask, u8 clr_mask)
{
    gpio_set_reg(gpio, reg, (gpio_get_reg(gpio, reg) & ~clr_mask) | set_mask);
}

/* ------------------------------------------------------------------------- *
 * exported functions (init)
 * ------------------------------------------------------------------------- */

int
gpio_core_xr17d158_init(gpio_core_t *core, u8 core_nr)
{
    int ret = -1;
    int nr_cores = sizeof(xr_data) / sizeof(xr_data_t);
    u32 mem_base, mem_end;
    xr_data_t *pdata;
    
    if (core_nr >= nr_cores) {
	ERR("%s: Unsupported core nr %u (max is %u)", XR_NAME, core_nr, nr_cores - 1);	
	goto bail;
    }

    core->private_data = pdata = &xr_data[core_nr];
    core->cleanup      = xr_cleanup;
   
    /* find hw */
    pdata->pci_dev = pci_find_slot(pdata->pci_bus, PCI_DEVFN(pdata->pci_slot, pdata->pci_func));
    if (!pdata->pci_dev
	|| pdata->pci_dev->vendor != XR_PCI_VENDOR_ID
	|| pdata->pci_dev->device != XR_PCI_DEVICE_ID) {
        ERR("%s: Failed to find PCI device (0x%04x:0x%04x) @ bus %u / slot %u / func %u",
	    XR_NAME, XR_PCI_VENDOR_ID, XR_PCI_DEVICE_ID, pdata->pci_bus, pdata->pci_slot, pdata->pci_func);
        goto bail;
    }
    
    mem_base = pci_resource_start(pdata->pci_dev, XR_PCI_BAR_NO);
    mem_end  = pci_resource_end(pdata->pci_dev, XR_PCI_BAR_NO);

    if (mem_base + XR_PCI_BAR_SIZE > (mem_end + 1)) {
	ERR("%s: Not enough mem space (have %u @ 0x%08x, need %u)",
	    XR_NAME, mem_end - mem_base, mem_base, XR_PCI_BAR_SIZE);
	goto bail;
    }

#if 0 /* already requested by serial driver! */
    if (request_mem_region(mem_base, XR_PCI_BAR_SIZE, XR_NAME) == NULL) {
	ERR("%s: Memory region @ 0x%08x (size %u) already in use",
	    XR_NAME, mem_base, XR_PCI_BAR_SIZE);
	goto bail;
    }
    pdata->mem_region_requested = 1;
#endif

    if ((pdata->regs = ioremap_nocache(mem_base, XR_PCI_BAR_SIZE)) == NULL) {
	ERR("%s: Could not remap PCI regs @ 0x%08x (size %d)",
	    XR_NAME, mem_base, XR_PCI_BAR_SIZE);
	goto bail;
    }

    if (request_irq(pdata->pci_dev->irq, xr_isr, SA_SHIRQ | SA_SAMPLE_RANDOM, XR_NAME, core) < 0) {
	ERR("%s: failed to request irq %d", XR_NAME, pdata->pci_dev->irq);
	goto bail;
    }
    pdata->isr_registered = 1;

    core->caps = GPIO_CAPS_IRQ;
    core->nr_bits = XR_NR_BITS;
    strncpy(core->name, XR_NAME, sizeof(core->name) - 1);
    core->name[sizeof(core->name) - 1] = '\0';

    core->get_input  = xr_get_input;
    core->get_output = xr_get_output;
    core->get_enable = xr_get_enable;
    core->set_output = xr_set_output;
    core->set_enable = xr_set_enable;
    core->irq_enable = xr_irq_enable;
    core->irq_edge   = xr_irq_edge;
    core->irq_ack    = NULL;

    ret = SUCCESS;

    INFO("GPIO core %s (%u) initialized", core->name, core_nr);
    
 bail:
    if (ret != 0) xr_cleanup(core);
    return ret;
}

/* ------------------------------------------------------------------------- *
 * core operations
 * ------------------------------------------------------------------------- */

static void
xr_cleanup(gpio_core_t *core)
{
    xr_data_t *pdata = (xr_data_t*) core->private_data;

    if (pdata == NULL) {
	return; // we were never really initialized
    }
    
    if (pdata->isr_registered) {
	free_irq(pdata->pci_dev->irq, core);
	pdata->isr_registered = 0;
    }

    if (pdata->regs != NULL) {
	iounmap((void*)pdata->regs);
	pdata->regs = NULL;
    }

    if (pdata->pci_dev && pdata->mem_region_requested) {
	release_mem_region(pci_resource_start(pdata->pci_dev, XR_PCI_BAR_NO), XR_PCI_BAR_SIZE);
    }

    INFO("GPIO core %s cleanup done", core->name);
}

static int
xr_get_input(gpio_core_t *core, u_long *value)
{
    xr_data_t *pdata = (xr_data_t*) core->private_data;
    u8 input_mask = gpio_get_reg(pdata, XR_REG_MPIOSEL);
  
    if (value) {
	*value = (gpio_get_reg(pdata, XR_REG_MPIOLVL) & input_mask) ^ (pdata->invert_input & input_mask);
    }

    return 0;
}

static int
xr_get_output(gpio_core_t *core, u_long *value)
{
    xr_data_t *pdata = (xr_data_t*) core->private_data;
    u8 output_mask = ~gpio_get_reg(pdata, XR_REG_MPIOSEL);

    if (value) *value = gpio_get_reg(pdata, XR_REG_MPIOLVL) & output_mask;

    return 0;
}

static int
xr_get_enable(gpio_core_t *core, u_long *value)
{
    xr_data_t *pdata = (xr_data_t*) core->private_data;

    /* register has inverse meaning - so negate it */
    if (value) *value = ~gpio_get_reg(pdata, XR_REG_MPIO3T);

    return 0;
}

static int
xr_set_output(gpio_core_t *core, u_long set, u_long clr)
{
    xr_data_t *pdata = (xr_data_t*) core->private_data;

    gpio_set_reg_bits(pdata, XR_REG_MPIOLVL, set & 0xFF, clr & 0xFF);

    return 0;
}

static int
xr_set_enable(gpio_core_t *core, u_long set, u_long clr)
{
    xr_data_t *pdata = (xr_data_t*) core->private_data;

    /* registers have inverse meaning - so swap set and clr */
    gpio_set_reg_bits(pdata, XR_REG_MPIO3T, clr & 0xFF, set & 0xFF);
    gpio_set_reg_bits(pdata, XR_REG_MPIOSEL, clr & 0xFF, set & 0xFF);

    return 0;
}

static int
xr_irq_enable(gpio_core_t *core, u_long set, u_long clr)
{
    xr_data_t *pdata = (xr_data_t*) core->private_data;

    u8 irq = (gpio_get_reg(pdata, XR_REG_MPIOINT) | set) & ~clr;
    u8 oe  = ~gpio_get_reg(pdata, XR_REG_MPIOSEL);

    if (irq & oe) {
	WARN("%s: Not all irq bits are set to input (MPIOINT=0x%02x, MPIOSEL=0x%02x)\n",
	     XR_NAME, irq, oe);
    }

    gpio_set_reg_bits(pdata, XR_REG_MPIOINT, set & 0xFF, clr & 0xFF);
    
    return 0;
}

static int
xr_irq_edge(gpio_core_t *core, u_long level, u_long edge,
	    u_long pos, u_long neg)
{
    xr_data_t *pdata = (xr_data_t*) core->private_data;

    if (level) {
	ERR("%s: driver does not support level triggered IRQs currently\n", XR_NAME);
	return -1;
    }
    
    gpio_set_reg_bits(pdata, XR_REG_MPIOINV, neg, pos);
    pdata->invert_input |= neg & 0xFF;
    pdata->invert_input &= ~pos & 0xFF;
    
    return 0;
}

/* ------------------------------------------------------------------------- *
 * internal/helper functions
 * ------------------------------------------------------------------------- */

static irqreturn_t
xr_isr(int irq, void *dev_id, struct pt_regs *regs)
{
    gpio_core_t *core = (gpio_core_t*) dev_id;
    xr_data_t *pdata = (xr_data_t*) core->private_data;
    u8 intr_mask = gpio_get_reg(pdata, XR_REG_MPIOINT);
    u8 int1 = gpio_get_reg(pdata, XR_REG_INT1);
    u_long value;
    
    if ((int1 & 0x06) != 0x06) {
	return IRQ_NONE; /* not our intr */
    }

    /* clear the irq bits */
    xr_get_input(core, &value);

    /* we cannot determine which MPIO triggered the interrupt - assume the irq mask */
    if (core->isr_cb) core->isr_cb(core, intr_mask);

    return IRQ_HANDLED;
}
