#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>
#include <linux/slab.h>

#include "debug.h"
#include "pp_i2c.h"
#include "i2c_kme.h"
#include "i2c_ipmi.h"
#include "i2c_adc.h"
#include "i2c_pci_adc.h"
#include "i2c_ddc.h"
#include "i2c_fp_led.h"
#include "i2c_fp_lcd.h"
#include "i2c_fp_bus.h"
#include "i2c_asics.h"
#include "i2c_faraday.h"
#include "i2c_ibm_iic.h"
#include "i2c_ktst_lcd_data.h"
#include "i2c_ktst_lcd_cmd.h"
#include "i2c_ipmb_test.h"
#include "i2c_common.h"

#define PP_I2C_NAME	"pp_i2c"

/* ------------------------------------------------------------------------- *
 * Module debugging and output stuff
 * ------------------------------------------------------------------------- */

#define DBG(fmt, x...)	I2C_DBG(PP_I2C_NAME, fmt, ##x)
#define DBG2(fmt, x...)	I2C_DBG2(PP_I2C_NAME, fmt, ##x)
#define WARN(fmt, x...)	I2C_WARN(PP_I2C_NAME, fmt, ##x)

/* ------------------------------------------------------------------------- *
 * function prototypes
 * ------------------------------------------------------------------------- */

static int pp_i2c_open(struct inode * inode, struct file * file);

/* ------------------------------------------------------------------------- *
 * devices as plugins
 * ------------------------------------------------------------------------- */

/* increase for more devices */
#define MAX_NO_PLUGINS	23

static struct i2c_device_plugin * plugins[MAX_NO_PLUGINS];

static int no_plugins = 0;

/* ------------------------------------------------------------------------- *
 * global structures
 * ------------------------------------------------------------------------- */

struct file_operations pp_i2c_ops = {
	owner:			THIS_MODULE,
	open:			pp_i2c_open,
};

/* ------------------------------------------------------------------------- *
 * Initialization
 * ------------------------------------------------------------------------- */

static __init int
pp_i2c_init(void)
{
	int i, r;
	int active_plugins = 0;
	
	DBG("Initializing module.\n");
	
	/* register the character device */	
	if ((r = register_chrdev(I2C_DEVICE_MAJOR, PP_I2C_NAME, &pp_i2c_ops)) < 0) {
		WARN("failed to register device (%d)\n", r);
		return -ENODEV;
	}
  
	i2c_kme_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
	i2c_adc_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
	i2c_ipmi_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);

#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI) || defined(LARA_KIMAMD) || defined(LARA_KIMINTEL)
	i2c_asics_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS, 5);
#endif
#ifdef PP_FEAT_DDC_DEVICE
	i2c_ddc_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
#endif
	
#ifdef PP_BOARD_KIRA
	i2c_faraday_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS, I2C_FARADAY_NO_ADAPTERS);
#else
	i2c_ibm_iic_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS, 1);
#endif
	
#ifdef PRODUCT_ASMIDC
	i2c_ipmb_test_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
#endif
	
#ifdef PRODUCT_FLASHX4
	i2c_fp_led_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
	i2c_fp_lcd_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
	i2c_fp_bus_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
#endif

#ifdef PP_FEAT_PCI_ADC
	i2c_pci_adc_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
#endif
    
#ifdef PRODUCT_KIMTESTERMST
	i2c_ktst_lcd_data_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
	i2c_ktst_lcd_cmd_init_plugin(plugins, &no_plugins, MAX_NO_PLUGINS);
#endif

	/* initialize all devices here */
	for (i = 0; i < no_plugins; i++) {
		if (!plugins[i]->init_func(plugins[i])) {
			plugins[i]->initialized = 1;
			active_plugins++;
		}
	}
	
	printk("module %s initialized. ", PP_I2C_NAME);
	if (active_plugins) {
		printk("Loaded devices:\n");
		for (i = 0; i < no_plugins; i++) {
			if (plugins[i]->initialized) printk("%s ", plugins[i]->name);
		}
		printk("\n");
	}
	else {
		printk("No deviced loaded!\n");
	}
	
	return 0;
}

static __exit void
pp_i2c_cleanup(void)
{
	int i, ret;
	
	for (i = 0; i < no_plugins; i++) {
		if (plugins[i]->initialized) plugins[i]->cleanup_func(plugins[i]);
	}
	
	/* deregister the character device */
	if ((ret = unregister_chrdev(I2C_DEVICE_MAJOR, PP_I2C_NAME)) < 0) {
		WARN("failed to deregister device (%d)\n", ret);
	}
  
	printk("module %s successfully unloaded.\n", PP_I2C_NAME);
}

/* ------------------------------------------------------------------------- *
 * Open an I2C device
 * ------------------------------------------------------------------------- */

static int pp_i2c_open(struct inode * inode, struct file * file) {
	int i;
	int minor = MINOR(inode->i_rdev);
	struct file_operations * ops = NULL;
	
	DBG2("Opening device with minor number %d\n", minor);
	for (i = 0; i < no_plugins; i++) {
		if (minor == plugins[i]->minor && plugins[i]->initialized) {
			ops = plugins[i]->fops;
			DBG2("Found plugin %s\n", plugins[i]->name);
			break;
		}
	}
	
	if (ops) {
		DBG2("Got file ops.\n");
		file->f_op = ops;
		return file->f_op->open(inode, file);
	}
	else {
		DBG("No file ops!\n");
		return -ENODEV;
	}
}

/* ------------------------------------------------------------------------- *
 * Linux kernel module stuff
 * ------------------------------------------------------------------------- */

#ifdef MODULE

MODULE_AUTHOR("thre@peppercon.de");
MODULE_DESCRIPTION("Linux kernel module for several I2C devices");
MODULE_LICENSE("GPL");

module_init(pp_i2c_init);
module_exit(pp_i2c_cleanup);

#endif	/* MODULE */

