/* ------------------------------------------------------------------------- *
 * I2C driver for DDC slave mode EDID transmission
 *
 * This is not your typical I2C driver, the semantics are different from the
 * other drivers (means 'quickhack') because the general i2c slave structure
 * is not sufficient to handle DDC in a sane way.
 *
 * As soon as another device needs more complicated slave operations,
 * the slave interface will be redesigned. Promised.
 *
 * -read/write is used to transfer the complete EDID structure (128byte)
 * -master transfers are not possible
 * -slave transfers with the device are handled by the driver itself
 * -poll, seek, ioctl are noops
 * ------------------------------------------------------------------------- */


#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>
#include <linux/slab.h>
#include <linux/poll.h>

#include "debug.h"
#include "i2c_common.h"

#define I2C_DDC_NAME	"i2c-ddc"

/* ------------------------------------------------------------------------- *
 * Module debugging and output stuff
 * ------------------------------------------------------------------------- */
#define DBG(fmt, x...)	I2C_DBG(I2C_DDC_NAME, fmt, ##x)
#define DBG2(fmt, x...)	I2C_DBG2(I2C_DDC_NAME, fmt, ##x)
#define WARN(fmt, x...)	I2C_WARN(I2C_DDC_NAME, fmt, ##x)

/* ------------------------------------------------------------------------- *
 * Constants
 * ------------------------------------------------------------------------- */
#define EDID_SIZE	128

/* ------------------------------------------------------------------------- *
 * Function prototypes
 * ------------------------------------------------------------------------- */

static int i2c_ddc_init(struct i2c_device_plugin *dev);
static void i2c_ddc_cleanup(struct i2c_device_plugin *dev);
static int i2c_ddc_i2c_attach_adapter(struct i2c_adapter *adapter);
static int i2c_ddc_i2c_detect_client(struct i2c_adapter *adapter, int address, 
				     unsigned short flags, int kind);
static int i2c_ddc_i2c_detach_client(struct i2c_client *client);
static void i2c_ddc_i2c_inc_use (struct i2c_client *client);
static void i2c_ddc_i2c_dec_use (struct i2c_client *client);
static int i2c_ddc_i2c_command(struct i2c_client *client,
			       unsigned int cmd, void *arg);
static int i2c_ddc_open(struct inode * inode, struct file * file);
static int i2c_ddc_release(struct inode * inode, struct file * file);
static loff_t i2c_ddc_llseek(struct file * file, loff_t offset, int whence);
static int i2c_ddc_read(struct file* file, char* buffer,
			size_t size, loff_t* offset);
static int i2c_ddc_write(struct file* file, const char* buffer,
			 size_t size, loff_t* offset);
static int i2c_ddc_slave_write_cb(struct i2c_slave_device *dev, char* buf, int size);
static int i2c_ddc_slave_read_cb(struct i2c_slave_device *dev, char* buf, int size);
static int i2c_ddc_slave_responsible_cb(struct i2c_slave_device *dev, unsigned int slave_addr, char* buf, int size);

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

struct file_operations i2c_ddc_ops = {
	owner:			THIS_MODULE,
	ioctl:			NULL,
	open:			i2c_ddc_open,
	release:		i2c_ddc_release,
	read:			i2c_ddc_read,
	write:			i2c_ddc_write,
	llseek:			i2c_ddc_llseek,
	poll:			NULL,
};

/* ------------------------------------------------------------------------- *
 * Module structures
 * ------------------------------------------------------------------------- */

static struct i2c_logical_device i2c_ddc_device = {
	name:			I2C_DDC_NAME,
#ifdef PP_BOARD_KIRA
	desired_adapter_name:	"FIA320 I2C 1",
#else
	desired_adapter_name:	"Asics I2C DDC",
#endif
	transmitter_mode:	I2C_MODE_MASTER,
	receiver_mode:		I2C_MODE_SLAVE,
	slave_address:		I2C_DDC_ADDRESS,
	master_address:		I2C_OWN_ADDRESS,
	master_bufsize:		I2C_MASTER_BUFFER_SIZE,
	i2c_initialized:	0,
	data:			NULL,
};

static struct i2c_device_plugin i2c_ddc_plugin = {
	name:		I2C_DDC_NAME,
	minor:		I2C_DDC_MINOR,
	address:	I2C_DDC_ADDRESS,
	initialized:	0,
	fops:		&i2c_ddc_ops,
	init_func:	i2c_ddc_init,
	cleanup_func:	i2c_ddc_cleanup,
};

/* ------------------------------------------------------------------------- *
 * driver structure
 * ------------------------------------------------------------------------- */

static struct i2c_driver i2c_ddc_driver = {
	name:			"i2c_ddc",
	id:			0xFFF4,	/* f000-ffff for local use */
	flags:			I2C_DF_NOTIFY,
	attach_adapter:		i2c_ddc_i2c_attach_adapter,
	detach_client:		i2c_ddc_i2c_detach_client,
	command:		i2c_ddc_i2c_command,
	inc_use:		i2c_ddc_i2c_inc_use,
	dec_use:		i2c_ddc_i2c_dec_use,
};

/* ------------------------------------------------------------------------- *
 * Slave device structure
 * ------------------------------------------------------------------------- */

static struct i2c_slave_device i2c_ddc_slave = {
	name:			"i2c_ddc",
	adapter:		NULL,	/* later filled up by init function */
	slave_write:		i2c_ddc_slave_write_cb,
	slave_read:		i2c_ddc_slave_read_cb,
	is_responsible:		i2c_ddc_slave_responsible_cb,
};

/* ------------------------------------------------------------------------- *
 * Variables
 * ------------------------------------------------------------------------- */
static u_int8_t edid_buffer[EDID_SIZE];
static u_int8_t edid_addr = 0;

/* ------------------------------------------------------------------------- *
 * Initialization
 * ------------------------------------------------------------------------- */

int
i2c_ddc_init_plugin(struct i2c_device_plugin ** plugins,
		    int * no_plugins, int max_no)
{
	if (*no_plugins >= max_no) {
		return -ENOMEM;
	}
	
	plugins[(*no_plugins)++] = &i2c_ddc_plugin;
	return 0;
}

static int
i2c_ddc_init(struct i2c_device_plugin *dev)
{
	DBG("init.\n");
	
	if (i2c_common_init(&i2c_ddc_device,
	    &i2c_ddc_driver, &i2c_ddc_slave) != 0) {
		goto fail;
	}
	
	DBG("initialization finished.\n");
	return 0;
fail:
	WARN("initialization failed.\n");
	return -1;
}

static void
i2c_ddc_cleanup(struct i2c_device_plugin *dev)
{
	DBG("cleanup.\n");
	
	i2c_common_cleanup(&i2c_ddc_device, &i2c_ddc_driver, &i2c_ddc_slave);
	
	DBG("cleanup finished.\n");
}

/* ------------------------------------------------------------------------- *
 * I2C functions
 * ------------------------------------------------------------------------- */

static void
i2c_ddc_i2c_inc_use (struct i2c_client *client)
{
}

static void
i2c_ddc_i2c_dec_use (struct i2c_client *client)
{
}

static int
i2c_ddc_i2c_command(struct i2c_client *client,
		    unsigned int cmd, void *arg)
{
	/* no commands defined */
	return 0;
}

static int
i2c_ddc_i2c_attach_adapter(struct i2c_adapter *adapter)
{
	return i2c_common_attach_adapter(&i2c_ddc_device,
		adapter, &i2c_ddc_i2c_detect_client);
}

static int
i2c_ddc_i2c_detach_client(struct i2c_client *client)
{
	return i2c_common_i2c_detach_client(&i2c_ddc_device, client);
}


static int
i2c_ddc_i2c_detect_client(struct i2c_adapter *adapter, int address, 
			  unsigned short flags, int kind)
{
	return i2c_common_i2c_detect_client(&i2c_ddc_device,
		&i2c_ddc_driver, adapter, address, flags, kind);
}


/* ------------------------------------------------------------------------- *
 * Device functions
 * ------------------------------------------------------------------------- */

static int
i2c_ddc_open(struct inode * inode, struct file * file)
{
	return i2c_common_open(&i2c_ddc_device, inode, file);
}

static int
i2c_ddc_release(struct inode * inode, struct file * file)
{
	return i2c_common_release(&i2c_ddc_device, inode, file);
}

static loff_t
i2c_ddc_llseek(struct file * file, loff_t offset, int whence)
{
    loff_t newpos;

    switch (whence) {
      case 0: /* SEEK_SET */
	  newpos = offset;
	  break;
      case 1: /* SEEK_CUR */
	  newpos = file->f_pos + offset;
	  break;
      case 2: /* SEEK_END */
	  newpos = EDID_SIZE - offset;
	  break;	  
      default:
	  return -EINVAL;
    }

    if (newpos < 0 || newpos >= EDID_SIZE) return -EINVAL;
    file->f_pos = newpos;

    return newpos;
}

static int
i2c_ddc_read(struct file* file, char* buffer,
	     size_t size, loff_t* offset)
{
    int count, ret = 0;

    if (file->f_flags & O_NONBLOCK) {
	if (down_trylock(&i2c_ddc_device.sem)) return -EAGAIN;
    } else {
	if (down_interruptible(&i2c_ddc_device.sem)) return -ERESTARTSYS;
    }

    /* handle EOF */
    if (*offset >= EDID_SIZE-1) {
	ret = 0;
	goto bail;
    }

    count = (*offset + size) > EDID_SIZE ? EDID_SIZE - *offset : size;

    if (copy_to_user(buffer, edid_buffer, count)) {
	ret = -EFAULT;
	goto bail;
    }

    *offset += count;
    ret = count;
    
 bail:
    up(&i2c_ddc_device.sem);
    return ret;

}

static int
i2c_ddc_write(struct file* file, const char* buffer,
	      size_t size, loff_t* offset)
{
    int count, ret = 0;
 
    if (file->f_flags & O_NONBLOCK) {
	if (down_trylock(&i2c_ddc_device.sem)) return -EAGAIN;
    } else {
	if (down_interruptible(&i2c_ddc_device.sem)) return -ERESTARTSYS;
    }

    /* handle EOF */
    if (*offset >= EDID_SIZE-1) {
	ret = 0;
	goto bail;
    }

    count = (*offset + size) > EDID_SIZE ? EDID_SIZE - *offset : size;

    if (copy_from_user(edid_buffer, buffer, count)) {
	ret = -EFAULT;
	goto bail;
    }

    *offset += count;
    ret = count;   
    
 bail:
    up(&i2c_ddc_device.sem);
    return ret;
}

/* ------------------------------------------------------------------------- *
 * I2C Slave mode callbacks
 * ------------------------------------------------------------------------- */

static int
i2c_ddc_slave_write_cb(struct i2c_slave_device *dev, char* buf, int size)
{
    if (size != 1) {
	WARN("unexpected size of ddc address (%d)\n", size);
    }

    DUMP_BUF("ddc_write_cb", buf, size);
    
    edid_addr = (u_int8_t) *buf;
    // reading the offset is necessary for some BIOSes which reads
    // some special bytes (e.g. EDID version) and decide whether to
    // continue
    // let us try whether we can read the offset reliable
    //
    // edid_addr = 0;
    DBG2("got edid address of %d\n", edid_addr);
    
    return size;
}

static int
i2c_ddc_slave_read_cb(struct i2c_slave_device *dev, char* buf, int size)
{
    int count = (edid_addr + size > EDID_SIZE) ? EDID_SIZE - edid_addr : size;

    DBG2("writing from edid_addr=%d, count=%d, beginning with %p\n",
	 edid_addr, count, &edid_buffer[edid_addr]);

    DUMP_BUF("ddc_read_cb", &edid_buffer[edid_addr], count);

    memcpy(buf, &edid_buffer[edid_addr], count);

    return count;
}

static int
i2c_ddc_slave_responsible_cb(struct i2c_slave_device *dev, unsigned int slave_addr, char* buf, int size)
{
    /* always responsible when our address matched */
    return 1;
}

