#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>
#include <linux/slab.h>
#include <linux/poll.h>

#include "debug.h"
#include "i2c_common.h"

#define I2C_IBM_IIC_NAME	"i2c-ibm_iic"
/* FIXME: still some redundancy here, replace wrapped _common calls with function pointers */

/* ------------------------------------------------------------------------- *
 * Module debugging and output stuff
 * ------------------------------------------------------------------------- */

#define DBG(fmt, x...)	I2C_DBG(I2C_IBM_IIC_NAME, fmt, ##x)
#define DBG2(fmt, x...)	I2C_DBG2(I2C_IBM_IIC_NAME, fmt, ##x)
#define WARN(fmt, x...)	I2C_WARN(I2C_IBM_IIC_NAME, fmt, ##x)

/* ------------------------------------------------------------------------- *
 * Constants
 * ------------------------------------------------------------------------- */

/* ------------------------------------------------------------------------- *
 * Function prototypes
 * ------------------------------------------------------------------------- */

static int i2c_ibm_iic_init(struct i2c_device_plugin *dev);
static void i2c_ibm_iic_cleanup(struct i2c_device_plugin *dev);
static int i2c_ibm_iic_i2c_attach_adapter(struct i2c_adapter *adapter);
static int i2c_ibm_iic_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
					 unsigned short flags,
#endif
					 int kind);
static int i2c_ibm_iic_i2c_detach_client(struct i2c_client *client);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void i2c_ibm_iic_i2c_inc_use (struct i2c_client *client);
static void i2c_ibm_iic_i2c_dec_use (struct i2c_client *client);
#endif
static int i2c_ibm_iic_i2c_command(struct i2c_client *client,
				   unsigned int cmd, void *arg);
static int i2c_ibm_iic_open(struct inode * inode, struct file * file);
static int i2c_ibm_iic_release(struct inode * inode, struct file * file);
static int i2c_ibm_iic_ioctl(struct inode * inode, struct file * file,
			     uint cmd, ulong arg);
static loff_t i2c_ibm_iic_llseek(struct file * file, loff_t offset, int whence);
static int i2c_ibm_iic_read(struct file* file, char* buffer,
			    size_t size, loff_t* offset);
static int i2c_ibm_iic_write(struct file* file, const char* buffer,
			     size_t size, loff_t* offset);
static unsigned int i2c_ibm_iic_poll(struct file * file, poll_table * wait);
static int i2c_ibm_iic_slave_write_cb(struct i2c_slave_device *dev, char* buf, int size);
static int i2c_ibm_iic_slave_read_cb(struct i2c_slave_device *dev, char* buf, int size);
static int i2c_ibm_iic_slave_responsible_cb(struct i2c_slave_device *dev, unsigned int slave_addr, char* buf, int size);

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

struct file_operations i2c_ibm_iic_ops = {
	owner:			THIS_MODULE,
	ioctl:			i2c_ibm_iic_ioctl,
	open:			i2c_ibm_iic_open,
	release:		i2c_ibm_iic_release,
	read:			i2c_ibm_iic_read,
	write:			i2c_ibm_iic_write,
	llseek:			i2c_ibm_iic_llseek,
	poll:			i2c_ibm_iic_poll,
};

/* ------------------------------------------------------------------------- *
 * Module structures
 * ------------------------------------------------------------------------- */

static struct i2c_logical_device i2c_ibm_iic_devices[I2C_IBM_IIC_NO_ADAPTERS];

static struct i2c_logical_device i2c_ibm_iic_device_template = {
	transmitter_mode:	I2C_MODE_MASTER,
	receiver_mode:		I2C_MODE_MASTER,
	slave_address:		0,
	master_address:		I2C_OWN_ADDRESS,
	master_bufsize:		I2C_MASTER_BUFFER_SIZE,
	i2c_initialized:	0,
	data:			NULL,
};

static struct i2c_device_plugin i2c_ibm_iic_plugins[I2C_IBM_IIC_NO_ADAPTERS];

static struct i2c_device_plugin i2c_ibm_iic_plugin_template = {
	address:	0,
	initialized:	0,
	fops:		&i2c_ibm_iic_ops,
	init_func:	i2c_ibm_iic_init,
	cleanup_func:	i2c_ibm_iic_cleanup,
};

/* ------------------------------------------------------------------------- *
 * driver structure
 * ------------------------------------------------------------------------- */

static struct i2c_driver i2c_ibm_iic_drivers[I2C_IBM_IIC_NO_ADAPTERS];

static struct i2c_driver i2c_ibm_iic_driver_template = {
	flags:			I2C_DF_NOTIFY,
	attach_adapter:		i2c_ibm_iic_i2c_attach_adapter,
	detach_client:		i2c_ibm_iic_i2c_detach_client,
	command:		i2c_ibm_iic_i2c_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	inc_use:		i2c_ibm_iic_i2c_inc_use,
	dec_use:		i2c_ibm_iic_i2c_dec_use,
#endif
};

/* ------------------------------------------------------------------------- *
 * Slave device structure
 * ------------------------------------------------------------------------- */

static struct i2c_slave_device i2c_ibm_iic_slaves[I2C_IBM_IIC_NO_ADAPTERS];

static struct i2c_slave_device i2c_ibm_iic_slave_template = {
	adapter:		NULL,	/* later filled up by init function */
	slave_write:		i2c_ibm_iic_slave_write_cb,
	slave_read:		i2c_ibm_iic_slave_read_cb,
	is_responsible:		i2c_ibm_iic_slave_responsible_cb,
};

/* ------------------------------------------------------------------------- *
 * Find the logical device from some information
 * ------------------------------------------------------------------------- */

static int get_device_index_from_plugin(struct i2c_device_plugin * dev)
{
	int i;
	
	for (i = 0; i < I2C_IBM_IIC_NO_ADAPTERS; i++) {
		if (&i2c_ibm_iic_plugins[i] == dev) {
			DBG2("Found adapter %s by plugin.\n", i2c_ibm_iic_devices[i].name);
			return i;
		}
	}
	DBG("No adapter found by plugin!\n");
	return -1;
}

static struct i2c_logical_device * get_device_from_inode(struct inode * inode)
{
	int i;
	int minor = MINOR(inode->i_rdev);
	
	for (i = 0; i < I2C_IBM_IIC_NO_ADAPTERS; i++) {
		if (i2c_ibm_iic_plugins[i].minor == minor) {
			DBG2("Found adapter %s by inode.\n", i2c_ibm_iic_devices[i].name);
			return &i2c_ibm_iic_devices[i];
		}
	}
	DBG("No adapter found by inode!\n");
	return NULL;
}

static struct i2c_logical_device * get_device_from_slave(struct i2c_slave_device * dev)
{
	int i;
	
	for (i = 0; i < I2C_IBM_IIC_NO_ADAPTERS; i++) {
		if (&i2c_ibm_iic_slaves[i] == dev) {
			DBG2("Found adapter %s by slave.\n", i2c_ibm_iic_devices[i].name);
			return &i2c_ibm_iic_devices[i];
		}
	}
	DBG("No adapter found by slave!\n");
	return NULL;
}

static int get_device_index_from_adapter(struct i2c_adapter * adap)
{
	int i;
	
	for (i = 0; i < I2C_IBM_IIC_NO_ADAPTERS; i++) {
		if (!strcmp(adap->name, i2c_ibm_iic_devices[i].desired_adapter_name)) {
			DBG2("Found adapter %s by adapter.\n", i2c_ibm_iic_devices[i].name);
			return i;
		}
	}
	DBG("No adapter found by adapter!\n");
	return -1;
}

static struct i2c_logical_device * get_device_from_client(struct i2c_client * cli)
{
	int i;
	char name[16];
	
	if (!cli->driver) {
		DBG("No adapter found by client, no driver there!\n");
		return NULL;
	}
	
	for (i = 0; i < I2C_IBM_IIC_NO_ADAPTERS; i++) {
		snprintf(name, sizeof(name), "i2c_ibm_iic_%d", i);
		if (!strcmp(cli->driver->name, name)) {
			DBG2("Found adapter %s by client.\n", i2c_ibm_iic_devices[i].name);
			return &i2c_ibm_iic_devices[i];
		}
	}
	DBG("No adapter found by client!\n");
	return NULL;
}

/* ------------------------------------------------------------------------- *
 * Initialization
 * ------------------------------------------------------------------------- */

int
i2c_ibm_iic_init_plugin(struct i2c_device_plugin ** plugins,
			int * no_plugins, int max_no, int no_ibm_iic_plugins)
{
	int i;
	
	if (no_ibm_iic_plugins > I2C_IBM_IIC_NO_ADAPTERS) {
		no_ibm_iic_plugins = I2C_IBM_IIC_NO_ADAPTERS;
	}
	
	for (i = 0; i < no_ibm_iic_plugins; i++) {
		/* initialize the plugin */
		memcpy(&i2c_ibm_iic_devices[i], &i2c_ibm_iic_device_template, sizeof(i2c_ibm_iic_device_template));
		snprintf(i2c_ibm_iic_devices[i].name, sizeof(i2c_ibm_iic_devices[i].name), I2C_IBM_IIC_NAME "-%d", i);
		snprintf(i2c_ibm_iic_devices[i].desired_adapter_name, sizeof(i2c_ibm_iic_devices[i].desired_adapter_name),
			 "IBM_IIC I2C %d", i);
		
		memcpy(&i2c_ibm_iic_plugins[i], &i2c_ibm_iic_plugin_template, sizeof(i2c_ibm_iic_plugin_template));
		snprintf(i2c_ibm_iic_plugins[i].name, sizeof(i2c_ibm_iic_plugins[i].name), I2C_IBM_IIC_NAME "-%d", i);
		i2c_ibm_iic_plugins[i].minor = I2C_IBM_IIC_FIRST_MINOR + i;
		
		memcpy(&i2c_ibm_iic_slaves[i], &i2c_ibm_iic_slave_template, sizeof(i2c_ibm_iic_slave_template));
		snprintf(i2c_ibm_iic_slaves[i].name, sizeof(i2c_ibm_iic_slaves[i].name), "i2c_ibm_iic_%d", i);

		memcpy(&i2c_ibm_iic_drivers[i], &i2c_ibm_iic_driver_template, sizeof(i2c_ibm_iic_driver_template));
		snprintf(i2c_ibm_iic_drivers[i].name, sizeof(i2c_ibm_iic_drivers[i].name), "i2c_ibm_iic_%d", i);
		i2c_ibm_iic_drivers[i].id = 0xFFE0 + i;

		/* add the plugin */
		if (*no_plugins >= max_no) {
			return -ENOMEM;
		}
		
		plugins[(*no_plugins)++] = &i2c_ibm_iic_plugins[i];
	}
	return 0;
}

static int
i2c_ibm_iic_init(struct i2c_device_plugin *dev)
{
	int index = get_device_index_from_plugin(dev);
	if (dev < 0) {
		WARN("Init: Plugin not found!\n");
		return -ENODEV;
	}
	
	DBG("init.\n");
	
	if (i2c_common_init(&i2c_ibm_iic_devices[index],
	    &i2c_ibm_iic_drivers[index], &i2c_ibm_iic_slaves[index]) != 0) {
		goto fail;
	}
	
	DBG("initialization finished.\n");
	return 0;
fail:
	WARN("initialization failed.\n");
	return -1;
}

static void
i2c_ibm_iic_cleanup(struct i2c_device_plugin *dev)
{
	int index = get_device_index_from_plugin(dev);
	if (dev < 0) {
		WARN("Init: Plugin not found!\n");
		return;
	}

	DBG("cleanup.\n");
	
	i2c_common_cleanup(&i2c_ibm_iic_devices[index],
		&i2c_ibm_iic_drivers[index], &i2c_ibm_iic_slaves[index]);
	
	DBG("cleanup finished.\n");
}

/* ------------------------------------------------------------------------- *
 * I2C functions
 * ------------------------------------------------------------------------- */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void
i2c_ibm_iic_i2c_inc_use (struct i2c_client *client)
{
}

static void
i2c_ibm_iic_i2c_dec_use (struct i2c_client *client)
{
}
#endif

static int
i2c_ibm_iic_i2c_command(struct i2c_client *client,
		    unsigned int cmd, void *arg)
{
	/* no commands defined */
	return 0;
}

static int
i2c_ibm_iic_i2c_attach_adapter(struct i2c_adapter *adapter)
{
	int index = get_device_index_from_adapter(adapter);
	if (index < 0) {
		return -ENODEV;
	}
	return i2c_common_attach_adapter(&i2c_ibm_iic_devices[index],
		adapter, &i2c_ibm_iic_i2c_detect_client);
}

static int
i2c_ibm_iic_i2c_detach_client(struct i2c_client *client)
{
	struct i2c_logical_device * dev = get_device_from_client(client);
	return dev ? i2c_common_i2c_detach_client(dev, client) : -ENODEV;
}


static int
i2c_ibm_iic_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			    unsigned short flags,
#endif
			    int kind)
{
	int index = get_device_index_from_adapter(adapter);
	if (index < 0) {
		return -ENODEV;
	}
	return i2c_common_i2c_detect_client(&i2c_ibm_iic_devices[index],
		&i2c_ibm_iic_drivers[index], adapter, address,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	       	flags,
#endif
	       	kind);
}


/* ------------------------------------------------------------------------- *
 * Device functions
 * ------------------------------------------------------------------------- */

static int
i2c_ibm_iic_open(struct inode * inode, struct file * file)
{
	struct i2c_logical_device * dev = get_device_from_inode(inode);
	return dev ? i2c_common_open(dev, inode, file) : -ENODEV;
}

static int
i2c_ibm_iic_release(struct inode * inode, struct file * file)
{
	struct i2c_logical_device * dev = get_device_from_inode(inode);
	return dev ? i2c_common_release(dev, inode, file) : -ENODEV;
}

static loff_t
i2c_ibm_iic_llseek(struct file * file, loff_t offset, int whence)
{
	return i2c_common_llseek((struct i2c_logical_device *)file->private_data,
		file, offset, whence);
}

static int
i2c_ibm_iic_ioctl(struct inode * inode, struct file * file,
	      uint cmd, ulong arg)
{
	return i2c_common_ioctl((struct i2c_logical_device *)file->private_data,
		inode, file, cmd, arg);
}

static int
i2c_ibm_iic_read(struct file* file, char* buffer,
	     size_t size, loff_t* offset)
{
	return i2c_common_read((struct i2c_logical_device *)file->private_data,
		file, buffer, size, offset);
}

static int
i2c_ibm_iic_write(struct file* file, const char* buffer,
	      size_t size, loff_t* offset)
{
	return i2c_common_write((struct i2c_logical_device *)file->private_data,
		file, buffer, size, offset);
}

static unsigned int
i2c_ibm_iic_poll(struct file * file, poll_table * wait)
{
	return i2c_common_poll((struct i2c_logical_device *)file->private_data,
		file, wait);
}

/* ------------------------------------------------------------------------- *
 * I2C Slave mode callbacks
 * ------------------------------------------------------------------------- */

static int
i2c_ibm_iic_slave_write_cb(struct i2c_slave_device *dev, char* buf, int size)
{
	struct i2c_logical_device * ldev = get_device_from_slave(dev);
	return ldev ? i2c_common_slave_write_cb(ldev, buf, size) : -ENODEV;
}

static int
i2c_ibm_iic_slave_read_cb(struct i2c_slave_device *dev, char* buf, int size)
{
	struct i2c_logical_device * ldev = get_device_from_slave(dev);
	return ldev ? i2c_common_slave_read_cb(ldev, buf, size) : -ENODEV;
}

static int
i2c_ibm_iic_slave_responsible_cb(struct i2c_slave_device *dev, unsigned int slave_addr, char* buf, int size)
{
	/* not responsible at the moment */
	return 0;
}

