/*****************************************************************************
 * smbus_arp.c
 *
 * SMBUS ARP implementation, a module able to enumerate devices on an I2C bus
 * and assign addresses to them
 *
 * FIXME: currently assuming that we are the only SMBUS ARP master on the bus
 * FIXME: this module does not support SMBUS hotplug/Notify ARP Master
 *****************************************************************************/

#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/module.h>

#include "smbus_arp.h"

/* FIXME: add locking */

#define noUNUSED_FUNCTIONS
#define LOG_COLORED
#define LOG_DBG
#include "log.h"

#define SUCCESS			0
#define ERROR			-1
#define ADD_PEC			1
#define DONT_ADD_PEC		0
#define DELAY()			udelay(100)
#define MASTER_RETRIES		5
#define MAX_RETRY_COUNT		10
#define I2C_MAX_DATA_SIZE	32
#define DEVNAME			"smbus_arp"

/* SMBUS ARP commands */
#define CMD_PREPARE_ARP		0x01
#define CMD_RESET_DEVICE	0x02
#define CMD_GET_UDID		0x03
#define CMD_ASSIGN_ADDRESS	0x04

#define SMBUS_ARP_ADDR		0x61
#define SMBUS_ADDRESS_SIZE	128

#define ARP_STATUS_FREE		0x00
#define ARP_STATUS_RESERVED	0x01
#define ARP_STATUS_BUSY		0x02

/* ------------------------------------------------------------------------- *
 * Internal ARP handling structures
 * ------------------------------------------------------------------------- */

typedef struct {
    smbus_udid_caps_t		caps;
    smbus_udid_version_t	version;
    uint16_t			vendor_id_le16;
    uint16_t			device_id_le16;
    uint8_t			interface_reserved;
    smbus_udid_interface_t	interface;
    uint16_t			subvendor_id_le16;
    uint16_t			subdevice_id_le16;
    uint32_t			specific_id_le32;
} __attribute__((packed)) smbus_udid_wire_t;

typedef struct {
    struct list_head	anchor;
    int			status;
    uint8_t		slave_addr;
    smbus_udid_t	udid;
} arp_device_t;

typedef struct {
    struct list_head	anchor;
    struct i2c_client  *client;					/* i2c client, one per adapter */
    uint8_t		address_pool[SMBUS_ADDRESS_SIZE];	/* free/busy/reserved addresses */
    struct list_head	arp_dev_list;				/* list of enumerated ARP devices */
} client_entry_t;


LIST_HEAD(client_list);
//static spinlock_t client_list_lock = SPIN_LOCK_UNLOCKED;
static int i2c_driver_initialized = 0;

/* reserved addresses not to use, according to
   SMBUS spec Appendix C and chapter 5.2 */
static uint8_t reserved_addresses[] = {
    0x01, /* CBUS */
    0x02, /* different bus format */
    0x03, /* reserved for future use */
    0x04, /* reserved for future use */
    0x05, /* reserved for future use */
    0x06, /* reserved for future use */
    0x07, /* reserved for future use */    
    0x08, /* smbus host */
    0x09, /* smart battery charger */
    0x0A, /* smart battery selector / system manager */
    0x0B, /* smart battery */
    0x0C, /* smbus alert response */
    0x28, /* ACCESS.bus host */
    0x2C, /* LCD controller */
    0x2D, /* CCFL backlight */
    0x37, /* ACCESS.bus default address */
    0x40, /* PCMCIA */
    0x41, /* PCMCIA */
    0x42, /* PCMCIA */
    0x43, /* PCMCIA */
    0x44, /* VGA */
    0x61, /* ARP slave */
    0x78, /* 10 bit addressing */
    0x79, /* 10 bit addressing */
    0x7A, /* 10 bit addressing */
    0x7B, /* 10 bit addressing */
    0x7C, /* reserved for future use */
    0x7D, /* reserved for future use */
    0x7E, /* reserved for future use */
    0x7F, /* reserved for future use */
    0x00  /* general call, also the last of this list */
};

/* ------------------------------------------------------------------------- *
 * PEC stuff
 * FIXME(miba): on 2.6 PEC handling can be replaced with kernel functions
 * ------------------------------------------------------------------------- */
#define crc8(x,crc) crc8table[(uint8_t)(crc ^ x)]
#define calc_crc8(data,len) update_crc8(data,len,0)

const static uint8_t crc8table[];
static uint8_t update_crc8 (uint8_t *data, size_t len, uint8_t crc);

/* ------------------------------------------------------------------------- *
 * Functions
 * ------------------------------------------------------------------------- */

static int  smbus_arp_mod_init(void);
static void smbus_arp_mod_cleanup(void);

static void cleanup_arp_devs(client_entry_t *entry);
static client_entry_t* find_client_from_adap(const struct i2c_adapter *adap);
static int find_free_address(client_entry_t *entry, uint8_t *address);

static int arp_cmd_prepare_arp(const client_entry_t *entry);
static int arp_cmd_get_udid(const client_entry_t *entry, arp_device_t *arp_dev);
static int arp_cmd_assign_address(const client_entry_t *entry, const smbus_udid_t *udid,
				  uint8_t address);
#ifdef UNUSED_FUNCTIONS
static int arp_cmd_reset_device(const client_entry_t *entry);
#endif

static void dump_udid(const smbus_udid_t *udid);

/* I2C low level routines */
static int smbus_arp_attach_adapter(struct i2c_adapter *adapter);
static int smbus_arp_detach_client(struct i2c_client *client);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void smbus_arp_inc_use (struct i2c_client *client);
static void smbus_arp_dec_use (struct i2c_client *client);
#endif

/* i2c read_write functions. Make sure the buffer has an additional
   byte room for the PEC */
static int i2c_access_write(struct i2c_adapter *adap, uint8_t slave_addr, char* buffer,
			    size_t size);
#ifdef UNUSED_FUNCTIONS 
static int i2c_access_read(struct i2c_adapter *adap, uint8_t slave_addr, char* buffer,
			   size_t size);
static int i2c_access_proc(struct i2c_adapter *adap, uint8_t slave_addr, char* buffer,
			   size_t out_size, size_t in_size);
#endif
static int i2c_read_block(struct i2c_adapter *adap, uint8_t slave_addr,
			  uint8_t command, uint8_t *byte_count, uint8_t *data);

static int i2c_write_block(struct i2c_adapter *adap, uint8_t slave_addr,
		           uint8_t command, uint8_t byte_count, int add_pec, const uint8_t * data);


static void dump (u_int8_t *buf, int size)
{
    int i, j;
    for (i=0; i<size; i+=16) {
        for (j=i; j<i+16; j++) {
            if (j<size) {
                printk ("%02x ", buf[j]);
            } else {
                printk ("   ");
            }
        }
        printk ("\n"); 
    }
}

EXPORT_SYMBOL(smbus_arp_trigger);
EXPORT_SYMBOL(smbus_arp_get_address);

/* ------------------------------------------------------------------------- *
 * I2C driver structure
 * ------------------------------------------------------------------------- */

static struct i2c_driver smbus_arp_i2c_driver = {
    name:		"SMBUS ARP",
    id:			0xFFD2,	/* f000-ffff for local use */
    flags:		I2C_DF_NOTIFY,
    attach_adapter:	smbus_arp_attach_adapter,
    detach_client:	smbus_arp_detach_client,
    command:		NULL,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    inc_use:		smbus_arp_inc_use,
    dec_use:		smbus_arp_dec_use,
#endif
};

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Michael Baumann <michael.baumann@raritan.com>");
MODULE_DESCRIPTION("Driver to handle to SMBUS ARP");

module_init(smbus_arp_mod_init);
module_exit(smbus_arp_mod_cleanup);

/* ------------------------------------------------------------------------- *
 * Initialization
 * ------------------------------------------------------------------------- */

static int
smbus_arp_mod_init(void)
{
    struct list_head *pos;
    int ret = SUCCESS, r;
    
    /* register i2c driver */
    if ((r = i2c_add_driver(&smbus_arp_i2c_driver))) {
	WARN("%s: i2c driver registration failed (%d).\n", DEVNAME, r);
	ret = -ENODEV;
	goto bail;
    }
    i2c_driver_initialized = 1;
    
    list_for_each(pos, &client_list) {
	client_entry_t* tmp_entry = list_entry(pos, client_entry_t, anchor);

	if (smbus_arp_trigger(tmp_entry->client->adapter) != SUCCESS) {
	    ERR("Error on I2C arp for adapter '%s'", tmp_entry->client->adapter->name);
	}
    }
	
 bail:
    return ret;
}

static void 
smbus_arp_mod_cleanup(void)
{
    /* delete the driver */
    if (i2c_driver_initialized == 1) {
	if (i2c_del_driver(&smbus_arp_i2c_driver)) {
	    WARN("%s: i2c driver deregistration failed.\n", DEVNAME);
	}
	i2c_driver_initialized = 0;;
    }
}

/* ------------------------------------------------------------------------- *
 * Exported functions
 * ------------------------------------------------------------------------- */

int
smbus_arp_trigger(const struct i2c_adapter *adap)
{
    client_entry_t* entry;
    int ret = -1, count = 0;

    if ((entry = find_client_from_adap(adap)) == NULL) {
	ERR("Could not find adapter '%s'", adap->name);
	goto bail;
    }

    /* new ARP run, cleanup old data */
    cleanup_arp_devs(entry);

   
    if (arp_cmd_prepare_arp(entry) != SUCCESS) {
	ERR("Could not issue prepare to ARP command");
	goto bail;
    }

    /* scan for devices and assign addresses from the pool */
    while (1) {
	arp_device_t *arp_dev;
	uint8_t new_address;

	if ((arp_dev = kmalloc(sizeof(arp_device_t), GFP_KERNEL)) == NULL) {
	    ERR("Could not allocate memory for arp device");
	}

	if (arp_cmd_get_udid(entry, arp_dev) != SUCCESS) {
	    /* we are done enumerating */
	    break;
	}

	DBG("Found device, vendor=0x%04x, device=0x%04x, address=0x%02x",
	    arp_dev->udid.vendor_id, arp_dev->udid.device_id, arp_dev->slave_addr >> 1);

	if (arp_dev->slave_addr != 0xFF) {
	    /* device already has an address */
	    uint8_t old_address = arp_dev->slave_addr >> 1;

	    if (arp_dev->udid.caps.address_type == SMBUS_UDID_ADDR_TYPE_FIXED) {
		DBG("   Address is fixed");
		/* existing fixed address, it will ignore our AssignAddress */
		new_address = arp_dev->slave_addr;
		entry->address_pool[new_address] = ARP_STATUS_BUSY;
	    } else {
		/* existing address, check if we need to reassign it */
		if (0 && entry->address_pool[old_address] == ARP_STATUS_FREE) {
		    entry->address_pool[old_address] = ARP_STATUS_BUSY;
		    new_address = old_address;
		} else {
		    /* existing address not free, find a new one */
		    if (find_free_address(entry, &new_address) != SUCCESS) {
			ERR("Unable to find a free I2C address");
			break;
		    }
		}
	    }
	} else {
	    if (find_free_address(entry, &new_address) != SUCCESS) {
		ERR("Unable to find a free I2C address");
		break;
	    }
	}
	
	DBG("Assigning address 0x%02x to vendor=0x%04x, device=0x%04x",
	    new_address, arp_dev->udid.vendor_id, arp_dev->udid.device_id);

	arp_dev->slave_addr = new_address;
	
	if (arp_cmd_assign_address(entry, &arp_dev->udid, arp_dev->slave_addr) != SUCCESS) {
	    WARN("Error assigning address 0x%02x to vendor=0x%04x, device=0x%04x",
		 new_address, arp_dev->udid.vendor_id, arp_dev->udid.device_id);
	}

	count++;
    }

    INFO("Nr. of SMBUS ARP devices on '%s': %d", adap->name, count);

    ret = SUCCESS;
    
 bail:
    return ret;
}

int
smbus_arp_get_address(const struct i2c_adapter *adap,
		      uint16_t vendor_id, uint16_t device_id)
{
    client_entry_t* entry;
    struct list_head *pos;
    int ret = -1;

    if ((entry = find_client_from_adap(adap)) == NULL) {
	ERR("Could not find adapter '%s'", adap->name);
	goto bail;
    }

    list_for_each(pos, &entry->arp_dev_list) {
	arp_device_t *arp_dev = (arp_device_t*) list_entry(pos, arp_device_t, anchor);

	if (arp_dev->udid.vendor_id == vendor_id &&
	    arp_dev->udid.device_id == device_id) {

	    ret = arp_dev->slave_addr;
	    DBG("Found I2C address 0x%02x for vendor 0x%04x, device 0x%04x",
		ret, vendor_id, device_id);

	    goto bail;
	}
    }

    DBG("I2C Address for vendor 0x%04x, device 0x%04x not found",
	vendor_id, device_id);
    
 bail:
    return ret;
}

/* ------------------------------------------------------------------------- *
 * Private functions
 * ------------------------------------------------------------------------- */

static void
cleanup_arp_devs(client_entry_t *entry)
{
    struct list_head *pos, *tmp;
    int i;
    
    list_for_each_safe(pos, tmp, &entry->arp_dev_list) {
	arp_device_t *arp_dev = list_entry(pos, arp_device_t, anchor);
	list_del(pos);

	kfree(arp_dev);
    }

    /* free address pool */
    memset(entry->address_pool, ARP_STATUS_FREE, sizeof(entry->address_pool));

    /* mark reserved addresses */
    for (i=0; i<sizeof(reserved_addresses); i++) {
	uint8_t address = reserved_addresses[i];
	if (address >= sizeof(entry->address_pool)) {
	    ERR("Reserved address 0x%02x exceeds maximum of 0x%02x",
		address, sizeof(entry->address_pool) - 1);
	    continue;
	}
	entry->address_pool[address] = ARP_STATUS_RESERVED;
    }    
}

static client_entry_t*
find_client_from_adap(const struct i2c_adapter *adap)
{
    client_entry_t* entry = NULL;
    struct list_head *pos;
    
    list_for_each(pos, &client_list) {
	client_entry_t* tmp_entry = (client_entry_t*) list_entry(pos, client_entry_t, anchor);

	if (tmp_entry->client->adapter == adap) {
	    entry = tmp_entry;
	    break;
	}	
    }

    return entry;
}

static int
find_free_address(client_entry_t *entry, uint8_t *address)
{
    int ret = ERROR, i;

    for (i=0; i<SMBUS_ADDRESS_SIZE; i++) {
	if (entry->address_pool[i] == ARP_STATUS_FREE) {
	    entry->address_pool[i] = ARP_STATUS_BUSY;
	    *address = i;
	    break;
	}
    }

    if (i == SMBUS_ADDRESS_SIZE) {
	goto bail;
    }
    
    ret = SUCCESS;

 bail:
    return ret;    
}
 
static int
arp_cmd_prepare_arp(const client_entry_t *entry)
{
    uint8_t buf[8] = { CMD_PREPARE_ARP };
    int ret;

    ret = i2c_access_write(entry->client->adapter, SMBUS_ARP_ADDR,
			   buf, 1);

    return (ret == 1) ? SUCCESS : ERROR;
}

static int arp_cmd_get_udid(const client_entry_t *entry,
			    arp_device_t *arp_dev)
{
    uint8_t data[I2C_MAX_DATA_SIZE], byte_count;
    smbus_udid_t *udid = &arp_dev->udid;
    smbus_udid_wire_t *udid_wire = (smbus_udid_wire_t*) data;
    int ret = ERROR;

    memset(&data, 0x00, sizeof(data));
    
    if (i2c_read_block(entry->client->adapter, SMBUS_ARP_ADDR,
		       CMD_GET_UDID, &byte_count, data) != 0) {
	DBG("arp_cmd_get_udid: i2c_read_block failed");
	goto bail;
    }

    /* sanity check */    
    if (byte_count != 17) {
	DBG("arp_cmd_get_udid: byte_count (%d) != 17", byte_count);
	goto bail;
    }

    /* copy UDID content */
    udid->caps = udid_wire->caps;
    udid->version = udid_wire->version;
    udid->vendor_id = le16_to_cpu(udid_wire->vendor_id_le16);
    udid->device_id = le16_to_cpu(udid_wire->device_id_le16);
    udid->interface_reserved = udid_wire->interface_reserved;
    udid->interface = udid_wire->interface;
    udid->subvendor_id = le16_to_cpu(udid_wire->subvendor_id_le16);
    udid->subdevice_id = le16_to_cpu(udid_wire->subdevice_id_le16);
    udid->specific_id = le32_to_cpu(udid_wire->specific_id_le32);

    arp_dev->slave_addr = data[16];

    dump(data, I2C_MAX_DATA_SIZE);
    
#ifdef LOG_DBG
    dump_udid(udid);
#endif

    ret = SUCCESS;
    
 bail:
    return ret;
}

static int
arp_cmd_assign_address(const client_entry_t *entry, const smbus_udid_t *udid,
		       uint8_t address)
{
    uint8_t data[I2C_MAX_DATA_SIZE];
    smbus_udid_wire_t *udid_wire = (smbus_udid_wire_t*) data;
    int ret = ERROR;

    memset(&data, 0x00, sizeof(data));
    
    /* copy UDID content */
    udid_wire->caps = udid->caps;
    udid_wire->version = udid->version;
    udid_wire->vendor_id_le16 = cpu_to_le16(udid->vendor_id);
    udid_wire->device_id_le16 = cpu_to_le16(udid->device_id);
    udid_wire->interface_reserved = udid->interface_reserved;
    udid_wire->interface = udid->interface;
    udid_wire->subvendor_id_le16 = cpu_to_le16(udid->subvendor_id);
    udid_wire->subdevice_id_le16 = cpu_to_le16(udid->subdevice_id);
    udid_wire->specific_id_le32 = cpu_to_le32(udid->specific_id);

    data[16] = address << 1;

    dump(data, I2C_MAX_DATA_SIZE);
    
    if (i2c_write_block(entry->client->adapter, SMBUS_ARP_ADDR,
			CMD_ASSIGN_ADDRESS, 17, ADD_PEC, data) != 0) {
	goto bail;
    }
    
    ret = SUCCESS;
    
 bail:
    return ret;
}

#ifdef UNUSED_FUNCTIONS
static int arp_cmd_reset_device(const client_entry_t *entry)
{
    uint8_t buf[8] = { CMD_RESET_DEVICE };
    int ret;

    ret = i2c_access_write(entry->client->adapter, SMBUS_ARP_ADDR,
			   buf, 1);

    return (ret == 1) ? SUCCESS : ERROR;
}
#endif

static void
dump_udid(const smbus_udid_t *udid)
{
    printk("UDID Caps:        Address Type=%d, PEC=%d\n"
	   "     Version:     UDID version=0x%02x, silicon=0x%02x\n"
	   "     Interface:   IPMI=%d, ASF=%d, OEM=%d, SMBus version=0x%01x\n"
	   "     VendorID:    0x%04x,    DeviceID 0x%04x\n"
	   "     SubVendorID: 0x%04x, SubDeviceID 0x%04x\n"
	   "     Specific ID: 0x%08x\n",
	   udid->caps.address_type, udid->caps.pec_supported,
	   udid->version.version, udid->version.silicon_rev,
	   udid->interface.ipmi, udid->interface.asf,
	   udid->interface.oem, udid->interface.smbus_version,
	   udid->vendor_id, udid->device_id,
	   udid->subvendor_id, udid->subdevice_id,
	   udid->specific_id);	   
}

/* ------------------------------------------------------------------------- *
 * I2C low level routines
 * ------------------------------------------------------------------------- */

static int
smbus_arp_attach_adapter(struct i2c_adapter *adapter)
{
    struct i2c_client* new_client = NULL;
    client_entry_t* new_client_entry = NULL;
    struct list_head *pos;
    int ret = 0;
    
    /* check if we already have a client for this adapter */
    list_for_each(pos, &client_list) {
	client_entry_t *entry = list_entry(pos, client_entry_t, anchor);

	if (entry->client->adapter == adapter) {
	    DBG("adapter '%s' already attached", adapter->name);
	    return 0;
	}	
    }

    /* create a client, attach and insert it */
    if (!(new_client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL))) {
    	ERR("Could not create client");
	ret = -ENOMEM;
    	goto bail;
    }
    if (!(new_client_entry = kmalloc(sizeof(client_entry_t), GFP_KERNEL))) {
    	ERR("Could not create client entry");
    	ret = -ENOMEM;
	goto bail_free;
    }
    memset(new_client, 0, sizeof(struct i2c_client));
    memset(new_client_entry, 0, sizeof(struct i2c_client));

    new_client->adapter = adapter;
    new_client->driver	= &smbus_arp_i2c_driver;
    new_client->flags	= 0;
    new_client->data	= new_client_entry;
    snprintf(new_client->name, sizeof(new_client->name), "CLIENT @ %s", adapter->name);
   
    if ((ret = i2c_attach_client_force(new_client))) {
    	goto bail_free;
    }    

    INIT_LIST_HEAD(&new_client_entry->arp_dev_list);
    INIT_LIST_HEAD(&new_client_entry->anchor);
    new_client_entry->client = new_client;

    list_add_tail(&new_client_entry->anchor, &client_list);

    DBG("i2c_attach_client '%s'", new_client->name);
    return ret;
    
 bail_free:
    kfree(new_client_entry);
    kfree(new_client);    
    
 bail:
    return ret;
}

static int
smbus_arp_detach_client(struct i2c_client *client)
{
    client_entry_t *client_entry = (client_entry_t*) client->data;
    int ret = 0;

    DBG("i2c_detach_client '%s'", client->name);

    if ((ret = i2c_detach_client(client_entry->client))) {
    	WARN("i2c client detach failed");
	goto remove;
    }
    
    kfree(client);

 remove:
    list_del(&client_entry->anchor);
    kfree(client_entry);
      
    return ret;   
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void
smbus_arp_inc_use (struct i2c_client *client)
{
}

static void
smbus_arp_dec_use (struct i2c_client *client)
{
}
#endif

static inline int
i2c_access_transfer(struct i2c_adapter * adapter, struct i2c_msg * msg, int num)
{
    int retries = 0;
    int ret;
    
    if (!adapter) {
        return -ENODEV;
    }
    
    do {
        ret = i2c_transfer(adapter, msg, num);
        if (ret > 0) return ret;
        DBG ("i2c_transfer failed, retry # %d", retries + 1);
        DELAY();
    } while (retries++ < MASTER_RETRIES);
    
    return ret;
}

static int
i2c_access_write(struct i2c_adapter *adap, uint8_t slave_addr,
		 char* buffer, size_t size)
{
    struct i2c_msg msg;
    int ret;
    
    memset(&msg, 0, sizeof(struct i2c_msg));

    buffer[size] = update_crc8(buffer, size, crc8table[slave_addr << 1]);
    
    msg.addr  = slave_addr;
    msg.buf   = buffer;
    msg.len   = size + 1;
    
    ret = i2c_access_transfer(adap, &msg, 1);

    if (ret < 1) return ret;    
    return size;
}

#ifdef UNUSED_FUNCTIONS
static int
i2c_access_read(struct i2c_adapter *adap, uint8_t slave_addr,
		char* buffer, size_t size)
{
    struct i2c_msg msg;
    int ret;
    
    memset(&msg, 0, sizeof(struct i2c_msg));
    
    msg.addr  = slave_addr;
    msg.buf   = buffer;
    msg.flags = I2C_M_RD;
    msg.len   = size;
    
    ret = i2c_access_transfer(adap, &msg, 1);

    /* FIXME: check PEC on reads */
        
    if (ret < 1) return ret;    
    return size;
}

static int
i2c_access_proc(struct i2c_adapter *adap, uint8_t slave_addr,
		char* buffer, size_t out_size, size_t in_size)
{
    struct i2c_msg msg[2];
    int ret;
    
    memset(&msg, 0, sizeof(struct i2c_msg));

    msg[0].addr  = slave_addr;
    msg[0].buf   = buffer;
    msg[0].len   = out_size;
        
    msg[1].addr  = slave_addr;
    msg[1].buf   = buffer;
    msg[1].flags = I2C_M_RD;
    msg[1].len   = in_size;
    
    ret = i2c_access_transfer(adap, msg, 2);

    if (ret < 1) return ret;
    return in_size;
}
#endif

/* block read/write */

/* NOTE: the i2c_smbus_... functions of the linux kernel only supports
 * messages up to 32 bytes length, block reads are not supported at all.
 * So we have to implement a SMBus emulation via I2C function calls here.
 */
static int i2c_write_block(struct i2c_adapter *adap,
			   uint8_t slave_addr,
		           uint8_t command,
		           uint8_t byte_count,
			   int add_pec,
		           const uint8_t * data)
{
    int ret = 0;

    /* sanity check */
    if (byte_count > I2C_MAX_DATA_SIZE) {
        WARN("i2c_write_block: byte count of %d exceeds limit of %d.",
             byte_count, I2C_MAX_DATA_SIZE);
        ret = -EINVAL;
        goto bail;
    }

    /* construct an I2C message
       byte 1: command
       byte 2: byte count
       byte 3 .. byte n: data */
    uint8_t msg_buf[2 + I2C_MAX_DATA_SIZE];
    struct i2c_msg msg = {
        addr:   slave_addr,
        flags:  0,
        len:    (add_pec == ADD_PEC) ? byte_count + 3 : byte_count + 2,
        buf:    msg_buf,
    };

    /* fill the I2C message */
    msg_buf[0] = command;
    msg_buf[1] = byte_count;
    memcpy(msg_buf + 2, data, byte_count);

    /* add packet error checking code */
    if (add_pec == ADD_PEC) {
	msg_buf[byte_count + 2] = update_crc8(msg_buf, byte_count + 2, crc8table[SMBUS_ARP_ADDR << 1]);
    }
    
    /* transfer the I2C message */
    int retry = MAX_RETRY_COUNT;
    do {
        ret = i2c_transfer(adap, &msg, 1);
        DELAY();
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during I2C transfer: %d", ret);
        else WARN("Error during I2C transfer: %d", ret);
    } while (retry-- > 0);
    if (ret < 0) goto bail;

bail:
    return ret;
}

static int i2c_read_block(struct i2c_adapter *adap,
			  uint8_t slave_addr,
			  uint8_t command,
		          uint8_t *byte_count,
		          uint8_t *data)
{
    int ret = 0;

    /* construct 2 I2C messages:
       message 1: write command byte
       message 2: read data bytes */    
    uint8_t msg_buf[1 + I2C_MAX_DATA_SIZE];
    struct i2c_msg msg[2] = {
        { slave_addr, 0, 1, &command },
        { slave_addr, I2C_M_RD, 1 + I2C_MAX_DATA_SIZE, msg_buf },
    };

    /* transfer the I2C messages */
    int retry = MAX_RETRY_COUNT;
    do {
        ret = i2c_transfer(adap, msg, 2);
        DELAY();
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during I2C transfer: %d", ret);
        else WARN("Error during I2C transfer: %d", ret);
    } while (retry-- > 0);
    if (ret < 0) goto bail;

    /* fill out return values;
       byte 1: byte count
       byte 2 .. byte n: data */
    *byte_count = msg_buf[0];

    /* sanity check */
    if (*byte_count > I2C_MAX_DATA_SIZE) {
        WARN("i2c_read_block: byte count of %d exceeds limit of %d.",
             *byte_count, I2C_MAX_DATA_SIZE);
        ret = -EINVAL;
        goto bail;
    }

    memcpy(data, msg_buf + 1, *byte_count);

bail:
    return ret;
}

			    
/* ------------------------------------------------------------------------- *
 * Calculate CRC-8 x^8+x^2+x^+1
 * ------------------------------------------------------------------------- */

const static uint8_t crc8table[] =
{
    0x00, 0x07, 0x0E, 0x09, 0x1C, 0x1B, 0x12, 0x15,
    0x38, 0x3F, 0x36, 0x31, 0x24, 0x23, 0x2A, 0x2D,
    0x70, 0x77, 0x7E, 0x79, 0x6C, 0x6B, 0x62, 0x65,
    0x48, 0x4F, 0x46, 0x41, 0x54, 0x53, 0x5A, 0x5D,
    0xE0, 0xE7, 0xEE, 0xE9, 0xFC, 0xFB, 0xF2, 0xF5,
    0xD8, 0xDF, 0xD6, 0xD1, 0xC4, 0xC3, 0xCA, 0xCD,
    0x90, 0x97, 0x9E, 0x99, 0x8C, 0x8B, 0x82, 0x85,
    0xA8, 0xAF, 0xA6, 0xA1, 0xB4, 0xB3, 0xBA, 0xBD,
    0xC7, 0xC0, 0xC9, 0xCE, 0xDB, 0xDC, 0xD5, 0xD2,
    0xFF, 0xF8, 0xF1, 0xF6, 0xE3, 0xE4, 0xED, 0xEA,
    0xB7, 0xB0, 0xB9, 0xBE, 0xAB, 0xAC, 0xA5, 0xA2,
    0x8F, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9D, 0x9A,
    0x27, 0x20, 0x29, 0x2E, 0x3B, 0x3C, 0x35, 0x32,
    0x1F, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0D, 0x0A,
    0x57, 0x50, 0x59, 0x5E, 0x4B, 0x4C, 0x45, 0x42,
    0x6F, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7D, 0x7A,
    0x89, 0x8E, 0x87, 0x80, 0x95, 0x92, 0x9B, 0x9C,
    0xB1, 0xB6, 0xBF, 0xB8, 0xAD, 0xAA, 0xA3, 0xA4,
    0xF9, 0xFE, 0xF7, 0xF0, 0xE5, 0xE2, 0xEB, 0xEC,
    0xC1, 0xC6, 0xCF, 0xC8, 0xDD, 0xDA, 0xD3, 0xD4,
    0x69, 0x6E, 0x67, 0x60, 0x75, 0x72, 0x7B, 0x7C,
    0x51, 0x56, 0x5F, 0x58, 0x4D, 0x4A, 0x43, 0x44,
    0x19, 0x1E, 0x17, 0x10, 0x05, 0x02, 0x0B, 0x0C,
    0x21, 0x26, 0x2F, 0x28, 0x3D, 0x3A, 0x33, 0x34,
    0x4E, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5C, 0x5B,
    0x76, 0x71, 0x78, 0x7F, 0x6A, 0x6D, 0x64, 0x63,
    0x3E, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2C, 0x2B,
    0x06, 0x01, 0x08, 0x0F, 0x1A, 0x1D, 0x14, 0x13,
    0xAE, 0xA9, 0xA0, 0xA7, 0xB2, 0xB5, 0xBC, 0xBB,
    0x96, 0x91, 0x98, 0x9F, 0x8A, 0x8D, 0x84, 0x83,
    0xDE, 0xD9, 0xD0, 0xD7, 0xC2, 0xC5, 0xCC, 0xCB,
    0xE6, 0xE1, 0xE8, 0xEF, 0xFA, 0xFD, 0xF4, 0xF3
};

static uint8_t 
update_crc8 (uint8_t *data, size_t len, uint8_t crc)
{
    int   i;

    for (i = 0; i < len; i++)
    {
        crc = crc8 (data[i], crc);
    }

    return crc;
}
