/**
 * lpc.h
 *
 * KIRA100 LPC (slave) device IOCTLs
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#ifndef __LPC_H__
#define __LPC_H__

#define LPC_MODE_KCS    1
#define LPC_MODE_SMIC   2
#define LPC_MODE_BT     3

#define LPC_MODE_SNOOP  10

#define LPC_DEFAULT_KCS_HOST_PORT_BASE  0xca2 /* ...0xca3 */
#define LPC_DEFAULT_SMIC_HOST_PORT_BASE 0xca9 /* ...0xcab */
#define LPC_DEFAULT_BT_HOST_PORT_BASE   0x0e4 /* ...0x0e6 */

typedef struct {
    int mode;
    unsigned char chan; /* ignored for snoop mode */
    unsigned short host_port;
} lpc_ioctl_init_t;

typedef struct {
    /* irq counting */
    int intr_cnt;
    int lpc_intr_cnt;
} lpc_ioctl_hk_t;

typedef struct {
    int reg;
    unsigned char val;
} lpc_ioctl_reg_t;

typedef struct {
    unsigned int addr;
    unsigned char val;
} lpc_ioctl_mem_t;

typedef struct {
    unsigned short port; /* 0: off */
    unsigned char irq; /* 0: no irq */
} lpc_ioctl_uart_cfg_t;

#define LPC_IOCTL_MAGIC 'L'
#define LPC_IOCTL_INIT          _IOW (LPC_IOCTL_MAGIC,  1, lpc_ioctl_init_t)
#define LPC_IOCTL_SET_HOST_ATN  _IO  (LPC_IOCTL_MAGIC,  2)
#define LPC_IOCTL_CLR_HOST_ATN  _IO  (LPC_IOCTL_MAGIC,  3)
#define LPC_IOCTL_MEM_READ      _IOWR(LPC_IOCTL_MAGIC, 10, lpc_ioctl_mem_t)
#define LPC_IOCTL_MEM_WRITE     _IOW (LPC_IOCTL_MAGIC, 11, lpc_ioctl_mem_t)
#define LPC_IOCTL_SET_UART_CFG  _IOW (LPC_IOCTL_MAGIC, 20, lpc_ioctl_uart_cfg_t)
#define LPC_IOCTL_GET_REG       _IOWR(LPC_IOCTL_MAGIC, 90, lpc_ioctl_reg_t)
#define LPC_IOCTL_SET_REG       _IOW (LPC_IOCTL_MAGIC, 91, lpc_ioctl_reg_t)
#define LPC_IOCTL_LOCK_NET      _IOW (LPC_IOCTL_MAGIC, 92, int)
#define LPC_IOCTL_GET_HK        _IOR (LPC_IOCTL_MAGIC, 99, lpc_ioctl_hk_t)

/*
 * Helper function to cover IOCTL (only in user mode)
 */
#ifndef __KERNEL__
#include <sys/ioctl.h>

static __inline__ int lpc_ioctl_init(int fd, int mode, unsigned char chan, unsigned short host_port)
{
    lpc_ioctl_init_t ioc = { .mode = mode, .chan = chan, .host_port = host_port, };
    return ioctl(fd, LPC_IOCTL_INIT, &ioc);
}

static __inline__ int lpc_ioctl_set_host_atn(int fd)
{
    return ioctl(fd, LPC_IOCTL_SET_HOST_ATN, NULL);
}

static __inline__ int lpc_ioctl_clr_host_atn(int fd)
{
    return ioctl(fd, LPC_IOCTL_CLR_HOST_ATN, NULL);
}

static __inline__ int lpc_ioctl_mem_read(int fd, unsigned int addr)
{
    lpc_ioctl_mem_t ioc = { .addr = addr, };
    int r = ioctl(fd, LPC_IOCTL_MEM_READ, &ioc);
    return r < 0 ? r : ioc.val;
}

static __inline__ int lpc_ioctl_mem_write(int fd, unsigned int addr, unsigned char val)
{
    lpc_ioctl_mem_t ioc = { .addr = addr, .val = val, };
    return ioctl(fd, LPC_IOCTL_MEM_WRITE, &ioc);
}

static __inline__ int lpc_ioctl_set_uart_cfg(int fd, unsigned short port, unsigned char irq)
{
    lpc_ioctl_uart_cfg_t ioc = { .port = port, .irq = irq, };
    return ioctl(fd, LPC_IOCTL_SET_UART_CFG, &ioc);
}

static __inline__ int lpc_ioctl_set_reg(int fd, int reg, unsigned char val)
{
    lpc_ioctl_reg_t ioc = { .reg = reg, .val = val, };
    return ioctl(fd, LPC_IOCTL_SET_REG, &ioc);
}

static __inline__ int lpc_ioctl_get_reg(int fd, int reg)
{
    lpc_ioctl_reg_t ioc = { .reg = reg, };
    int r = ioctl(fd, LPC_IOCTL_GET_REG, &ioc);
    return r < 0 ? r : ioc.val;
}

static __inline__ int lpc_ioctl_lock_net(int fd, int lock)
{
    return ioctl(fd, LPC_IOCTL_LOCK_NET, &lock);
}

#endif /* !__KERNEL__ */

#endif /* !__LPC_H__ */
