/**
 * lpc_snoop.c
 *
 * KIRA100 LPC (slave) driver for host io port snooping
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>

#include "lpc_core.h"
#include "lpc_snoop.h"

#define SUCCESS 0

/*
 * Global data
 */

typedef struct {
    lpc_t       *lpc;
    uint16_t    port;
} snoop_t;

static snoop_t g_snoop;

/*
 * Public functions
 */

#define snoop_data_avail()  (!(lpc_get_reg(snoop->lpc, LPC_HOST_STATUS) & LPC_SNOOP_EMPTY))
#define snoop_get_data()    (lpc_get_reg(snoop->lpc, LPC_SNOOP_FIFO))
#define snoop_set_addr(a) { \
    lpc_set_reg(snoop->lpc, LPC_SNOOP_ADDRH, ((a) >> 8) & 0xff); \
    lpc_set_reg(snoop->lpc, LPC_SNOOP_ADDRL, ((a) >> 0) & 0xff); \
}


int snoop_init(lpc_t *lpc, uint16_t port, void **psnoop_data)
{
    snoop_t *snoop = &g_snoop;

    if (snoop->lpc) return -EBUSY; // can only be used exclusively

    snoop->lpc = lpc;
    snoop->port = port;
    *psnoop_data = snoop;

    /* pre-empty FIFO... */
    while (snoop_data_avail()) snoop_get_data();

    /* program snoop port */
    snoop_set_addr(port);

    return SUCCESS;
}

int snoop_cleanup(void *snoop_data)
{
    snoop_t *snoop = snoop_data;
    snoop_set_addr(0);
    snoop->lpc = NULL;
    return SUCCESS;
}

int snoop_read(void *snoop_data, uint8_t *buf, int size)
{
    snoop_t *snoop = snoop_data;
    int i;
    for (i = 0; i < size && snoop_data_avail(); i++) buf[i] = snoop_get_data();
    return i;
}

int snoop_can_read(void *snoop_data)
{
    snoop_t *snoop = snoop_data;
    return snoop_data_avail();
}
