#include <linux/if_ether.h>
#include <linux/ip.h>

#include "mac_cache.h"

#define ENTRIES 10

typedef struct {
    uint32_t ip;
    uint8_t mac[ETH_ALEN];
    int last_used;
} mac_cache_t;

mac_cache_t mac_cache[ENTRIES];

/*
 * Add source MAC/IP combination from skb to MAC cache; overwrite
 * oldest existing entry in case cache is full.
 */
void mac_cache_add(struct sk_buff *skb)
{
    struct ethhdr *eth = (struct ethhdr *)skb->data;
    struct iphdr *ip = (struct iphdr *)(skb->data + ETH_HLEN);
    int i;
    mac_cache_t *oldest = mac_cache;

    if (eth->h_proto != __constant_htons(ETH_P_IP)) return;

    for (i=0; i<ENTRIES; i++) {
	if (mac_cache[i].ip == ip->saddr) {
	    // existing entry for this IP; update
	    oldest = &mac_cache[i];
	    break;
	}
	if (oldest->ip) {
	    if (!mac_cache[i].ip || mac_cache[i].last_used < oldest->last_used) {
		oldest = &mac_cache[i];
	    }
	}
    }

    oldest->ip = ip->saddr;
    memcpy(oldest->mac, eth->h_source, ETH_ALEN);
    oldest->last_used = jiffies;
}

/*
 * Lookup destination IP in cache and update the skb's destination MAC.
 */
int mac_cache_lookup(struct sk_buff *skb)
{
    struct ethhdr *eth = (struct ethhdr *)skb->data;
    struct iphdr *ip = (struct iphdr *)(skb->data + ETH_HLEN);
    int i;

    if (eth->h_proto != __constant_htons(ETH_P_IP)) return -1;

    for (i=0; i<ENTRIES; i++) {
	if (mac_cache[i].ip == ip->daddr) {
	    memcpy(eth->h_dest, mac_cache[i].mac, ETH_ALEN);
	    mac_cache[i].last_used = jiffies;
	    return 0;
	}
    }

    return -1;
}

