/***************************************************************************** 
 * Linux kernel module to access the erla RTC EEPROM directly
 *
 * Author: Michael Baumann (miba@peppercon.de)
 *****************************************************************************/
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/pci.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <linux/iobuf.h>
#endif
#include <linux/types.h>
#include <linux/ioctl.h>
#include <asm/fcntl.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/semaphore.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#ifdef __powerpc__
# include <asm/ppc4xx_i2c.h>
#endif
#endif
#include "rtc_eeprom.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

/* define this to compile the read_ccr function */
#undef USE_CCR_READ

#define RTC_EEP_MAJOR			((u8)243)
#define RTC_EEP_MAX_SIZE		512
#define RTC_CCR_I2C_ADDR		0x6F
#define RTC_EEP_I2C_ADDR		0x57
#define RTC_WRITE_RETRY_COUNT		3
#define RTC_WRITE_CYCLE_WAIT		5
#define RTC_CONTROL_REG			0x10
#define RTC_STATUS_REG			0x3F

#ifdef SUCCESS
#  undef  SUCCESS
#endif
#define SUCCESS 0

#ifdef DEBUG
# define D(fmt, args...)        { printk(fmt, ##args); }
#else
# define D(fmt, args...)        { }
#endif

/* ------------------------------------------------------------------------- *
 * global variables
 * ------------------------------------------------------------------------- */

typedef struct {
    char			name[15];
    char			i2c_initialized;   
    long long			size;
    unsigned char		buffer[RTC_EEP_MAX_SIZE];
    struct semaphore		sem;
    unsigned char		eep_unlocked;
} rtc_eep_dev_t;

static rtc_eep_dev_t dev = {
    name:			"rtc_eep",
    size:			RTC_EEP_MAX_SIZE,
    i2c_initialized:		0,
    sem:			{},
    eep_unlocked:		0
};

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

#ifdef MODULE
int		init_module(void);
void		cleanup_module(void);
#endif

int		rtc_eep_init(void);
static int	rtc_eep_cleanup(void);

static int	rtc_eep_ioctl(struct inode *, struct file *, uint, ulong);
static int	rtc_eep_open(struct inode *, struct file *);
static int	rtc_eep_release(struct inode *, struct file *);
static int	rtc_eep_read(struct file *, char *, size_t, loff_t *);
static int	rtc_eep_write(struct file *, const char *, size_t, loff_t *);
static loff_t	rtc_eep_llseek(struct file *, loff_t, int);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void rtc_eep_i2c_inc_use (struct i2c_client *client);
static void rtc_eep_i2c_dec_use (struct i2c_client *client);
#endif
static int  rtc_eep_i2c_command(struct i2c_client *client, unsigned int cmd, void *arg);
static int  rtc_eep_i2c_attach_adapter(struct i2c_adapter *adapter);
static int  rtc_eep_i2c_detach_client(struct i2c_client *client);
static int  rtc_eep_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
				      unsigned short flags,
#endif
				      int kind);

static int  ccr_byte_write(int addr, unsigned char byte);
static int  eep_byte_write(int addr, unsigned char byte);
#ifdef USE_CCR_READ
static int  ccr_read(int addr, unsigned char* buf, int length);
#endif
static int  eep_read(int addr, unsigned char* buf, int length);


/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

#ifdef MODULE
MODULE_AUTHOR("miba@peppercon.de");
MODULE_DESCRIPTION("Linux kernel module for accessing the RTC EEPROM");
MODULE_SUPPORTED_DEVICE("/dev/rtc_eep");
MODULE_LICENSE("GPL");

/*
 * Initialize the module
 */
int init_module(void)
{
    return rtc_eep_init();
}

/*
 * Cleanup - unregister the appropriate file from /proc
 */
void cleanup_module(void)
{
    int r;

    if ((r = unregister_chrdev(RTC_EEP_MAJOR, dev.name)) < 0) {
	printk("%s: failed to unregister device (%d)\n", dev.name, r);
    }
    
    if (rtc_eep_cleanup()) {
	printk("%s: failed to cleanup device\n", dev.name);
    }    
}
#endif	/* MODULE */

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations rtc_eep_ops = {
    owner:   THIS_MODULE,
    ioctl:   rtc_eep_ioctl,
    open:    rtc_eep_open,
    release: rtc_eep_release,
    read:    rtc_eep_read,
    write:   rtc_eep_write,
    llseek:  rtc_eep_llseek
};

/* ------------------------------------------------------------------------- *
 * i2c client stuff
 * ------------------------------------------------------------------------- */

static struct i2c_driver rtc_eep_driver={
    name:               "rtc_eep",
    id:                 0xFFF0,                 /* f000-ffff for local use */
    flags:              I2C_DF_NOTIFY,
    attach_adapter:     rtc_eep_i2c_attach_adapter,
    detach_client:      rtc_eep_i2c_detach_client,
    command:            rtc_eep_i2c_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    inc_use:            rtc_eep_i2c_inc_use,
    dec_use:            rtc_eep_i2c_dec_use
#endif
};

typedef struct {
    struct i2c_client*	client;
} rtc_eep_data_t;

static unsigned short normal_i2c[] = { RTC_CCR_I2C_ADDR, I2C_CLIENT_END };
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static unsigned short normal_i2c_range[] = { RTC_CCR_I2C_ADDR, RTC_CCR_I2C_ADDR, I2C_CLIENT_END };
#endif
I2C_CLIENT_INSMOD;

static rtc_eep_data_t* gdata = NULL;

/* ------------------------------------------------------------------------- *
 * driver initialization
 * ------------------------------------------------------------------------- */

int __init
rtc_eep_init(void)
{
    int rc=SUCCESS, r;
  
    /* ---- register the character device ------------------------------ */

    if ((r = register_chrdev(RTC_EEP_MAJOR, dev.name, &rtc_eep_ops)) < 0) {
	printk("%s: failed to register device (%d)\n", dev.name, r);
	rc = -ENODEV;
	goto fail;
    }
  
    /* register i2c driver */
    if ((r = i2c_add_driver(&rtc_eep_driver))) {
	printk("%s: i2c registration failed (%d).\n", dev.name, r);
	rc = -ENODEV;
	rtc_eep_cleanup();
        goto fail;
    }
    dev.i2c_initialized++;

    if (gdata == NULL) {
	printk("%s: no client detected.\n", dev.name);
	rc = -ENODEV;
	rtc_eep_cleanup();
	goto fail;
    }
    
    init_MUTEX(&dev.sem);
    printk("%s: kernel module loaded successfully!\n", dev.name);

    return SUCCESS;
    
 fail:
    rtc_eep_cleanup();
    printk("%s: kernel module didn't load successfully %d!\n", dev.name,rc);
    return rc;
}

static int
rtc_eep_cleanup(void)
{
    int ret;
    
    if (dev.i2c_initialized == 1) {
	if ((ret = i2c_del_driver(&rtc_eep_driver))) {
	    printk("%s: Driver registration failed, module not removed.\n", dev.name);
	    return ret;
	}
	dev.i2c_initialized--;
    }

    return SUCCESS;
}

/* ------------------------------------------------------------------------- *
 * the driver operations
 * ------------------------------------------------------------------------- */

static int
rtc_eep_ioctl(struct inode * inode, struct file * file, uint cmd, ulong arg)
{
    switch (cmd) {
      default:
	  return -EINVAL;
    }
}

static int
rtc_eep_open(struct inode * inode, struct file * file)
{   
    if (down_interruptible(&dev.sem)) return -ERESTARTSYS;

    if (!dev.eep_unlocked) {
	D("RTC EEP unlock\n");
	
	/* unlock Register write access */
	ccr_byte_write(RTC_STATUS_REG, 0x02); /* WEL  */
	ccr_byte_write(RTC_STATUS_REG, 0x06); /* RWEL */
	
	/* unlock EEPROM array */
	ccr_byte_write(RTC_CONTROL_REG, 0x00);
	mdelay(10);

	dev.eep_unlocked = 1;
    }    

    up(&dev.sem);
    return SUCCESS;
}

static int
rtc_eep_release(struct inode * inode, struct file * file)
{
    if (down_interruptible(&dev.sem)) return -ERESTARTSYS;

    if (dev.eep_unlocked) {
	D("RTC EEP lock\n");
    
	/* unlock Register write access */
	ccr_byte_write(RTC_STATUS_REG, 0x02); /* WEL  */
	ccr_byte_write(RTC_STATUS_REG, 0x06); /* RWEL */
	
	/* lock EEPROM array */
	ccr_byte_write(RTC_CONTROL_REG, 0xE0);
	mdelay(10);

	dev.eep_unlocked = 0;
    }

    up(&dev.sem);
    return SUCCESS;
}

static int
rtc_eep_read(struct file* file, char* buffer, size_t size, loff_t* offset)
{
    int count, left, ret;
    long long ofs;
    long dbgofs;
    char* buf = dev.buffer;

    if (file->f_flags & O_NONBLOCK) {
	if (down_trylock(&dev.sem)) return -EAGAIN;
    } else {
	if (down_interruptible(&dev.sem)) return -ERESTARTSYS;
    }

    ofs = *offset; dbgofs = *offset & 0xFFFFFFFF;
    D("RTC EEP read ofs %ld, size %d\n", dbgofs, size);
	
    /* handle EOF */
    if (*offset >= dev.size-1) {
	ret = 0;
	goto cleanup;
    }

    left = count = (*offset + size) > dev.size ? dev.size - *offset : size;
    
    while (left > 0) {
	if (eep_read(ofs, buf, 1)) {
	    printk("error ofs %ld\n", dbgofs);
	    break;
	}
	schedule();
	ofs++;dbgofs++;buf++;left--;
    }
      
    if (left) {
	ret = -EIO;
	goto cleanup;
    }

    if (copy_to_user(buffer, dev.buffer, count)) {
	ret = -EFAULT;
	goto cleanup;
    }
		     
    *offset += count;
    ret = count;
    
 cleanup:
    up(&dev.sem);
    return ret;
}

static int
rtc_eep_write(struct file* file, const char* buffer, size_t size, loff_t* offset)
{
    int count, left, ret = 0;
    int retry = RTC_WRITE_RETRY_COUNT;
    long long ofs;
    long dbgofs;
    char* buf;

    if (file->f_flags & O_NONBLOCK) {
	if (down_trylock(&dev.sem)) return -EAGAIN;
    } else {
	if (down_interruptible(&dev.sem)) return -ERESTARTSYS;
    }

    ofs = *offset; dbgofs = *offset & 0xFFFFFFFF;
    D("RTC EEP write ofs %ld, size %d\n", dbgofs, size);

    /* handle EOF */
    if (*offset >= dev.size-1) {
	ret = 0;
	goto cleanup;
    }

    left = count = (*offset + size) > dev.size ? dev.size - *offset : size;
    
    if (copy_from_user(dev.buffer, buffer, count)) {
	ret = -EFAULT;
	goto cleanup;
    }

    buf = dev.buffer;
    while (left > 0) {
	while (retry-- > 0 && eep_byte_write(ofs, *buf)) {
	    printk("error ofs %ld, retry %d\n", dbgofs, RTC_WRITE_RETRY_COUNT - retry);
	    /* wait write cycle time */
	    schedule();
	    mdelay(RTC_WRITE_CYCLE_WAIT);
	}
	if (retry == 0) {
	    break;
	} else {
	    mdelay(RTC_WRITE_CYCLE_WAIT);
	    retry = RTC_WRITE_RETRY_COUNT;
	}
	ofs++;dbgofs++;buf++;left--;
    }

    if (left) {
	ret = -EIO;
	goto cleanup;
    }
    
    *offset += count;
    ret = count;

 cleanup:
    up(&dev.sem);
    return ret;
}

static loff_t
rtc_eep_llseek(struct file * file, loff_t offset, int whence)
{
    loff_t newpos;

    switch (whence) {
      case 0: /* SEEK_SET */
	  newpos = offset;
	  break;
      case 1: /* SEEK_CUR */
	  newpos = file->f_pos + offset;
	  break;
      case 2: /* SEEK_END */
	  newpos = dev.size - offset;
	  break;	  
      default:
	  return -EINVAL;
    }

    if (newpos < 0) return -EINVAL;
    file->f_pos = newpos;

    return newpos;
}

/* ---------------------------- i2c functions ------------------------ */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void
rtc_eep_i2c_inc_use (struct i2c_client *client)
{
}

static void
rtc_eep_i2c_dec_use (struct i2c_client *client)
{
}
#endif

static int
rtc_eep_i2c_command(struct i2c_client *client,
		    unsigned int cmd, void *arg)
{
    /* no commands defined */
    return 0;
}

static int
rtc_eep_i2c_attach_adapter(struct i2c_adapter *adapter)
{
    D("i2c adapter attached, calling probe: %s\n", adapter->name);
#ifdef __powerpc__
    return i2c_probe_force(adapter, &addr_data, &rtc_eep_i2c_detect_client);
#else
    return i2c_probe(adapter, &addr_data, &rtc_eep_i2c_detect_client);
#endif
}

static int
rtc_eep_i2c_detach_client(struct i2c_client *client)
{
    int err;

    if ((err = i2c_detach_client(client))) {
	printk("i2c client detach failed\n");
	goto fail;
    }

    kfree(client);
    
    return SUCCESS;

 fail:
    return err;
}

static int
rtc_eep_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			  unsigned short flags,
#endif
			  int kind)
{
    struct i2c_client* new_client = NULL;
    rtc_eep_data_t* data;
    int err=0;

    D("i2c detect client addr %04x, flags %x, kind %d\n", address, flags, kind);
    
    if (!(new_client = kmalloc(sizeof(struct i2c_client)+sizeof(rtc_eep_data_t),
			       GFP_KERNEL))) {
	err = -ENOMEM;
	goto fail;
    }

    gdata = data = (rtc_eep_data_t*) (new_client + 1);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    (void) flags; /* keep compiler silent */
    strcpy(new_client->name, rtc_eep_driver.name);
    new_client->data = data;
    new_client->id  = 0; /* ??? */
#else
    strlcpy(new_client->name, rtc_eep_driver.name, I2C_NAME_SIZE);
    i2c_set_clientdata(new_client, data);
#endif
    new_client->addr	= address;
    new_client->adapter = adapter;
    new_client->driver  = &rtc_eep_driver;
    new_client->flags   = 0;

    if ((err = i2c_attach_client(new_client))) {
	goto fail_free;
    }    

    data->client = new_client;
    
    return SUCCESS;

 fail_free:    
    if (new_client) { kfree(new_client); new_client = NULL; }
    
 fail:
	return err;    
}

#ifdef USE_CCR_READ
static int
ccr_read(int addr, unsigned char* buf, int length)
{
#define NUM_MSGS 2
    struct i2c_msg msg[NUM_MSGS];
    unsigned char addr_msg[] = { (addr & 0xFF00) >> 8, addr & 0x00FF };
    int ret;

    memset(msg, 0, NUM_MSGS * sizeof(*msg));

    msg[0].addr = RTC_CCR_I2C_ADDR;
    msg[0].buf  = addr_msg;
    msg[0].len	= 2;

    msg[1].addr = RTC_CCR_I2C_ADDR;
    msg[1].buf  = buf;
    msg[1].flags= I2C_M_RD;
    msg[1].len  = length;

    D("Reading %d bytes from %04x for CCR...", length, addr);
    ret = i2c_transfer(gdata->client->adapter, msg, NUM_MSGS);
    D("ret %d\n", ret);

    return (ret == NUM_MSGS ? 0 : -1);
#undef NUM_MSGS
}
#endif

static int
eep_read(int addr, unsigned char* buf, int length)
{
#define NUM_MSGS 2
    struct i2c_msg msg[NUM_MSGS];
    unsigned char addr_msg[] = { (addr & 0xFF00) >> 8, addr & 0x00FF };
    int ret;

    memset(msg, 0, 2*sizeof(struct i2c_msg));

    msg[0].addr = RTC_EEP_I2C_ADDR;
    msg[0].buf  = addr_msg;
    msg[0].len	= 2;

    msg[1].addr = RTC_EEP_I2C_ADDR;
    msg[1].buf  = buf;
    msg[1].flags= I2C_M_RD;
    msg[1].len  = length;

    D("Reading %d bytes from %04x for EEP...", length, addr);
    ret = i2c_transfer(gdata->client->adapter, msg, NUM_MSGS);
    D("ret %d\n", ret);

    return (ret == NUM_MSGS ? 0 : -1);
#undef NUM_MSGS    
}

static int
ccr_byte_write(int addr, unsigned char byte)
{
    struct i2c_msg msg;
    unsigned char buf[] = { (addr & 0xFF00) >> 8, addr & 0x00FF, byte };
    int ret;

    memset(&msg, 0, sizeof(msg));
    msg.addr = RTC_CCR_I2C_ADDR;
    msg.buf = buf;
    msg.len = 3;

    D("Writing %02x to %04x for CCR...", byte, addr);
    ret = i2c_transfer(gdata->client->adapter, &msg, 1);
    D("ret %d\n", ret);

    return (ret == 1 ? 0 : -1);
}

static int
eep_byte_write(int addr, unsigned char byte)
{
    struct i2c_msg msg;
    unsigned char buf[] = { (addr & 0xFF00) >> 8, addr & 0x00FF, byte };
    int ret;

    memset(&msg, 0, sizeof(msg));
    msg.addr = RTC_EEP_I2C_ADDR;
    msg.buf = buf;
    msg.len = 3;

    D("Writing %02x to %04x for EEP...", byte, addr);
    ret = i2c_transfer(gdata->client->adapter, &msg, 1);
    D("ret %d\n", ret);

    return (ret == 1 ? 0 : -1);
}
