/**
 * tco_i2c_core.h
 *
 * Low level funktions for Intel's TCO Pass-Thru via SMBus
 *
 * (c) 2006 Peppercon AG, Thomas Rendelmann <thre@peppercon.de>
 */

#include <linux/i2c.h>
#include <linux/delay.h>
#include <asm/irq.h>

#include "tco_core.h"
#include "fml_ophir.h"

#define NAME "TCO_I2C"
#define LOG_COLORED
#define noLOG_DBG
#include "log.h"

/* ------------------------------------------------------------------------- *
 * constants
 * ------------------------------------------------------------------------- */

#define DEV_NAME            "tco_i2c"
#define I2C_ADAPTER_NAME    "FIA320 I2C 2"

#define DELAY() udelay(100)
#define RETRIES 3

#define SUCCESS 0

#define I2C_MAX_DATA_SIZE   32
#define I2C_SLAVE_ADDR      ((uint8_t)0x64)
#define I2C_SLAVE_ADDR_2ND  ((uint8_t)0x62)

#define DEF_IRQ             23

/* ------------------------------------------------------------------------- *
 * some irq functions for KIRA
 * ------------------------------------------------------------------------- */

extern void cpe_int_set_irq(unsigned int irq, int mode, int level);
extern void cpe_int_disable_irq(unsigned int base, unsigned int irq);
extern void cpe_int_enable_irq(unsigned int base, unsigned int irq);

/* ------------------------------------------------------------------------- *
 * function prototypes
 * ------------------------------------------------------------------------- */

/* tco specific */
static int i2c_init(struct net_device *dev, tco_t *tco);
static void i2c_cleanup(tco_t *tco);
static int i2c_core_init(struct net_device *dev, tco_t *tco);
static void i2c_core_release(struct net_device *dev, tco_t *tco);
static void i2c_enable_client_int(tco_t *tco, int en);
static int i2c_check_int_source(tco_t *tco);
static int i2c_write_block(tco_t *tco, u8 command,
		           u8 byte_count, const u8 * data);
static int i2c_read_block(tco_t *tco, u8 command,
		          u8 *byte_count, u8 *data);

/* i2c specific */
static int i2c_tco_i2c_attach_adapter(struct i2c_adapter *adapter);
static int i2c_tco_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
				     unsigned short flags,
#endif
				     int kind);
static int i2c_tco_i2c_detach_client(struct i2c_client *client);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void i2c_tco_i2c_inc_use (struct i2c_client *client);
static void i2c_tco_i2c_dec_use (struct i2c_client *client);
#endif
static int i2c_tco_i2c_command(struct i2c_client *client,
			       unsigned int cmd, void *arg);

/* other */
static void i2c_mod_isr(int irq, void *dev_id, struct pt_regs *regs);

/* ------------------------------------------------------------------------- *
 * private data
 * ------------------------------------------------------------------------- */

typedef struct i2c_mod_priv_s {
    tco_t               tco;

    /* i2c specific stuff */
    struct i2c_client*  i2c_client;
    int                 isr_registered;
} i2c_priv_t;

i2c_priv_t i2c_mod_priv = {
    tco: {
        name:               DEV_NAME,
        initialized:        0,
        max_data_size:      I2C_MAX_DATA_SIZE,
        slave_addr:         I2C_SLAVE_ADDR,

        init:               i2c_init,
        cleanup:            i2c_cleanup,
        core_init:          i2c_core_init,
        core_release:       i2c_core_release,
        enable_client_int:  i2c_enable_client_int,
        check_int_source:   i2c_check_int_source,
        write_block:        i2c_write_block,
        read_block:         i2c_read_block,
    },
};

tco_t *tco = (tco_t*)&i2c_mod_priv;

static struct i2c_driver i2c_tco_driver = {
    name:		DEV_NAME,
    id:			0xFFD0,	/* f000-ffff for local use */
    flags:		I2C_DF_NOTIFY,
    attach_adapter:	i2c_tco_i2c_attach_adapter,
    detach_client:	i2c_tco_i2c_detach_client,
    command:		i2c_tco_i2c_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    inc_use:		i2c_tco_i2c_inc_use,
    dec_use:		i2c_tco_i2c_dec_use,
#endif
};

/* ------------------------------------------------------------------------- *
 * exported functions
 * ------------------------------------------------------------------------- */

/* initialize on module loading */
static int i2c_init(struct net_device *dev, tco_t *tco) {
    int r, rc = 0;
    i2c_priv_t *priv = (i2c_priv_t *)tco;
    
    /* register i2c driver */
    if ((r = i2c_add_driver(&i2c_tco_driver))) {
    	WARN("%s: i2c driver registration failed (%d).", dev->name, r);
    	rc = -ENODEV;
    	goto fail;
    }
    tco->initialized = 1;
    
    /* check if the driver is installed */
    if (priv->i2c_client == NULL) {
        /* The probed driver is not included in the fpga ip core. This
         * happens quite often, so dont issue a warning but a debug message. */
        WARN("%s: device not present, skipping", dev->name);
    
        // use ENXIO instead of ENODEV to distinguish between 'bad' and regular initialization errors
    	rc = -ENXIO;
    	goto fail;
    }
    
 fail:
    return rc;
}

static void i2c_cleanup(tco_t *tco) {

    /* delete the driver */
    if (tco->initialized) {
    	if (i2c_del_driver(&i2c_tco_driver)) {
            WARN("%s: i2c driver deregistration failed.", tco->name);
    	}
    	tco->initialized = 0;
    }
}

/* initialize core on first usage */
static int i2c_core_init(struct net_device *dev, tco_t *tco) {
    i2c_priv_t *priv = (i2c_priv_t *)tco;
    int r, ret = 0;

    // disable IRQ source
    i2c_enable_client_int(tco, 0);

    DBG("Requesting IRQ %d", DEF_IRQ);
    cpe_int_set_irq(DEF_IRQ, LEVEL, L_ACTIVE);
    if ((r = request_irq(DEF_IRQ, i2c_mod_isr, 0, DEV_NAME, dev)) < 0) {
        ERR("failed to request irq %d (err %d)", DEF_IRQ, r);
        ret = -EBUSY;
        goto bail;
    }
    priv->isr_registered = 1;

 bail:
    return ret;
}

/* cleanup core on last usage */
static void i2c_core_release(struct net_device *dev, tco_t *tco) {
    i2c_priv_t *priv = (i2c_priv_t *)tco;

    if (priv->isr_registered) {
        free_irq(DEF_IRQ, dev);
        priv->isr_registered = 0;
    }
}

/* block read/write */

/* NOTE: the i2c_smbus_... functions of the linux kernel only supports
 * messages up to 32 bytes length, block reads are not supported at all.
 * So we have to implement a SMBus emulation via I2C function calls here.
 */
static int i2c_write_block(tco_t *tco,
		           u8 command,
		           u8 byte_count,
		           const u8 * data) {
    i2c_priv_t *priv = (i2c_priv_t *)tco;
    int ret = 0;

    /* sanity check */
    if (byte_count > I2C_MAX_DATA_SIZE) {
        WARN("i2c_write_block: byte count of %d exceeds limit of %d.",
             byte_count, I2C_MAX_DATA_SIZE);
        ret = -EINVAL;
        goto bail;
    }

    DBG("i2c_write_block: len: %d, data (%02x, %02x, %02x)",
        byte_count, data[0], data[1], data[2]);

    /* construct an I2C message
       byte 1: command
       byte 2: byte count
       byte 3 .. byte n: data */
    u8 msg_buf[2 + I2C_MAX_DATA_SIZE];
    struct i2c_msg msg = {
        addr:   tco->slave_addr,
        flags:  0,
        len:    byte_count + 2, /* 2 bytes header + data */
        buf:    msg_buf,
    };

    /* fill the I2C message */
    msg_buf[0] = command;
    msg_buf[1] = byte_count;
    memcpy(msg_buf + 2, data, byte_count);

    /* transfer the I2C message */
    int retry = RETRIES;
    do {
        ret = i2c_transfer(priv->i2c_client->adapter, &msg, 1);
        DELAY();
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during I2C transfer: %d", ret);
        else WARN("Error during I2C transfer: %d", ret);
    } while (retry-- > 0);
    if (ret < 0) goto bail;

bail:
    return ret;
}

static int i2c_read_block(tco_t *tco,
		          u8 command,
		          u8 *byte_count,
		          u8 *data) {
    i2c_priv_t *priv = (i2c_priv_t *)tco;
    int ret = 0;

    /* construct 2 I2C messages:
       message 1: write command byte
       message 2: read data bytes */    
    u8 msg_buf[1 + I2C_MAX_DATA_SIZE];
    struct i2c_msg msg[2] = {
        { tco->slave_addr, 0, 1, &command },
        { tco->slave_addr, I2C_M_RD, 1 + I2C_MAX_DATA_SIZE, msg_buf },
    };

    /* transfer the I2C messages */
    int retry = RETRIES;
    do {
        ret = i2c_transfer(priv->i2c_client->adapter, msg, 2);
        DELAY();
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during I2C transfer: %d", ret);
        else WARN("Error during I2C transfer: %d", ret);
    } while (retry-- > 0);
    if (ret < 0) goto bail;

    /* fill out return values;
       byte 1: byte count
       byte 2 .. byte n: data */
    *byte_count = msg_buf[0];

    /* sanity check */
    if (*byte_count > I2C_MAX_DATA_SIZE) {
        WARN("i2c_read_block: byte count of %d exceeds limit of %d.",
             *byte_count, I2C_MAX_DATA_SIZE);
        ret = -EINVAL;
        goto bail;
    }

    memcpy(data, msg_buf + 1, *byte_count);

    DBG("i2c_read_block: cmd: %02x, len: %d, data (%02x, %02x, %02x)",
        command, *byte_count, data[0], data[1], data[2]);

bail:
    return ret;
}

/* enable/disable client interrupt */
static void i2c_enable_client_int(tco_t *tco, int en) {
//#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#if 0
    if (en) {
        cpe_int_enable_irq(CPE_IC_VA_BASE, DEF_IRQ);
    } else {
        cpe_int_disable_irq(CPE_IC_VA_BASE, DEF_IRQ);
    }
#else
    // TODO: check _why_ this also work sfor kernel 2.4
    if (en) {
        enable_irq(DEF_IRQ);
    } else {
        disable_irq_nosync(DEF_IRQ);
    }
#endif
}

static int i2c_check_int_source(tco_t *tco)
{
    i2c_priv_t *priv = (i2c_priv_t *)tco;

    // check where the SMBus alert comes from (by ARA)
    u8 ara = 0;
    struct i2c_msg msg = { 0x0c, I2C_M_RD, 1, &ara };

    if (i2c_transfer(priv->i2c_client->adapter, &msg, 1) < 0) {
        ara = 0;
    }

    ara >>= 1; // make it a 7bit address
    DBG("ARA = %02x", ara);

    // ESB's second SMBus device: make a master read to satisfy alert
    if (ara == I2C_SLAVE_ADDR_2ND) {
        uint8_t count;
        tco_recv_pkt_t recv_pkt;

        WARN("Alert of sesond device detected");

        tco->slave_addr = I2C_SLAVE_ADDR_2ND;
        /*
         * HACK ALERT ;-)
         * The check_int_source() method is solely needed by fml_ophir.c, but
         * since this driver can also be used with fml_i825xx.c we must not
         * depend on ophir_recv_any() here.
         */
#if 0
        if (ophir_recv_any(tco, &count, &recv_pkt) != SUCCESS) {
            ERR("error reading from 2nd SMBus device");
        }
#endif
        if (tco->read_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_RECV_ANY,
                        &count, (uint8_t*)&recv_pkt) != SUCCESS);
        tco->slave_addr = I2C_SLAVE_ADDR;

        return -1;
    }

    return 0;
}

/* ------------------------------------------------------------------------- *
 * I2C stuff
 * ------------------------------------------------------------------------- */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static unsigned short normal_i2c[] = { I2C_SLAVE_ADDR, I2C_CLIENT_END };
static unsigned short normal_i2c_range[] = { I2C_SLAVE_ADDR, I2C_SLAVE_ADDR, I2C_CLIENT_END };
static unsigned short probe_i2c[] = { I2C_CLIENT_END };
static unsigned short probe_i2c_range[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c_range[] = { I2C_CLIENT_END };
// TODO: we could probe here, not force...
static unsigned short force_i2c[] = { ANY_I2C_BUS, I2C_SLAVE_ADDR, I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c, normal_i2c_range,
	probe_i2c, probe_i2c_range,
	ignore_i2c, ignore_i2c_range,
	force_i2c
};
#else
static unsigned short normal_i2c[] = { I2C_SLAVE_ADDR, I2C_CLIENT_END };
static unsigned short probe_i2c[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c[] = { I2C_CLIENT_END };
// TODO: we could probe here, not force...
static unsigned short force_i2c[] = { ANY_I2C_BUS, I2C_SLAVE_ADDR, I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c, probe_i2c, ignore_i2c, force_i2c
};
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void i2c_tco_i2c_inc_use (struct i2c_client *client) {
}

static void i2c_tco_i2c_dec_use (struct i2c_client *client) {
}
#endif

static int i2c_tco_i2c_command(struct i2c_client *client,
		               unsigned int cmd, void *arg) {
    /* no commands defined */
    return 0;
}

static int i2c_tco_i2c_attach_adapter(struct i2c_adapter *adapter) {
    /* don't attach the client if it is already attached */
    if (i2c_mod_priv.i2c_client) {
        DBG("Device already has adapter attached.");
        return 0;
    }
    
    DBG("i2c adapter attached, calling probe: %s", adapter->name);
    
    /* don't attach the client if adapter name doesn't match */
    if (strcmp(I2C_ADAPTER_NAME, adapter->name) != 0) {
        DBG("Expected adapter: %s, skipping", I2C_ADAPTER_NAME);
        return 0;
    }
    
    return i2c_probe_force(adapter, &addr_data, &i2c_tco_i2c_detect_client);
}

static int i2c_tco_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			             unsigned short flags,
#endif
			             int kind) {
    struct i2c_client* new_client = NULL;
    int err = 0;
    
    DBG("i2c detect client addr 0x%04x, kind %d", address, kind);
    
    if (!(new_client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL))) {
    	err = -ENOMEM;
    	goto fail;
    }
    memset(new_client, 0, sizeof(struct i2c_client));
    
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    strcpy(new_client->name, i2c_tco_driver.name);
    new_client->data = &i2c_mod_priv;
    new_client->id  = 0; /* ??? */
#else
    strlcpy(new_client->name, i2c_tco_driver.name, I2C_NAME_SIZE);
    i2c_set_clientdata(new_client, &i2c_mod_priv);
#endif
    new_client->addr = address;
    new_client->adapter = adapter;
    new_client->driver = &i2c_tco_driver;
    new_client->flags = 0;

    if ((err = i2c_attach_client_force(new_client))) {
    	goto fail_free;
    }    
    
    i2c_mod_priv.i2c_client = new_client;
    
    return 0;
    
 fail_free:    
    if (new_client) {
    	kfree(new_client);
    	new_client = NULL;
    }
    
 fail:
    return err;
}

static int i2c_tco_i2c_detach_client(struct i2c_client *client) {
    int err;
    i2c_priv_t *priv = client->data;
    
    DBG("i2c detach client %s", client->name);

    if ((err = i2c_detach_client(client))) {
    	WARN("i2c client detach failed");
    	goto fail;
    }
    
    kfree(client);

 fail:
    priv->i2c_client = NULL;
    
    return err;
}

/* ------------------------------------------------------------------------- *
 * interrupt handling
 * ------------------------------------------------------------------------- */

static void i2c_mod_isr(int irq, void *dev_id, struct pt_regs *regs) {
    tco_recv_cb(tco);

#ifndef __arm__ 
    return IRQ_HANDLED;    
#endif
}
