/***************************************************************************
 *   Copyright (C) 2004 by Ralf Guenther                                   *
 *   rgue@peppercon.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/socket.h>

#include <fml.h>

/*****************************/
/* buffer offsets...       */
/*****************************/
#define REG_OFFSET  0x0
#define BUF0_OFFSET 0x100 
#define BUF1_OFFSET 0x140
#define BUF2_OFFSET 0x180
#define BUF3_OFFSET 0x1c0

/*****************************/
/* register offsets...       */
/*****************************/
#define MGR_ADDR 0x0          /* general register */
#define MCR0_ADDR 0x1         /* command register */
#define MBA0_ADDR 0x2         /* ram buffer address register */
#define MGSR_ADDR 0x3         /* status register */
#define MPRER_ADDR 0x4        /* devider register */
#define MTOUT_ADDR 0x5        /* FML timeout value in system clock cycles! */
#define MID1_ADDR 0x6         /* FML-Core ID1 register */
#define MID2_ADDR 0x7         /* FML-Core ID2 register */


/*****************************/
/* define register values... */
/*****************************/

/* two ID registers to find the core */
#define FML_ID1 0x464d4c2d
#define FML_ID2 0x436f7265

/* values for the MGR-Register */
#define MGO  (0x1)
#define MRST (0x1 << 1)
#define MALLIEN (0x1 << 4)
#define MFMLIEN (0x1 << 5)
#define MCOREIEN (0x1 << 6)

/* values for the MCR0-Register */
#define MSTC (0x1<<0)
#define MSPM (0x1<<1)
#define MNC  (0x1<<2)
#define MRDC (0x1<<4)
#define MWRC (0x1<<5)
#define MPC  (0x1<<6)
#define MIACK (0x1<<9)
#define BTCNT_EN (0x1<<11)
#define MBUFSEL_MASK (0x3<< 12)
#define MBUFSEL_SHIFT 12
#define MBC_MASK (0xf<16)
#define MBC_SHIFT 16                             

/* values of the MGSR-Register */
#define MERR             (0x1 << 0)
#define MAB              (0x1 << 1)
#define CORE_DONE        (0x1 << 7)
#define MCORE_IRQ        (0x1 << 16)
#define OVERFLOW         (0x1 << 19)
#define GET_MBWD(a)      (( a & 0xff00 ) >> 8)
#define GET_RD_BNTCNT(a)      (( a & 0xff000000 ) >> 24) 

static int find_core(int sock)
{
    if (fml_ioctl_get_reg(sock, MID1_ADDR) != FML_ID1
     || fml_ioctl_get_reg(sock, MID2_ADDR) != FML_ID2) {
        return -1;
    }
    return 0;
}

static void dump_regs(int sock, const char* txt)
{
    int mgr  = fml_ioctl_get_reg(sock, MGR_ADDR);
    int mcr0 = fml_ioctl_get_reg(sock, MCR0_ADDR);
    int mgsr = fml_ioctl_get_reg(sock, MGSR_ADDR);
    int dbg = fml_ioctl_get_reg(sock, MID2_ADDR);
    if (mgsr & MCORE_IRQ) fml_ioctl_set_reg(sock, MCR0_ADDR, mcr0 | MIACK);
    printf("%s: mgr=%08x mcr0=%08x mgsr=%08x dbg=%08x\n", txt, mgr, mcr0, mgsr, dbg);
}

int main(int argc, char *argv[])
{
    int sock;

    argc = argc; argv = argv;
    
    printf("Opening socket...\n");
    sock = socket(PF_INET, SOCK_DGRAM, 0);
    if (sock == -1) {
        printf("ERROR: open failed (%s)\n", strerror(errno));
        exit(-1);
    }

    printf("Checking FML core...\n");
    if (find_core(sock)) {
        printf("FML core not found (%s)\n", strerror(errno));
        exit(-1);
    }

    dump_regs(sock, "origin");

#if 0    
    // reset
    fml_ioctl_set_reg(sock, 0, MGR_ADDR, MRST);
    while (fml_ioctl_get_reg(sock, 0, MGR_ADDR) & MRST) {
        dump_regs(sock, "waiting for reset");
        sleep(0);
    }
    dump_regs(sock, "reset");

    // init
    fml_ioctl_set_reg(sock, 0, MPRER_ADDR, 1);
    fml_ioctl_set_reg(sock, 0, MGR_ADDR, MGO);
    dump_regs(sock, "init");

    // request status
    unsigned char saddr = 0x63 << 1;
    unsigned char cmd = 0xdd;
    unsigned char len = 1;
    unsigned char data[] = { 0 };
    fml_ioctl_set_reg(sock, 0, BUF0_OFFSET + 0, saddr | (cmd << 8) | (len << 16) | (data[0] << 24));
    int i, j;
    for (i = 1, j = 1; i < len; i += 4, j++)
        fml_ioctl_set_reg(sock, 0, BUF0_OFFSET + j, data[i] | (data[i + 1] << 8) | (data[i + 2] << 16) | (data[i + 3] << 24));
    fml_ioctl_set_reg(sock, 0, MCR0_ADDR, MSTC | MWRC | ((len + 3) << MBC_SHIFT));

/*
//shouldn't we start the cmd by GO?????
    fml_ioctl_set_reg(sock, 0, MGR_ADDR, MGO);
    while (fml_ioctl_get_reg(sock, 0, MGR_ADDR) & MGO) {
        dump_regs(sock, "waiting for transmission");
        sleep(0);
    }
*/
    dump_regs(sock, "transmit");
#endif

    {
        fml_ioctl_raw_pkt_t pkt = {
            .saddr = 0x63,
            .cmd = 0xdd,
            .len = 1,
            .data = { 0 },
        };

        if (fml_ioctl_raw_send(sock, &pkt) < 0) {
            printf("ERROR: fml_ioctl_tco_send failed (%s)\n", strerror(errno));
        }
    }

    while (1) {
        dump_regs(sock, "idle");
        sleep(1);
    }

    close(sock);
    return 0;
}
