/* linux includes */
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/crypto.h>
#include <asm/scatterlist.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
# include <asm/kmap_types.h>
# include <asm/highmem.h>
#endif

/* local includes */
#include "peppercon_usb.h"
#include "pp_usb.h"
#include "hal_isp1181.h"
#include "hid.h"
#ifdef PP_FEAT_MASS_STORAGE
#include "mass-storage-bo.h"
#include "scsi.h"
#endif /* PP_FEAT_MASS_STORAGE */

/* ep 3 mouse, ep 4 kbd */
extern usb_driver_t g_ud;
#define PEPPERCON_USB_MAX_PACKET_SIZE 64

#define PEPPERCON_USB_DEFAULT_SERIAL_NUMBER \
    "1" "\0" "2" "\0" "3" "\0" "4" "\0" \
    "5" "\0" "6" "\0" "7" "\0" "8" "\0" \
    "9" "\0" "0" "\0" "1" "\0" "2" "\0" \
    "3" "\0" "4" "\0" "5" "\0" "6" "\0" \
    "7" "\0" "8" "\0" "9" "\0" "0" "\0" \
    "1" "\0" "2" "\0" "3" "\0" "4" "\0" \
    "5" "\0" "6" "\0" "7" "\0" "8" "\0" \
    "9" "\0" "0" "\0" "1" "\0" "2" "\0"

/* callback functions */
static void peppercon_usb_config_endpoints(void);
static void peppercon_usb_device_init(void);
static int peppercon_send_data(const uint8_t channel, const uint8_t type, const char* data, const int length);
static void peppercon_non_std_descriptors(uint16_t wValue, uint16_t wIndex, uint16_t wLength);


static int HID_MouseInterfaceNumber;
static int HID_KeyboardInterfaceNumber;
#ifdef PP_FEAT_MASS_STORAGE
static int MassStorageInterfaceNumber[PP_FEAT_USB_MASS_STORAGE_NO];
#endif /* PP_FEAT_MASS_STORAGE */


/* local functions */
static void peppercon_set_mouse_parameter(usb_device_mouse_type_t mouse_type);
static void peppercon_generate_serial_number(usb_driver_t *ud);
static int peppercon_build_configuration_descriptor(usb_driver_t *ud);
static int peppercon_build_string_descriptors(usb_driver_t *ud);
static int peppercon_setup_interfaces_and_endpoints(usb_driver_t *ud);

/* local data */
static USB_DEVICE_DESCRIPTOR  PepperconDeviceDescriptor;
static const LANGID_DESCRIPTOR  LANGID_Descriptor;
static const MANUFACTURER_DESCRIPTOR  Manufacturer_Descriptor;
static const PRODUCT_DESCRIPTOR  Product_Descriptor;
static const SERIAL_DESCRIPTOR  Serial_Descriptor;

static USB_CONFIGURATION_DESCRIPTOR Configuration_Descriptor;

static USB_INTERFACE_DESCRIPTOR Interface_Descriptor_Mouse;
static USB_HID_DESCRIPTOR HID_Descriptor_Mouse;
static USB_ENDPOINT_DESCRIPTOR Endpoint_Descriptor_Mouse;

static USB_INTERFACE_DESCRIPTOR Interface_Descriptor_Keyboard;
static USB_HID_DESCRIPTOR HID_Descriptor_Keyboard;
static USB_ENDPOINT_DESCRIPTOR Endpoint_Descriptor_Keyboard;

#ifdef PP_FEAT_MASS_STORAGE
static USB_INTERFACE_DESCRIPTOR Interface_Descriptor_MassStorage;
static USB_ENDPOINT_DESCRIPTOR Endpoint_Descriptor_MassStorage_In;
static USB_ENDPOINT_DESCRIPTOR Endpoint_Descriptor_MassStorage_Out;
#endif /* PP_FEAT_MASS_STORAGE */

const char* current_mouse_descriptor;
int current_mouse_descriptor_length;
int current_mouse_report_length;
#ifdef PP_FEAT_MASS_STORAGE
int mass_storage_enabled[PP_FEAT_USB_MASS_STORAGE_NO];
usb_device_type_t mass_storage_type[PP_FEAT_USB_MASS_STORAGE_NO];
#endif /* PP_FEAT_MASS_STORAGE */
int hid_enabled;

int peppercon_usb_init(usb_driver_t *ud, usb_device_mouse_type_t mouse_type, int * enable_mass_storage, usb_device_type_t* mass_storage_types ) {

    int i;
    int ret;

    /* some default stuff */
    {
	int ms_enabled = 0;
#ifdef PP_FEAT_MASS_STORAGE
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    mass_storage_enabled[i] = enable_mass_storage[i];
	    ms_enabled |= mass_storage_enabled[i];
	    mass_storage_type[i] = mass_storage_types[i];
	}
#endif /* PP_FEAT_MASS_STORAGE */
	hid_enabled = (mouse_type != peppercon_mouse_type_none);
	UD("hid_enabled: %d\n", hid_enabled);
	
	/* sanity check */
	if (!ms_enabled && !hid_enabled) {
	    /* we must emulate at least one interface */
	    UD("neither mass storage nor hid enabled, but there must be at least one interface!");
	    ud->initialized = 0;
	} else {
	    ud->initialized = 1;
	}
    }

    for (i = 0; i < MAX_INTERFACES; i++) {
	ud->class_interface_request[i] = NULL;
    }
    for (i = EP1; i <= EP14; i++) {
	    ud->ep_int_functions[i] = NULL;
    }
    for (i = EP1; i <= EP14; i++) {
	ud->eps[i].used = 0;
    }
    ud->vendor_device_request = NULL;
    
    /* set the appropriate mouse descriptor */
    if (hid_enabled) peppercon_set_mouse_parameter(mouse_type);
    
    /* set global parameters */
    if (hid_enabled) {
	ud->send_data = peppercon_send_data;
    } else {
	ud->send_data = NULL;
    }
    ud->usb_max_packet_size = PEPPERCON_USB_MAX_PACKET_SIZE;

    /* set the device descriptor */
    ud->DeviceDescriptor = (char*)&PepperconDeviceDescriptor;

    /* build the configuration descriptor */
    if ((ret = peppercon_build_configuration_descriptor(ud)) != 0) {
    	return ret;
    }
    
    /* build the string descriptors */
    if ((ret = peppercon_build_string_descriptors(ud)) != 0) {
    	return ret;
    }
    
    /* setup the interfaces and endpoints */
    if ((ret = peppercon_setup_interfaces_and_endpoints(ud)) != 0) {
    	return ret;
    }
    
    /* set some function pointers */
    ud->config_endpoints = peppercon_usb_config_endpoints; /* configures endpoints 1 to 14 */
    ud->usb_init = peppercon_usb_device_init;
    ud->non_std_descriptors = peppercon_non_std_descriptors;
    ud->get_data_length = NULL;
    
#ifdef PP_FEAT_MASS_STORAGE
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
    	if (mass_storage_enabled[i]) {
	    ms_bo_init();
	    break;
    	}
    }
#endif /* PP_FEAT_MASS_STORAGE */

    /* build the serial number string */
    peppercon_generate_serial_number(ud);

    return 0;
}

#define CONFIG_DESCRIPTOR_MAX_SIZE	512

static uint8_t ConfigDescriptorSpace[CONFIG_DESCRIPTOR_MAX_SIZE];

/* add a descriptor to the configuration descriptor
   returns 0 if okay or <0 on error */
static int add_to_config_descriptor(usb_driver_t *ud, char * desc, int size) {
    USB_CONFIGURATION_DESCRIPTOR * config = (USB_CONFIGURATION_DESCRIPTOR *)ud->ConfigDescriptor;
    int curr_pos = le16_to_cpu(config->wTotalLength);
    
    /* size check */
    if ((curr_pos + size) > CONFIG_DESCRIPTOR_MAX_SIZE) {
    	return -1;
    }
    
    /* add the new descriptor */
    memcpy(&ud->ConfigDescriptor[curr_pos], desc, size);
    
    /* adjust the size */
    config->wTotalLength = cpu_to_le16(curr_pos + size);
    
    return 0;
}

static int peppercon_build_configuration_descriptor(usb_driver_t *ud) {
    USB_CONFIGURATION_DESCRIPTOR * config;
    int curr_intf = 0;
    int ms = 0;
    
    ud->ConfigDescriptor = ConfigDescriptorSpace;
    config = (USB_CONFIGURATION_DESCRIPTOR *)ud->ConfigDescriptor;
    memset(ud->ConfigDescriptor, 0, CONFIG_DESCRIPTOR_MAX_SIZE);
    
    /* add the configuration descriptor */
    if (add_to_config_descriptor(ud, (char *)&Configuration_Descriptor, Configuration_Descriptor.bLength)) {
    	printk("Could not add configuration descriptor.\n");
    	return -1;
    }
    
#ifdef PP_FEAT_MASS_STORAGE
    /* add all mass storage devices */
    for (ms = 0; ms < PP_FEAT_USB_MASS_STORAGE_NO; ms++) {
    	if (mass_storage_enabled[ms]) {
    	    /* adjust the bInterfaceNumber and bEndpointAddress numbers */
    	    Interface_Descriptor_MassStorage.bInterfaceNumber = curr_intf;
    	    Endpoint_Descriptor_MassStorage_In.bEndpointAddress = 0x81 + 2 * ms;
    	    Endpoint_Descriptor_MassStorage_Out.bEndpointAddress = 0x02 + 2 * ms;
	    
	    if (mass_storage_type[ms] == PP_USB_IMAGE_TYPE_FLOPPY) {
		Interface_Descriptor_MassStorage.bInterfaceSubClass = USB_MASS_STORAGE_SUBCLASS_SCSI_TRANS_CS;
		//Interface_Descriptor_MassStorage.bInterfaceSubClass = USB_MASS_STORAGE_SUBCLASS_UFI;
	    } else {
		Interface_Descriptor_MassStorage.bInterfaceSubClass = USB_MASS_STORAGE_SUBCLASS_SCSI_TRANS_CS;
	    }
	    
    	    /* add the descriptors */
    	    if (add_to_config_descriptor(ud, (char *)&Interface_Descriptor_MassStorage, Interface_Descriptor_MassStorage.bLength) ||
    	        add_to_config_descriptor(ud, (char *)&Endpoint_Descriptor_MassStorage_In, Endpoint_Descriptor_MassStorage_In.bLength) ||
    	        add_to_config_descriptor(ud, (char *)&Endpoint_Descriptor_MassStorage_Out, Endpoint_Descriptor_MassStorage_Out.bLength)) {
    	        
    	        printk("Could not add mass storage interface descriptor %d.\n", ms);
    	        return -1;
    	    }
    	    
    	    MassStorageInterfaceNumber[ms] = curr_intf;

    	    /* go on with the next interface number */
    	    curr_intf++;
    	    config->bNumInterfaces++;
    	} else {
    	    MassStorageInterfaceNumber[ms] = -1;
    	}
    }
#endif /* PP_FEAT_MASS_STORAGE */
    
    /* add the HID descriptors */
    if (hid_enabled) {
    	/* first, add the keyboard */
    	Interface_Descriptor_Keyboard.bInterfaceNumber = curr_intf;
    	
    	if (add_to_config_descriptor(ud, (char *)&Interface_Descriptor_Keyboard, Interface_Descriptor_Keyboard.bLength) ||
    	    add_to_config_descriptor(ud, (char *)&HID_Descriptor_Keyboard, HID_Descriptor_Keyboard.bLength) ||
    	    add_to_config_descriptor(ud, (char *)&Endpoint_Descriptor_Keyboard, Endpoint_Descriptor_Keyboard.bLength)) {
    	    
    	    printk("Could not add keyboard interface descriptor %d.\n", ms);
    	    return -1;
    	}
    	
    	HID_KeyboardInterfaceNumber = curr_intf;
    	curr_intf++;
    	config->bNumInterfaces++;
    	
    	/* second, add the mouse */
    	Interface_Descriptor_Mouse.bInterfaceNumber = curr_intf;
    	
    	if (add_to_config_descriptor(ud, (char *)&Interface_Descriptor_Mouse, Interface_Descriptor_Mouse.bLength) ||
    	    add_to_config_descriptor(ud, (char *)&HID_Descriptor_Mouse, HID_Descriptor_Mouse.bLength) ||
    	    add_to_config_descriptor(ud, (char *)&Endpoint_Descriptor_Mouse, Endpoint_Descriptor_Mouse.bLength)) {
    	    
    	    printk("Could not add mouse interface descriptor %d.\n", ms);
    	    return -1;
    	}
    	
    	HID_MouseInterfaceNumber = curr_intf;
    	curr_intf++;
    	config->bNumInterfaces++;
    	
    } else {
    	HID_MouseInterfaceNumber = -1;
    	HID_KeyboardInterfaceNumber = -1;
    }
    return 0;
}

static int peppercon_build_string_descriptors(usb_driver_t *ud) {
    int i;

    /* clear the descriptors */
    for (i = 4; i < MAX_STRING_DESCRIPTORS; i++) {
	ud->StringDescriptors[i] = NULL;
    }
    
    /* set the language descriptor */
    ud->StringDescriptors[0] = (char*)&LANGID_Descriptor;
    ud->StringDescriptors[1] = (char*)&Manufacturer_Descriptor;
    ud->StringDescriptors[2] = (char*)&Product_Descriptor;
    ud->StringDescriptors[3] = (char*)&Serial_Descriptor;
        
    return 0;
}

static int peppercon_setup_interfaces_and_endpoints(usb_driver_t *ud) {
    
    if (HID_MouseInterfaceNumber >= 0) {
    	ud->class_interface_request[HID_MouseInterfaceNumber] = hid_mouse_handle_classrequest;
    }
    if (HID_KeyboardInterfaceNumber >= 0) {
    	ud->class_interface_request[HID_KeyboardInterfaceNumber] = hid_keyboard_handle_classrequest;
    }
    
    if (hid_enabled) {
	ud->eps[EP1 + 2 * PP_FEAT_USB_MASS_STORAGE_NO].used = 1;
	ud->eps[EP1 + 2 * PP_FEAT_USB_MASS_STORAGE_NO].ep_type = INTERRUPT_IN;
	ud->eps[EP2 + 2 * PP_FEAT_USB_MASS_STORAGE_NO].used = 1;
	ud->eps[EP2 + 2 * PP_FEAT_USB_MASS_STORAGE_NO].ep_type = INTERRUPT_IN;
    }
    
#ifdef PP_FEAT_MASS_STORAGE
    {
	int i;
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    if (mass_storage_enabled[i]) {
		ud->eps[EP_MS_BULK_IN(i)].used = 1;
		ud->eps[EP_MS_BULK_IN(i)].ep_type = BULK_IN;
		ud->eps[EP_MS_BULK_IN(i)].status = BULK_IDLE;

		ud->eps[EP_MS_BULK_OUT(i)].used = 1;
		ud->eps[EP_MS_BULK_OUT(i)].ep_type = BULK_OUT;
		ud->eps[EP_MS_BULK_OUT(i)].status = BULK_IDLE;

		if (MassStorageInterfaceNumber[i] >= 0) {
		    ud->class_interface_request[MassStorageInterfaceNumber[i]] = ms_bo_handle_classrequest;
		}
	    }
	}
    }
#endif /* PP_FEAT_MASS_STORAGE */

    return 0;
}

#pragma pack(1)
const int KeyboardReportLength = 8;
const unsigned char KeyboardReportDescriptor[] = {
    0x05, 0x01,                    // USAGE_PAGE (Generic Desktop)
    0x09, 0x06,                    // USAGE (Keyboard)
    0xa1, 0x01,                    // COLLECTION (Application)

    /* modifier keys */
    0x05, 0x07,                    //   USAGE_PAGE (Keyboard)
    0x19, 0xe0,                    //   USAGE_MINIMUM (Keyboard LeftControl)
    0x29, 0xe7,                    //   USAGE_MAXIMUM (Keyboard Right GUI)
    0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
    0x25, 0x01,                    //   LOGICAL_MAXIMUM (1)
    0x75, 0x01,                    //   REPORT_SIZE (1)
    0x95, 0x08,                    //   REPORT_COUNT (8)
    0x81, 0x02,                    //   INPUT (Data,Var,Abs)
    0x95, 0x01,                    //   REPORT_COUNT (1)
    0x75, 0x08,                    //   REPORT_SIZE (8)
    0x81, 0x01,                    //   INPUT (Cnst,Ary,Abs)

    /* normal keys */
    0x19, 0x00,                    //   USAGE_MINIMUM (Reserved (no event indicated))
    0x29, 0xe7,                    //   USAGE_MAXIMUM (Keyboard Right GUI)
    0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
    0x26, 0xe7, 0x00,              //   LOGICAL_MAXIMUM (231)
    0x75, 0x08,                    //   REPORT_SIZE (8)
    0x95, 0x06,                    //   REPORT_COUNT (6)
    0x81, 0x00,                    //   INPUT (Data,Ary,Abs)

    /* LEDs */
    0x05, 0x08,                    //   USAGE_PAGE (LEDs)
    0x19, 0x01,                    //   USAGE_MINIMUM (Num Lock)
    0x29, 0x05,                    //   USAGE_MAXIMUM (Kana)
    0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
    0x25, 0x01,                    //   LOGICAL_MAXIMUM (1)
    0x75, 0x01,                    //   REPORT_SIZE (1)
    0x95, 0x05,                    //   REPORT_COUNT (5)
    0x91, 0x02,                    //   OUTPUT (Data,Var,Abs)
    0x95, 0x01,                    //   REPORT_COUNT (1)
    0x75, 0x03,                    //   REPORT_SIZE (3)
    0x91, 0x01,                    //   OUTPUT (Cnst,Ary,Abs)

    0xc0                           // END_COLLECTION
};

const int MouseAbsReportLength = 6;
const unsigned char  MouseReportDescriptorAbsolute[] = {
    0x05, 0x01,                    // USAGE_PAGE (Generic Desktop)
    0x09, 0x02,                    // USAGE (Mouse)
    0xa1, 0x01,                    // COLLECTION (Application)
    0x09, 0x01,                    //   USAGE (Pointer)
    0xa1, 0x00,                    //   COLLECTION (Physical)

    // buttons start
    0x05, 0x09,                    //     USAGE_PAGE (Button)
    0x19, 0x01,                    //     USAGE_MINIMUM (Button 1)
    0x29, 0x03,                    //     USAGE_MAXIMUM (Button 3)
    0x15, 0x00,                    //     LOGICAL_MINIMUM (0)
    0x25, 0x01,                    //     LOGICAL_MAXIMUM (1)
    0x95, 0x03,                    //     REPORT_COUNT (3)
    0x75, 0x01,                    //     REPORT_SIZE (1)
    0x81, 0x02,                    //     INPUT (Data,Var,Abs)
    0x95, 0x01,                    //     REPORT_COUNT (1)
    0x75, 0x05,                    //     REPORT_SIZE (5)
    0x81, 0x01,                    //     INPUT (Cnst,Ary,Abs)

    /* X and Y */
    0x05, 0x01,                    //     USAGE_PAGE (Generic Desktop)
    0x09, 0x30,                    //     USAGE (X)
    0x09, 0x31,                    //     USAGE (Y)
    0x16, 0x00, 0x00,                    //     LOGICAL_MINIMUM (-126)
    0x26, 0xff, 0x7f,                    //     LOGICAL_MAXIMUM (127)
    0x36, 0x00, 0x00,  // phys. min
    0x46, 0xFF, 0x7F,  // phys. min
    0x66, 0x00, 0x00,
    0x75, 0x10,                    //     REPORT_SIZE (10)
    0x95, 0x02,                    //     REPORT_COUNT (2)
    0x81, 0x22,                    //     INPUT (Data,Var,Abs)

    /* W */
    0x05, 0x01,                    //     USAGE_PAGE (Generic Desktop)
    0x09, 0x38,                    //     USAGE (Z)
    
    0x15, 0x81,                    //     LOGICAL_MINIMUM (-127)
    0x25, 0x7f,                    //     LOGICAL_MAXIMUM (127)
#if 1
    0x35, 0x81,                    //     PHYSICAL_MINIMUM (-127)
    0x45, 0x7f,                    //     PHYSICAL_MAXIMUM (127)
#endif
    0x75, 0x08,                    //     REPORT_SIZE (8)
    0x95, 0x01,                    //     REPORT_COUNT (1)
    0x81, 0x06,                    //     INPUT (Data,Var,Rel)

    0xc0,                          //   END_COLLECTION
    0xc0                           // END_COLLECTION
};

const int MouseRelReportLength = 4;
const unsigned char  MouseReportDescriptorRelative[] = {
    0x05, 0x01,                    // USAGE_PAGE (Generic Desktop)
    0x09, 0x02,                    // USAGE (Mouse)
    0xa1, 0x01,                    // COLLECTION (Application)
    0x09, 0x01,                    //   USAGE (Pointer)
    0xa1, 0x00,                    //   COLLECTION (Physical)

    // buttons start
    0x05, 0x09,                    //     USAGE_PAGE (Button)
    0x19, 0x01,                    //     USAGE_MINIMUM (Button 1)
    0x29, 0x03,                    //     USAGE_MAXIMUM (Button 3)
    0x15, 0x00,                    //     LOGICAL_MINIMUM (0)
    0x25, 0x01,                    //     LOGICAL_MAXIMUM (1)
    0x95, 0x03,                    //     REPORT_COUNT (3)
    0x75, 0x01,                    //     REPORT_SIZE (1)
    0x81, 0x02,                    //     INPUT (Data,Var,Abs)
    0x95, 0x01,                    //     REPORT_COUNT (1)
    0x75, 0x05,                    //     REPORT_SIZE (5)
    0x81, 0x03,                    //     INPUT (Cnst,Ary,Abs)

    /* X and Y */
    0x05, 0x01,                    //     USAGE_PAGE (Generic Desktop)
    0x09, 0x30,                    //     USAGE (X)
    0x09, 0x31,                    //     USAGE (Y)
    0x09, 0x38,                    //     USAGE (W)
    0x15, 0x81,                    //     LOGICAL_MINIMUM (-126)
    0x25, 0x7f,                    //     LOGICAL_MAXIMUM (127)
    0x75, 0x08,                    //     REPORT_SIZE (10)
    0x95, 0x03,                    //     REPORT_COUNT (2)
    0x81, 0x06,                    //     INPUT (Data,Var,Rel)
    0xc0,                          //   END_COLLECTION
    0xc0                           // END_COLLECTION
};

static void peppercon_set_mouse_parameter(usb_device_mouse_type_t mouse_type) {
    if (mouse_type == peppercon_mouse_type_absolute) {
	UD("absolute \n");
	current_mouse_descriptor = MouseReportDescriptorAbsolute;
	current_mouse_descriptor_length = sizeof(MouseReportDescriptorAbsolute);
	current_mouse_report_length = MouseAbsReportLength;
    } else {
	if (mouse_type != peppercon_mouse_type_relative) {
	    printk("unknown mouse parameter - set to default\n");
	}
	current_mouse_descriptor = MouseReportDescriptorRelative;
	current_mouse_descriptor_length = sizeof(MouseReportDescriptorRelative);
	current_mouse_report_length = MouseRelReportLength;
    }
    HID_Descriptor_Mouse.w1stDescLength = cpu_to_le16(current_mouse_descriptor_length);
}

void peppercon_non_std_descriptors(uint16_t wValue, uint16_t wIndex, uint16_t wLength) {
    uint8_t length;

    switch((wValue & 0xFF00) >> 8) {
      case TYPE_REPORT_DESCRIPTOR:
	  if (wIndex == HID_MouseInterfaceNumber) {
		length = current_mouse_descriptor_length;
		Chap9_BurstTransmitEP0(current_mouse_descriptor, length);
	  } else if (wIndex == HID_KeyboardInterfaceNumber) {
		length = sizeof(KeyboardReportDescriptor);
		/* send always the report protocol descriptor regardless of the protocol (boot/report) F.3 */
		Chap9_BurstTransmitEP0((uint8_t*)&KeyboardReportDescriptor, length);
	  } else {
		Chap9_StallEP0InControlRead();
	  }
	  break;
      case TYPE_HID_DESCRIPTOR:
	  UD("windex of hid_desc: %x\n", wIndex);
	  if (wIndex == HID_MouseInterfaceNumber) {
		memcpy(&length, &HID_Descriptor_Mouse, 1);
		Chap9_BurstTransmitEP0((uint8_t*)&HID_Descriptor_Mouse, length);
	  } else if (wIndex == HID_KeyboardInterfaceNumber) {
		memcpy(&length, &HID_Descriptor_Keyboard, 1);
		Chap9_BurstTransmitEP0((uint8_t*)&HID_Descriptor_Keyboard, length);
	  } else {
		Chap9_StallEP0InControlRead();
	  }
	  break;
      case TYPE_PHYSICAL_DESCRIPTOR:
	  /* fall through */
      default:
	  Chap9_StallEP0InControlRead();
    }
}

void peppercon_usb_device_init(void) {
    if (hid_enabled) hid_init();
#ifdef PP_FEAT_MASS_STORAGE
    {
	int i;
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    if (mass_storage_enabled[i]) {
		ms_bo_init();
		break;
	    }
	}
    }
#endif /* PP_FEAT_MASS_STORAGE */
}

static int peppercon_send_data(const uint8_t channel, const uint8_t type, const char* data, const int length) {
    if (hid_enabled) {
	if (type == PP_KM_INPUT_KEYBOARD) {
	    hid_keyboard_set_report(data, length);
	} else if (type == PP_KM_INPUT_MOUSE) {
	    hid_mouse_set_pointer(data, length);
	} else {
	    return -1;
	}
    } else {
	return -1;
    }
    return 0;
}

static void peppercon_usb_config_endpoints(void) {
    uint8_t i;
    uint8_t mode;

#ifdef PP_FEAT_MASS_STORAGE
    int ms;
    for (ms = 0; ms < PP_FEAT_USB_MASS_STORAGE_NO; ms++) {
    	if (mass_storage_enabled[ms]) {
	    mode =
	    	ISP1181B_EP_CONFIG_FIFOEN |
	    	ISP1181B_EP_CONFIG_EPIN |
	    	(0x3);
	    hal_SetEndpointConfig(EP_MS_BULK_IN(ms), mode);
	    mode =
	    	ISP1181B_EP_CONFIG_FIFOEN |
	    	(0x3);
	    hal_SetEndpointConfig(EP_MS_BULK_OUT(ms), mode);
    	} else {
	    for (i = EP_MS_BULK_IN(ms); i <= EP_MS_BULK_OUT(ms); i++) {
	    	mode = 0;
	    	hal_SetEndpointConfig(i, mode);
	    }
    	}
    }
#endif /* PP_FEAT_MASS_STORAGE */
    
    if (hid_enabled) {
	/* mouse use EP 3 and 4 at the moment */
	for (i = EP1 + 2 * PP_FEAT_USB_MASS_STORAGE_NO; i <= EP2 + 2 * PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    mode =
		ISP1181B_EP_CONFIG_FIFOEN |
		ISP1181B_EP_CONFIG_EPIN |
		(0x1 & 0xf);
	    hal_SetEndpointConfig(i, mode);
	}
    }

    for (i = EP3 + 2 * PP_FEAT_USB_MASS_STORAGE_NO; i <= EP14; i++) {
	mode = 0;
	hal_SetEndpointConfig(i, mode);
    }
}

/* Serial number generation */
#define offset_in_page(p) ((unsigned long)(p) & ~PAGE_MASK)

static void* get_logical_address(const void * address) {
    pgd_t *pgd;
    pmd_t *pmd;
    pte_t *ptep, pte;
    unsigned long kaddr = (unsigned long)address;
    unsigned int adr;
    
    adr = (unsigned long)address;
    
    pgd = pgd_offset_k(adr);
    if (!pgd_none(*pgd)) {
    	pmd = pmd_offset(pgd, adr);
    	if (!pmd_none(*pmd)) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    	    ptep = pte_offset(pmd, adr);
#else
	    ptep = ((pte_t *) kmap(pmd_page(*(pmd))) + pte_index(adr));
#endif
    	    pte = *ptep;
    	    if(pte_present(pte)) {
    	        kaddr  = (unsigned long) page_address(pte_page(pte));
    	        kaddr |= (adr & (PAGE_SIZE - 1));
    	    }
    	}
    }
    
    return (void*) kaddr;
}

static void do_crypto_digest_update(struct crypto_tfm *tfm, const void *address, unsigned int length) {
    int no;
    struct scatterlist sg[2];
    void* logical_address1;
    void* logical_address2;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    inc_preempt_count();
#endif

    memset(&sg, 0, sizeof(sg));
    
    logical_address1 = get_logical_address(address);
    
    sg[0].page = virt_to_page(logical_address1);
    sg[0].offset = offset_in_page(logical_address1);

    if (sg[0].offset + length <= PAGE_SIZE) {
    	sg[0].length = length;
    	sg[1].page = 0;
    	sg[1].offset = 0;
    	sg[1].length = 0;
    	no = 1;
    } else {
    	sg[0].length = PAGE_SIZE - sg[0].offset;
    	logical_address2 = get_logical_address(address + sg[0].length);
    	sg[1].length = length - sg[0].length;
    	sg[1].page = virt_to_page(logical_address2);
    	sg[1].offset = offset_in_page(logical_address2);
    	no = 2;
    }
    
    crypto_digest_update(tfm, sg, no);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    dec_preempt_count();
#endif
}

static void peppercon_generate_serial_number(usb_driver_t *ud) {
    /* we calculate the serial number as an MD5 hash over all our descriptors */
    int i, j;
    char md5sum[16];
    char md5sum_text[33];
    struct crypto_tfm *tfm;
    
    SERIAL_DESCRIPTOR * serial_descriptor =
    	(SERIAL_DESCRIPTOR *) ud->StringDescriptors[PepperconDeviceDescriptor.iSerialNumber];
    int serial_length = sizeof(serial_descriptor->bString) / 2;
    
    if (serial_descriptor == NULL) {
    	return;
    }
    
    /* first, clear the serial number because the hash is also built on
       the current serial */
    for (i = 0, j = 0; i < serial_length; i++, j += 2) {
    	serial_descriptor->bString[j] = '0';
    	serial_descriptor->bString[j+1] = '\0';
    }
    
    /* initialize md5 calculator */
    tfm = crypto_alloc_tfm("md5", 0);
    
    if (tfm) {
    	/* calculate MD5 checksum */

    	/* initialize it */
    	memset(md5sum, 0, sizeof(md5sum));
    	crypto_digest_init(tfm);
    	
    	/* add all descriptors */
    	do_crypto_digest_update(tfm, ud->DeviceDescriptor,
    	    ((PUSB_DEVICE_DESCRIPTOR)ud->DeviceDescriptor)->bLength);
    	do_crypto_digest_update(tfm, ud->ConfigDescriptor,
    	    le16_to_cpu(((PUSB_CONFIGURATION_DESCRIPTOR)ud->ConfigDescriptor)->wTotalLength));
    	for (i = 0; i < MAX_STRING_DESCRIPTORS; i++) {
    	    if (ud->StringDescriptors[i]) {
    	    	do_crypto_digest_update(tfm, ud->StringDescriptors[i],
    	    	    ((PUSB_STRING_DESCRIPTOR)ud->StringDescriptors[i])->bLength);
    	    }
    	}
    	
    	/* finalize it */
    	crypto_digest_final(tfm, md5sum);

    	/* make string out of md5 sum */
    	for (i = 0; i < sizeof(md5sum); i++) {
    	    sprintf(&md5sum_text[i*2], "%02X", md5sum[i] & 0xff);
    	}
    	
    	/* now write the md5 sum as new serial number */
    	for (i = 0, j = 0; i < serial_length; i++, j += 2) {
    	    serial_descriptor->bString[j] = md5sum_text[i];
    	    serial_descriptor->bString[j+1] = '\0';
    	}
    	
    	UD("MD5 sum as USB serial number: %s\n", md5sum_text);
    	
    	crypto_free_tfm(tfm);
    }
    else {
    	/* error, use default serial number */
    	memcpy(serial_descriptor->bString,
    	    PEPPERCON_USB_DEFAULT_SERIAL_NUMBER,
    	    sizeof(serial_descriptor->bString));
    	
    	printk("Could not calculate MD5 sum, using default USB serial number.\n");
    }
}

/* Descriptor data */
#pragma pack(1)
static USB_DEVICE_DESCRIPTOR  PepperconDeviceDescriptor = {
    sizeof(USB_DEVICE_DESCRIPTOR), /* bLength */
    TYPE_DEVICE_DESCRIPTOR,        /* bDescriptorType */
    __constant_cpu_to_le16(0x0110),     /* bcdUSB USB Version 1.1 */
    0,                             /* bDeviceClass */
    0,                             /* bDeviceSubclass */
    0,                             /* bDeviceProtocol */
    PEPPERCON_USB_MAX_PACKET_SIZE, /* bMaxPacketSize Bytes */
    __constant_cpu_to_le16(0x14dd),     /* idVendor */
    __constant_cpu_to_le16(0x0002),     /* idProduct */
    __constant_cpu_to_le16(0x0001),     /* bcdDevice */
    1,                             /* iManufacturer String Index */
    2,                             /* iProduct String Index */
    3,                             /* iSerialNumber String Index */
    1                              /* bNumberConfigurations */
};

static USB_CONFIGURATION_DESCRIPTOR Configuration_Descriptor = {
	sizeof(USB_CONFIGURATION_DESCRIPTOR),             /* bLength */
	TYPE_CONFIGURATION_DESCRIPTOR,                    /* bDescriptorType */
	0,                                                /* wTotalLength, filled during runtime */
	0,                                                /* bNumInterfaces, filled during runtime */
	1,                                                /* bConfigurationValue */
	0,                                                /* iConfiguration String Index */
	0x80,                                             /* bmAttributes Bus Powered, No Remote Wakeup */
	0x32                                              /* bMaxPower, 100mA */
};

static USB_INTERFACE_DESCRIPTOR Interface_Descriptor_Mouse = {
	sizeof(USB_INTERFACE_DESCRIPTOR), /* bLength */
	TYPE_INTERFACE_DESCRIPTOR,        /* bDescriptorType */
	0,                                /* bInterfaceNumber, set at runtime */
	0,                                /* bAlternateSetting */
	1,                                /* bNumEndpoints */
	0x03,                             /* bInterfaceClass (Vendor specific) */
	0x00,                             /* bInterfaceSubClass: No Subclass */
	0x02,                             /* bInterfaceProtocol: Mouse */
	0                                 /* iInterface String Index */
};

static USB_HID_DESCRIPTOR HID_Descriptor_Mouse = {
	sizeof(USB_HID_DESCRIPTOR),       /* bLength */
	TYPE_HID_DESCRIPTOR,              /* bDescriptorType */
	__constant_cpu_to_le16(0x0101),        /* wHIDClassSpecComp */
	0x00,                             /* Country */
	0x01,                             /* bNumDescriptors */
	0x22,                             /* b1stDescType */
	__constant_cpu_to_le16(0),             /* w1stDescLength */
};

static USB_ENDPOINT_DESCRIPTOR Endpoint_Descriptor_Mouse = {
	sizeof(USB_ENDPOINT_DESCRIPTOR),  /* bLength */
	TYPE_ENDPOINT_DESCRIPTOR,         /* bDescriptorType */
	0x81 + 2 * PP_FEAT_USB_MASS_STORAGE_NO,       /* bEndpoint Address EP1 IN */
	0x03,                             /* bmAttributes - Interrupt */
	__constant_cpu_to_le16(0x0008),        /* wMaxPacketSize */
	0x0A                              /* bInterval */
};

static USB_INTERFACE_DESCRIPTOR Interface_Descriptor_Keyboard = {
	sizeof(USB_INTERFACE_DESCRIPTOR), /* bLength */
	TYPE_INTERFACE_DESCRIPTOR,        /* bDescriptorType */
	1,                                /* bInterfaceNumber, set at runtime */
	0,                                /* bAlternateSetting */
	1,                                /* bNumEndpoints */
	0x03,                             /* bInterfaceClass (Vendor specific) */
	0x01,                             /* bInterfaceSubClass: Boot Interface Subclass */
	0x01,                             /* bInterfaceProtocol: Keyboard */
	0                                 /* iInterface String Index */
};

static USB_HID_DESCRIPTOR HID_Descriptor_Keyboard = {
	sizeof(USB_HID_DESCRIPTOR),                    /* bLength */
	TYPE_HID_DESCRIPTOR,                           /* bDescriptortype */
	__constant_cpu_to_le16(0x101),                      /* wHIDClassSpecComp */
	0x00,                                          /* Country */
	0x01,                                          /* bNumDescriptors */
	0x22,                                          /* b1stDescType */
	__constant_cpu_to_le16(sizeof(KeyboardReportDescriptor)), /* w1stDescLength */
};

static USB_ENDPOINT_DESCRIPTOR Endpoint_Descriptor_Keyboard = {
	sizeof(USB_ENDPOINT_DESCRIPTOR),  /* bLength */
	TYPE_ENDPOINT_DESCRIPTOR,         /* bDescriptorType */
	0x82 + 2 * PP_FEAT_USB_MASS_STORAGE_NO,       /* bEndpoint Address EP1 IN */
	0x03,                             /* bmAttributes - Interrupt */
	__constant_cpu_to_le16(0x0008),        /* wMaxPacketSize */
	0x0A                              /* bInterval */
};

#ifdef PP_FEAT_MASS_STORAGE
static USB_INTERFACE_DESCRIPTOR Interface_Descriptor_MassStorage = {
	sizeof(USB_INTERFACE_DESCRIPTOR), /* bLength */
	TYPE_INTERFACE_DESCRIPTOR,        /* bDescriptorType */
	2,                                /* bInterfaceNumber, set at runtime */
	0,                                /* bAlternateSetting */
	2,                                /* bNumEndpoints */
	0x08,                             /* bInterfaceClass (Mass Storage) */
	0x06,                             /* bInterfaceSubClass: SCSI transparent command set */
	0x50,                             /* bInterfaceProtocol: BOT */
	0                                 /* iInterface String Index */
};

static USB_ENDPOINT_DESCRIPTOR Endpoint_Descriptor_MassStorage_In = {
	sizeof(USB_ENDPOINT_DESCRIPTOR),  /* bLength */
	TYPE_ENDPOINT_DESCRIPTOR,         /* bDescriptorType */
	0x81,                             /* bEndpoint Address EP IN, set at runtime */
	0x02,                             /* bmAttributes - Interrupt */
	__constant_cpu_to_le16(0x0040),        /* wMaxPacketSize */
	0x00                              /* bInterval */
};

static USB_ENDPOINT_DESCRIPTOR Endpoint_Descriptor_MassStorage_Out = {
	sizeof(USB_ENDPOINT_DESCRIPTOR),  /* bLength */
	TYPE_ENDPOINT_DESCRIPTOR,         /* bDescriptorType */
	0x02,                             /* bEndpoint Address EP OUT, set at runtime */
	0x02,                             /* bmAttributes - Interrupt */
	__constant_cpu_to_le16(0x0040),        /* wMaxPacketSize */
	0x00                              /* bInterval */
};
#endif /* PP_FEAT_MASS_STORAGE */

static const LANGID_DESCRIPTOR  LANGID_Descriptor = { /* LANGID String Descriptor 0 */
    sizeof(LANGID_DESCRIPTOR),                 /* bLength */
    TYPE_STRING_DESCRIPTOR,                    /* bDescriptorType */
    __constant_cpu_to_le16(0x0409)                  /* LANGID US English */
};

#ifdef PRODUCT_MSIDC
static const MANUFACTURER_DESCRIPTOR  Manufacturer_Descriptor = { /* ManufacturerString 1 */
    sizeof(MANUFACTURER_DESCRIPTOR),                       /* bLenght */
    TYPE_STRING_DESCRIPTOR,                                /* bDescriptorType */
    "M\0S\0I\0 \0"                                         /* ManufacturerString in UNICODE */
};
#else /* !PRODUCT_MSIDC */
static const MANUFACTURER_DESCRIPTOR  Manufacturer_Descriptor = { /* ManufacturerString 1 */
    sizeof(MANUFACTURER_DESCRIPTOR),                       /* bLenght */
    TYPE_STRING_DESCRIPTOR,                                /* bDescriptorType */
#if defined(OEM_BELKIN)
    "B\0e\0l\0k\0i\0n\0 \0 \0 \0 \0 \0 \0"                 /* ManufacturerString in UNICODE */
#else
    "P\0e\0p\0p\0e\0r\0c\0o\0n\0 \0A\0G\0"                 /* ManufacturerString in UNICODE */
#endif
};
#endif /* !PRODUCT_MSIDC */

static const PRODUCT_DESCRIPTOR  Product_Descriptor = {  /* ManufacturerString 1 */
    sizeof(PRODUCT_DESCRIPTOR),                   /* bLenght */
    TYPE_STRING_DESCRIPTOR,                       /* bDescriptorType */
    "M\0u\0l\0t\0i\0D\0e\0v\0i\0c\0e\0\0\0"         /* ManufacturerString in UNICODE */
};

static const SERIAL_DESCRIPTOR  Serial_Descriptor = {        /* SerialString 2 */
    sizeof(SERIAL_DESCRIPTOR),                        /* bLenght */
    TYPE_STRING_DESCRIPTOR,                           /* bDescriptorType */
    PEPPERCON_USB_DEFAULT_SERIAL_NUMBER
 };
#pragma pack()

