#include <fcntl.h>
#include <unistd.h>
#include <sys/reboot.h>
#include <sys/ioctl.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <pp/base.h>
#include <pp/i2c.h>
#include <pp/cfg.h>
#include <pp/firmware.h>
#include <pp/um.h>
#include <liberic_config.h>
#include <liberic_notify.h>
#include <liberic_misc.h>
#include <liberic_net.h>
#include <liberic_pthread.h>
#include <openssl/hmac.h>

#if defined(PP_FEAT_HARDWAREREVISION)
# include <pp_gpio.h>
# include <lara.h>
#endif /* PP_FEAT_HARDWAREREVISION */

#ifdef PP_FEAT_RPCCFG
# include <pp/rpc_ripc.h>
#endif /* PP_FEAT_RPCCFG */

#define LM75_FILE_PATH			"/proc/driver/lm75"
#define BOARD_TYPE_FILE_PATH		"/proc/sys/kernel/boardtype"
#define PRODUCT_TYPE_FILE_PATH		"/proc/sys/kernel/producttype"

extern int fw_update_sem_id;
extern struct sembuf fw_update_lock[2];
extern struct sembuf fw_update_unlock[1];

static int board_reset_running = 0;

/*
 * Programs using liberic_misc should not be forced to link
 * liberic_net that needs openssl
 */
PP_SYM_WEAK int
eric_net_close(BIO * bio UNUSED, eric_net_close_mode_t close_mode UNUSED)
{
    return -1;
}

/*
 * resetter() - thread function that performs the actual reset
 */
static void *
resetter(void * arg UNUSED)
{
    usleep(3000000);
    sync();
    reboot(RB_AUTOBOOT);

    /* we shouldn't reach this, but just in case ... */
    eric_config_avoid_flushing(0);
    pp_firmware_erla_release_update_lock_no_ctx();
    board_reset_running = 0;

    pthread_exit(NULL);
}

int
eric_misc_reset_is_running(void) 
{
    return board_reset_running;
}
/*
 * eric_misc_trigger_board_reset() - triggers resetting the board
 *
 *   parameters:	session - the session context
 *			bio     - a bio that gets freed
 *
 *   return value:	 0 - no error
 *			-1 - permission denied
 *			-2 - firmware upgrade running
 *			-3 - internal error
 */
int
eric_misc_trigger_board_reset(eric_session_int_id_t session)
{
    char logmsg[PP_NOTIFY_MAX_MSG_LEN+1];
    static pthread_t reset_thread;
    int do_log = 1;

    if (session) {
	const char * user = eric_session_get_user(session);
	if(PP_ERR == pp_um_user_has_permission(user, "reset",
                                               pp_acl_raasip_yes_str)){
	    eric_notify_security_violation(session);
	    return -1;
	}
	snprintf(logmsg, sizeof(logmsg), "Board Reset performed by user '%s'.", user);
    } else {
	snprintf(logmsg, sizeof(logmsg), "Board Reset performed by a background task.");
    }

    /* prevent restarting the e-RIC during a firmware upgrade */
    if (PP_SUCCED(pp_firmware_erla_test_and_set_update_lock_no_ctx())) {
	  board_reset_running = 1;
	  eric_config_flush();	
	  eric_config_avoid_flushing(1);
#if defined(PP_FEAT_RPCCFG)
	  do_log = pp_rpc_am_i_master();
#endif
	  if (do_log) eric_notify_post_event(logmsg, "device", PP_NOTIFY_EVENT_GENERIC);
	  if (eric_pthread_create(&reset_thread, 1 /*detached*/, 64 * 1024, resetter, NULL)) {
	      pp_log("%s(): Cannot create reset thread.\n", ___F);
	      return -3;
	  }
	  return 0;
    } else if (errno == EAGAIN) {
	return board_reset_running ? 0 : -2;
    } else {
	return -3;
    }
}

char *
eric_misc_get_boardname(void)
{
    char *boardname = NULL;

    if (PP_FAILED(pp_cfg_get(&boardname, "device.devname"))) {
	boardname = strdup("");
    }
        
    return boardname;
}

char *
eric_misc_get_hardware_rev_str(void)
{
#define MAX_HWID_STR_LEN 4
    char * revname = NULL;
    unsigned char rev = eric_misc_get_hardware_rev_int();

    if (rev != 0) {
	revname = malloc(MAX_HWID_STR_LEN);
	snprintf(revname, MAX_HWID_STR_LEN, "%02X", rev);
    }
    
    return revname;
#undef MAX_HWID_STR_LEN
}

unsigned char
eric_misc_get_hardware_rev_int()
{
    unsigned char rev = 0;
#if defined(PP_FEAT_HARDWAREREVISION)
    ioctl(eric_fd, ERICIOCGETHARDWAREREVISION, &rev);
#endif /* PP_FEAT_HARDWAREREVISION */
    return rev;
}
 
static int
eric_misc_xfer_panel_id(int do_set, unsigned int panel_id)
{
    int ret = 0;
    /* be silent, gcc! */
    (void)do_set;
    (void)panel_id;
#ifdef PRODUCT_ERIC2
    u_char i2c_handle;
    int error;
    /* 
     * Short functional description: We access a chip (DS3904) via I2C that
     * contains three digitally tunable resistors that are between the LTGIOx
     * and the ground. The LTGIOx have pull-ups additionally. Each resistor is
     * accessed via a register in the chip. The tuning range is 0-20kB
     * (0x00-0x7f). The special value 0x80 is the high impedance state. We only
     * use 0x00 and 0x80. Therefore the chip is operating as a digital switch.
     * If we write a 0x80 then the LTGIOx is high, if we write 0x00 it is low.
     *
     * NOTE: Between write cycles we have to wait ~20ms!
     */

    i2c_handle = pp_i2c_open("ibm-0", &error);
    if (error == PP_I2C_ERROR) {
	pp_log_err("%s: error opening i2c device\n", ___F);
	goto bail;
    }
    
#  define PANEL_ID_SWITCH_I2C_ADDR	0x50
    ret = -1;

    if (do_set) {		
	pp_i2c_tx_byte_data(i2c_handle, PANEL_ID_SWITCH_I2C_ADDR, 0xf8,
			    (panel_id & 1) ? 0x80 : 0, &error);
	if (error == PP_I2C_ERROR) { goto bail; }
	usleep(20000);
	pp_i2c_tx_byte_data(i2c_handle, PANEL_ID_SWITCH_I2C_ADDR, 0xf9,
			    (panel_id & 2) ? 0x80 : 0, &error);
	if (error == PP_I2C_ERROR) { goto bail; }
	usleep(20000);
	pp_i2c_tx_byte_data(i2c_handle, PANEL_ID_SWITCH_I2C_ADDR, 0xfa,
			    (panel_id & 4) ? 0x80 : 0, &error);
	if (error == PP_I2C_ERROR) { goto bail; }
	usleep(20000);

	ret = 0;
    } else {
	u_char r0 = pp_i2c_rx_byte_data(i2c_handle, PANEL_ID_SWITCH_I2C_ADDR, 0xf8, &error);
	if (error == PP_I2C_ERROR) { goto bail; }
	u_char r1 = pp_i2c_rx_byte_data(i2c_handle, PANEL_ID_SWITCH_I2C_ADDR, 0xf9, &error);
	if (error == PP_I2C_ERROR) { goto bail; }
	u_char r2 = pp_i2c_rx_byte_data(i2c_handle, PANEL_ID_SWITCH_I2C_ADDR, 0xfa, &error);
	if (error == PP_I2C_ERROR) { goto bail; }
	ret = (r2 ? 4 : 0) | (r1 ? 2 : 0) | (r0 ? 1 : 0);
    }   
 bail:
    if (ret == -1) {
	pp_log_err("%s: error communicating with i2c device\n", ___F);
    }
#endif /* PRODUCT_ERIC2 */ 
    return ret;
}

int
eric_misc_get_panel_id()
{
    return eric_misc_xfer_panel_id(0, 0);
}

int
eric_misc_set_panel_id(unsigned int panel_id)
{
    return eric_misc_xfer_panel_id(1, panel_id);
}

int
pp_misc_get_kvm_node(void)
{
#ifdef PRODUCT_RIPCKIMXN
    static int have_eth0_phy = -1;
    static int have_eth1_phy = -1;

    if (have_eth0_phy == -1) have_eth0_phy = pp_eth_have_phy("eth0");
    if (have_eth1_phy == -1) have_eth1_phy = pp_eth_have_phy("eth1");

    if (have_eth0_phy && !have_eth1_phy) {
	return 0;
    } else if (!have_eth0_phy && have_eth1_phy) {
	return 1;
    } else {
	pp_log("%s(): Cannot determine channel - assume 0.\n", ___F);
	return 0;
    }
#else /* !PRODUCT_RIPCKIMXN */
    return 0;
#endif /* !PRODUCT_RIPCKIMXN */
}

int
pp_misc_get_kvm_node_cnt(void)
{
    int ret;
#ifdef PRODUCT_RIPCKIMXN
    ret = 2;
#else /* !PRODUCT_RIPCKIMXN */
    ret = 1;
#endif /* !PRODUCT_RIPCKIMXN */
    return ret;
}
