/**
 * Copyright 2001 Peppercon AG
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 *
 * Description: That is the actual parser using the keytable
 *              in the keytable.c file. The search algorithm
 *              is pretty stupid for the moment, as we just
 *              do a linear search. However this save us from
 *              building up hashtables and we have around
 *              120 entries only, that is not that much
 *
 *              The parser tries ot parse strings with the
 *              following syntax:
 *
 *                  <keycode>[+|-|>|[*]<keycode>]*
 *
 *              Any keycode and the delemitters are returned
 *              in the syntax tree, what is a simple array in
 *              in our case... ;-)
 */
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <pp/base.h>
#include <pp/tokenizer.h>
#include <liberic_misc.h>
#include "keytable.c" // only needed here, so just include

/* ------ some private methods ----------- */

/* parses a complete string to an array of key numbers */
static vector_t* eric_parse_key_string(char* string, char** errortok,
				       int include_delis);

/* parses a single symbol to the associated keynumber */
static int eric_parse_key_sym(char* string);

/*
 * forms the required applet parameter from a key description string
 * and its key numvber representation
 */
static char* eric_get_key_seq(const vector_t* keystrokes);

/* ------- public methods implemented --------- */

/**
 * transforms a complete button key definition to the
 * associated sequence of key numbers that can be given
 * to the applet for a button key, the key string is going
 * to be destroyed, so it needs to be duplicated in case
 * this is not tolerable
 */
char *
eric_misc_get_key_appletparam(const char * key, char** errortok, int include_delis)
{
    char *p, *k, *key_dup;
    vector_t * keyvec;

    assert(key);

    key_dup = strdup(key);
    k = pp_trim_string(key_dup);
    
    // check whether there is a prefix,
    if ((p = strchr(k, ' ')) != NULL) {
	// yes there is, check for confirm
	if (!strncmp(k, KEYCONFIRM_STR, strlen(KEYCONFIRM_STR))) {
	    k = pp_trim_string(p+1); // correct prefix, so use rest as key string
	}
    }
    if ((keyvec = eric_parse_key_string(k, errortok, include_delis)) != NULL) {
	p = eric_get_key_seq(keyvec);
	vector_delete(keyvec);
	//printf("keyappletparam: key '%s' became '%s'\n", key_dup, p);
    } else {
	p = NULL;
    }

    free(key_dup);

    return p;
}

/**
 * similar to above, but returns a sequence of keypresses/releases
 * as a vector of keycodes, caller has to free this
 */
vector_t *
eric_misc_key_get_keysequence(const char* key, char** errortok)
{
    char *key_dup;
    vector_t* keyvec = NULL;

    assert(key);

    key_dup = strdup(key);
    pp_trim_string(key_dup);
    
    keyvec = eric_parse_key_string(key_dup, errortok, 1);

    free(key_dup);

    return keyvec;
}

/* ------- private methods implemented --------- */

/**
 * parses a complete string to an array of key numbers
 * and delemitters. There is a small logic applied
 * for the delemitters, see head of the file for syntax!
 * we don't know the count of key numbers so add the found
 * number to a vector that increases size dynamically
 * in case of an error we return the error token in
 * errortok
 */
static vector_t *
eric_parse_key_string(char* string, char** errortok, int include_delis)
{
    tokenizer_t tok;
    char* token, ct, lastdeli = '\0';
    vector_t* vec = vector_new(NULL, 16, NULL);
    tokenizer_new(&tok, string, "+-*>", include_delis ? TOKENIZER_DELI_RET : TOKENIZER_DELI_NORET);
    while(tokenizer_has_next(&tok)) {
	int key;
	token = pp_trim_string(tokenizer_get_next(&tok));
	if(0 == strlen(token)) continue; // ignore empty tokens
	switch(ct = token[0]) {
	  case '+':
	  case '-':
	      if(lastdeli != '\0') goto error;
	      lastdeli = ct;
	      vector_add_int(vec, ct=='+' ? KEYDELI_ADD : KEYDELI_SEPERATE);
	      break;
	  case '>':
	      if(lastdeli == '+' || lastdeli == '-') goto error;
	      lastdeli = ct;
	      vector_add_int(vec, KEYDELI_RELEASE_LAST);
	      break;
	  case '*':
	      if(lastdeli == '\0') goto error;
	      lastdeli = ct;
	      vector_add_int(vec, KEYDELI_PAUSE);
	      break;
	  default:
	      lastdeli = '\0';
	      if((key = eric_parse_key_sym(token)) < 0) goto error;
	      vector_add_int(vec, key);
	}
    }
    return vec;
 error:
    if (errortok) *errortok = token ? strdup(token) : NULL;
    vector_delete(vec);
    return NULL;
}

/**
 * parses a single symbol to the associated keynumber
 * currently we lineary search the whole table,
 * that's quite stupid but OK, as the table is not that big
 */
static int
eric_parse_key_sym(char* key)
{
    int i, k;
    char* t = pp_strtoupper(strdup(key));
    for(i=0; i<KEYNO; ++i) {
	for(k=0; k<MAXKEYSYMS; ++k) {
	    const char* s = keytable[i][k];
	    if(NULL == s) break;
	    if(!strcmp(s, t)) goto found;
	}
    }
    i = -1; // we havn't found anyting
 found:
    free(t);
    return i;
}

/**
 * forms the required applet parameter from a key description string
 * and its key numvber representation,
 * the return string is newly allocated and has to be freed by the
 * calles
 */
static char *
eric_get_key_seq(const vector_t* vec)
{
    // "<keynr>+"
    unsigned int i, l = 3*vector_size(vec) + 1;
    char buf[3], *param = (char*)malloc(l * sizeof(char));
    assert(param);
    *param = '\0';
    for(i=0; i<vector_size(vec); ++i) {
	sprintf(buf, "%2.2x ", vector_get_int(vec, i));
	strcat(param, buf);
    }
    return param;
}

// little standalone test just for fun
#ifdef KEY_PARSER_TEST

int
main(int argc, char** argv)
{
    char *key, *code, *error;
    printf("button key parser test... \n");
    key = argv[1];
    //while(1) // memory leak test
    { 
        printf("parsing string: %s\n", key);
        if (NULL == (code = eric_misc_get_key_appletparam(key, &error))) {
	    printf("error parsing '%s': near token '%s'\n", key, error);
	    return 1;
        }
        printf("applet key:     %s\n", key);
        printf("applet code:    %s\n", code);
        free(code);
    }
    return 0;
}
#endif
