#include <stdio.h>
#include <pp/bio.h>
#if defined(PP_FEAT_DRIVE_REDIRECTION)
#include <pp/usb.h>
#endif /* PP_FEAT_DRIVE_REDIRECTION */
#if defined(PP_FEAT_REMOTE_CONSOLE)
#include <pp/rfb.h>
#endif /* PP_FEAT_REMOTE_CONSOLE */
#include <liberic_webs.h>
#include <liberic_net.h>

void
eric_net_demuxer_handle_connection(pp_net_conn_data_t * conn_data)
{
    int r;
    /*
     * In case of an connect to port 80/443 we have the following BIO chain:
     *
     *     Port  80: socket_bio
     *     Port 443: ssl_bio - rfb_check_bio - socket_bio
     *
     * If we detect the RFB protocol in the rfb_check_bio than SSL is not
     * in use and we remove the ssl_bio. Otherwise we push another
     * rfb_check_bio in front of ssl_bio:
     *
     *     Port  80: rfb_check_bio2 - socket_bio
     *     Port 443: rfb_check_bio2 - ssl_bio - rfb_check_bio - socket_bio
     *
     * If we detect the RFB protocol in the rfb_check_bio2 than we
     * create a new RFB client else we assume a HTTP request.
     */

    /*
     * In case of an connect to the configured HTTPS port we have the
     * following BIO chain:
     *
     *     ssl_bio - rfb_check_bio - socket_bio
     *
     * If we detect the RFB protocol in the rfb_check_bio than SSL is not
     * in use and we remove the ssl_bio. Otherwise we push another
     * rfb_check_bio in front of ssl_bio:
     *
     *     rfb_check_bio2 - ssl_bio - rfb_check_bio - socket_bio
     *
     * If we detect the RFB protocol in the rfb_check_bio2 than we
     * create a new RFB client else we assume a HTTP request.
     */
    if ((r = pp_bio_do_rfb_check(conn_data->bio)) == BIO_PP_PROTO_RFB || r == BIO_PP_PROTO_MSP) {
	BIO * b = BIO_pop(conn_data->bio);
	BIO_free(conn_data->bio);
	conn_data->ssl = 0;
	conn_data->bio = b;
    } else if (r == BIO_PP_PROTO_HTTPS) {
	BIO * rfb_check_bio = pp_bio_new_rfb_check(pp_rfb_check_cb);
	conn_data->ssl = 1;
	conn_data->bio = BIO_push(rfb_check_bio, conn_data->bio);
    } else {
	eric_net_close(conn_data->bio, NOWAIT);
	free(conn_data);
	return;
    }
    if ((r = pp_bio_do_rfb_check(conn_data->bio)) == BIO_PP_PROTO_HTTPS) {
	conn_data->protocol_type = PP_NET_PROTOCOL_TYPE_HTTPS;
	eric_webs_handle_connection(conn_data);
#if defined(PP_FEAT_REMOTE_CONSOLE)
    } else if (r == BIO_PP_PROTO_RFB) {
	conn_data->protocol_type = PP_NET_PROTOCOL_TYPE_RFB;
	pp_rfb_handle_connection(conn_data);
#endif
#if defined(PP_FEAT_DRIVE_REDIRECTION)
    } else if (r == BIO_PP_PROTO_MSP) {
	conn_data->protocol_type = PP_NET_PROTOCOL_TYPE_MSP;
	pp_usb_msp_handle_connection(conn_data);
#endif
    } else {
	eric_net_close(conn_data->bio, NOWAIT);
	free(conn_data);
    }
}
