/**
 * esb2_tpt_net.c
 *
 * Network connection handler for Intel's ESB2 TCP Pass-Through sideband
 * interfave via FML
 * 
 * (c) 2005 Peppercon AG, 2005/07/26, Ralf Guenther <rgue@peppecon.de>
 */

#include <pthread.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/poll.h>
#include <openssl/ssl.h>

#include <pp/ipmi.h>
#include <pp/bio.h>
#include <pp/cfg.h>
#include <esb2_tpt.h>
#include <pp/rfb.h>
#include <pp/dropbear.h>
#include <liberic_pthread.h>
#include <liberic_notify.h>

#include "esb2_tpt_net.h"

#define DEV_FILE_NAME   "/dev/esb2_tpt"
#define ESB2_TPT_CHAN   1
#define ESB2_I2C_DELAY  250000 // usecs

static struct {
    pp_net_conn_data_t *conn_data;
    int fd;
    int running;
    int ssh_running;
    int last_tpt_status;
    pthread_t thread;
    pthread_t ssh_thread;
    int port;
} esb2 = {
    .conn_data = NULL,
    .fd = -1,
    .running = 0,
    .ssh_running = 0,
    .last_tpt_status = -2, // == uninit
};

static void* esb2_tpt_handle(void* ctx);
static void* esb2_tpt_ssh_thread(void* ctx);

static int esb2_tpt_open_connection(void);
static void esb2_tpt_cleanup_connection(void);

static int enable_tpt(void)
{
    pp_ipmi_parameter_t p = { .is_cmdline = 0, };
    pp_ipmi_return_t r;
    int e, enabled = 0, standby = 0, multi_push = 0;
    char *proto = NULL;

    // contact BMC via IPMB
    usleep(ESB2_I2C_DELAY);
    e = 0;
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_BMC, PP_IPMI_BMC_SUBCMD_INFO, &p, &r, &e, NULL))) {
        pp_log("ASMI: Failed to find Intel BMC on IPMB (%s)\n", pp_ipmi_get_error_string(e));
        return PP_ERR;
    } else {
        pp_log("ASMI: Intel BMC found on IPMB: manu=%ld prod=%ld ipmiver=%s\n",
                    r.data.bmc_info.manufacturer_id,
                    r.data.bmc_info.product_id,
                    r.data.bmc_info.ipmi_version.buf);
    }

    // get TPT config
    usleep(ESB2_I2C_DELAY);
    e = 0;
    p.data.oem_intel_get_tpt_cfg.chan = ESB2_TPT_CHAN;
    p.data.oem_intel_get_tpt_cfg.mask = PP_IPMI_OEM_INTEL_TPT_CFG_ENABLES
                                      | PP_IPMI_OEM_INTEL_TPT_CFG_PORT
                                      | PP_IPMI_OEM_INTEL_TPT_CFG_FML_OP_MODE
                                      | PP_IPMI_OEM_INTEL_TPT_CFG_DEF_CONN;
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_OEM_INTEL_TPT, PP_IPMI_OEM_INTEL_GET_TPT_CFG, &p, &r, &e, NULL))) {
        pp_log("ASMI: Failed to get TPT config (err=%d)\n", e);
        return PP_ERR;
    } else {
        pp_ipmi_oem_intel_get_tpt_cfg_result_t *cfg = &r.data.oem_intel_get_tpt_cfg;
        pp_log("ASMI: TPT cfg en=%d sb=%d defport=%d multipush=%d defproto=%s\n",
            cfg->cfg.enabled,
            cfg->cfg.in_standby,
            cfg->cfg.port,
            cfg->cfg.fml_multi_push_mode,
            cfg->cfg.def_ssl ? "ssl" : "tcp");

        if (cfg->mask & PP_IPMI_OEM_INTEL_TPT_CFG_ENABLES) {
            enabled = cfg->cfg.enabled;
            standby = cfg->cfg.in_standby;
        } else pp_log("ASMI: Failed to get TPT config value 'TPT enabled'\n");

        if (cfg->mask & PP_IPMI_OEM_INTEL_TPT_CFG_FML_OP_MODE) multi_push = cfg->cfg.fml_multi_push_mode;
        else pp_log("ASMI: Failed to get TPT config value 'FML Operating Mode'\n");
    }

    // enable TPT (if not already done)
    if (!enabled || !standby) {
        usleep(ESB2_I2C_DELAY);
        e = 0;
        p.data.oem_intel_set_tpt_cfg.chan = ESB2_TPT_CHAN;
        p.data.oem_intel_set_tpt_cfg.mask = PP_IPMI_OEM_INTEL_TPT_CFG_ENABLES;
        p.data.oem_intel_set_tpt_cfg.cfg.enabled = 1;
        p.data.oem_intel_set_tpt_cfg.cfg.in_standby = 1;
        if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_OEM_INTEL_TPT, PP_IPMI_OEM_INTEL_SET_TPT_CFG, &p, &r, &e, NULL))) {
            pp_log("ASMI: Failed to enable TPT (err=%d)\n", e);
            return PP_ERR;
        } else {
            pp_log("ASMI: TPT enabled\n");
        }
    }

    // activate FML Multiple Push (if not already done)
    if (!multi_push) {
        usleep(ESB2_I2C_DELAY);
        e = 0;
        p.data.oem_intel_set_tpt_cfg.chan = ESB2_TPT_CHAN;
        p.data.oem_intel_set_tpt_cfg.mask = PP_IPMI_OEM_INTEL_TPT_CFG_FML_OP_MODE;
        p.data.oem_intel_set_tpt_cfg.cfg.fml_multi_push_mode = 1;
        if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_OEM_INTEL_TPT, PP_IPMI_OEM_INTEL_SET_TPT_CFG, &p, &r, &e, NULL))) {
            pp_log("ASMI: Failed to enable FML Multiple Push mode (err=%d)\n", e);
            return PP_ERR;
        } else {
            pp_log("ASMI: FML Multiple Push mode enabled\n");
        }
    }

    pp_cfg_get(&proto, "network.asmi_tpt_protocol");
    if (pp_strcmp_safe(proto, "tcp") == 0) {
        p.data.oem_intel_tpt_listen.protocol = PP_IPMI_OEM_INTEL_PROTO_TCPIP_ONLY;
    } else if (pp_strcmp_safe(proto, "ssl") == 0) {
        p.data.oem_intel_tpt_listen.protocol = PP_IPMI_OEM_INTEL_PROTO_SSLTLS;
    } else {
        p.data.oem_intel_tpt_listen.protocol = PP_IPMI_OEM_INTEL_PROTO_DEFAULT;
    }

    // activate TPT
    usleep(ESB2_I2C_DELAY);
    e = 0;
    p.data.oem_intel_tpt_listen.chan = ESB2_TPT_CHAN;
    p.data.oem_intel_tpt_listen.port = esb2.port;
    p.data.oem_intel_tpt_listen.listen = 1;
    // p.data.oem_intel_tpt_listen.protocol = see above
    p.data.oem_intel_tpt_listen.cipher_suite = 0;
    p.data.oem_intel_tpt_listen.linger = PP_IPMI_OEM_INTEL_LINGER_DEFAULT;
    p.data.oem_intel_tpt_listen.coalescing = PP_IPMI_OEM_INTEL_COALESCING_DEFAULT;
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_OEM_INTEL_TPT, PP_IPMI_OEM_INTEL_TPT_LISTEN, &p, &r, &e, NULL))) {
        pp_log("ASMI: Failed to activate TPT (err=%d)\n", e);
        return PP_ERR;
    } else {
        pp_log("ASMI: TPT activated\n");
    }

    // get TPT status
    usleep(ESB2_I2C_DELAY);
    e = 0;
    p.data.oem_intel_get_tpt_status.chan = ESB2_TPT_CHAN;
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_OEM_INTEL_TPT, PP_IPMI_OEM_INTEL_GET_TPT_STATUS, &p, &r, &e, NULL))) {
        pp_log("ASMI: Failed to get TPT status (err=%d)\n", e);
        return PP_ERR;
    } else {
        pp_ipmi_oem_intel_get_tpt_status_result_t *status = &r.data.oem_intel_get_tpt_status;
        pp_log("ASMI: TPT status=%.2x port=%d ip=%d.%d.%d.%d\n",
                    status->status,
                    status->port,
                    status->remote_ip[0],
                    status->remote_ip[1],
                    status->remote_ip[2],
                    status->remote_ip[3]);
    }

    return PP_SUC;
}

static int esb2_tpt_open_connection(void)
{
    // open device
    esb2.fd = open(DEV_FILE_NAME, O_RDWR);
    if (esb2.fd == -1) {
        pp_log("ASMI: Failed to open FML device file '%s' (err=%d).\n", DEV_FILE_NAME, errno);
        goto fail;
    }

    esb2.conn_data = (pp_net_conn_data_t*)malloc(sizeof(pp_net_conn_data_t));
    if (esb2.conn_data == NULL) {
	goto fail;
    }

    esb2.conn_data->protocol_type = PP_NET_PROTOCOL_TYPE_ASMI_TPT,
    esb2.conn_data->bio = pp_bio_new_esb2_tpt(esb2.fd, BIO_NOCLOSE);

    if (!esb2.conn_data->bio) {
        pp_log("ASMI: Failed to create BIO.\n");
        goto fail;
    }

    return PP_SUC;

fail:
    esb2_tpt_cleanup_connection();
    return PP_ERR;
}

static void esb2_tpt_cleanup_connection(void)
{
    if (esb2.conn_data) {
        if (esb2.conn_data->bio) BIO_free(esb2.conn_data->bio);
        free(esb2.conn_data);
        esb2.conn_data = NULL;
    }
    if (esb2.fd != -1) {
        close(esb2.fd);
        esb2.fd = -1;
    }
}

int esb2_tpt_init(void)
{
    int enabled;
    u_short port = 0;

    pp_cfg_is_enabled(&enabled, "network.asmi_tpt_enabled");
    if (enabled) pp_cfg_get_ushort(&port, "network.asmi_tpt_port");
    if (port == 0) return 0;
    esb2.port = port;

    // register new log category
    if (PP_FAILED(eric_notify_register_log_object("asmi", "ASMI", PP_NOTIFY_EVENT_ALL_ENABLE))) {
        pp_log("ASMI: Failed to register log object 'ASMI'.\n");
    }

    if (esb2.conn_data) esb2_tpt_cleanup();

    if (PP_FAILED(esb2_tpt_open_connection())) {
	goto fail;
    }

    // setup BMC via IPMB
    if (PP_FAILED(enable_tpt())) { // logging will be done inside
	pp_log("ASMI: Failed to enable TPT.\n");
	// don't close the file descriptore here, we might be in an evalboard here
	// which needs the TPT kernel module for testing
	goto fail_no_close;
    }

    // start listener thread
    esb2.running = 1;
    if (eric_pthread_create(&esb2.thread, 0, 128 * 1024,
                            esb2_tpt_handle,
                            NULL) != 0) {
        pp_log("ASMI: Cannot create listener thread.\n");
        esb2.running = 0;
        goto fail;
    }

    return 0;

fail:
    esb2_tpt_cleanup();
fail_no_close:
    return -1;
}

int esb2_tpt_cleanup(void)
{
    if (esb2.ssh_running) {
        esb2.ssh_running = 0;
        if (pthread_join(esb2.ssh_thread, NULL) != 0) {
            pp_log("ASMI: Failed to stop ssh thread (err=%d).\n", errno);
        }
    }
 
    if (esb2.running) {
        esb2.running = 0;
        if (pthread_join(esb2.thread, NULL) != 0) {
            pp_log("ASMI: Failed to stop listener thread (err=%d).\n", errno);
        }
    }

    esb2_tpt_cleanup_connection();
    return 0;
}

int eric_net_tpt_get_status(void) // declared and exported in liberic_net.h
{
    return esb2.last_tpt_status;
}

static void* esb2_tpt_ssh_thread(void* ctx UNUSED)
{
    if (esb2.conn_data) {
    	pp_dropbear_handle_connection(esb2.conn_data);
    }
    return 0;
}

static void* esb2_tpt_handle(void* ctx UNUSED)
{
    int res, status, retries;
    esb2_tpt_ioctl_tpt_stat_t tpt_stat = { .chan = ESB2_TPT_CHAN, };

    while (esb2.running) {
	if (!esb2.ssh_running) {
	    //if the ssh thread isn't running (and thus there is no connection yet), poll for incoming data
	    struct pollfd pfd = { .fd = esb2.fd, .events = POLLIN, };
	    poll(&pfd, 1, 1000);

	    if (pfd.revents & POLLIN) {
		// a client is connected, proceed with conn establishment
		if (!esb2.ssh_running) {
		    esb2.ssh_running = 1;
		    if (eric_pthread_create(&esb2.ssh_thread, 0, 128 * 1024, esb2_tpt_ssh_thread, NULL) != 0) {
			pp_log("ASMI: Cannot create SSH thread.\n");
			esb2.ssh_running = 0;
			esb2.running = 0;
			continue;
		    }
		}
	    }
	} else {
	    //if the ssh thread already runs, do not poll to avoid CPU time waste
	    usleep(1000000);
	}

	retries = 3;
	do {
	    res = esb2_tpt_ioctl_get_tpt_stat(esb2.fd, &tpt_stat);
	    retries--;
	} while ((res < 0) && (retries > 0));

	if (res < 0) {
	    pp_log("ASMI: failed to get TPT status\n");
	}
	status = res < 0 ? -1 : tpt_stat.status;

	if (status != esb2.last_tpt_status) {
	    const char *msg;

	    pp_log("ASMI: TPT state changed (old state: %i, new state: %i, ip=%d.%d.%d.%d)\n",
		    esb2.last_tpt_status, status, tpt_stat.remote_ip[0], tpt_stat.remote_ip[1],
		    tpt_stat.remote_ip[2], tpt_stat.remote_ip[3]);

	    switch (status) {
		case -1:
		    msg = "TPT not available"; break;
		case ESB2_TPT_STAT_CLOSED:
		    msg = "TPT connection closed"; break;
		case ESB2_TPT_STAT_LISTENING:
		    msg = "TPT listening"; break;
		case ESB2_TPT_STAT_CONNECTED:
		    msg = "TPT connection active";
		    if (!esb2.ssh_running) {
			esb2.ssh_running = 1;
			if (eric_pthread_create(&esb2.ssh_thread, 0, 128 * 1024, esb2_tpt_ssh_thread, NULL) != 0) {
			    pp_log("ASMI: Cannot create SSH thread.\n");
			    esb2.ssh_running = 0;
			    esb2.running = 0;
			    continue;
			}
		    }
		    break;
		case ESB2_TPT_STAT_RECV_TIMEOUT:
		    msg = "TPT terminated (receive inactivity timeout)"; break;
		case ESB2_TPT_STAT_TRANSM_TIMEOUT:
		    msg = "TPT terminated (transmit inactivity timeout)"; break;
		case ESB2_TPT_STAT_TERMINATED:
		    msg = "TPT terminated by remote console"; break;
		case ESB2_TPT_STAT_STANDBY:
		    msg = "TPT closed due to standby power state"; break;
		case ESB2_TPT_STAT_LOST_PACKET:
		    msg = "TPT packets lost"; break;
		case ESB2_TPT_STAT_DISABLED:
		    msg = "TPT disabled"; break;
		case ESB2_TPT_STAT_AUTH_FAILED:
		    msg = "TPT terminated (SSL authentication failure)"; break;
		case ESB2_TPT_STAT_INTEGRITY_FAILED:
		    msg = "TPT terminated (SSL integrity failure)"; break;
		case ESB2_TPT_STAT_DECRYPT_FAILED:
		    msg = "TPT terminated (SSL decryption failure)"; break;
		case ESB2_TPT_STAT_NO_LINK:
		    msg = "TPT no network link"; break;
		default:
		    msg = "TPT unknown state";
	    }
	    eric_notify_post_event(msg, "asmi", PP_NOTIFY_EVENT_GENERIC);

	    if ((esb2.last_tpt_status == ESB2_TPT_STAT_CONNECTED) && esb2.ssh_running) {
		//connection dropped, notify the kernel module so it can send EOF to the ssh server
		esb2_tpt_ioctl_connlost(esb2.fd);
		if (pthread_join(esb2.ssh_thread, NULL) != 0) {
		    pp_log("ASMI: Failed to stop ssh thread (err=%d).\n", errno);
		}
		//connection has been closed by the SSH server, so reopen it
		if (PP_FAILED(esb2_tpt_open_connection())) {
		    esb2.running = 0;
		}
		esb2.ssh_running = 0;
	    }
	    esb2.last_tpt_status = status;
	}
	if (ESB2_TPT_IS_CLOSED(status)) {
	    pp_log("ASMI: connection is closed (status = %d), re-opening\n", status);
	    // connection lost: re-setup BMC via IPMB
	    if (PP_FAILED(enable_tpt())) { // logging will be done inside
		pp_log("ASMI: Failed to enable TPT.\n");
		esb2.running = 0;
	    }
	}
    }
    return 0;
}
