#include <pthread.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/x509v3.h>
#include <pp/base.h>
#include <pp/bio.h>
#include <liberic_pthread.h>
#include <pp/rfb.h>
#include <liberic_cert.h>
#include <liberic_net.h>

#include "wlan.h"
#include <iwlib.h>

static pthread_mutex_t g_essids_mtx = PTHREAD_MUTEX_INITIALIZER;

/* Type of headers we know about (basically union iwreq_data) */
#define IW_HEADER_TYPE_NULL	0	/* Not available */
#define IW_HEADER_TYPE_CHAR	2	/* char [IFNAMSIZ] */
#define IW_HEADER_TYPE_UINT	4	/* __u32 */
#define IW_HEADER_TYPE_FREQ	5	/* struct iw_freq */
#define IW_HEADER_TYPE_ADDR	6	/* struct sockaddr */
#define IW_HEADER_TYPE_POINT	8	/* struct iw_point */
#define IW_HEADER_TYPE_PARAM	9	/* struct iw_param */
#define IW_HEADER_TYPE_QUAL	10	/* struct iw_quality */


static const char	standard_event_hdr[] = {
	IW_HEADER_TYPE_ADDR,	/* IWEVTXDROP */
	IW_HEADER_TYPE_QUAL,	/* IWEVQUAL */
	IW_HEADER_TYPE_POINT,	/* IWEVCUSTOM */
	IW_HEADER_TYPE_ADDR,	/* IWEVREGISTERED */
	IW_HEADER_TYPE_ADDR,	/* IWEVEXPIRED */
};

/* Headers for the various requests */
static const char standard_ioctl_hdr[] = {
	IW_HEADER_TYPE_NULL,	/* SIOCSIWCOMMIT */
	IW_HEADER_TYPE_CHAR,	/* SIOCGIWNAME */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWNWID */
	IW_HEADER_TYPE_PARAM,	/* SIOCGIWNWID */
	IW_HEADER_TYPE_FREQ,	/* SIOCSIWFREQ */
	IW_HEADER_TYPE_FREQ,	/* SIOCGIWFREQ */
	IW_HEADER_TYPE_UINT,	/* SIOCSIWMODE */
	IW_HEADER_TYPE_UINT,	/* SIOCGIWMODE */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWSENS */
	IW_HEADER_TYPE_PARAM,	/* SIOCGIWSENS */
	IW_HEADER_TYPE_NULL,	/* SIOCSIWRANGE */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWRANGE */
	IW_HEADER_TYPE_NULL,	/* SIOCSIWPRIV */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWPRIV */
	IW_HEADER_TYPE_NULL,	/* SIOCSIWSTATS */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWSTATS */
	IW_HEADER_TYPE_POINT,	/* SIOCSIWSPY */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWSPY */
	IW_HEADER_TYPE_POINT,	/* SIOCSIWTHRSPY */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWTHRSPY */
	IW_HEADER_TYPE_ADDR,	/* SIOCSIWAP */
	IW_HEADER_TYPE_ADDR,	/* SIOCGIWAP */
	IW_HEADER_TYPE_NULL,	/* -- hole -- */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWAPLIST */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWSCAN */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWSCAN */
	IW_HEADER_TYPE_POINT,	/* SIOCSIWESSID */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWESSID */
	IW_HEADER_TYPE_POINT,	/* SIOCSIWNICKN */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWNICKN */
	IW_HEADER_TYPE_NULL,	/* -- hole -- */
	IW_HEADER_TYPE_NULL,	/* -- hole -- */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWRATE */
	IW_HEADER_TYPE_PARAM,	/* SIOCGIWRATE */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWRTS */
	IW_HEADER_TYPE_PARAM,	/* SIOCGIWRTS */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWFRAG */
	IW_HEADER_TYPE_PARAM,	/* SIOCGIWFRAG */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWTXPOW */
	IW_HEADER_TYPE_PARAM,	/* SIOCGIWTXPOW */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWRETRY */
	IW_HEADER_TYPE_PARAM,	/* SIOCGIWRETRY */
	IW_HEADER_TYPE_POINT,	/* SIOCSIWENCODE */
	IW_HEADER_TYPE_POINT,	/* SIOCGIWENCODE */
	IW_HEADER_TYPE_PARAM,	/* SIOCSIWPOWER */
	IW_HEADER_TYPE_PARAM,	/* SIOCGIWPOWER */
};
static const unsigned int standard_ioctl_num = sizeof(standard_ioctl_hdr);
static const unsigned int standard_event_num = sizeof(standard_event_hdr);

/* Size (in bytes) of various events */
static const int event_type_size[] = {
	IW_EV_LCP_LEN,		/* IW_HEADER_TYPE_NULL */
	0,
	IW_EV_CHAR_LEN,		/* IW_HEADER_TYPE_CHAR */
	0,
	IW_EV_UINT_LEN,		/* IW_HEADER_TYPE_UINT */
	IW_EV_FREQ_LEN,		/* IW_HEADER_TYPE_FREQ */
	IW_EV_ADDR_LEN,		/* IW_HEADER_TYPE_ADDR */
	0,
	IW_EV_POINT_LEN,	/* Without variable payload */
	IW_EV_PARAM_LEN,	/* IW_HEADER_TYPE_PARAM */
	IW_EV_QUAL_LEN,		/* IW_HEADER_TYPE_QUAL */
};


static pthread_t wlan_thread;
static volatile int watcher_should_die;
void *wlan_watcher(void * arg);
static void print_scanning_info(void);

#define MAX_ESSIDS 10
static char* g_essids[MAX_ESSIDS+1] = {NULL, NULL, NULL, NULL, NULL,
				       NULL, NULL, NULL, NULL, NULL, NULL};

static char* wlan_if="ath0";

void start_wlan_watcher() {
    const char * fn = ___F;
    watcher_should_die = 0;
    if (eric_pthread_create(&wlan_thread, 0, 65536,
			    wlan_watcher, NULL) != 0) {
	pp_log("%s(): Cannot create wlan-thread.\n", fn);
    } else {
	printf("started wlan watcher\n");
    }
}

void stop_wlan_watcher() {
    watcher_should_die = 1;
    pthread_join(wlan_thread, NULL);
}


/* this code is inspired by wireless_tools */

int
iw_get_range_info(int		skfd,
		  char *	ifname,
		  iwrange *	range)
{
  struct iwreq		wrq;
  char			buffer[sizeof(iwrange) * 2];	/* Large enough */
  static int iw_ignore_version = 0;
  
  /* Cleanup */
  memset(buffer, 0, sizeof(buffer));

  wrq.u.data.pointer = (caddr_t) buffer;
  wrq.u.data.length = sizeof(buffer);
  wrq.u.data.flags = 0;
  if(iw_get_ext(skfd, ifname, SIOCGIWRANGE, &wrq) < 0)
    return(-1);

  /* Copy stuff at the right place, ignore extra */
  memcpy((char *) range, buffer, sizeof(iwrange));

  /* Lots of people have driver and tools out of sync as far as Wireless
   * Extensions are concerned. It's because /usr/include/linux/wireless.h
   * and /usr/src/linux/include/linux/wireless.h are different.
   * We try to catch this stuff here... */
  if(!iw_ignore_version)
    {
      /* For new versions, we can check the version directly, for old versions
       * we use magic. 300 bytes is a also magic number, don't touch... */
      if((WIRELESS_EXT > 10) && (wrq.u.data.length >= 300))
	{
#if WIRELESS_EXT > 10
	  /* Version verification - for new versions */
	  if(range->we_version_compiled != WIRELESS_EXT)
	    {
	      fprintf(stderr, "Warning: Driver for device %s has been compiled with version %d\n", ifname, range->we_version_compiled);
	      fprintf(stderr, "of Wireless Extension, while this program is using version %d.\n", WIRELESS_EXT);
	      fprintf(stderr, "Some things may be broken...\n\n");
	    }
	  /* Driver version verification */
	  if(range->we_version_compiled < range->we_version_source)
	    {
	      fprintf(stderr, "Warning: Driver for device %s recommend version %d of Wireless Extension,\n", ifname, range->we_version_source);
	      fprintf(stderr, "but has been compiled with version %d, therefore some driver features\n", range->we_version_compiled);
	      fprintf(stderr, "may not be available...\n\n");
	    }
#endif /* WIRELESS_EXT > 10 */
	}
      else
	{
	  /* Version verification - for old versions */
	  if(wrq.u.data.length != sizeof(iwrange))
	    {
	      fprintf(stderr, "Warning: Driver for device %s has been compiled with an ancient version\n", ifname);
	      fprintf(stderr, "of Wireless Extension, while this program is using version %d.\n", WIRELESS_EXT);
	      fprintf(stderr, "Some things may be broken...\n\n");
	    }
	}
    }
  /* Don't complain twice.
   * In theory, the test apply to each individual driver, but usually
   * all drivers are compiled from the same kernel, and most often
   * problem is the system/glibc headers. */
  iw_ignore_version = 1;

  /* Note : we are only trying to catch compile difference, not source.
   * If the driver source has not been updated to the latest, it doesn't
   * matter because the new fields are set to zero */

  return(0);
}



void n_iw_ether_ntop(const struct ether_addr* eth, int length, char* buf) {
    snprintf(buf, length, "%02X:%02X:%02X:%02X:%02X:%02X",
	    eth->ether_addr_octet[0], eth->ether_addr_octet[1],
	    eth->ether_addr_octet[2], eth->ether_addr_octet[3],
	    eth->ether_addr_octet[4], eth->ether_addr_octet[5]);
}


char* eric_net_wlan_get_ap_mac(void) {
    char* mac = NULL;
    int sock;
    struct iwreq wrq;
    char mac00[6] = {0,0,0,0,0,0};
    char macff[6] = {0xff,0xff,0xff,0xff,0xff,0xff};
   	 
    sock = socket(PF_INET, SOCK_DGRAM, 0);
    if (sock < 0) return NULL;
   	 
    /* Get AP address */
    if(iw_get_ext(sock, wlan_if, SIOCGIWAP, &wrq) >= 0)	{
	if (memcmp(&wrq.u.ap_addr.sa_data, mac00, 6) && memcmp(&wrq.u.ap_addr.sa_data, macff, 6)) {
#define MAX_MAC_SIZE 20
	    mac = malloc(MAX_MAC_SIZE); // more is better than less ;-)
	    n_iw_ether_ntop((const struct ether_addr *)&wrq.u.ap_addr.sa_data, MAX_MAC_SIZE, mac);
#undef MAX_MAC_SIZE
	}
    }
    close(sock);
    return mac;
}



int eric_net_wlan_get_signal_level(void) {
    struct iwreq wrq;
    iwstats wstats;
    int sock;
    // because we have an own socket, and only local variables, I assume we are thread save
    sock = socket(PF_INET, SOCK_DGRAM, 0);
    if (sock < 0) return -1;
    
    wrq.u.data.pointer = (caddr_t) &wstats;
    wrq.u.data.length = 0;
    wrq.u.data.flags = 1;		/* Clear updated flag */
    strncpy(wrq.ifr_name, wlan_if, IFNAMSIZ);
    if(iw_get_ext(sock, wlan_if, SIOCGIWSTATS, &wrq) < 0) {
	close(sock);
	return -1;
    }
    printf("level: %d, ret: %d\n", wstats.qual.level, wstats.qual.level - 0x100);
    close(sock);
    return (wstats.qual.level - 0x100)*(-1);
}

char** eric_net_wlan_get_essids(void) {
    int i;
    MUTEX_LOCK(&g_essids_mtx);
    for (i = 0; i < MAX_ESSIDS; i++) {
	free(g_essids[i]);
	g_essids[i] = NULL;
    }
    print_scanning_info();
    MUTEX_UNLOCK(&g_essids_mtx);
    return g_essids;
}

void wlan_add_essid(char* essid) {
    int i;
    for (i = 0; i < MAX_ESSIDS; i++) {
	if (g_essids[i] == NULL) {
	    g_essids[i] = strdup(essid);
	    break;
	}
    }
}

void *
wlan_watcher(void * arg UNUSED)
{
    struct iwreq		wrq;
    iwstats wstats;
    int sock;
    char quality[10];
    struct iw_range	range;
    int qualint;
    
    sock = socket(PF_INET, SOCK_DGRAM, 0);
    if (sock < 0) exit(1);
    
    while (!watcher_should_die) {
	sleep(1);	
	wrq.u.data.pointer = (caddr_t) &wstats;
	wrq.u.data.length = 0;
	wrq.u.data.flags = 1;		/* Clear updated flag */
	strncpy(wrq.ifr_name, wlan_if, IFNAMSIZ);
	if(iw_get_ext(sock, wlan_if, SIOCGIWSTATS, &wrq) < 0)
	    continue;
	if(iw_get_range_info(sock, wlan_if, &range) < 0)
	    continue;
	if (range.max_qual.qual == 0)
	    continue;
	
	//printf("q: %d/%d\n", wstats.qual.qual, range.max_qual.qual);
	/* just convert the wlan stats into meaningful values */

	{
	    int maxval = 50;
	    int minval = 25;
	    
	    if (wstats.qual.qual > maxval) { wstats.qual.qual = maxval; }
	    if (wstats.qual.qual < minval) { wstats.qual.qual = minval; }
	
	    qualint = ((wstats.qual.qual-minval) * 100) / (maxval-minval);
	}
	snprintf(quality, sizeof(quality), "%d", qualint);

	/* at the moment there is only one channel on all KIM boards */
	/* send_command uses internally for_all_clients - if there are no clients,
	   than it doesn't matter */
	pp_rfb_send_command(-1, "wlan_quality", quality);
    }
    if (watcher_should_die) {
	close(sock);
	return NULL;
    }
    /* In case of an error we should exit the whole process */
    exit(1);
}


static inline int
print_scanning_token(struct iw_event *	event,	/* Extracted token */
		     int		ap_num,	/* AP number */
		     struct iw_range *	iwrange UNUSED,	/* Range info */
		     int		has_range UNUSED)
{
  /* Now, let's decode the event */
  switch(event->cmd) {
    case SIOCGIWAP:
	ap_num++;
	break;
    case SIOCGIWNWID:
    case SIOCGIWFREQ:
    case SIOCGIWMODE:
    case SIOCGIWNAME:
	break;
    case SIOCGIWESSID:
      {
	char essid[IW_ESSID_MAX_SIZE+1];
	if((event->u.essid.pointer) && (event->u.essid.length))
	  memcpy(essid, event->u.essid.pointer, event->u.essid.length);
	essid[event->u.essid.length] = '\0';
	if(event->u.essid.flags) {
	    wlan_add_essid(essid);
	}
      }
      break;
    case SIOCGIWENCODE:
    case SIOCGIWRATE:
    case IWEVQUAL:
    default:
	break;
  }	/* switch(event->cmd) */

  /* May have changed */
  return(ap_num);
}

/*------------------------------------------------------------------*/
/*
 * Perform a scanning on one device
 */
static void
print_scanning_info(void)
{
  struct iwreq		wrq;
  unsigned char		buffer[IW_SCAN_MAX_DATA];	/* Results */
  struct timeval	tv;				/* Select timeout */
  int			timeout = 5000000;		/* 5s */
  int sock;

  sock = socket(PF_INET, SOCK_DGRAM, 0);
  if (sock < 0) exit(1);
  
  /* Init timeout value -> 250ms*/
  tv.tv_sec = 0;
  tv.tv_usec = 250000;

  /*
   * Here we should look at the command line args and set the IW_SCAN_ flags
   * properly
   */
  wrq.u.param.flags = IW_SCAN_DEFAULT;
  wrq.u.param.value = 0;		/* Later */

  /* Initiate Scanning */
  if(iw_set_ext(sock, wlan_if, SIOCSIWSCAN, &wrq) < 0)
    {
      if(errno != EPERM)
	{
	  fprintf(stderr, "%-8.8s  Interface doesn't support scanning : %s\n\n",
		  wlan_if, strerror(errno));
	  return;
	}
      /* If we don't have the permission to initiate the scan, we may
       * still have permission to read left-over results.
       * But, don't wait !!! */
      tv.tv_usec = 0;
    }
  timeout -= tv.tv_usec;

  /* Forever */
  while(1)
    {
      fd_set		rfds;		/* File descriptors for select */
      int		last_fd;	/* Last fd */
      int		ret;

      /* Guess what ? We must re-generate rfds each time */
      FD_ZERO(&rfds);
      last_fd = -1;

      /* In here, add the rtnetlink fd in the list */

      /* Wait until something happens */
      ret = select(last_fd + 1, &rfds, NULL, NULL, &tv);

      /* Check if there was an error */
      if(ret < 0)
	{
	  if(errno == EAGAIN || errno == EINTR)
	    continue;
	  fprintf(stderr, "Unhandled signal - exiting...\n");
	  return;
	}

      /* Check if there was a timeout */
      if(ret == 0)
	{
	  /* Try to read the results */
	  wrq.u.data.pointer = buffer;
	  wrq.u.data.flags = 0;
	  wrq.u.data.length = sizeof(buffer);
	  if(iw_get_ext(sock, wlan_if, SIOCGIWSCAN, &wrq) < 0)
	    {
	      /* Check if results not available yet */
	      if(errno == EAGAIN)
		{
		  /* Restart timer for only 100ms*/
		  tv.tv_sec = 0;
		  tv.tv_usec = 100000;
		  timeout -= tv.tv_usec;
		  if(timeout > 0)
		    continue;	/* Try again later */
		}

	      /* Bad error */
	      fprintf(stderr, "%-8.8s  Failed to read scan data : %s\n\n",
		      wlan_if, strerror(errno));
	      return;
	    }
	  else
	    /* We have the results, go to process them */
	    break;
	}

      /* In here, check if event and event type
       * if scan event, read results. All errors bad & no reset timeout */
    }

  if(wrq.u.data.length)
    {
      struct iw_event		iwe;
      struct stream_descr	stream;
      int			ap_num = 1;
      int			ret;
      struct iw_range		range;
      int			has_range;

      has_range = (iw_get_range_info(sock, wlan_if, &range) >= 0);
      iw_init_event_stream(&stream, buffer, wrq.u.data.length);
      do
	{
	  /* Extract an event and print it */
	  ret = iw_extract_event_stream(&stream, &iwe);
	  if(ret > 0)
	    ap_num = print_scanning_token(&iwe, ap_num, &range, has_range);
	}
      while(ret > 0);
    }
  else
    printf("%-8.8s  No scan results\n", wlan_if);

  return;
}

void
iw_init_event_stream(struct stream_descr *	stream,	/* Stream of events */
		     char *			data,
		     int			len)
{
  /* Cleanup */
  memset((char *) stream, '\0', sizeof(struct stream_descr));

  /* Set things up */
  stream->current = data;
  stream->end = data + len;
}


int
iw_extract_event_stream(struct stream_descr *	stream,	/* Stream of events */
			struct iw_event *	iwe)	/* Extracted event */
{
  int		event_type = 0;
  unsigned int	event_len = 1;		/* Invalid */
  char *	pointer;
  /* Don't "optimise" the following variable, it will crash */
  unsigned	cmd_index;		/* *MUST* be unsigned */

  /* Check for end of stream */
  if((stream->current + IW_EV_LCP_LEN) > stream->end)
    return(0);

#if 0
  printf("DBG - stream->current = %p, stream->value = %p, stream->end = %p\n",
	 stream->current, stream->value, stream->end);
#endif

  /* Extract the event header (to get the event id).
   * Note : the event may be unaligned, therefore copy... */
  memcpy((char *) iwe, stream->current, IW_EV_LCP_LEN);

#if 0
  printf("DBG - iwe->cmd = 0x%X, iwe->len = %d\n",
	 iwe->cmd, iwe->len);
#endif

  /* Check invalid events */
  if(iwe->len <= IW_EV_LCP_LEN)
    return(-1);

  /* Get the type and length of that event */
  if(iwe->cmd <= SIOCIWLAST)
    {
      cmd_index = iwe->cmd - SIOCIWFIRST;
      if(cmd_index < standard_ioctl_num)
	event_type = standard_ioctl_hdr[cmd_index];
    }
  else
    {
      cmd_index = iwe->cmd - IWEVFIRST;
      if(cmd_index < standard_event_num)
	event_type = standard_event_hdr[cmd_index];
    }
  /* Unknown events -> event_type=0 => IW_EV_LCP_LEN */
  event_len = event_type_size[event_type];

  /* Check if we know about this event */
  if(event_len <= IW_EV_LCP_LEN)
    {
      /* Skip to next event */
      stream->current += iwe->len;
      return(2);
    }
  event_len -= IW_EV_LCP_LEN;

  /* Set pointer on data */
  if(stream->value != NULL)
    pointer = stream->value;			/* Next value in event */
  else
    pointer = stream->current + IW_EV_LCP_LEN;	/* First value in event */

#if 0
  printf("DBG - event_type = %d, event_len = %d, pointer = %p\n",
	 event_type, event_len, pointer);
#endif

  /* Copy the rest of the event (at least, fixed part) */
  if((pointer + event_len) > stream->end)
    {
      /* Go to next event */
      stream->current += iwe->len;
      return(-2);
    }
  memcpy((char *) iwe + IW_EV_LCP_LEN, pointer, event_len);

  /* Skip event in the stream */
  pointer += event_len;

  /* Special processing for iw_point events */
  if(event_type == IW_HEADER_TYPE_POINT)
    {
      /* Check the length of the payload */
      if((iwe->len - (event_len + IW_EV_LCP_LEN)) > 0)
	/* Set pointer on variable part (warning : non aligned) */
	iwe->u.data.pointer = pointer;
      else
	/* No data */
	iwe->u.data.pointer = NULL;

      /* Go to next event */
      stream->current += iwe->len;
    }
  else
    {
      /* Is there more value in the event ? */
      if((pointer + event_len) <= (stream->current + iwe->len))
	/* Go to next value */
	stream->value = pointer;
      else
	{
	  /* Go to next event */
	  stream->value = NULL;
	  stream->current += iwe->len;
	}
    }
  return(1);
}
