#include <fcntl.h>
#include <pthread.h>
#include <stdio.h>
#include <unistd.h>
#include <pp/base.h>
#include <liberic_notify.h>

#include "list.h"

#define LIST_LOG_FILE_PATH		"/var/log/evtlog"
#define LIST_LOG_FILE_MAX_ENTRIES	1000
#define LIST_LOG_MAX_ENTRIES		LIST_LOG_FILE_MAX_ENTRIES

typedef struct {
    u_int id;
    time_t date;
    char * event;
    char * desc;
} log_list_entry_t;

static int initialized = 0;
static FILE * logfp = NULL;
static log_list_entry_t ** log_ring;
static u_int next_id = 0;
static u_int first_offs;
static u_int next_offs;
static u_int file_entry_cnt;
static pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;

static void log_ring_append_unlocked(time_t date, const char * event, const char * desc);
static int log_file_open(void);
static int log_file_close(void);
static int log_file_delete(void);
static int log_file_load(void);
static int log_file_save(void);

int
list_log_init(void)
{
    MUTEX_LOCK(&lock);
    if (!initialized) {
	log_ring = calloc(LIST_LOG_MAX_ENTRIES, sizeof(log_list_entry_t*));
	first_offs = next_offs = file_entry_cnt = 0;
	log_file_open();
	log_file_load();
	initialized = 1;
    }
    MUTEX_UNLOCK(&lock);
    return PP_SUC;
}

void
list_log_cleanup(void)
{
    u_int i;
    MUTEX_LOCK(&lock);
    for (i = 0; i < LIST_LOG_MAX_ENTRIES; ++i) {
        free(log_ring[i]->event);
        free(log_ring[i]->desc);
	free(log_ring[i]);
    }
    free(log_ring);
    log_ring = NULL;
    first_offs = next_offs = file_entry_cnt = 0;
    log_file_close();
    initialized = 0;
    MUTEX_UNLOCK(&lock);
}

/**
 * clear internal log list and truncate local log file
 */
void
eric_notify_list_log_clear(void)
{
    u_int i;
    MUTEX_LOCK(&lock);
    for (i = 0; i < LIST_LOG_MAX_ENTRIES; ++i) {
	log_list_entry_t * entry = log_ring[i];
        if (entry != NULL) {
            free(entry->event);
            free(entry->desc);
            free(entry);
            log_ring[i] = NULL;
        }
    }
    first_offs = next_offs = file_entry_cnt = 0;
    next_id = 0;

    log_file_close();
    log_file_delete();
    log_file_open();
    MUTEX_UNLOCK(&lock);
}

u_int
eric_notify_list_get_entry(u_int id, time_t * date_p, char ** event_p, char ** desc_p)
{
    log_list_entry_t * entry;
    u_int idx, id_next;

    MUTEX_LOCK(&lock);

    idx = id % LIST_LOG_MAX_ENTRIES;
    entry = log_ring[idx];

    if (entry && entry->id > id) {
	entry = log_ring[first_offs];
    }

    if (entry && entry->id >= id) {
	*date_p = entry->date;
	*event_p = strdup(entry->event);
	*desc_p = strdup(entry->desc);
	id_next = entry->id + 1;
    } else {
	/* end of list reached */
	*date_p = 0;
	*event_p = NULL;
	*desc_p = NULL;
	id_next = id;
    }

    MUTEX_UNLOCK(&lock);
    return id_next;
}

int
eric_notify_list_get_entry_exact(u_int id, time_t * date_p, char ** event_p, char ** desc_p)
{
    log_list_entry_t *entry;
    u_int idx;

    if ( id > eric_notify_list_get_last_id() ||
	 id < eric_notify_list_get_first_id() ) {
	return PP_ERR;
    }

    MUTEX_LOCK(&lock);

    idx = id % LIST_LOG_MAX_ENTRIES;
    entry = log_ring[idx];
   
    if (!entry || entry->id > id) {
	MUTEX_UNLOCK(&lock);
	return PP_ERR;
    }

    *date_p = entry->date;
    *event_p = strdup(entry->event);
    *desc_p = strdup(entry->desc);

    MUTEX_UNLOCK(&lock);
    return PP_SUC;    
}
    
u_int
eric_notify_list_get_last_id()
{
    u_int id = 0;

    MUTEX_LOCK(&lock);
    if (next_id > 0) {
	id = next_id - 1;
    }
    MUTEX_UNLOCK(&lock);

    return id;
}

u_int
eric_notify_list_get_first_id()
{
    u_int id = 0;
    log_list_entry_t *entry;

    MUTEX_LOCK(&lock);
    
    entry = log_ring[first_offs];
    
    if ( entry ) {
	id = entry->id;
    } else {
	id = 0;
    }
    MUTEX_UNLOCK(&lock);

    return id;
}

int
eric_notify_list_have_more_entries(u_int id, int direction)
{
    log_list_entry_t * entry;
    u_int idx;
    int ret;

    MUTEX_LOCK(&lock);

    if (direction == PP_NOTIFY_LIST_FORWARD) {
	idx = (id + 1) % LIST_LOG_MAX_ENTRIES;
	entry = log_ring[idx];
	ret = entry && entry->id == (id + 1);
    } else {
	if (id > 0) {
	    idx = (id - 1) % LIST_LOG_MAX_ENTRIES;
	    entry = log_ring[idx];
	    ret = entry && entry->id == (id - 1);
	} else {
	    ret = 0;
	}
    }

    MUTEX_UNLOCK(&lock);
    return ret;
}


int
list_log_event(time_t date, const char * event, const char * desc)
{
    int ret;
    char *tmp_desc;

    MUTEX_LOCK(&lock);

    log_ring_append_unlocked(date, event, desc);
    
    if (file_entry_cnt >= LIST_LOG_FILE_MAX_ENTRIES) {
	log_file_close();
	log_file_delete();
	log_file_open();
	log_file_save();
    }

    /* escape NL and CR - will be reversed when parsing */
    tmp_desc = pp_str_escape('\\', "\n\r", "nr", desc);

    if (logfp != NULL
	&& fprintf(logfp, "%lu\t%s\t%s\n", date, event, tmp_desc) > 0
	&& fflush(logfp) == 0
	&& !feof(logfp)
	&& !ferror(logfp)) {
	++file_entry_cnt;
	ret = PP_SUC;
    } else {
	if (logfp) {
	    clearerr(logfp);
	    pp_log("%s(): fprintf() or fflush() failed\n", ___F);
	} else {
	    pp_log("%s(): logfile not open\n", ___F);
	}
	ret = PP_ERR;
    }

    MUTEX_UNLOCK(&lock);

    free(tmp_desc);

    return ret;
}

static void
log_ring_append_unlocked(time_t date, const char * event, const char * desc)
{
    log_list_entry_t * next_entry = log_ring[next_offs];

    if (next_entry) {
	free(next_entry->event);
	free(next_entry->desc);
	if (first_offs == next_offs) {
	    first_offs = (first_offs + 1) % LIST_LOG_MAX_ENTRIES;
	}
    } else {
	next_entry = calloc(1, sizeof(log_list_entry_t));
	log_ring[next_offs] = next_entry;
    }
    next_entry->id = next_id++;
    next_entry->date = date;
    next_entry->event = strdup(event);
    next_entry->desc = strdup(desc);
    next_offs = (next_offs + 1) % LIST_LOG_MAX_ENTRIES;
}

static int
log_file_open(void) 
{
    assert(logfp == NULL);
    if ((logfp = fopen(LIST_LOG_FILE_PATH, "a+")) == NULL) {
	pp_log_err("%s(): fopen(%s) failed", ___F, LIST_LOG_FILE_PATH);
	return PP_ERR;
    }        
    return PP_SUC;
}

static int
log_file_close(void) 
{
    if (logfp != NULL) {
	fclose(logfp);
	logfp = NULL;
    }
    return PP_SUC;
}

static int
log_file_delete(void) 
{
    assert(logfp == NULL);
    if (unlink(LIST_LOG_FILE_PATH) != 0) {
	if (errno != ENOENT) {
	    pp_log_err("%s(): unlink(%s) failed", ___F, LIST_LOG_FILE_PATH);
	}
	return PP_ERR;
    }
    return PP_SUC;
}

static int
log_file_load(void) 
{
    fpos_t fpos;
    u_int line;        
    time_t date;
    char date_str[20];
    char event[64];
    char desc[256];
    char line_buf[20+64+256];
    char *tmp_desc;

    if (fgetpos(logfp, &fpos) != 0) {
	pp_log_err("%s(): fgetpos(%s) failed", ___F, LIST_LOG_FILE_PATH);
	return PP_ERR;
    }

    rewind(logfp);

    for (line = 0; !feof(logfp) && !ferror(logfp); line++) {
	/* mind the buffer sizes! */
	fgets(line_buf, sizeof(line_buf), logfp);
	int n = sscanf(line_buf, "%19[^\t]\t%63[^\t]\t%255[^\n]\n", date_str, event, desc);

	/* unescape NL and CR */
	tmp_desc = pp_str_unescape('\\', "\n\r", "nr", desc);
	strncpy (desc, tmp_desc, sizeof(desc));
	free(tmp_desc);
	
	if (n >= 0 && n < 3) {
	    pp_log("%s(): Parse error in line %u of '%s'!\n", ___F, line + 1, LIST_LOG_FILE_PATH);
	    while (!feof(logfp) && !ferror(logfp) && fgetc(logfp) != '\n');
	} else if (n == 3) {
	    struct tm tm;
	    if (strptime(date_str, "%m/%d/%Y %R:%S", &tm) != NULL) {
		date = mktime(&tm);
	    } else {
		date = pp_strtoul_10(date_str, 0, NULL);
	    }
	    log_ring_append_unlocked(date, event, desc);
	    ++file_entry_cnt;
	}
    }

    if (fsetpos(logfp, &fpos) != 0) {
	pp_log_err("%s(): fsetpos(%s) failed", ___F, LIST_LOG_FILE_PATH);
	return PP_ERR;
    }
    
    return PP_SUC;
}

static int
log_file_save(void)
{
    u_int i;
    int ret = PP_SUC;

    file_entry_cnt = 0;

    for (i = 0; i < LIST_LOG_MAX_ENTRIES; ++i) {
	log_list_entry_t * entry = log_ring[(first_offs + i) % LIST_LOG_MAX_ENTRIES];

	if (entry && entry->event && entry->desc) {
	    if (logfp != NULL
		&& fprintf(logfp, "%lu\t%s\t%s\n", entry->date, entry->event, entry->desc) > 0
		&& fflush(logfp) == 0
		&& !feof(logfp)
		&& !ferror(logfp)) {
		++file_entry_cnt;
	    } else {
		if (logfp) {
		    clearerr(logfp);
		    pp_log("%s(): fprintf() or fflush() failed\n", ___F);
		} else {
		    pp_log("%s(): logfile not open\n", ___F);
		}
		ret = PP_ERR;
		break;
	    }
	}
    }
    return ret;
}

