#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <semaphore.h>
#include <sys/param.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <liberic_pthread.h>
#include <liberic_notify.h>

#include "nfs.h"

#define NFS_LOG_DIR   "/var/nfs"
#define NFS_LOG_FILE  "event_log"

#define MAX_EVENT_NAME_LEN 63
#define MAX_EVENT_DESC_LEN PP_NOTIFY_MAX_MSG_LEN

typedef struct {
    time_t date;
    char event[PP_NOTIFY_MAX_EVENT_NAME_LEN];
    char desc[PP_NOTIFY_MAX_EVENT_DESC_LEN];
} nfs_queue_entry_t;

static int initialized = 0;
static sem_t nfs_sem;
static pp_queue_t * nfs_queue;
static pthread_mutex_t nfs_queue_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_t nfs_thread;
static int nfs_thread_started = 0;
static int nfs_thread_stop = 0;
static FILE * logfp;
static char nfs_filepath[MAXPATHLEN + 1];
static int nfs_mounted = 0;

static void * nfs_event_thread(void* arg);
static int log_file_open(const char * filename);
static int log_file_close(void);

int 
nfs_log_init(void)
{
    int ret = PP_ERR;

    if (!initialized) {
	if (sem_init(&nfs_sem, 0, 0)) {
	    pp_log_err("%s(): semaphore init failed.\n", ___F);
	    goto bail;
	}
    
	if ((nfs_queue = pp_queue_alloc(0)) == NULL) {
	    pp_log("%s(): pp_queue_alloc() failed.\n", ___F);
	    goto bail;
	}

	if (PP_FAILED(eric_notify_nfs_mount())) {
	    goto bail;      
	}

	initialized = 1;
    }
    ret = PP_SUC;
 bail:
    if (ret != PP_SUC) nfs_log_cleanup();
    return ret;
}

void
nfs_log_cleanup(void)
{
    eric_notify_nfs_unmount();
    if (nfs_queue) pp_queue_free(nfs_queue);
    sem_destroy(&nfs_sem); /* FIXME: what happens if never initialized? */
    initialized = 0;
}

int
nfs_log_reconf_ch(pp_cfg_chg_ctx_t * ctx)
{
    int ret = PP_SUC;

    eric_notify_nfs_unmount();
    if (PP_FAILED(eric_notify_nfs_mount())) {
	pp_strstream_t strbuf;
	pp_strstream_init(&strbuf);
	pp_strappendf(&strbuf, "Mounting NFS share failed!");
	*(ctx->retstr) = pp_strstream_buf(&strbuf);
	ret = PP_ERR;
    }

    return ret;
}

int
nfs_log_event(time_t date, const char *event, const char *desc)
{
    int ret = PP_ERR;

    if (nfs_thread_started) {
	nfs_queue_entry_t * entry = malloc(sizeof(nfs_queue_entry_t));
	entry->date = date;
	snprintf(entry->event, sizeof(entry->event), "%s", event);
	snprintf(entry->desc, sizeof(entry->desc), "%s", desc);
    
	MUTEX_LOCK(&nfs_queue_mtx);
	if (PP_SUCCED(pp_queue_enqueue(nfs_queue, entry, 0, 0))) {
	    sem_post(&nfs_sem);
	}
	MUTEX_UNLOCK(&nfs_queue_mtx);

	ret = PP_SUC;
    } else {
	pp_log("NFS thread not initialized\n");
    }

    return ret;
}

int
eric_notify_nfs_mount(void)
{
    char command[MAXPATHLEN + 1];
    int nfs_enabled;
    char * nfs_server = NULL;
    char * nfs_share = NULL;
    char * nfs_file = NULL;
    int status, ret = PP_ERR;

    pp_cfg_is_enabled(&nfs_enabled, "log.nfs.enabled");
    if (!nfs_enabled) return PP_SUC;

    if (PP_FAILED(pp_cfg_get_nodflt(&nfs_server, "log.nfs.server"))
	|| PP_FAILED(pp_cfg_get_nodflt(&nfs_share, "log.nfs.share"))
	|| PP_FAILED(pp_cfg_get_nodflt(&nfs_file, "log.nfs.file"))) {
        pp_log("%s(): Missing parameter for NFS mount.\n", ___F);
        goto bail;
    }

    snprintf(command, sizeof(command), "mount -o nolock,soft %s:/%s %s", nfs_server, nfs_share, NFS_LOG_DIR);
    
    MUTEX_LOCK(&nfs_queue_mtx);

    status = pp_system(command);
    if (WEXITSTATUS(status) == 0) {
	nfs_mounted = 1;
	if (PP_SUCCED(log_file_open(nfs_file))) {
	    if (!nfs_thread_started) {
		nfs_thread_stop = 0;
		nfs_thread_started = 1;
		if (eric_pthread_create(&nfs_thread, 0, 128 * 1024, nfs_event_thread, NULL) == 0) {
		    ret = PP_SUC;
		} else {
		    pp_log_err("%s(): nfs thread creation failed.", ___F);
		    nfs_thread_started = 0;
		}
	    }
	}
    } else {
	pp_log_err("%s(): pp_system(mount) failed (exit != 0).", ___F);
    }

    MUTEX_UNLOCK(&nfs_queue_mtx);

 bail:
    free(nfs_server);
    free(nfs_share);
    free(nfs_file);
    return ret;
}

int
eric_notify_nfs_unmount(void)
{
    char command[MAXPATHLEN+1];
    int status, ret = PP_SUC;
    
    if (nfs_mounted ) {
    
	MUTEX_LOCK(&nfs_queue_mtx);

	if (nfs_thread_started) {
	    nfs_thread_stop = 1;
	    sem_post(&nfs_sem);
	    pthread_join(nfs_thread, NULL);
	    nfs_thread_started = 0;
	}

	log_file_close();
    
	snprintf(command, sizeof(command), "umount -f %s", NFS_LOG_DIR);
	status = pp_system(command);
	if (WEXITSTATUS(status) == 0) {
	    nfs_mounted = 0;
	} else {
	    pp_log_err("%s(): pp_system(umount) failed (exit != 0).", ___F);
	    ret = PP_ERR;
	}

	MUTEX_UNLOCK(&nfs_queue_mtx);
	
    }
    return ret;
}

static void *
nfs_event_thread(void * arg UNUSED)
{    
    pp_queue_entry_t * queue_entry;
    nfs_queue_entry_t * log_entry;
    struct stat st;
    
    while (!nfs_thread_stop) {
	sem_wait(&nfs_sem);
	if (nfs_thread_stop) break;

 	MUTEX_LOCK(&nfs_queue_mtx);
        queue_entry = pp_queue_dequeue(nfs_queue);
        MUTEX_UNLOCK(&nfs_queue_mtx);
	log_entry = (nfs_queue_entry_t *)queue_entry->data;
	
	// file is still present?
	if (fstat(fileno(logfp), &st) < 0) {	
	    log_file_close();
	    log_file_open(NULL);
	}
	
	if (log_entry->event && log_entry->desc) {
	    char date_str[32];
	    eric_notify_get_time_string(log_entry->date, date_str, sizeof(date_str));
	    if (logfp == NULL
		|| fprintf(logfp, "%s\t%s\t%s\n", date_str, log_entry->event, log_entry->desc) == 0
		|| fflush(logfp) != 0
		|| feof(logfp)
		|| ferror(logfp)) {
		clearerr(logfp);
		if (logfp) {
		    pp_log("%s(): fprintf() or fflush() failed\n", ___F);
		} else {
		    pp_log("%s(): logfile not open\n", ___F);
		}
	    }
	}
	free(queue_entry->data);
	free(queue_entry);
    }

    return NULL;
}

static int
log_file_open(const char * filename)
{
    if (filename) snprintf(nfs_filepath, sizeof(nfs_filepath), "%s/%s", NFS_LOG_DIR, filename);
    if ((logfp = fopen(nfs_filepath, "a+")) == NULL) {
	pp_log_err("%s(): fopen(%s) failed", ___F, nfs_filepath);
	return PP_ERR;
    }        
    return PP_SUC;
}

static int
log_file_close(void) 
{
    if (logfp != NULL) {
	fclose(logfp);
	logfp = NULL;
    }
    return PP_SUC;
}
