#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/param.h>
#include <semaphore.h>
#include <pp/smtp.h>
#include <pp/cfg.h>
#include <pp/vector.h>
#include <pp/base.h>
#include <liberic_notify.h>
#include <liberic_config.h>
#include <liberic_pthread.h>
#ifdef PRODUCT_KX2
#include <pp/hal_kx2.h>
#endif /* PRODUCT KX2 */
#include "nfs.h"

#include "list.h"
#include "nfs.h"
#include "smtp.h"
#include "snmptrap.h"
#include "rsyslog.h"

static pp_hash_t * event_types_h = NULL;

static void evt_type_lst_entry_free(void* _list_entry);
static u_int get_log_media(const char *event_id);

/* FIXME: (rwa) unclean signature - void pointer circumvents type check! */
int
eric_notify_post_event(const void * data, const char *evt_name, int evt_flag)
{
    time_t t = time(NULL);
    log_event_t event;
    evt_type_entry_t * entry = NULL;
    u_int log_media = 0;
    char *desc_tmp = NULL;
    char *event_msg = NULL;
    char *subj_msg = NULL;
#if defined(OEM_BELKIN) && defined(PRODUCT_LARA)
    time_t duration;
#endif /* OEM_BELKIN && PRODUCT_LARA */
    
    memset(&event, 0, sizeof(event));
    /* legacy events are login, host power and security events, 
       they are former snmp traps with own MIB entries and need 
       different event handling */
    if (evt_flag == PP_NOTIFY_EVENT_LEGACY) {
        event = *(const log_event_t*)data;
    /* all other event are generic, data has to be a character string */
    } else if (evt_flag == PP_NOTIFY_EVENT_GENERIC) {
        event.trapoid = evt_name;
        event.eventdata.generictrapdata.trap_desc = (const char *) data;
    }
       
    /* get event group description */
    if ((entry = eric_notify_get_event_object(evt_name)) == NULL) {
	pp_log("%s(): log event object not registered: %s\n", ___F, evt_name);
        return PP_ERR;
    }
    
    event.longname = entry->longname;         
    log_media = get_log_media(evt_name);
    
    if (log_media & PP_NOTIFY_EVENT_SNMP_ENABLE) {
        snmp_log_trap_send(&event);
    }
        
    if (!(log_media & PP_NOTIFY_EVENT_SMTP_ENABLE)
	&& !(log_media & PP_NOTIFY_EVENT_LIST_ENABLE)
	&& !(log_media & PP_NOTIFY_EVENT_NFS_ENABLE)) {
        return PP_SUC;
    }
        
    /* legacy event handling */        
    if (strstr(event.trapoid, "Loginfailed")) {

#if !(defined(OEM_BELKIN) && defined(PRODUCT_LARA))
	if (asprintf(&desc_tmp, "User '%s' failed to log in from IP address %s",
		     event.eventdata.logintrapdata.loginname, 
		     event.eventdata.logintrapdata.remote_host) == -1) {
	    pp_log_err("%s(): asprintf()", ___F);
	    goto finish;
	}
#else /* defined(OEM_BELKIN) && defined(PRODUCT_LARA) */
	asprintf (&subj_msg, "User '%s' from IP address %s failed to log in!",
		event.eventdata.logintrapdata.loginname, 
		event.eventdata.logintrapdata.remote_host);
	asprintf (&event_msg, "%s", subj_msg);
	asprintf (&desc_tmp, "Details:\n  %s\n", subj_msg);
#endif

    } else if (strstr(event.trapoid, "UserBlocked")) {
	asprintf (&subj_msg, "User '%s' from IP address %s  has been blocked.",
		event.eventdata.logintrapdata.loginname, 
		event.eventdata.logintrapdata.remote_host);
	asprintf (&event_msg, "%s", subj_msg);
	asprintf (&desc_tmp, "Details:\n  %s\n", subj_msg);
        
    } else if (strstr(event.trapoid, "Loginsuccess")) {

#if !(defined(OEM_BELKIN) && defined(PRODUCT_LARA))
	if (asprintf(&desc_tmp, "User '%s' logged in from IP address %s",
		     event.eventdata.logintrapdata.loginname,
		     event.eventdata.logintrapdata.remote_host) == -1) {
	    pp_log_err("%s(): asprintf()", ___F);
	    goto finish;
	}

#else /* defined(OEM_BELKIN) && defined(PRODUCT_LARA) */
	asprintf (&subj_msg, "User '%s' logged in from IP address %s",
		event.eventdata.logintrapdata.loginname, 
		event.eventdata.logintrapdata.remote_host);
	asprintf (&event_msg, "%s", subj_msg);
	asprintf (&desc_tmp, "Details:\n  %s\n", subj_msg);
#endif
    } else if (strstr(event.trapoid, "Logout")) {

#if defined(OEM_BELKIN) && defined(PRODUCT_LARA)
	/* default Email subject */
	asprintf (&subj_msg, "User '%s' from IP address %s logged out.",
		event.eventdata.logintrapdata.loginname, 
		event.eventdata.logintrapdata.remote_host);
	asprintf (&event_msg, "%s", subj_msg);
	/* print session duration string */
	duration = event.eventdata.logintrapdata.session_duration;
	asprintf(&desc_tmp, 
		"Details:\n"
		"  %s\n"
		"  Session duration: %ldh, %ldmin, %lds\n",
		subj_msg, 
		duration / 3600, duration % 3600 / 60, duration % 60);
#else
	goto finish;
#endif
    } else if (strstr(event.trapoid, "SecurityViolation")) {
	if (asprintf(&desc_tmp, "User '%s' causes a security violation on Host %s",
		     event.eventdata.logintrapdata.loginname,
		     event.eventdata.logintrapdata.remote_host) == -1) {
	    pp_log_err("%s(): asprintf()", ___F);
	    goto finish;
	}
    } else if (strstr(event.trapoid, "HostPower")) {
        desc_tmp = strdup("Host power state changed.\n");              
    } else if (strstr(event.trapoid, "HostReset")) {    
        desc_tmp = strdup("Host reset was performed.\n");
#if defined(PRODUCT_KX2)
        pp_hal_kx2_beep(1);
	sleep(1);
        pp_hal_kx2_beep(1);
#endif	
    } else if (strstr(event.trapoid, "IpmiAlert")) {
        if (asprintf(&desc_tmp, "%s %s %s\n", event.eventdata.ipmitrapdata.alert_host,
		     event.eventdata.ipmitrapdata.evnt_type,
		     event.eventdata.ipmitrapdata.evnt_desc) == -1) {
	    pp_log_err("%s(): asprintf()", ___F);
	    goto finish;
	}
    /* generic event handling */                                          
    } else {                                                                  
        desc_tmp = strdup(event.eventdata.generictrapdata.trap_desc);
#if defined (PRODUCT_KX2)
    if (event.eventcode >= ERIC_EC_System_Startup &&
	event.eventcode <= ERIC_EC_EVENT_End) {
	goto finish;
    }
#endif
    }

    if ((log_media & PP_NOTIFY_EVENT_LIST_ENABLE) && PP_FAILED(list_log_event(t, event.longname, desc_tmp))) {
	pp_log("%s(): list_log_event() failed.\n", ___F);
    }
        
    if ((log_media & PP_NOTIFY_EVENT_SMTP_ENABLE) && PP_FAILED(smtp_log_event(t, event.longname, desc_tmp))) {
	pp_log("%s(): smtp_log_event() failed.\n", ___F);
    }

    if ((log_media & PP_NOTIFY_EVENT_NFS_ENABLE) && PP_FAILED(nfs_log_event(t, event.longname, desc_tmp))) {
	pp_log("%s(): nfs_log_event() failed.\n", ___F);
    }

#if defined(PP_FEAT_REMOTE_SYSLOG)
    if ((log_media & PP_NOTIFY_EVENT_SYSLOG_ENABLE) && PP_FAILED(syslog_log_event(t, event.longname, desc_tmp))) {
        pp_log("%s(): syslog_log_event() failed.\n", ___F);
    }
#endif    

 finish:
    free(desc_tmp);             
    free(event_msg);             
    free(subj_msg);             
        
    return PP_SUC;
}

void
eric_notify_security_violation(eric_session_int_id_t session)
{
    log_event_t event;

    if (session) {	
	event.eventdata.logintrapdata.loginname = eric_session_get_user(session);
	event.eventdata.logintrapdata.remote_host = eric_session_get_ip_str(session);
	event.trapoid = "SecurityViolation";
	eric_notify_post_event(&event, "security", PP_NOTIFY_EVENT_LEGACY);
    }
}

void eric_notify_power_supply_change(int outlet)
{
    log_event_t event; 
    int stringsize = 25;
    char powersupplystring[stringsize];
    snprintf ( powersupplystring ,  stringsize , "%s%d", "PowerOutlet" , outlet );
    event.eventcode = ERIC_EC_Device_PowerSupply_Status_Changed;
    event.trapname = "powerNotification";
    event.objectname = "DKX";
    event.objectinstance = "DominionKX";
    event.trapoid = "powerNotification";	
    event.eventdata.powersupplystatustrapdata.outletname = powersupplystring;
    event.eventdata.powersupplystatustrapdata.status =  "OFF"	;
    eric_notify_post_event((void*)&event, "device", PP_NOTIFY_EVENT_LEGACY);
}
/**
 * get the count of event group entries
 * thats not the count of log entries in the internal log,
 * but the count of event groups, like auth, device, etc.
 * these groups have to be registrated at the begin of eric process
 */
int
eric_notify_get_object_cnt(void)
{
    return event_types_h ? pp_hash_get_entry_count(event_types_h) : 0;
}

/**
 * just wrappers
 */
evt_type_entry_t*
eric_notify_get_first_event_obj(void)
{
    return event_types_h ? pp_hash_get_first_entry(event_types_h) : NULL;  
}

/**
 * dito
 */
evt_type_entry_t*
eric_notify_get_next_event_obj(void)
{
    return event_types_h ? pp_hash_get_next_entry(event_types_h) : NULL;
}

/**
 * dito
 */
evt_type_entry_t*
eric_notify_get_event_object(const char *name)
{
    return event_types_h ? pp_hash_get_entry(event_types_h, name) : NULL;
}

/**
 * register an log event group 
 */
int 
eric_notify_register_log_object(const char *short_name, const char *long_name, int flags)
{
    char* config_val;
    evt_type_entry_t *object;  
    
    if (!event_types_h) event_types_h = pp_hash_create(30);              
    
    object = calloc(1, sizeof(evt_type_entry_t));
    object->name = strdup(short_name);
    object->longname = strdup(long_name);

    if (PP_SUCCED(pp_cfg_get_nodflt(&config_val, "log.event[%s]", short_name))) {
	if (strstr(config_val, "log")) object->list_enabled = 1;
	if (strstr(config_val, "snmp")) object->snmp_enabled = 1;
	if (strstr(config_val, "smtp")) object->smtp_enabled = 1;
	if (strstr(config_val, "nfs")) object->nfs_enabled = 1;
	if (strstr(config_val, "syslog")) object->syslog_enabled = 1;
	free(config_val);
    }
    
    object->flags = flags;
    
    pp_hash_set_entry(event_types_h, short_name, object, evt_type_lst_entry_free);
    
    return PP_SUC;
}

static void
evt_type_lst_entry_free(void* list_entry)
{
    evt_type_entry_t *entry = (evt_type_entry_t *) list_entry;

    if (entry) {
        free(entry->name);
	free(entry->longname);
	free(entry);
    }
}

static u_int
get_log_media(const char * event_id)
{
    u_int ret = 0;
    
    if (event_id) {
	evt_type_entry_t * evt_type = eric_notify_get_event_object(event_id);
	int enabled;

	pp_cfg_is_enabled(&enabled, "log.snmp.enabled");
	if (enabled && evt_type->snmp_enabled && (evt_type->flags & PP_NOTIFY_EVENT_SNMP_ENABLE)) {
	    ret |= PP_NOTIFY_EVENT_SNMP_ENABLE;
	}

	pp_cfg_is_enabled(&enabled, "log.smtp.enabled");
	if (enabled && evt_type->smtp_enabled && (evt_type->flags & PP_NOTIFY_EVENT_SMTP_ENABLE)) {
	    ret |= PP_NOTIFY_EVENT_SMTP_ENABLE;
	}

	pp_cfg_is_enabled(&enabled, "log.list.enabled"); 
	if (enabled && evt_type->list_enabled && (evt_type->flags & PP_NOTIFY_EVENT_LIST_ENABLE)) {
	    ret |= PP_NOTIFY_EVENT_LIST_ENABLE;
	}

	pp_cfg_is_enabled(&enabled, "log.nfs.enabled"); 
	if (enabled && evt_type->nfs_enabled && (evt_type->flags & PP_NOTIFY_EVENT_NFS_ENABLE)) {
	    ret |= PP_NOTIFY_EVENT_NFS_ENABLE;
	}

	pp_cfg_is_enabled(&enabled, "log.syslog.enabled");
        if (enabled && evt_type->nfs_enabled && (evt_type->flags & PP_NOTIFY_EVENT_SYSLOG_ENABLE)) {
            ret |= PP_NOTIFY_EVENT_SYSLOG_ENABLE;
        }
    }

    return ret;
}
