#include <fcntl.h>
#include <pthread.h>
#include <regex.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include "liberic_notify.h"
#include "snmptrap.h"
#if !defined(PRODUCT_KX2)
#define MIB_PATH "/lib/snmp/mibs/LARA-MIB.txt"
#else
#define MIB_PATH "/lib/snmp/mibs/RemoteKVMDevice-MIB.txt"
#endif
#define MAXCMDLEN 512
#define TRAP_DEBUG 0

static int initialized = 0;
char eric_snmp_mibname[64];

int
snmp_log_init(void)
{
    int mibfd = -1;
    int ret = PP_ERR;

    if (!initialized) {
	char buffer[1024];
	regex_t rgexp;
	regmatch_t rgmatch[1];
	int r;

	if ((mibfd = open(MIB_PATH, O_RDONLY)) < 0) {
	    pp_log_err("%s(): open(%s) failed", ___F, MIB_PATH);
	    goto done;
	}

	if (read(mibfd, buffer, sizeof(buffer)) < 0) {
	    pp_log_err("%s(): read(%s) failed", ___F, MIB_PATH);
	    goto done;
	}

	if ((r = regcomp(&rgexp, "DEFINITIONS ::= BEGIN", REG_ICASE)) != 0) {
	    pp_log("%s(): regcomp() failed with %d\n", ___F, r);
	    goto done;
	}

	if ((r = regexec(&rgexp, buffer, 1, rgmatch, 0)) != 0) {
	    if (r == REG_NOMATCH) {
		pp_log("%s(): regexec() found no match. Check syntax of '%s'!", ___F, MIB_PATH);
	    } else {
		pp_log("%s(): regexec() failed with %d\n", ___F, r);
	    }
	    goto done;
	}

	if (rgmatch[0].rm_so >= 0) {
	    size_t size = min((u_int)rgmatch[0].rm_so, sizeof(eric_snmp_mibname));
	    snprintf(eric_snmp_mibname, size, "%s", buffer);
	} else {
	    assert(0);
	    goto done;
	}
	regfree(&rgexp);

	initialized = 1;
    }

    ret = PP_SUC;

 done:
    if (mibfd >= 0) close(mibfd);
    return ret;
}

void
snmp_log_cleanup(void)
{
    initialized = 0;
    *eric_snmp_mibname = '\0';
}

static int
do_send_trap(const char * cmd)
{
    int r;

#if TRAP_DEBUG
    pp_log("%s(): execute: %s\n", ___F, cmd);
#endif

    r = pp_system(cmd);

#if TRAP_DEBUG
    pp_log("%s(): pp_system() returned with %d\n", ___F, r);
#endif

    return  r ? PP_ERR : PP_SUC;
}

/*
 * FIXME: (rwa) Traps disabled because not referenced anywhere yet
 *              We might need to discuss first who to use the traps
 *              in some generic way via eric_notify_post_event.
 */
#if defined(PRODUCT_KX2) 

// System Events 
// rebootStarted
static int
sendSystemShutdownTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logininfo.loginname);
	ret = do_send_trap(cmd);
    }
    return ret;
}

// rebootCompleted
static int
sendSystemStartupTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendFactoryResetTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logininfo.loginname,
		 eric_snmp_mibname, event->eventdata.logininfo.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendStartManagementTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logininfo.loginname,
		 eric_snmp_mibname, event->eventdata.logininfo.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendStopManagementTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logininfo.loginname,
		 eric_snmp_mibname, event->eventdata.logininfo.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

// Device Events

static int
sendDeviceUpdateStartedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s "
		 "%s::imageVersion s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logininfo.loginname,
		 eric_snmp_mibname, event->eventdata.logininfo.remote_host,
		 eric_snmp_mibname, event->eventdata.upgradetrapdata.imageversion);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendDeviceUpdateCompletedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s "
		 "%s::imageversion s %s "
		 "%s::status s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logininfo.loginname,
		 eric_snmp_mibname, event->eventdata.logininfo.remote_host,
		 eric_snmp_mibname, event->eventdata.upgradetrapdata.imageversion,
		 eric_snmp_mibname, event->eventdata.upgradetrapdata.status);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendConfigBackupTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s "
		 "%s::fileType s %s "
		 "%s::fileVersion s %s "
		 "%s::status s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logininfo.loginname,
		 eric_snmp_mibname, event->eventdata.logininfo.remote_host,
		 eric_snmp_mibname, event->eventdata.backuprestoretrapdata.type,
		 eric_snmp_mibname, event->eventdata.backuprestoretrapdata.fileversion,
		 eric_snmp_mibname, event->eventdata.backuprestoretrapdata.status);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendConfigRestoreTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s "
		 "%s::fileType s %s "
		 "%s::fileVersion s %s "
		 "%s::status s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logininfo.loginname,
		 eric_snmp_mibname, event->eventdata.logininfo.remote_host,
		 eric_snmp_mibname, event->eventdata.backuprestoretrapdata.type,
		 eric_snmp_mibname, event->eventdata.backuprestoretrapdata.fileversion,
		 eric_snmp_mibname, event->eventdata.backuprestoretrapdata.status);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendPowerSupplyStatusChangedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::outletName s %s "
		 "%s::status s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.powersupplystatustrapdata.outletname,
		 eric_snmp_mibname, event->eventdata.powersupplystatustrapdata.status);
	ret = do_send_trap(cmd);
    }
    return ret;
}

// Port Events

static int
sendPortStatusChangedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];
    char *devName =NULL;
    /* FIXME: devName must be set somewhere */

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::deviceName s %s "
		 "%s::portName s %s "
		 "%s::portStatus s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, devName,
		 eric_snmp_mibname, event->eventdata.porttrapdata.portname,
		 eric_snmp_mibname, event->eventdata.porttrapdata.portstatus);
	ret = do_send_trap(cmd);
    }
    return ret;
}

// User Events

static int
sendUserAddedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::tartetUser s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.usertrapdata.username,
		 eric_snmp_mibname, event->eventdata.usertrapdata.targetuser);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendUserChangedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::tartetUser s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.usertrapdata.username,
		 eric_snmp_mibname, event->eventdata.usertrapdata.targetuser);
	ret = do_send_trap(cmd);
    }
    return ret;
}


static int
sendUserDeletedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::tartetUser s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.usertrapdata.username,
		 eric_snmp_mibname, event->eventdata.usertrapdata.targetuser);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendUserPasswordChangedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::targetUser s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.usertrapdata.username,
		 eric_snmp_mibname, event->eventdata.usertrapdata.targetuser,
		 eric_snmp_mibname, event->eventdata.logininfo.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

// Group Events

static int
sendGroupAddedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if ( ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::groupName s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.grouptrapdata.username,
		 eric_snmp_mibname, event->eventdata.grouptrapdata.groupname);
	ret = do_send_trap(cmd);
    }
    return ret;
}


static int
sendGroupModifiedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::groupName s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.grouptrapdata.username,
		 eric_snmp_mibname, event->eventdata.grouptrapdata.groupname);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendGroupDeletedTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::groupName s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.grouptrapdata.username,
		 eric_snmp_mibname, event->eventdata.grouptrapdata.groupname);
	ret = do_send_trap(cmd);
    }
    return ret;
}

//Access Events

static int
sendUserLoginTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logintrapdata.loginname,
		 eric_snmp_mibname, event->eventdata.logintrapdata.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendUserLogoutTrap(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logintrapdata.loginname,
		 eric_snmp_mibname, event->eventdata.logintrapdata.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

// userAuthenticationFailure
static int
sendAccessLoginFailed(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logintrapdata.loginname,
		 eric_snmp_mibname, event->eventdata.logintrapdata.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendAccessConnectionLost(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logintrapdata.loginname,
		 eric_snmp_mibname, event->eventdata.logintrapdata.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendAccessConnectionTimeout(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::ipAddress s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logintrapdata.loginname,
		 eric_snmp_mibname, event->eventdata.logintrapdata.remote_host);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendAccessPortConnect(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::portName s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logintrapdata.loginname,
		 eric_snmp_mibname, event->eventdata.porttrapdata.portname);
	ret = do_send_trap(cmd);
    }
    return ret;
}

static int
sendAccessPortDisconnect(const log_event_t * event, const char *my_ip,
			const char * ip, const char * community)
{
    int ret = PP_ERR;
    char cmd[MAXCMDLEN];

    if (ip && community && my_ip) {
	snprintf(cmd, sizeof(cmd),
		 "/bin/snmptrap -v 2c -c %s %s '' %s::%s "
		 "%s::objectName s %s "
		 "%s::objectInstance s %s "
		 "%s::userName s %s "
		 "%s::portName s %s ",
		 community, ip, eric_snmp_mibname, event->trapname,
		 eric_snmp_mibname, event->objectname, 
		 eric_snmp_mibname, event->objectinstance,
		 eric_snmp_mibname, event->eventdata.logintrapdata.loginname,
		 eric_snmp_mibname, event->eventdata.porttrapdata.portname);
	ret = do_send_trap(cmd);
    }
    return ret;
}
#endif /* defined(PRODUCT_KX2) */

int
snmp_log_trap_send(const log_event_t * event)
{
    int ret;

    /*time_t cfg_mtime;*/
    u_int i;
    static pthread_mutex_t snmp_trap_mtx = PTHREAD_MUTEX_INITIALIZER;    
    static char * my_ip = NULL;
    static int eric_auth_traps_enabled;
    static int host_traps_enabled;
    static int ipmi_traps_enabled;
    static u_int trap_dest_cnt = 0;
    static char ** trap_dest_ips = NULL;
    static char ** trap_communities = NULL;    
    
#if TRAP_DEBUG
    pp_log("%s(): Generate Trap with oid: '%s'\n", ___F, event->trapoid);
#endif

    if (event->trapoid == NULL) {
	return -1;
    }
    /* FIXME: we shouldn't lock for the whole time */
    MUTEX_LOCK(&snmp_trap_mtx);

    /* (re)read snmp trap config if necessary */
    if (trap_dest_cnt == 0/*||eric_config_global_changed_since(&cfg_mtime)*/) {
	/* free old data */	
	free(my_ip); my_ip = NULL;
	for (i = 0; i < trap_dest_cnt; i++) {
	    if (trap_dest_ips) { free(trap_dest_ips[i]); }
	    if (trap_communities) { free(trap_communities[i]); }
	}
	free(trap_dest_ips); trap_dest_ips = NULL;
	free(trap_communities); trap_communities = NULL;
	
	/* get new data */
	if (PP_FAILED(pp_cfg_get_nodflt(&my_ip, "network.ipaddr"))) {
	    goto fail; /* we have no ipv4 address - so we do nothing here! */
	}

	pp_cfg_is_enabled(&eric_auth_traps_enabled, "snmp.trap.auth");
	pp_cfg_is_enabled(&host_traps_enabled, "snmp.trap.host");
	pp_cfg_is_enabled(&ipmi_traps_enabled, "snmp.trap.ipmi");
	pp_cfg_get_uint(&trap_dest_cnt, "snmp.trap.dest._s_");

	if (trap_dest_cnt > 0) {
	    trap_dest_ips = malloc(trap_dest_cnt * sizeof(char *));
	    trap_communities = malloc(trap_dest_cnt * sizeof(char *));
	    
	    for (i = 0; i < trap_dest_cnt; ++i) {
		char * ip = NULL;
		char * community = NULL;
		pp_cfg_get_nodflt(&ip, "snmp.trap.dest[%u].ip", i);
		pp_cfg_get_nodflt(&community, "snmp.trap.dest[%u].community", i);
		if (ip && community) {
		    trap_dest_ips[i] = ip;
		    trap_communities[i] = community;
		} else {
		    free(ip);
		    free(community);
		    --trap_dest_cnt;
		}
	    }
	}
    }

#if !defined(PRODUCT_KX2) 
    char cmd[MAXCMDLEN];
    if (strstr(event->trapoid, "Loginfailed") ||
	strstr(event->trapoid, "Loginsuccess") ||
	strstr(event->trapoid, "SecurityViolation")) {        
        
#if TRAP_DEBUG
        pp_log("%s(): user: %s\n", ___F,
		event->eventdata.logintrapdata.loginname);
#endif
	for (i = 0; i < trap_dest_cnt; i++) {
	    if (trap_dest_ips[i] && trap_dest_ips[i][0] &&
	        trap_communities[i] && trap_communities[i][0]) {                
                snprintf(cmd, sizeof(cmd),
			 "/bin/snmptrap -c %s %s %s::%s "
			 "%s 6 1 '' %s::userLoginName "
			 "s '%s' %s::remoteHost a %s &",
			 trap_communities[i], trap_dest_ips[i],
			 eric_snmp_mibname, event->trapoid,
			 my_ip, eric_snmp_mibname,
			 event->eventdata.logintrapdata.loginname,
			 eric_snmp_mibname,
			 event->eventdata.logintrapdata.remote_host);
		ret = do_send_trap(cmd);
	    }
	}        
    } else if (strstr(event->trapoid, "HostPower") ||
	       strstr(event->trapoid, "HostReset")) {        
	for (i = 0; i < trap_dest_cnt; i++) {
	    if (trap_dest_ips[i] && trap_dest_ips[i][0] &&
	        trap_communities[i] && trap_communities[i][0]) {
                snprintf(cmd, sizeof(cmd), "/bin/snmptrap -c %s %s "
			 "%s::%s %s 6 1 '' ", 
			 trap_communities[i], trap_dest_ips[i],
			 eric_snmp_mibname,
			 event->trapoid, my_ip);
		ret = do_send_trap(cmd);
	    }
	}        
    } else if (strstr(event->trapoid, "IpmiAlert")) {         
        for (i = 0; i < trap_dest_cnt; i++) {
	    if (trap_dest_ips[i] && trap_dest_ips[i][0] &&
		trap_communities[i] && trap_communities[i][0]) {
                snprintf(cmd, sizeof(cmd), "/bin/snmptrap -c %s %s "
			    "%s::%s %s 6 1 '' "
			    "%s::petTime s '%s' "
			    "%s::petSource s '%s' "
			    "%s::petType s '%s' "
			    "%s::petDesc s '%s' &", 
			    trap_communities[i], trap_dest_ips[i], eric_snmp_mibname,
			    event->trapoid, my_ip,
			    eric_snmp_mibname, event->eventdata.ipmitrapdata.evnt_time,
			    eric_snmp_mibname, event->eventdata.ipmitrapdata.alert_host,
			    eric_snmp_mibname, event->eventdata.ipmitrapdata.evnt_type,
			    eric_snmp_mibname, event->eventdata.ipmitrapdata.evnt_desc);
		ret = do_send_trap(cmd);
	    }
        }
    } else {
#if TRAP_DEBUG
	pp_log("%s(): Generic: %s\n", ___F,
		event->eventdata.generictrapdata.trap_desc);
#endif
	for (i = 0; i < trap_dest_cnt; i++) {	    
	    if (trap_dest_ips[i] && trap_dest_ips[i][0] &&
		trap_communities[i] && trap_communities[i][0]) {                
		snprintf(cmd, sizeof(cmd),
			 "/bin/snmptrap -c %s %s %s::%s "
			 "%s 6 1 '' "
			 "%s::eventType s '%s' "
			 "%s::eventDesc s '%s' &",
			 trap_communities[i], trap_dest_ips[i], eric_snmp_mibname, "Generic",
			 my_ip, eric_snmp_mibname, event->longname, eric_snmp_mibname, 
			 event->eventdata.generictrapdata.trap_desc);
		ret = do_send_trap(cmd);
	    }
	}
    }
#else /* PRODUCT_KX2 */

    for (i = 0; i < trap_dest_cnt; i++) 
    {
	switch (event->eventcode)
	{
	    case ERIC_EC_System_Startup:
		ret = sendSystemStartupTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_System_Shutdown:
		ret = sendSystemShutdownTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_System_Factory_Reset:
		ret = sendFactoryResetTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_System_Start_Management:
		ret = sendStartManagementTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_System_Stop_Management:
		ret = sendStopManagementTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    // Device Events
	    case ERIC_EC_Device_Update_Started:
		ret = sendDeviceUpdateStartedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Device_Update_Completed:
		ret = sendDeviceUpdateCompletedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Device_Config_Backup:
		ret = sendConfigBackupTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Device_Config_Restore:
		ret = sendConfigRestoreTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Device_PowerSupply_Status_Changed:
		ret = sendPowerSupplyStatusChangedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    // Port Events

	    case ERIC_EC_Port_Status_Changed:
		ret = sendPortStatusChangedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    // User Events

	    case ERIC_EC_User_Added:
		ret = sendUserAddedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_User_Changed:
		ret = sendUserChangedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_User_Deleted:
		ret = sendUserDeletedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_User_Password_Changed:
		ret = sendUserPasswordChangedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    // Group Events

	    case ERIC_EC_Group_Added:
		ret = sendGroupAddedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Group_Changed:
		ret = sendGroupModifiedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Group_Deleted:
		ret = sendGroupDeletedTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    // Access Events

	    case ERIC_EC_Access_Login:
		ret = sendUserLoginTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Access_Logout:
		ret = sendUserLogoutTrap(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Access_Login_Failed:
		ret = sendAccessLoginFailed(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Access_Connection_Lost:
		ret = sendAccessConnectionLost(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Access_Connection_Timeout:
		ret = sendAccessConnectionTimeout(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Access_Port_Connect:
		ret = sendAccessPortConnect(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;
	    case ERIC_EC_Access_Port_Disconnect:
		ret = sendAccessPortDisconnect(event, my_ip, trap_dest_ips[i], trap_communities[i]);
		break;

	    default: 
		// dont do anything.
		break;
	}
    }
#endif

 fail:
    MUTEX_UNLOCK(&snmp_trap_mtx);

    return 0;
}
