#ifndef _LIBERIC_SESSION_H
#define _LIBERIC_SESSION_H

#include <stdarg.h>
#include <sys/types.h>

#define ERIC_ID_LENGTH       64 /* must be multiple of 8 */
#define ERIC_MAX_USER_LENGTH 64
#define ERIC_RESPONSE_LENGTH 32

#define ERIC_SESSION_ERR_NO_ERROR		0
#define ERIC_SESSION_ERR_INTERNAL_ERROR		1
#define ERIC_SESSION_ERR_MAXIMUM_REACHED	2

typedef unsigned int eric_session_int_id_t;

int eric_session_init(u_int _max_sessions, u_int timeout);
void eric_session_cleanup(void);
unsigned int eric_session_get_max_count(void);
unsigned int eric_session_get_count(void);

/* @param radius_sid    sid returned from radius during authentication, 0 otherwise */
eric_session_int_id_t eric_session_create(const char * user,
					  const char * ip_str,
					  const char * nickname,
					  int radius_sid,
					  int* error);

/* @param radius_sid    sid returned from radius during authentication, 0 otherwise */
eric_session_int_id_t eric_session_create_with_id(const char * user,
						  const char * ip_str,
						  const char * nickname,
						  const char* id,
						  int radius_sid,
						  int* error);

eric_session_int_id_t eric_session_get_by_id(const char * id);
eric_session_int_id_t eric_session_get_by_challenge_response(const char *challenge, const char * response);
eric_session_int_id_t eric_session_get_by_login(const char *login);
void eric_session_release(eric_session_int_id_t);
void eric_session_for_all(void (*cb)(eric_session_int_id_t, va_list), ...);
void eric_session_touch(eric_session_int_id_t s);
const char * eric_session_get_id(eric_session_int_id_t s);
time_t eric_session_get_creation_time(eric_session_int_id_t s);
u_int eric_session_get_idle_time(eric_session_int_id_t s);
const char * eric_session_get_user(eric_session_int_id_t s);
const char * eric_session_get_nickname(eric_session_int_id_t s);
int eric_session_get_gid(eric_session_int_id_t s);
void eric_session_set_user(eric_session_int_id_t s, const char * user);
void eric_session_set_gid(eric_session_int_id_t s, int gid);
const char * eric_session_get_ip_str(eric_session_int_id_t s);
int eric_session_get_radius_sid(eric_session_int_id_t id);
int eric_session_add_destroy_cb(eric_session_int_id_t s,
				void (*arbitrary_destroy_cb)(eric_session_int_id_t sid));
int eric_session_remove_destroy_cb(eric_session_int_id_t sid,
				   void (*arbitrary_destroy_cb)(eric_session_int_id_t sid));

/**
 * For remote authentication, local databse doesn't has user record.
 * So group information is retrieved and saved in session, thus 
 * permission can be got from session.
 * Note: This API is called by pp_um_user_has_permission() for remotely 
 *       authenticated user.
 *
 * @param s                  session id
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   the desired permission value,
 *                           (eg. 'view' or 'control')
 *
 * @return                   PP_SUC if the user is this session  has the permission
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int eric_session_has_permission(eric_session_int_id_t s,
                                const char* permission_name,
                                const char* permission_value);
/**
 * Get a permission for a session.
 * Remember to free returned permission_value!
 *
 * @param s                  session id
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   return value, not set in PP_ERR case
 *
 * @return                   PP_SUC if the groups permission could be retrived,
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int eric_session_get_permission(eric_session_int_id_t s,
                                const char* permission_name,
                                char** permission_value);


/**
 * encoder descriptor
 *
 * Defines the callbacks that an forensic encoder has to
 * implement to connect to the session library. Callbacks in
 * this structure may be NULL.
 */
typedef struct {
    /**
     * Close an encoder session
     * 
     * @param id          The session ID of the session to be closed
     */
    void (*close_encoder_session)(eric_session_int_id_t id);

    /**
     * A user session has been established
     * 
     * @param id          The session ID of the new session
     */
    void (*encoder_session_opened)(eric_session_int_id_t id);

    /**
     * A user session has been closed
     * 
     * @param id          The session ID of the closed session
     */
    void (*encoder_session_closed)(eric_session_int_id_t id);
} eric_session_encoder_desc_t;

int eric_session_connect_encoder(eric_session_encoder_desc_t* encoder);

#endif /* _LIBERIC_SESSION_H */
