
/*****************************************************************************
 * Filename:    cmds_diag.c
 * Author:      Yong Zou
 *
 * DESCRIPTION: Implements diagnostic  Terminal Commands.
 *
 ****************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h> 
#include <string.h>
#include <unistd.h>

#include <terminal.h>

#include "commands.h"

#include <sys/types.h>
#include <sys/ioctl.h>

#include <pp/diagnos.h>

/*---------------- CmdSetlog ----------------*/

#define NUM_SETLOG_OPTIONS 4

static const char* CmdSetlogOptions[NUM_SETLOG_OPTIONS] = {
    "module",
    "level",
    "verbose",
    "vfield"
};

static const char* SetlogLevelOptions[] = {
    "trace",
    "debug",
    "info",
    "warn",
    "err"
};

static const char* SetlogVfieldOptions[] = {
    "ts",
    "fl",
    "mn",
    "thr"
};

static long CmdSetlog(struct arg *arg, term_cl_t * clp) 
{
    char *options[NUM_SETLOG_OPTIONS];
    int i = 0, j, found, module_id = -1, verbose;
    _dbl_log_level loglevel = DBL_TRACE_LEVEL;
    _dbl_vfield vfield = DBL_VF_TIMESTAMP;
    
    memset(&options[0], 0, sizeof(options));
    while(arg[i].type != t_endoflist){
	// parsing options
	found = 0;
	for(j=0; j<NUM_SETLOG_OPTIONS; j++){
	    if(strcmp(arg[i].value.s, CmdSetlogOptions[j]) == 0){
                found = 1; 
		i++;
                if ((arg[i].type == t_string) && (arg[i].value.s != NULL)) {
                    options[j] = arg[i].value.s;
                    i++;
                }
                else {
                    eric_term_printf(clp,
                        "Command[setlog] invalid parameter. Please try help setlog\r\n\n");
                    return 0;
                }
		break;
            }
	}

	if (found == 0) {
		eric_term_printf(clp,
                        "Command[setlog] invalid parameter. Please try help setlog\r\n\n");
                    return 0;
	}
    }

    if(options[0])
    {
        // find module id for the specific module name
        module_id = dbl_get_module_id(options[0]);
        if(module_id < 0){
	     eric_term_printf(clp,
                "Command[setlog] module is not found. Please check\r\n\n");
            return 0;
	}
    }

    if(options[1]) {
	// log level control
	found = 0;
	for(i=0; i<(int)(sizeof(SetlogLevelOptions)/sizeof(char *)); i++){
	    if(strcmp(options[1], SetlogLevelOptions[i]) == 0){
		found = 1;
		loglevel += i;
		break;
	    }
	}

	if(!found){
	    eric_term_printf(clp,
                "Command[setlog] invalid loglevel. Please try help setlog\r\n\n");
            return 0;
	}

	if(options[0])
	{
	    // set loglevle for specific module
	    dbl_set_level(module_id, loglevel);
	    eric_term_printf(clp,
                "module[%s] has been set loglevel to [%s]. \r\n\n",
		options[0], options[1]);
	}
	else
	{
	    // set loglevel for all modules
	    dbl_set_level_all(loglevel);
	    eric_term_printf(clp,
                "All modules have been set loglevel to [%s]. \r\n\n",
                options[1]);
	}
    }
	    
    if(options[2]){
	// verbose control
	if(strcmp(options[2], "on")==0) verbose = 1;
	else if(strcmp(options[2], "off")==0) verbose = 0;
	else {
	    eric_term_printf(clp,
                "Command[setlog] invalid verbose value. Please try help setlog\r\n\n");
            return 0;
	}

	if(options[3]){
	    // specific vfield
	    found = 0;
	    for(i=0; i<(int)(sizeof(SetlogVfieldOptions)/sizeof(char *)); i++){
		if(strcmp(options[3], SetlogVfieldOptions[i]) == 0){
		    found = 1;
		    vfield += i;
		    break;
		}
	    }

	    if(!found){
		eric_term_printf(clp,
		    "Command[setlog] invalid vfield. Please try help setlog\r\n\n");
		return 0;
	    }

	    dbl_set_verbose(vfield, verbose);
	    eric_term_printf(clp,
                    "vfield[%s] has been set [%s]. \r\n\n",
                    options[3], options[2]);
	}
	else {
	    dbl_set_verbose_all(verbose);
	    eric_term_printf(clp,
                    "All vfields have been set to [%s]. \r\n\n",
                    options[2]);
	}
    }

    if(!options[1] && !options[2])
    {
	// display log setting status
	if(options[0]){
	    // display specific module log setting
	    dbl_snapshot_m(module_id, (void *)clp, (pp_diag_oprintf_t)eric_term_printf);
	}
	else {
	    // display all modules log setting
	    dbl_snapshot((void *)clp, (pp_diag_oprintf_t)eric_term_printf);
	}

	// display verbose setting
    }

    return 0;
}

static int CmdSetlogHelp(const char *cmd, term_cl_t * clp) 
{
    eric_term_printf(clp,
		    "\r\nUsage: %s [module <module_name>] [level <trace | debug | info | warn | err>] [verbose <on | off>] [vfield <ts | mn |fl | thr>] \r\n\n"
		    "  set log level, set verbose, display log setting (level, verbose).\r\n"
		    "  verbose field options: \r\n"
		    "  ts   -- Timestamp \r\n"
		    "  mn   -- module name \r\n"
		    "  fl   -- file name and line number \r\n"
		    "  thr  -- thread ID. \r\n"
		    "  NOTE: \r\n"
                    "  * If module is not specified, it applies all modules. \r\n"
		    "  * verbose control is system wide. It ignores module if specified. \r\n"
		    "  * If vfield is not specified, it applies all verbose fields. \r\n\n", cmd);          
    return 0;
}


/*---------------- CmdViewstats ----------------*/

static long CmdViewstats(struct arg *arg, term_cl_t * clp)
{
    if ((arg[0].type == t_string) && (arg[0].value.s != NULL)) {
	if(strcmp(arg[0].value.s, "module") == 0) {
	    if((arg[1].type == t_string) && (arg[1].value.s != NULL)) {
		// view status for specific module
		if(dbl_take_module_snapshot(arg[1].value.s, clp, (pp_diag_oprintf_t)eric_term_printf) < 0) {
		    eric_term_printf(clp,
				    "Command[viewstats] module[%s] can not be found \r\n\n",
				    arg[1].value.s);
		    return 0;

		}
	    }
	    else {
		eric_term_printf(clp,
				"Command[viewstats] invalid parameter, please try help viewstats\r\n\n");
		return 0;
	    }
	}
	else {
	    eric_term_printf(clp,
			    "Command[viewstats] invalid parameter, please try help viewstats\r\n\n");
            return 0;
	}
    }
    else if(arg[0].type == t_endoflist) {
        // view all modules status
        dbl_take_module_snapshot(NULL, clp, (pp_diag_oprintf_t)eric_term_printf);
    }
    else {
	eric_term_printf(clp,
                        "Command[viewstats] invalid parameter, please try help viewstats\r\n\n");
        return 0;
    }

    return 0;
}

static int CmdViewstatsHelp(const char *cmd, term_cl_t * clp)
{
    eric_term_printf(clp,
                    "\r\nUsage: %s [module <module_name>] \r\n\n"
                    "  display specific module or all module status. \r\n",
                    cmd);
    return 0;
}

static long CmdTakeSnapshot(UNUSED struct arg *arg, UNUSED term_cl_t *clp)
{
    take_snapshot();
    eric_term_printf(clp,
                    "Snapshot is done! \r\n\n");

    return 0;
}

static int CmdTakeSnapshotHelp(UNUSED const char *cmd, UNUSED term_cl_t *clp)
{
    eric_term_printf(clp,
		    "\r\nUsage: snapshot \r\n\n"
		    "  Collect all debug infomation into a snapshot file. \r\n");
                       
    return 0;
}

/* 
 * Diagnostic commands defines
 */
static struct command DiagCmds[] = {
    {"setlog",		NULL, CmdSetlog,    CmdSetlogHelp},
    {"viewstats",       NULL, CmdViewstats, CmdViewstatsHelp},
    {"snapshot",        NULL, CmdTakeSnapshot, CmdTakeSnapshotHelp},
};


/*
 * register all Diagnostic commands into term
 */
void
add_diag_cmds(term_cl_t * clp)
{
    unsigned int i;

    for (i = 0; i < sizeof(DiagCmds) / sizeof(struct command); i++) {
        term_insert_command(&DiagCmds[i], clp);
    }
}


void
remove_diag_cmds(term_cl_t * clp)
{
    unsigned int i;

    for (i = 0; i < sizeof(DiagCmds) / sizeof(struct command); i++) {
        term_remove_command(&DiagCmds[i], clp);
    }
}




