#ifndef _TERM_INTERNAL_H
#define _TERM_INTERNAL_H

#include <pthread.h>
#include <semaphore.h>
#include <pp/base.h>
#include <liberic_net.h>
#include <liberic_session.h>
#include <openssl/bio.h>
#include <features.h>

#ifdef PP_FEAT_CLP
#include <pp/clp.h>
#endif

#include "liberic_term.h"
#include "telnetd/defs.h"

#define c_ESC		0x1b
#define c_BS		0x08
#define c_RETURN	0x0d
#define c_DEL		0x7f
#define c_LF		0x0a

#define MAX_LOGIN_TRIES		3
#define MAX_ESC_PARAMS		16
#define MAX_KEYCODE_LIST_LEN	16
#define HOTKEY_SEQ_EXIT		1
#define HOTKEY_SEQ_BREAK	2
#define PASSTHROUGH_PORT_1	0
#define PASSTHROUGH_PORT_2	1

#define tempbuflen 256
#define maxargs	30

typedef enum {
    SUB_STATE_LOGIN,
    SUB_STATE_PASSWD,
    SUB_STATE_AUTH,
    SUB_STATE_SHELL,
    SUB_STATE_TERMINAL,
    SUB_STATE_TERMINAL_FIXED,
    SUB_STATE_CLP,
} sub_state_t;

typedef enum {
    ESC_STATE_NORMAL,
    ESC_STATE_ESC,
    ESC_STATE_SQUARE,
    ESC_STATE_GET_PARAMS,
    ESC_STATE_GOT_PARAMS,
    ESC_STATE_FUNC_KEY,
    ESC_STATE_HOTSEQ,
} esc_state_t;

struct command;

struct _term_cl_t {
    pp_net_conn_data_t * conn_data;
    int net_fd;
    int serial_id;
    int serial_fd;
    char client_ip[INET6_ADDRSTRLEN];
    char i_buf[1024];
    char * i_buf_end;
    char * i_buf_cur;
    char login[64], *loginp;
    char passwd[64], *passwdp;
    int login_tries;
    eric_session_int_id_t session;
    sub_state_t sub_state;
    esc_state_t esc_state;
    unsigned int esc_params[MAX_ESC_PARAMS];
    const char * prompt;

    /* eshell stuff */
    unsigned int old_gw, old_gh;
    unsigned int old_w, old_h;
    unsigned int old_cur_x, old_cur_y;
    unsigned char *old_buf, *new_buf;
    int NumCmds;
    char *argptrfld[maxargs];
    char **argv;
    char argc;
    char ArgStrLine[tempbuflen];
    struct command *MainCmd;
    pthread_t dumper_thd;
    int exit_dumper;
    int asc2key_map_id;
    int npar;
    unsigned int last_attrib;
    int g1_active;
    int cursor_visible;
    int hstype;
    int hsidx;
    int enable_console_input;
    
    /* telnetd stuff */
    int state;
    char options[256];
    char do_dont_resp[256];
    char will_wont_resp[256];
    int	flowmode;	/* current flow control state */
    char *terminaltype;
    char netibuf[512], *netip;
    ssize_t ncc;
    struct _clocks clocks;
    char terminalname[41];
    unsigned char subbuffer[512], *subpointer, *subend;

#ifdef PP_FEAT_CLP
    int clp_enabled;
    /* CLP parser session */
    pp_clp_session_t *clp_session;
#endif
};

struct arg {
   int	type;
   union {
      char *s;
      long *n;
   }value;
};

struct command {
   char	  cmd[16];
   struct command *subcmd;
   long	  (*cmdfunc)(struct arg *arg, term_cl_t * clp);
   int	  (*helpfunc)(const char *cmd, term_cl_t * clp);
};

extern const char * login_prompt;
extern const char * passwd_prompt;
extern const char * shell_prompt;

int esh_init(term_cl_t * clp);
int esh_cleanup(term_cl_t * clp);
int esh_parse_and_execute_cmd(term_cl_t * clp);
void switch_to_sub_state(term_cl_t * clp, sub_state_t sub_state, const char * prompt);
int send_keycodes(const char * keys) __attribute__((weak));;
void add_eric_cmds(term_cl_t * clp);
void remove_eric_cmds(term_cl_t * clp);

#if defined(PP_FEAT_DIAGNOS)
void add_diag_cmds(term_cl_t * clp);
void remove_diag_cmds(term_cl_t * clp);
#endif // #if defined(PP_FEAT_DIAGNOS)

int start_telnet(term_cl_t * clp);

/* net */

ssize_t net_read(term_cl_t * clp, void * buf, size_t len, int no_timeout);
ssize_t net_write(term_cl_t * clp, const void * buf, size_t len, int no_timeout);

static inline int
net_getchar(term_cl_t * clp)
{
    if (net_read(clp, clp->i_buf, 1, 1) < 0) return -1;
    return clp->i_buf[0];
}

static inline void
net_putchar(term_cl_t * clp, char c)
{
    if (net_write(clp, &c, 1, 0) < 0) pthread_exit(NULL);
}


/* serial */

int serial_init(void);
void serial_cleanup(void);
int serial_attach_client(term_cl_t * clp);
int serial_detach_client(term_cl_t * clp);
int serial_write_string(term_cl_t* clp, const char* c);
int serial_write_byte(term_cl_t* clp, char c);
int serial_write_break(term_cl_t* clp, int duration);

#endif /* _TERM_INTERNAL_H */
