/*
 * default.c -- Default URL handler. Includes support for ASP.
 *
 * Copyright (c) GoAhead Software Inc., 1995-2000. All Rights Reserved.
 *
 * See the file "license.txt" for usage and redistribution license requirements
 *
 */

/******************************* Description *********************************/

/*
 * This module provides default URL handling and Active Server Page support.
 *
 * In many cases we don't check the return code of calls to websWrite as
 * it is easier, smaller and non-fatal to continue even when the requesting
 * browser has gone away.
 */

#include "wsIntrn.h"
#include "eric_base.h"
#include <pp/intl.h>

static char * websDefaultPage;	/* Default page name */
static char * websDefaultDir;		/* Default Web page directory */

#if defined(PP_FEAT_WS_MANAGEMENT)

static websErrorType wsmanErrors[] = {
    { 200, "OK" },
    { 500, "WS-Management Fault" },
    { 0, NULL }
};

/*
 * Return the error message for a given code
 */

static const char *
wsmanErrorMsg(int code)
{
    websErrorType *ep;

    for (ep = wsmanErrors; ep->code; ep++) {
	if (code == ep->code) {
	    return ep->msg;
	}
    }
    a_assert(0);
    return "";
}

#endif /* PP_FEAT_WS_MANAGEMENT */

/*
 * Process a default URL request. This will validate the URL and handle "../"
 * and will provide support for Active Server Pages. As the handler is the
 * last handler to run, it always indicates that it has handled the URL 
 * by returning 1. 
 */

int
websDefaultHandler(webs_t wp, char *urlPrefix UNUSED, char *webDir UNUSED,
		   int arg UNUSED, char *url, char *path,
		   char *query UNUSED)
{
    websStatType	sbuf;
    char *		lpath = NULL;
    char *		tmp;
    char *		date;
    int			nchars, nchars_path;

    if (wp->download_cb == NULL) {
	/*
	 * Validate the URL and ensure that ".."s don't give access to
	 * unwanted files
	 */
	if (websValidateUrl(wp, path) < 0) {
	    websError(wp, 500, _("Invalid URL %s"), url);
	    return 1;
	}
	lpath = wp->lpath;
	nchars = strlen(lpath) - 1;
	if (lpath[nchars] == '/' || lpath[nchars] == '\\') {
	    lpath[nchars] = '\0';
	}

	nchars_path = strlen(path) - 1;	

	/*
	 * If the file is a directory, redirect using the nominated
	 * default page
	 */
	if (websPageIsDirectory(lpath) ||
	    path[nchars_path] == '/' || path[nchars_path] == '\\') {
	    
	    nchars = strlen(path);
	    if (path[nchars-1] == '/' || path[nchars-1] == '\\') {
		path[--nchars] = '\0';
	    }
	    nchars += strlen(websDefaultPage) + 2;
	    fmtAlloc(&tmp, nchars, "%s/%s", path, websDefaultPage);
	    websRedirect(wp, tmp, NULL);
	    bfreeSafe(B_L, tmp);
	    return 1;
	}
	/*
	 * Open the document. Stat for later use.
	 */	
	if (websPageOpen(wp, lpath, path, O_RDONLY, 0666) < 0) {
	    websError(wp, 400, _("Cannot open URL <b>%s</b>"), url);
	    return 1;
	}
	/*
	 * Stat document for later use.
	 */
	if (websPageStat(wp, lpath, path, &sbuf) < 0) {
	    websError(wp, 400, _("Cannot stat page for URL <b>%s</b>"), url);
	}
    }

    /*
     * Output the normal HTTP response header
     */
    if ((date = websGetDateString(NULL)) != NULL) {
#if defined(PP_FEAT_WS_MANAGEMENT)
	if (wp->wsman_code) {
	    websWrite(wp, "HTTP/1.0 %d %s\r\nDate: %s\r\n",
		      wp->wsman_code,
		      wsmanErrorMsg(wp->wsman_code),
		      date);
	} else {
#endif /* PP_FEAT_WS_MANAGEMENT */
	    websWrite(wp, "HTTP/1.0 200 OK\r\nDate: %s\r\n", date);
#if defined(PP_FEAT_WS_MANAGEMENT)
    	}
#endif /* PP_FEAT_WS_MANAGEMENT */

	/*
	 * By license terms the following line of code must not be
	 * modified.
	 */
	websWrite(wp, "Server: %s\r\n", WEBS_NAME);
	bfree(B_L, date);
    }
    wp->flags |= WEBS_HEADER_DONE;

    if (wp->download_cb && wp->download_name) {
	websWrite(wp, "Content-Disposition: attachment; filename=\"%s\"\r\n",
		  wp->download_name);
    }
  
    /* FIXME: generic method for special files */
    if ((wp->flags & WEBS_ASP) || wp->download_cb || 
        (strstr(url, "screenshot")) ) {
	
	if ((wp->flags & WEBS_SECURE) &&
	    wp->userAgent && strstr(wp->userAgent, "MSIE")) {
	    /*
	     * MSIE has a bug that prevents a file from beeing saved to disk
	     * if requested via HTTPS. This is a workaround.
            */    
	    websWrite(wp, "Pragma: public\r\nCache-Control: public\r\n");
	} else {	      
	    websWrite(wp, "Pragma: no-cache\r\nCache-Control: no-cache\r\n");
	}
	if (strstr(url, "screenshot")) {
	    websWrite(wp, "Expires: -1\r\n");
	}
    } else {
	if ((date = websGetDateString(&sbuf)) != NULL) {
	    websWrite(wp, "Last-modified: %s\r\n", date);
	    bfree(B_L, date);
	}
	wp->numbytes = sbuf.size;
    }

    if (wp->numbytes) {
	websWrite(wp, "Content-length: %d\r\n", wp->numbytes);
    }
    const char * encoding = pp_intl_get_charset();
    if (encoding && !strncasecmp(wp->type, "text", strlen("text"))) {
        websWrite(wp, "Content-Type: %s; charset=%s\r\n", wp->type, encoding);
    } else {
        websWrite(wp, "Content-Type: %s\r\n", wp->type);
    }

    if ((wp->flags & WEBS_KEEP_ALIVE) && !(wp->flags & WEBS_ASP)) {
	websWrite(wp, "Connection: keep-alive\r\n");
    }
    websWrite(wp, "\r\n");

    /*
     * All done if the browser did a HEAD request
     */
    if (wp->flags & WEBS_HEAD_REQUEST) {
	websDone(wp, 200);
	return 1;
    }
    
    if (wp->download_cb) {
	wp->download_cb(wp);
    } else if (wp->flags & WEBS_ASP) {
	if (websAspRequest(wp, lpath) >= 0) {
	    websDone(wp, 200);
	}
    } else {
	default_download_cb(wp);
    }

    return 1;
}

/*
 * Validate the URL path and process ".." path segments. Return -1 if the URL
 * is bad.
 */

int
websValidateUrl(webs_t wp, char *path)
{
    char	*parts[64];		/* Array of ptr's to URL parts */
    char	*token, *dir, *lpath;
    char	*tok_ptr;
    int		i, len, npart;

    a_assert(wp);
    a_assert(path);

    dir = wp->dir;
    if (dir == NULL || *dir == '\0') {
	return -1;
    }

    /*
     *	Copy the string so we don't destroy the original
     */
    path = bstrdup(B_L, path);
    websDecodeUrl(path, path, strlen(path));

    len = npart = 0;
    parts[0] = NULL;
    token = strtok_r(path, "/", &tok_ptr);

    /*
     *	Look at each directory segment and process "." and ".." segments
     *	Don't allow the browser to pop outside the root web. 
     */
    while (token != NULL) {
	if (strcmp(token, "..") == 0) {
	    if (npart > 0) {
		npart--;
	    }

	} else if (strcmp(token, ".") != 0) {
	    parts[npart] = token;
	    len += strlen(token) + 1;
	    npart++;
	}
	token = strtok_r(NULL, "/", &tok_ptr);
    }

    /*
     *	Create local path for document. Need extra space all "/" and null.
     */
    if (npart || (strcmp(path, "/") == 0) || (path[0] == '\0')) {
	lpath = balloc(B_L, (strlen(dir) + 1 + len + 1) * sizeof(char));
	strcpy(lpath, dir);

	for (i = 0; i < npart; i++) {
	    strcat(lpath, "/");
	    strcat(lpath, parts[i]);
	}
	websSetRequestLpath(wp, lpath);
	bfree(B_L, path);
	bfree(B_L, lpath);

    } else {
	bfree(B_L, path);
	return -1;
    }
    return 0;
}

void
mem_image_download_cb(webs_t wp)
{
    if (wp->download_image != NULL &&
	websWriteBlock(wp, wp->download_image, wp->numbytes) > 0) {

	wp->written += wp->numbytes;
    }

    if (wp->written >= wp->numbytes) {
	websDone(wp, 200);
    } else {
	websDone(wp, 0);
    }
}

#if defined(PP_FEAT_WS_MANAGEMENT)

void
wsman_download_cb(webs_t wp)
{
    mem_image_download_cb(wp);

    if (wp->written >= wp->numbytes) {
    	free(wp->wsman_response);
    	wp->wsman_response = NULL;
    }
}

#endif

void
default_download_cb(webs_t wp)
{
    int len;
    char * buf;

    a_assert(wp);

    /*
     * Note: websWriteBlock may return less than we wanted. It will
     *       return -1 on a socket error
     */
    if ((buf = balloc(B_L, PAGE_READ_BUFSIZE)) == NULL) {
	websError(wp, 200, _("Can't get memory"));
    } else {
	while ((len = websPageReadData(wp, buf, PAGE_READ_BUFSIZE)) > 0) {
	    if (websWriteBlock(wp, buf, len) < 0) { break; }
	    wp->written += len;
	}
	/*
	 * Safety. If we are at EOF, we must be done
	 */
	if (len == 0) {
	    a_assert(wp->written >= wp->numbytes);
	    wp->written = wp->numbytes;
	}
	bfree(B_L, buf);
    }

    /*
     * We're done if an error, or all bytes output
     */
    if (wp->written >= wp->numbytes) {
	websDone(wp, 200);
    } else {
	websDone(wp, 0);
    }
}

/* 
 * Closing down. Free resources.
 */

void websDefaultClose()
{
    if (websDefaultPage) {
	bfree(B_L, websDefaultPage);
	websDefaultPage = NULL;
    }
    if (websDefaultDir) {
	bfree(B_L, websDefaultDir);
	websDefaultDir = NULL;
    }
}

/*
 * Get the default page for URL requests ending in "/"
 */

char *
websGetDefaultPage()
{
    return websDefaultPage;
}

/*
 * Get the default web directory
 */

char *
websGetDefaultDir()
{
    return websDefaultDir;
}

/*
 * Set the default page for URL requests ending in "/"
 */

void
websSetDefaultPage(const char * page)
{
    a_assert(page && *page);

    if (websDefaultPage) {
	bfree(B_L, websDefaultPage);
    }
    websDefaultPage = bstrdup(B_L, page);
}

/*
 * Set the default web directory
 */

void
websSetDefaultDir(const char * dir)
{
    a_assert(dir && *dir);
    if (websDefaultDir) {
	bfree(B_L, websDefaultDir);
    }
    websDefaultDir = bstrdup(B_L, dir);
}
