/* 
 * ejIntrn.h -- Ejscript(TM) header
 *
 * Copyright (c) GoAhead Software, Inc., 1992-2000
 *
 * See the file "license.txt" for information on usage and redistribution
 */

#ifndef _EJINTRN_H
#define _EJINTRN_H 1

/******************************* Description *********************************/

/* 
 * GoAhead Ejscript(TM) header. This defines the Ejscript API and internal
 * structures.
 */

/******************************** Includes ***********************************/

#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>

#include "uemf.h"
#include "ej.h"

/********************************* Defines ***********************************/

/*
 * Constants
 */
#define EJ_INC			110	/* Growth for tags/tokens */
#define EJ_SCRIPT_INC		1023	/* Growth for ej scripts */
#define EJ_OFFSET		1	/* hAlloc doesn't like 0 entries */
#define EJ_MAX_RECURSE		100	/* Sanity for maximum recursion */

/*
 * Ejscript Lexical analyser tokens
 */
#define TOK_ERR			-1	/* Any error */
#define TOK_LPAREN		1	/* ( */
#define TOK_RPAREN		2	/* ) */
#define TOK_IF			3	/* if */
#define TOK_ELSE		4	/* else */
#define TOK_LBRACE		5	/* { */
#define TOK_RBRACE		6	/* } */
#define TOK_LOGICAL		7	/* ||, &&, ! */
#define TOK_EXPR		8	/* +, -, /, % */
#define TOK_SEMI		9	/* ; */
#define TOK_LITERAL		10	/* literal string */
#define TOK_FUNCTION		11	/* function name */
#define TOK_NEWLINE		12	/* newline white space */
#define TOK_ID			13	/* function name */
#define TOK_EOF			14	/* End of script */
#define TOK_COMMA		15	/* Comma */
#define TOK_VAR			16	/* var */
#define TOK_ASSIGNMENT		17	/* = */
#define TOK_FOR			18	/* for */
#define TOK_INC_DEC		19	/* ++, -- */
#define TOK_RETURN		20	/* return */

/*
 * Expression operators
 */
#define EXPR_LESS		1	/* < */
#define EXPR_LESSEQ		2	/* <= */
#define EXPR_GREATER		3	/* > */
#define EXPR_GREATEREQ		4	/* >= */
#define EXPR_EQ			5	/* == */
#define EXPR_NOTEQ		6	/* != */
#define EXPR_PLUS		7	/* + */
#define EXPR_MINUS		8	/* - */
#define EXPR_DIV		9	/* / */
#define EXPR_MOD		10	/* % */
#define EXPR_LSHIFT		11	/* << */
#define EXPR_RSHIFT		12	/* >> */
#define EXPR_MUL		13	/* * */
#define EXPR_ASSIGNMENT		14	/* = */
#define EXPR_INC		15	/* ++ */
#define EXPR_DEC		16	/* -- */
#define EXPR_BOOL_COMP		17	/* ! */
/*
 * Conditional operators
 */
#define COND_AND		1	/* && */
#define COND_OR			2	/* || */
#define COND_NOT		3	/* ! */

/*
 * States
 */
#define STATE_ERR		-1	/* Error state */
#define STATE_EOF		1	/* End of file */
#define STATE_COND		2	/* Parsing a "(conditional)" stmt */
#define STATE_COND_DONE		3
#define STATE_RELEXP		4	/* Parsing a relational expr */
#define STATE_RELEXP_DONE	5
#define STATE_EXPR		6	/* Parsing an expression */
#define STATE_EXPR_DONE		7
#define STATE_STMT		8	/* Parsing General statement */
#define STATE_STMT_DONE		9
#define STATE_STMT_BLOCK_DONE	10	/* End of block "}" */
#define STATE_ARG_LIST		11	/* Function arg list */
#define STATE_ARG_LIST_DONE	12
#define STATE_DEC_LIST		16	/* Declaration list */
#define STATE_DEC_LIST_DONE	17
#define STATE_DEC		18
#define STATE_DEC_DONE		19

#define STATE_RET		20	/* Return statement */

#define STATE_BEGIN		STATE_STMT

/*
 * Flags. Used in ej_t and as parameter to parse()
 */
#define FLAGS_EXE		0x1	/* Execute statements */
#define FLAGS_VARIABLES		0x2	/* Allocated variables store */
#define FLAGS_FUNCTIONS		0x4	/* Allocated function store */

/*
 * Function call structure
 */
typedef struct {
    char	*fname;			/* Function name */
    char	**args;			/* Args for function (halloc) */
    int		nArgs;			/* Number of args */
} ejfunc_t;

/*
 * EJ evaluation block structure
 */
typedef struct ejEval {
    ringq_t	tokbuf;			/* Current token */
    ringq_t	script;			/* Input script for parsing */
    char	*putBackToken;		/* Putback token string */
    int		putBackTokenId;		/* Putback token ID */
    char	*line;			/* Current line */
    int		lineLength;		/* Current line length */
    int		lineNumber;		/* Parse line number */
    int		lineColumn;		/* Column in line */
} ejinput_t;

/*
 * Per Ejscript session structure
 */
typedef struct ej {
    ejinput_t	*input;			/* Input evaluation block */
    sym_fd_t	functions;		/* Symbol table for functions */
    sym_fd_t	*variables;		/* hAlloc list of variables */
    int		variableMax;		/* Number of entries */
    ejfunc_t	*func;			/* Current function */
    char	*result;		/* Current expression result */
    char	*error;			/* Error message */
    const char	*token;			/* Pointer to token string */
    int		tid;			/* Current token id */
    int		eid;			/* Halloc handle */
    int		flags;			/* Flags */
    int		userHandle;		/* User defined handle */
    const char	*user;			/* name of the current user */
    webs_t	wp;			/* webs_t */
} ej_t;

/******************************* Prototypes **********************************/

int	ejOpenBlock(int eid);
int	ejCloseBlock(int eid, int vid);
char	*ejEvalBlock(int eid, char *script, char **emsg);
int	ejRemoveGlobalFunction(int eid, const char *name);
void	*ejGetGlobalFunction(int eid, const char *name);
int 	ejSetGlobalFunctionDirect(sym_fd_t functions, const char *name, 
				  int (*fn)(int eid, void *handle,
					    int argc, char **argv));
void 	ejError(ej_t* ep, const char* fmt, ...);
void	ejSetUserHandle(int eid, int handle);
int	ejGetUserHandle(int eid);
int	ejGetLineNumber(int eid);
char	*ejGetResult(int eid);
void	ejSetLocalVar(int eid, const char *var, const char *value);
void	ejSetGlobalVar(int eid, const char *var, const char *value);

int 	ejLexOpen(ej_t* ep);
void 	ejLexClose(ej_t* ep);
int	ejLexOpenScript(ej_t* ep, char *script);
void 	ejLexCloseScript(ej_t* ep);
void 	ejLexSaveInputState(ej_t* ep, ejinput_t* state);
void 	ejLexFreeInputState(ej_t* ep, ejinput_t* state);
void 	ejLexRestoreInputState(ej_t* ep, ejinput_t* state);
int	ejLexGetToken(ej_t* ep, int state);
void	ejLexPutbackToken(ej_t* ep, int tid, const char *s);

sym_fd_t	ejGetVariableTable(int eid);
sym_fd_t	ejGetFunctionTable(int eid);

int	ejEmfOpen(int eid);
void	ejEmfClose(int eid);

int	ejEmfDbRead(int eid, void *handle, int argc, char **argv);
int	ejEmfDbReadKeyed(int eid, void *handle, int argc, char **argv);
int	ejEmfDbTableGetNrow(int eid, void *handle, int argc, char **argv);
int	ejEmfDbDeleteRow(int eid, void *handle, int argc, char **argv);
int	ejEmfTrace(int eid, void *handle, int argc, char **argv);
int	ejEmfDbWrite(int eid, void *handle, int argc, char **argv);
int	ejEmfDbCollectTable(int eid, void *handle, int argc, char **argv);

#endif /* _EJINTRN_H */
