#include <unistd.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <time.h>

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <pp/hal_common.h>
#include <pp/firmware.h>
#if defined(PP_FEAT_AD_ATMEL) || defined(PP_FEAT_PCI_ADC)
 #include <pp/sense.h>
#endif
#if defined(PP_FEAT_MASS_STORAGE)
#include <pp/usb.h>
#endif /* PP_FEAT_MASS_STORAGE */
#if defined(PP_FEAT_WS_MANAGEMENT)
#include <pp/wsman.h>
#endif /* PP_FEAT_MASS_STORAGE */
#include <pp/intl.h>
#include <liberic_misc.h>
#include <liberic_notify.h>
#if defined(PP_FEAT_REMOTE_CONSOLE)
#include <pp/kvm.h>
#include <pp/km.h>
#include <pp/rfb.h>
#endif /* PP_FEAT_REMOTE_CONSOLE */
#include <liberic_session.h>
#include <liberic_net.h>
// really needed? #include <lara.h>
#include "eric_base.h"
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_internal.h"
#include "tmpl_firmware_common.h"
#ifdef PP_FEAT_MASS_STORAGE
#include "tmpl_vfloppy_common.h"
#endif
#include "webs.h"
#include "wsIntrn.h"
#include "gd_images.h"

pp_hash_i_t* webs_session_data = NULL;

eric_webs_encoder_desc_t *encoder_hook = NULL;

static const char* UA_ID_WINCE        = "Windows CE";
static const char* UA_ID_WINCE_JEODE  = "Jeode";
static const char* UA_PREFIX_WINCE    = "/pie";
static const char* UA_PREFIX_DEFAULT  = "";
static const int   UA_MAXWIDTH_WINCE  = 640;
static const int   UA_MAXHEIGHT_WINCE = 400;

static const char* OEM_DIR_PREFIX     = "/oem";

/* URLs which should not be taken from the oem directory
   btu are copied to RAM before, sync with fs-skeleton/etc/rc.oem */
static const char* const oem_exceptions[]   = { "/fwupdate.asp", NULL };

static int eric_url_handler(webs_t wp, char * urlPrefix, char * webdir,
			    int arg, char * url, char * path, char * query);
static int tmpl_begin_end_asp(int eid, webs_t wp, int argc, char **argv);
static int img_button_asp(int eid, webs_t wp, int argc, char **argv);
static int img_link_button_asp(int eid, webs_t wp, int argc, char **argv);
static int get_session_id_asp(int eid, webs_t wp, int argc, char **argv);

static int rfbport_asp(int eid, webs_t wp, int argc, char **argv);

static int get_response_msg_asp(int eid, webs_t wp, int argc, char **argv);
static int write_connected_users_asp(int eid, webs_t wp, int argc, char **argv);
static int write_connected_users_bar_asp(int eid, webs_t wp, int argc,char **argv);

static int write_voltages_asp(int eid, webs_t wp, int argc, char **argv);

static const char* findUserAgentPrefix(webs_t);

static int is_oem_exception(const char* url);
static void ericSetWebsRequestPath(webs_t wp, const char* uaprefix);

static int get_var_asp(int eid, webs_t wp, int argc, char **argv);
static int get_var_by_idx_asp(int eid, webs_t wp, int argc, char **argv);
static int html_esc_asp(int eid, webs_t wp, int argc, char **argv);

static int rand_asp(int eid, webs_t wp, int argc, char **argv);
static int button_asp(int eid, webs_t wp, int argc, char **argv);
static int get_hardware_rev_asp(int eid, webs_t wp, int argc, char **argv);
static int get_flashid_asp(int eid, webs_t wp, int argc, char **argv);
static int get_user_name_asp(int eid, webs_t wp, int argc, char **argv);
static int get_group_name_asp(int eid, webs_t wp, int argc, char **argv);
static int get_help_file_asp(int eid, webs_t wp, int argc, char **argv);

#ifdef PP_FEAT_USBPOWERED
static int get_external_power_status_asp(int eid, webs_t wp, int argc, char **argv);
#endif

static int use_raritan_desgin_asp(int eid, webs_t wp, int argc, char** argv);
static int sessioninfo_asp(int eid, webs_t wp, int argc, char** argv);
static int voltagesinfo_asp(int eid, webs_t wp, int argc, char** argv);

static int is_cluster_enabled_asp(int eid, webs_t wp, int argc, char **argv);
static int is_cluster_active_asp(int eid, webs_t wp, int argc, char **argv);
static int is_rfb_up_on_node_asp(int eid, webs_t wp, int argc, char **argv);

static void send_support_data(webs_t wp);

static int get_kvm_node_asp(int eid, webs_t wp, int argc, char **argv);
static int get_kvm_node_cnt_asp(int eid, webs_t wp, int argc, char **argv);

#if defined(PP_FEAT_WS_MANAGEMENT)
static int handle_wsman(webs_t wp);
#endif

static int get_host_info_asp(int eid, webs_t wp, int argc, char ** argv);
static int is_empty_asp(int eid, webs_t wp, int argc, char ** argv);

int
eric_init(void)
{
    if (forms_init() == -1
	|| translator_init() == -1
	|| acl_tmpl_init() == -1
	|| acl_ports_tmpl_init() == -1
#if defined(PP_FEAT_PDU)
	|| acl_outlets_tmpl_init() == -1
#endif
#if defined(PP_FEAT_EXTENDED_PORT_ACLS)
	|| acl_xports_tmpl_init() == -1
#endif /* PP_FEAT_EXTENDED_PORT_ACLS */
#if defined(PP_FEAT_REMOTE_CONSOLE)
	|| atx_reset_durations_tmpl_init() == -1
#endif /* PP_FEAT_REMOTE_CONSOLE */
	|| auth_tmpl_init() == -1
	|| auth_ldap_tmpl_init() == -1
	|| auth_radius_tmpl_init() == -1
	|| debug_memory_tmpl_init() == -1
	|| debug_misc_tmpl_init() == -1
#if defined(PP_FEAT_REMOTE_CONSOLE)
	|| debug_vsc_tmpl_init() == -1
#endif /* PP_FEAT_REMOTE_CONSOLE */
	|| device_prod_info_tmpl_init() == -1
	|| firmware_upload_tmpl_init() == -1
	|| firmware_update_tmpl_init() == -1
	|| flashdisk_usage_tmpl_init() == -1
	|| general_ping_tmpl_init() == -1
        || ipmi_alerts_tmpl_init() == -1
        || ipmi_filter_list_tmpl_init() == -1
        || ipmi_policy_list_tmpl_init() == -1
        || ipmi_policy_edit_tmpl_init() == -1
        || ipmi_lan_dest_list_tmpl_init() == -1
        || ipmi_filter_edit_tmpl_init() == -1
        || ipmi_lan_dest_edit_tmpl_init() == -1
        || ipmi_lan_dest_options_tmpl_init() == -1
        || ipmi_chassis_control_tmpl_init() == -1
	|| ipmi_chassis_info_tmpl_init() == -1
        || ipmi_chassis_locator_tmpl_init() == -1
        || ipmi_chassis_opma_tmpl_init() == -1
	|| ipmi_events_tmpl_init() == -1
	|| ipmi_fru_info_tmpl_init() == -1
        || ipmi_sel_tmpl_init() == -1
        || ipmi_sensors_tmpl_init() == -1
	|| ipmi_status_tmpl_init() == -1
	|| ipmi_tmpl_init() == -1
#if defined(PP_FEAT_REMOTE_CONSOLE)
	|| km_tmpl_init() == -1
	|| kvm_tmpl_init() == -1
	|| kvm_power_ctrl_tmpl_init() == -1
#endif /* PP_FEAT_REMOTE_CONSOLE */
#if defined(PP_FEAT_TARGET_PORT_SWITCHING)
	|| kvm_port_list_tmpl_init() == -1
#endif /* PP_FEAT_TARGET_PORT_SWITCHING */
	|| log_tmpl_init() == -1
	|| log_list_tmpl_init() == -1
	|| log_assign_tmpl_init() == -1
	|| log_nfs_tmpl_init() == -1
	|| log_smtp_tmpl_init() == -1
	|| log_snmp_tmpl_init() == -1
	|| net_basic_tmpl_init() == -1
	|| net_dyndns_tmpl_init() == -1
	|| net_lanif_tmpl_init() == -1
	|| net_misc_tmpl_init() == -1
	|| net_wlan_tmpl_init() == -1
	|| net_wlan_status_tmpl_init() == -1
	|| net_wlan_wep_tmpl_init() == -1
	|| net_wlan_wpa_tmpl_init() == -1
	|| net_wlan_wpa_psk_tmpl_init() == -1
	|| net_wlan_wpa_eap_peap_mschapv2_tmpl_init() == -1
	|| oem_dev_tmpl_init() == -1
	|| oem_misc_tmpl_init() == -1
	|| oem_upload_file_tmpl_init() == -1
#if defined(PP_FEAT_REMOTE_CONSOLE)
	|| power_control_ipmi_tmpl_init() == -1
	|| power_control_state_tmpl_init() == -1
	|| power_cycle_offtime_tmpl_init() == -1
	|| power_control_psu_adapt_tmpl_init() == -1
	|| power_control_direct_tmpl_init() == -1
	|| power_control_intern_tmpl_init() == -1
	|| power_control_ports_tmpl_init() == -1
	|| power_device_cfg_tmpl_init() == -1
	|| power_switch_tmpl_init() == -1
	|| pswitch_status_tmpl_init() == -1
	|| rc_tmpl_init() == -1
	|| rc_encoding_tmpl_init() == -1
	|| rc_general_tmpl_init() == -1
	|| rc_home_refresh_tmpl_init() == -1
	|| rc_hotkeys_tmpl_init() == -1
	|| rc_misc_tmpl_init() == -1
	|| rc_mouse_hotkey_tmpl_init() == -1
	|| rc_preview_tmpl_init() == -1
	|| rc_type_tmpl_init() == -1
	|| sec_crypt_kvm_tmpl_init() == -1
#endif /* PP_FEAT_REMOTE_CONSOLE */
	|| reset_tmpl_init() == -1
#ifdef PRODUCT_PDU
	|| rpc_common_tmpl_init() == -1
        || rpc_alert_dests_tmpl_init() == -1
        || rpc_alert_events_tmpl_init() == -1
        || rpc_alert_policies_tmpl_init() == -1
        || rpc_outlet_detail_tmpl_init() == -1
        || rpc_outlet_list_tmpl_init() == -1
        || rpc_outlet_setup_tmpl_init() == -1
#endif /* PRODUCT_PDU */
	|| sec_antibf_tmpl_init() == -1
	|| sec_crypt_web_tmpl_init() == -1
	|| sec_fw_tmpl_init() == -1
	|| sec_group_acl_tmpl_init() == -1
	|| sec_loglim_tmpl_init() == -1
	|| sec_server_cert_csr_tmpl_init() == -1
	|| sec_server_cert_upload_tmpl_init() == -1
#if defined(PP_FEAT_STRONG_PASSWORDS)
        || sec_strong_pw_tmpl_init() == -1
#endif /* PP_FEAT_STRONG_PASSWORDS */
	|| serial_1_tmpl_init() == -1
	|| serial_2_tmpl_init() == -1
	|| serial_blade_tmpl_init() == -1
	|| smtp_tmpl_init() == -1
	|| sn_tmpl_init() == -1
	|| snmp_agent_tmpl_init() == -1
	|| ssh_tmpl_init() == -1
	|| target_list_tmpl_init() == -1
	|| telnet_tmpl_init() == -1
	|| time_tmpl_init() == -1
	|| time_local_tmpl_init() == -1
	|| time_ntp_tmpl_init() == -1
	|| um_groups_tmpl_init() == -1
	|| um_pwchange_tmpl_init() == -1
	|| um_users_tmpl_init() == -1
	|| language_tmpl_init() == -1
#ifdef PP_FEAT_MASS_STORAGE
	|| vfloppy_active_tmpl_init() == -1
	|| vfloppy_extern_tmpl_init() == -1
	|| vfloppy_intern_tmpl_init() == -1
	|| vfloppy_options_tmpl_init() == -1
	|| vfloppy_drvredir_tmpl_init() == -1
#endif /* PP_FEAT_MASS_STORAGE */
#if defined(PP_FEAT_REMOTE_CONSOLE)
	|| video_adc_tmpl_init() == -1
	|| video_custom_modes_tmpl_init() == -1
	|| video_local_port_tmpl_init() == -1
	|| video_panel_tmpl_init() == -1
#endif /* PP_FEAT_REMOTE_CONSOLE */
	){
	return -1;
    }
    websUrlHandlerDefine("/", NULL, 0, eric_url_handler, WEBS_HANDLER_FIRST);

    websAspDefine("tmplBeginEnd", tmpl_begin_end_asp);
    websAspDefine("getSessionID", get_session_id_asp);
    websAspDefine("rfbport", rfbport_asp);
    websAspDefine("getResponseMsg", get_response_msg_asp);
    websAspDefine("writeConnectedUsers", write_connected_users_asp);
    websAspDefine("writeVoltages", write_voltages_asp);
    websAspDefine("getVar", get_var_asp);
    websAspDefine("getVarByIdx", get_var_by_idx_asp);
    websAspDefine("htmlEsc", html_esc_asp);
    websAspDefine("imgButton", img_button_asp);
    websAspDefine("imgLinkButton", img_link_button_asp);
    websAspDefine("getInternalVar", internal_var_asp);
    websAspDefine("rand", rand_asp);
    websAspDefine("getHardwareRev", get_hardware_rev_asp);
    websAspDefine("ppButton", button_asp);
    websAspDefine("getFlashID", get_flashid_asp);
    websAspDefine("getUserName", get_user_name_asp);
    websAspDefine("getGroupName", get_group_name_asp);
    websAspDefine("getHelpFile", get_help_file_asp);

#ifdef PP_FEAT_USBPOWERED
    websAspDefine("getExternalPowerStatus", get_external_power_status_asp);
#endif
    websAspDefine("getKvmNodeCnt", get_kvm_node_cnt_asp);
    websAspDefine("getKvmNode", get_kvm_node_asp);
    
    websAspDefine("useRaritanDesign", use_raritan_desgin_asp);
    websAspDefine("SessionInfo", sessioninfo_asp);
    websAspDefine("writeVoltagesSideBar", voltagesinfo_asp);
    websAspDefine("writeConnectedUsersSideBar", write_connected_users_bar_asp);
    
    websAspDefine("isClusterEnabled", is_cluster_enabled_asp);
    websAspDefine("isClusterActive", is_cluster_active_asp);
    websAspDefine("isRfbUpOnNode", is_rfb_up_on_node_asp);

    eric_notify_register_log_object("auth", pp_intl_translate(N_("Authentication")), PP_NOTIFY_EVENT_ALL_ENABLE);
    eric_notify_register_log_object("security", pp_intl_translate(N_("Security")), PP_NOTIFY_EVENT_ALL_ENABLE);
#if !((defined(PRODUCT_WINGLE) || defined(PRODUCT_RC1)) && defined(OEM_RARITAN)) && !defined(PRODUCT_MSIDC)
    eric_notify_register_log_object("host", pp_intl_translate(N_("Host Control")), PP_NOTIFY_EVENT_ALL_ENABLE);
#endif
    eric_notify_register_log_object("device", pp_intl_translate(N_("Board Message")), PP_NOTIFY_EVENT_ALL_ENABLE);

#if defined(PRODUCT_KX2)
    eric_notify_register_log_object("diag", pp_intl_translate(N_("Debug Message")), PP_NOTIFY_EVENT_LIST_ENABLE | PP_NOTIFY_EVENT_NFS_ENABLE);
#endif

    websAspDefine("getHostInfo", get_host_info_asp);
    websAspDefine("isEmpty", is_empty_asp);
    return 0;
}

void
eric_cleanup(void)
{
#ifdef PRODUCT_PDU
    rpc_common_tmpl_cleanup();
#endif
}

int eric_webs_connect_encoder(eric_webs_encoder_desc_t* encoder) {
    assert(encoder_hook == NULL);
    
    encoder_hook = encoder;
    return PP_SUC;
}

eric_webs_session_data_t *
eric_webs_alloc_session_data(void)
{
    eric_webs_session_data_t * sd;

    if ((sd = (eric_webs_session_data_t *)malloc(sizeof(*sd))) == NULL) {
	pp_log_err("%s(): malloc()", ___F);
    } else {
	memset(sd, 0, sizeof(*sd));
    }

    return sd;
}

void
eric_webs_free_session_data_by_id(eric_session_int_id_t sid) {
    pp_hash_delete_entry_i(webs_session_data, sid);
    return;
}

void
eric_webs_free_session_data(void * _s)
{
    eric_webs_session_data_t *sd = (eric_webs_session_data_t *) _s;

    pp_firmware_erla_ctx_free(sd->firmware_ctx);

#ifdef PP_FEAT_MASS_STORAGE
    {
	u_int ms_index;
	for (ms_index = 0; ms_index < PP_FEAT_USB_MASS_STORAGE_NO; ++ms_index) {
	    if ( vfloppy_session_check(sd->session, ms_index, 0) ) {
		int error;
		
		pp_usb_ms_unset_image(ms_index, &error);
		/*
		 * There is not necessarily a web page going to the user
		 * any more; can't tell him about the error.
		 * set_response(wp, ERIC_RESPONSE_ERROR, "Discarding the floppy from virtual USB drive failed");
		 */
		vfloppy_discard(ms_index);
	    }
	}
    }
#endif

    free(sd->local_edit_tag);
    free(sd); /* this must be the last */
}

/*
 * Note:    (from geo@peppercon.de)
 * This function cannot be used in the form handlers.
 * E.g. it's not possible to save form data and redirect
 * the user to another page.
 */
void
eric_redirect(webs_t wp, const char * url, const char * cookie)
{
    int no_killfrm = !pp_strcmp_safe(websGetVar(wp, "killfrm", NULL), "no");
    char new_url[256];

    if (no_killfrm) snprintf(new_url, sizeof(new_url), "%s?killfrm=no", url);
    websRedirect(wp, no_killfrm ? new_url : url, cookie);
}

static int
eric_url_handler(webs_t wp, char * urlPrefix UNUSED, char * webdir UNUSED,
		 int arg UNUSED, char * url, char * path UNUSED, char * query UNUSED)
{
    char * _url = NULL, * url_cut, *referer = NULL, *referer_cut = NULL;
    int url_cut_len;
    u_char curr_unit = 0;
    u_short curr_port = 0;
    int ret = 0;
    const char * redirect, * session;
    const char * strval;
    const char* uaprefix; /* user agent prefix string, in case there is a
			     special version available for a particular UA */    
    char uid_buffer[10];
    uaprefix = findUserAgentPrefix(wp);
    
    if (*url == '\0' || strcmp(url, "/") == 0) {
	eric_redirect(wp, "/auth.asp", NULL);
	return 1;
    }
    
    _url = strdup(url);
    url_cut = &_url[strspn(_url, "/")]; /* drop leading slashes */
    url_cut[strcspn(url_cut, "?")] = '\0'; /* drop GET parameters */
    url_cut_len = strlen(url_cut);
    
    websSetVar(wp, "__user__", wp->user);
    wp->uid = wp->user ? pp_um_user_get_uid(wp->user) : -1;
    snprintf(uid_buffer, 9, "%d", wp->uid);
    websSetVar(wp, "__uid__", uid_buffer);

    if (!strcmp(url_cut, "auth.asp") && form_was_submitted(wp)) {
        if (login_form(wp) != -1) {
		/* success, we were redirected */
                ret = 1;
	        goto finally_done;
	}
	/* error, set request path */
	goto finally;
    }
    
#if defined(PP_FEAT_WS_MANAGEMENT)
    if (!strcmp(url_cut, "wsman")) {
    	ret = handle_wsman(wp);
    	goto finally_done;
    }
#endif

    /* files that are unrestricted (*.jar is a MacOS java workaround!) */
    if (!strcmp(url_cut, "style.asp") ||
	!strcmp(url_cut, "favicon.ico") ||
	!strcmp(url_cut, "life_id.html") ||
        !strcmp(url_cut, "prod_info.asp") ||
        !strcmp(url_cut, "ldap_cdl.tgz") ||
	(strstr(url_cut,"gd_")!=url_cut && 
	    url_cut_len > 4 && 
	    (!strcmp(&url_cut[url_cut_len-4], ".gif") ||
		 !strcmp(&url_cut[url_cut_len-4], ".jpg") ||
		 !strcmp(&url_cut[url_cut_len-4], ".ico") ||
		 !strcmp(&url_cut[url_cut_len-4], ".dll") ||
		 !strcmp(&url_cut[url_cut_len-4], ".jar")))) {
	goto finally; // HACK ALERT: only 1 logo.gif, uaprefix for it!
    }

    redirect = websGetVar(wp, "redirect", "yes");
    if (!wp->auth_ok || (!strcmp(url_cut, "auth.asp") && !strcmp(redirect, "no"))) {
	if (strcmp(url_cut, "auth.asp")) {
            session = websGetVar(wp, "session", "yes");
	    char buf[32];
	    if (!strcmp(url_cut, "webs_cron.asp")) {
                if (!strcmp(session, "no")) {
                    // return error to let the javascript reload the page
		    goto finally;
		} else {
		    snprintf(buf, sizeof(buf), "/webs_cron.asp?session=%s", "no");
		    eric_redirect(wp, buf, NULL);
		    ret = 1;
		    goto finally_done;
		}
	    } else {
		snprintf(buf, sizeof(buf), "/auth.asp?redirect=%s", redirect);
		eric_redirect(wp, buf, NULL);
		ret = 1;
		goto finally_done;
	    }
	}
	goto finally;
    }

    /* file(s) that don't update session timeout */
    if (strcmp(url_cut, "webs_cron.asp")) {
	eric_session_touch(wp->session);
    }

    if (!wp->upload_failed) {
	set_response(wp, ERIC_RESPONSE_OK, NULL);
    }

    {
	const char * target_user = websGetVar(wp, "__target_user__", NULL);
	if (target_user == NULL) {
	    wp->target_user = bstrdup(B_L, wp->user);
	    wp->target_uid = wp->uid;
	    websSetVar(wp, "__target_user__", wp->target_user);
	} else {
	    wp->target_user = bstrdup(B_L, target_user);
	    wp->target_uid = pp_um_user_get_uid(target_user);
	}
    }

    {
	const char * target_group = websGetVar(wp, "__target_group__", NULL);
	if (target_group == NULL) {
	    char * target_group_from_um;
	    if (PP_FAILED(pp_um_user_get_group(wp->target_user, &target_group_from_um))){
		/* this is <Unknown> by default */
		/* TODO: get from cfg/um ? */
		wp->target_group = bstrdup(B_L, "<Unknown>");
	    } else {
		wp->target_group = bstrdup(B_L, target_group_from_um);
		free(target_group_from_um);
	    }
	    websSetVar(wp, "__target_group__", wp->target_group);
	} else {
	    wp->target_group = bstrdup(B_L, target_group);
	}
	wp->target_gid = pp_um_group_get_gid(wp->target_group);
    }

#if defined(KIRA_RPC)
    {
        int target_outlet = pp_strtol_10(websGetVar(wp, "__target_outlet__",
                                                    NULL), PP_ERR, NULL);
        
	if (target_outlet == PP_ERR) {
            /* outlets are counted 1 - e.g. 8, so fallback is not 0! */
	    wp->target_outlet = 1;
	    websSetVar(wp, "__target_outlet__", "1");
	} else {
	    wp->target_outlet = target_outlet;
	}
    }
#endif /* KIRA_RPC */

#if defined(PP_FEAT_REMOTE_CONSOLE)
    pp_kvm_get_unit_port_for_video_link(0, &curr_unit, &curr_port);
#endif

    strval = websGetVar(wp, "__target_unit__", NULL);
    if (strval == NULL) {
	char var[5];
        snprintf(var, sizeof(var), "%d", curr_unit);
        websSetVar(wp, "__target_unit__", var);
	wp->target_unit = curr_unit;
    } else {
	wp->target_unit = pp_strtoul_10(strval, 0, NULL);
    }

    strval = websGetVar(wp, "__target_port__", NULL);
    if (strval == NULL) {
	char var[5];
	snprintf(var, sizeof(var), "%d", curr_port);
	websSetVar(wp, "__target_port__", var);
	wp->target_port = curr_port;
    } else {
	wp->target_port = pp_strtoul_10(strval, 0, NULL);
    }
    
    if (wp->sd->local_edit_tag) {
	websSetVar(wp, "__tag__", wp->sd->local_edit_tag);
    }

    if(NULL != (referer = pp_strdup_safe(websGetVar(wp, "HTTP_REFERER",
                                                    NULL)))) {
        referer_cut = strrchr(referer, '/') + 1; /* drop leading slashes */
        referer_cut[strcspn(referer_cut, "?")] = '\0'; /* drop GET parameters */
    }
    
    if (!strcmp(url_cut, "auth.asp")) {
	eric_redirect(wp, "/home.asp", NULL);
	ret = 1;
	goto finally_done;
    } else if (!strcmp(url_cut, "logout")) {
	if (wp->session != 0) {
	    eric_net_ssl_auth_logout(wp->bio);
#if defined(PP_FEAT_REMOTE_CONSOLE)
	    pp_rfb_do_disconnect(wp->session);
#endif
	    eric_session_release(wp->session);
	}
	eric_redirect(wp, "/auth.asp?redirect=no", NULL);
	ret = 1;
	goto finally_done;
    } else  if (eric_misc_reset_is_running()) {
	// a reset is in progress. continue normally
        goto finally;
    } else  if (pp_firmware_erla_is_update_locked_no_ctx()) {
	if (strcmp(url_cut, "fwupdprog.asp")) {
            session = websGetVar(wp, "fwupd", "yes");
	    char buf[32];
	    if (!strcmp(url_cut, "webs_cron.asp")) {
                if (!strcmp(session, "no")) {
                    // return error to let the javascript reload the page
		    goto finally;
		} else {
		    snprintf(buf, sizeof(buf), "/webs_cron.asp?fwupd=%s", "no");
		    eric_redirect(wp, buf, NULL);
		    ret = 1;
		    goto finally_done;
		}
	    } else {
		eric_redirect(wp, "/fwupdprog.asp", NULL);
		ret = 1;
		goto finally_done;
	    }
	}
    } else if(strcmp(url_cut, "pwchangeforced.asp") &&
              pp_um_user_need_pw_change(wp->user) == PP_SUC &&
              (!pp_strcmp_safe(referer_cut, "pwchangeforced.asp") ||
               !pp_strcmp_safe(referer_cut, "auth.asp"))) {
        /**
         * only force password change if
         * - we do not already want to open change password dialog
         * - we need password change and
         * - we just logged in (referer is auth.asp) or
         * - we tried to leave password change dialog
         *       (referer is pwchangeforced.asp) 
         */
        eric_redirect(wp, "/pwchangeforced.asp", NULL);
        ret = 1;
        goto finally_done;
    } else if (!strcmp(url_cut, "supportdata.xml")) {
	send_support_data(wp);
    } else if (strstr(url_cut, "gd_")==url_cut) {
	send_gd_content(wp,url_cut+3);
    }
    else {
	form_handler(wp);
    }

    wp->upload_failed = 0;
 finally:
    /* calculate request path depending on prefix */
    ericSetWebsRequestPath(wp, uaprefix);
    
 finally_done:
    /* when redirected, no further webs action allowed */
    free(_url);
    free(referer);
    return ret;
}

/* usage: button("Text", "filename", "action") */
static int
button_asp(int eid, webs_t wp UNUSED, int argc, char **argv)
{
    const char * lang = pp_intl_get_language_shortname();
    char * buf;
    
    if (argc < 3) return -1;
    
    if (asprintf(&buf, "<input type=\"image\" src=\"button_%s.%s.gif\" name=\"action_%s\" value=\"%s\">", argv[1], lang, argv[2], argv[0]) < 0) {
	buf = NULL;
    }

    ejSetResult(eid, buf);
    free(buf);
    
    return 0;    
}

static int
get_session_id_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED)
{
    const char * id = NULL;

    if (wp->auth_ok) id = eric_session_get_id(wp->session);
    ejSetResult(eid, id);

    return 0;
}

static int
rand_asp(int eid, UNUSED webs_t wp, int UNUSED argc, UNUSED char **argv)
{
#define RANDMAXSTR 20
    char result[RANDMAXSTR];
    long int x;

    x = random();
    snprintf(result, RANDMAXSTR, "%ld", x);
    
    ejSetResult(eid, result);
    return 0;
#undef RANDMAXSTR
}

static int
rfbport_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    char * result;
 
    pp_cfg_get(&result, "network.https_port");
    ejSetResult(eid, result);
    free(result);

    return 0;
}

static int
get_response_msg_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED)
{
#ifdef PP_FEAT_RARITAN_DESIGN
    const char * fmt = "<br><div class=\"rsp_msg_wrap\"><div class=\"%s\">%s</div><div class=\"rsp_message\">%s %s</div></div><br><div class=\"tmpl_end\"></div>";
    const char * fmt_ok = "<br><div class=\"rsp_msg_wrap\"><div class=\"rsp_message\">%s</div><div class=\"rsp_message\">%s</div></div><br><div class=\"tmpl_end\"></div>";
#else /* !PP_FEAT_RARITAN_DESIGN */
    const char * fmt = "<span class=\"%s\">%s</span><br><span class=\"rsp_message\">%s</span>%s";
    const char * fmt_ok = "<span class=\"rsp_message\">%s</span>%s";
#endif /* !PP_FEAT_RARITAN_DESIGN */
    char * buf, * bulk_buf = gen_response_msg_for_bulk(wp);
    const char * msg, * bulk_msg = bulk_buf ? bulk_buf : "";
    int r = 0;
    
    if (wp->response_msg == NULL) wp->response_msg = "";
    msg = wp->response_msg;

    if (wp->response_code == ERIC_RESPONSE_OK) {
	r = asprintf(&buf, fmt_ok, *msg ? msg : "&nbsp;", bulk_msg);
    } else if (wp->response_code == ERIC_RESPONSE_ERROR) {
	r = asprintf(&buf, fmt, "error", _("Error: "),
		     *msg ? msg : _("An unknown error occured."), bulk_msg);
    } else if (wp->response_code == ERIC_RESPONSE_WARNING) {
	r = asprintf(&buf, fmt, "warning", _("Warning: "),
		     *msg ? msg : _("An unknown warning occured."), bulk_msg);
    }
    if (r < 0) buf = NULL;

#ifdef PP_FEAT_RARITAN_DESIGN
    if (wp->response_msg[0] == '\0') {
        ejSetResult(eid, "");
    } else {
        ejSetResult(eid, buf ? buf : "&nbsp;");
    }
#else /* !PP_FEAT_RARITAN_DESIGN */
    ejSetResult(eid, buf ? buf : "&nbsp;");
#endif /* !PP_FEAT_RARITAN_DESIGN */

    free(bulk_buf);
    free(buf);
    
    return 0;
}

static int
get_host_info_asp(int eid, webs_t wp UNUSED, int argc, char ** argv)
{
    const char * res = "unknown";
    
#if defined(PP_FEAT_PCI_ADC) || defined(PP_FEAT_AD_ATMEL)
    if (argc == 1) {
	if (!strcmp(argv[0], "power")) {
	    int host_power;
	    if (eric_misc_get_host_power(&host_power) != -1) {
		res = host_power ? "yes" : "no";
	    }
	}
	if (!strcmp(argv[0], "pwrsupplypower")) {
	    int host_pwrsupplypower;
	    if (eric_misc_get_host_pwrsupplypower(&host_pwrsupplypower) != -1) {
		res = host_pwrsupplypower ? "yes" : "no";
	    }
	}
    }
#else /* !PP_FEAT_PCI_ADC && !PP_FEAT_AD_ATMEL */
   (void)argv;
   (void)argc;
#endif /* !PP_FEAT_PCI_ADC && !PP_FEAT_AD_ATMEL */

    ejSetResult(eid, res);
    return 0;
}

static void
write_session_info(eric_session_int_id_t s, va_list args)
{
    char timestr[32];
    char outstr[256];
    int idlemins = eric_session_get_idle_time(s) / 60;
    webs_t wp = va_arg(args, webs_t);
    int * n = va_arg(args, int *);
    int rc_active;
    int rc_exclusive;
    char * user = escape_html(eric_session_get_user(s));
    char * nick = escape_html(eric_session_get_nickname(s));
    
#if defined(PP_FEAT_REMOTE_CONSOLE)
    rc_active = pp_rfb_is_active(0, s);
    rc_exclusive = rc_active && (pp_rfb_get_exclusive_session_info(s)
				 == PP_RFB_SESSION_IS_EXCLUSIVE);
#else
    rc_active = 0;
    rc_exclusive = 0;
#endif

    if (idlemins > 1)
	snprintf(timestr, 32, _("%2d min idle"), idlemins);
    else
	snprintf(timestr, 32, _("active"));

    if (nick && strlen(nick) > 0) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td class=\"normal\">%s \"%s\"</td>", user, nick);
    } else {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td class=\"normal\">%s</td>", user);
    }
    *n += websWrite(wp, outstr);
	
    snprintf(outstr, sizeof(outstr),
	     "<td class=\"normal\"> (%s) </td>"
	     "<td class=\"normal\">%s%s</td>"
	     "<td class=\"normal\">%s</td>"
	     "</tr>",
	     eric_session_get_ip_str(s),
	     rc_active ? _("RC") : "" ,
	     rc_exclusive ? _(" (exclusive)") : "",
	     timestr);
 
    *n += websWrite(wp, outstr);
    free(user);
    free(nick);
}

static void
write_session_info_bar(eric_session_int_id_t s, va_list args)
{
    char timestr[32];
    char outstr[256];
    int idlemins = eric_session_get_idle_time(s) / 60;
    webs_t wp = va_arg(args, webs_t);
    int * n = va_arg(args, int *);
    int rc_active;
    int rc_exclusive;
    char * user = escape_html(eric_session_get_user(s));
    char * nick = escape_html(eric_session_get_nickname(s));
    
#if defined(PP_FEAT_REMOTE_CONSOLE)
    rc_active = pp_rfb_is_active(0, s);
    rc_exclusive = rc_active && (pp_rfb_get_exclusive_session_info(s)
				 == PP_RFB_SESSION_IS_EXCLUSIVE);
#else
    rc_active = 0;
    rc_exclusive = 0;
#endif
    
    if (idlemins > 1)
	snprintf(timestr, 32, _("%2d min idle"), idlemins);
    else
	snprintf(timestr, 32, _("active"));

    if (nick && strlen(nick) > 0) {
	snprintf(outstr, sizeof(outstr),
		 "%s \"%s\" ", user, nick);
    } else {
	snprintf(outstr, sizeof(outstr),
		 "%s ", user);
    }
    *n += websWrite(wp, outstr);
	
    snprintf(outstr, sizeof(outstr),
	     "(%s) <br />"
	     "&nbsp;&nbsp;&nbsp; %s "
	     "%s <br />",
	     eric_session_get_ip_str(s),
	     rc_active ? _("RC") : "" ,
	     timestr);

    *n += websWrite(wp, outstr);
    free(user);
    free(nick);
}

static int
write_connected_users_asp(int eid UNUSED, webs_t wp, int argc UNUSED,
			  char **argv UNUSED)
{
    int n = 0;

    n+=websWrite(wp, "<table border=\"0\" cellpadding=\"2\" cellspacing=\"0\">");
    eric_session_for_all(write_session_info, wp, &n);
    n+=websWrite(wp, "</table>");

    return n;

}

static int
write_connected_users_bar_asp(int eid UNUSED, webs_t wp, int argc UNUSED,
			  char **argv UNUSED)
{
    int n = 0;
    if (wp->session != 0) {
	n+=websWrite(wp, "<div class=\"sidebar_row_content\">");
	eric_session_for_all(write_session_info_bar, wp, &n);
	n+=websWrite(wp, "</div>");
    }
    return n;
}

static int
write_voltages_asp(int eid UNUSED, webs_t wp, int argc UNUSED,
			  char **argv UNUSED)
{
    int n = 0;
    
#if defined(PP_FEAT_AD_ATMEL) || defined(PP_FEAT_PCI_ADC)
    float raw_value;
    char outstr[256];

    n+=websWrite(wp, "<table class=\"normal\" border=\"0\" cellpadding=\"2\" cellspacing=\"0\">");

    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_12V) ) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td>%s</td><td> %.2f V</td></tr>\n", _("PCI Voltage (+12V)"), raw_value);
                 n+=websWrite(wp, outstr);
    }
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_5V) ) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td>%s</td><td> %.2f V</td></tr>\n", _("PCI Voltage (+5V)"),  raw_value);
                 n+=websWrite(wp, outstr);
    }
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_VIO) ) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td>%s</td><td> %.2f V</td></tr>\n",_("PCI Voltage (VIO)"), raw_value);
                 n+=websWrite(wp, outstr);
    }
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_3V_PCI) ) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td>%s</td><td> %.2f V</td></tr>\n",_("PCI Voltage (PCI +3.3V)"), raw_value);
                 n+=websWrite(wp, outstr);
    }
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_3V_AUX) ) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td>%s</td><td> %.2f V</td></tr>\n",_("PCI Voltage (AUX +3.3V)"), raw_value);
                 n+=websWrite(wp, outstr);
    }
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_3V_PLANE) ) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td>%s</td><td> %.2f V</td></tr>\n",_("Plane Voltage (+3.3V)"), raw_value);
                 n+=websWrite(wp, outstr);
    }
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_12V_NEG) ) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td>%s</td><td> %.2f V</td></tr>\n",_("PCI Voltage (-12V)"), raw_value);
                 n+=websWrite(wp, outstr);
    }
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_5V_EXTERNAL_PWR_SUPPLY) ) {
	snprintf(outstr, sizeof(outstr),
		 "<tr><td>%s</td><td> %.2f V</td></tr>\n",_("EXT_PWR_SUP (0V||+5V)"), raw_value);
                 n+=websWrite(wp, outstr);
    }
    n+=websWrite(wp, "</table>");
#else
    (void)wp;
#endif
    return n;
}

static int
voltagesinfo_asp(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char** argv UNUSED)
{
    int n = 0;

    if (wp->session != 0) {
    
#if defined(PP_FEAT_AD_ATMEL) || defined(PP_FEAT_PCI_ADC)
    int c = 0;
    float raw_value = 0,
	  raw_value12v = 0,
	  raw_value5v = 0,
	  raw_value3v = 0,
	  raw_value12vneg = 0;
    char outstr[256];

    n+=websWrite(wp, "<div class=\"sidebar_row_content\">");

    if (pp_sense_read_adc_value_raw(&raw_value12v, PCI_VOLTAGE_12V)) raw_value12v = 0;
    if (pp_sense_read_adc_value_raw(&raw_value5v, PCI_VOLTAGE_5V)) raw_value5v = 0;
    if (pp_sense_read_adc_value_raw(&raw_value3v, PCI_VOLTAGE_3V_PCI)) raw_value3v = 0;
    if (pp_sense_read_adc_value_raw(&raw_value12vneg, PCI_VOLTAGE_12V_NEG)) raw_value12vneg = 0;

    /* sometimes we measure +24V on PCI 12V, +5.5V on PCI 3.3V... do not show such measurement errors in webfrontend */ 

    if (((raw_value12v > 4) && (raw_value12v < 15)) ||
	((raw_value5v > 2) && (raw_value5v < 7)) ||
	((raw_value3v > 1.5) && (raw_value3v < 4.8))) c = 1;

    if (c) {
	if (raw_value12v < 15) {
	    snprintf(outstr, sizeof(outstr),
		     "<div class=\"sidebar_table_replace\">"
		     "<div class=\"sidebar_table_replace_left\">%s</div>"
		     "<div class=\"sidebar_table_replace_right\" %s> %.2f V</div></div>\n",
		     _("PCI +12V"), (raw_value12v > 0) ? "": "style=\"color: Red;\"", raw_value12v);
	    n+=websWrite(wp, outstr);
	}

	if (raw_value5v < 7) {	
	    snprintf(outstr, sizeof(outstr),
		     "<div class=\"sidebar_table_replace\">"
		     "<div class=\"sidebar_table_replace_left\">%s</div>"
		     "<div class=\"sidebar_table_replace_right\" %s> %.2f V</div></div>\n",
		     _("PCI +5V"), (raw_value5v > 0) ? "": "style=\"color: Red;\"", raw_value5v);
	    n+=websWrite(wp, outstr);
	}
	
	if (raw_value3v < 4.8) {
	    snprintf(outstr, sizeof(outstr),
		     "<div class=\"sidebar_table_replace\">"
		     "<div class=\"sidebar_table_replace_left\">%s</div>"
		     "<div class=\"sidebar_table_replace_right\" %s> %.2f V</div></div>\n",
		     _("PCI +3.3V"), (raw_value3v > 0) ? "": "style=\"color: Red;\"", raw_value3v);
	    n+=websWrite(wp, outstr);
	}

    }
    
    if ((raw_value12vneg < -9) && (raw_value12vneg > -15)) {
	snprintf(outstr, sizeof(outstr),
		 "<div class=\"sidebar_table_replace\">"
		 "<div class=\"sidebar_table_replace_left\">%s</div>"
		 "<div class=\"sidebar_table_replace_right\"> %.2f V</div></div>\n",
		 _("PCI -12V"), raw_value12vneg);
	n+=websWrite(wp, outstr);
    }	
    
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_3V_AUX) ) {
	if (raw_value > 0) {
	    snprintf(outstr, sizeof(outstr),
		     "<div class=\"sidebar_table_replace\">"
		     "<div class=\"sidebar_table_replace_left\">%s</div>"
		     "<div class=\"sidebar_table_replace_right\"> %.2f V</div></div>\n",
		     _("AUX +3.3V"), raw_value);
                 n+=websWrite(wp, outstr);
	}
    }
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_5V_EXTERNAL_PWR_SUPPLY) ) {
	snprintf(outstr, sizeof(outstr),
		 "<div class=\"sidebar_table_replace\">"
		 "<div class=\"sidebar_table_replace_left\">%s</div>"
		 "<div class=\"sidebar_table_replace_right\">%s</div></div>\n",
		 _("Ext.Power Supply"), (raw_value > 0) ? _("Yes") : _("No"));
	     n+=websWrite(wp, outstr);
    }
    n+=websWrite(wp, "<div class=\"tmpl_end\"></div></div>");
#endif

    }
    return n;
}

static const char *
findUserAgentPrefix(webs_t wp)
{
    /* check for Windows CE */
    if (wp->userAgent && ( (strstr(wp->userAgent, UA_ID_WINCE) != NULL) ||
			   (strstr(wp->userAgent, UA_ID_WINCE_JEODE) != NULL) )) {
	/* yeah, check whether size is really small */
	if(wp->uadim.width < UA_MAXWIDTH_WINCE
	   || wp->uadim.height < UA_MAXHEIGHT_WINCE) {
	    /* uups, we really got something like a pda running CE */
	    return UA_PREFIX_WINCE;
	}
    }
    return UA_PREFIX_DEFAULT;
}

static int
is_oem_exception(const char* url)
{
    int ret = 0;
    const char* const *ex = oem_exceptions;

    if (!url) goto bail;
    
    while(*ex) {
	if (!strcmp(*ex, url)) {
	    ret = 1;
	    goto bail;
	}
	ex++;
    }
    
 bail:
    return ret;
}
		
static void
ericSetWebsRequestPath(webs_t wp, const char* uaprefix)
{
    char* npath, *nlpath;
    struct stat statbuf;

    //pp_log("%s(): start: url=%s dir=%s path=%s\n", ___F, wp->url, wp->dir,wp->path);

    // handle special user agent directory
    if(uaprefix != NULL && *uaprefix != '\0') {
	// take current dir and path
	// add the prefix to path and write it back
	npath = balloc(B_L, strlen(wp->path) + strlen(uaprefix) + 1);
	strcpy(npath, uaprefix);
	strcat(npath, wp->path);
	websSetRequestPath(wp, wp->dir, npath);
	bfree(B_L, npath);
    }

    // if path exists in special oem subdirectory, use it
    npath = balloc(B_L, strlen(wp->path) + strlen(OEM_DIR_PREFIX) + 1);
    strcpy(npath, OEM_DIR_PREFIX);
    strcat(npath, wp->path);

    // get complete local path
    nlpath = balloc(B_L, strlen(npath) + strlen(wp->dir) + 1);
    strcpy(nlpath, wp->dir);
    strcat(nlpath, npath);

    //pp_log("%s(): oem: npath=%s nlpath=%s\n", ___F, npath, nlpath);

    if (is_oem_exception(wp->url) || stat(nlpath, &statbuf) < 0) {
	// oem path doesn't exist, so use the normal one
	websSetRequestPath(wp, wp->dir, wp->path);
	goto bail;
    }

    websSetRequestPath(wp, wp->dir, npath);
    
 bail:
    bfree(B_L, nlpath);
    bfree(B_L, npath);
    //pp_log("%s(): end: url=%s dir=%s path=%s\n", ___F, wp->url, wp->dir, wp->path);
}

static int
get_var_asp(int eid, webs_t wp, int argc, char **argv)
{
    /* <realm>: "webs" (webs variable) or "conf" (from config fs).  If empty,
     * "webs" will be used, and "conf" as fallback.  */
    const char *var_name = argv[0]; 
    const char *realm = argv[1]; 
    const char *val = NULL;
    char * val_cfg = NULL;
    
    if (argc == 1) {
	/* default */
	val_cfg = get_config_var(wp, var_name);
	val = websGetVar(wp, var_name, val_cfg); /* fallback is val_cfg */
    } else if ((argc == 2) && !strcmp(realm, "conf")) {
	/* only from config fs */
	val = val_cfg = get_config_var(wp, var_name);
    } else if ((argc == 2) && !strcmp(realm, "webs")) {
	/* only webs var */
	val = websGetVar(wp, var_name, NULL);
    }
    ejSetResult(eid, val);
    free(val_cfg);
    return 0;
}

static int
get_var_by_idx_asp(int eid, webs_t wp, int argc, char **argv)
{
    /* <realm>: "webs" (webs variable) or "conf" (from config fs).  If empty,
     * "webs" will be used, and "conf" as fallback.  */
    char full_name[FV_CFGKEY_MAX_LEN + 1];
    const char *var_name = argv[0];
    const char *idx = argv[1]; 
    const char *realm = argv[2];
    const char *val = NULL;
    char * val_cfg = NULL;

    if (argc >= 2) {
        form_var_assoc_name(var_name, idx, full_name, sizeof(full_name));
    }
    
    if (argc == 2) {
	/* default */
	val_cfg = get_config_var(wp, full_name);
	val = websGetVar(wp, full_name, val_cfg); /* fallback is val_cfg */
    } else if ((argc == 3) && !strcmp(realm, "conf")) {
	/* only from config fs */
	val = val_cfg = get_config_var(wp, full_name);
    } else if ((argc == 3) && !strcmp(realm, "webs")) {
	/* only webs var */
	val = websGetVar(wp, full_name, NULL);
    }
    ejSetResult(eid, val);
    free(val_cfg);
    return 0;
}

static int
html_esc_asp(int eid, webs_t wp UNUSED, int argc, char **argv)
{
    if (argc == 1) {
	char * esc_html = escape_html(argv[0]);
	if (esc_html) {
	    ejSetResult(eid, esc_html);
	    free(esc_html);
	} else {
	    ejSetResult(eid, "");
	}
	return 0;
    }
    return -1;
}

static int
get_hardware_rev_asp(int eid, webs_t wp UNUSED, int argc UNUSED,
		     char **argv UNUSED)
{
    char result[64];
    char *rev;

    rev = eric_misc_get_hardware_rev_str();

    if (rev) {
	/* make LARA 1.4 obvious for people */
	if (!strcmp(rev, "06")) {
	    snprintf(result, sizeof(result), _("%s (Rev 1.4)"), rev);
	} else {
	    snprintf(result, sizeof(result), "%s", rev);
	}

	ejSetResult(eid, result);
	free(rev);
    } else {
	ejSetResult(eid, _("Unknown"));
    }

    return 0;
}

static int
get_flashid_asp(int eid, webs_t wp UNUSED, int argc UNUSED,
		char **argv UNUSED)    
{
    char* rev = pp_hal_common_get_flash_factory_id();

    if (rev != NULL) {
	ejSetResult(eid, rev);
	free(rev);
    } else {
	ejSetResult(eid, "Unknown");
    }
    
    return 0;
}

static int
get_user_name_asp(int eid, webs_t wp UNUSED, int argc, char **argv)
{
    char *name = NULL;
    int uid;

    uid = pp_strtol_10(argv[0], -1, NULL);
    if (argc == 1 && uid >= 0) {
        /* we got a uid, now get the name */
        pp_cfg_get_nodflt(&name, "user[%u].login", uid);
    }
        
    ejSetResult(eid, name);
    free(name);

    return 0;
}

static int
get_group_name_asp(int eid, webs_t wp UNUSED, int argc, char **argv)
{
    char *name = NULL;
    int gid;

    gid = pp_strtol_10(argv[0], -1, NULL);
    if (argc == 1 && gid >= 0) {
        /* we got a gid, now get the name */
        pp_cfg_get_nodflt(&name, "group[%u].name", gid);
    }
        
    ejSetResult(eid, name);
    free(name);

    return 0;
}

static int
get_help_file_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    // find the current language
    const char *lang = pp_intl_get_language_shortname();
    if (lang == NULL || *lang == '\0') goto default_lang;

    char help_lang[20];
    snprintf(help_lang, sizeof(help_lang), "help_%s.asp", lang);
    
    // try to find a matching language file
    char fname1[PATH_MAX], fname2[PATH_MAX];
    snprintf(fname1, sizeof(fname1), "%s/%s", wp->dir, help_lang);
    snprintf(fname2, sizeof(fname2), "%s%s/%s", wp->dir, OEM_DIR_PREFIX, help_lang);
    
    struct stat statbuf;
    if (stat(fname1, &statbuf) == 0 || stat(fname2, &statbuf) == 0) {
        ejSetResult(eid, help_lang);
        return 0;
    }

 default_lang:
    ejSetResult(eid, "help.asp");
    return 0;
}

static int
tmpl_begin_end_asp(int eid UNUSED, webs_t wp, int argc, char **argv)
{
    int ret = 0;

    if (argc == 4) {
	char * operation = argv[0];
	char * template = argv[1];
	char * no_perm_html = argv[2];
	char * disabled_ldap_html = argv[3];
	int can_view = 0, can_control = 0, is_root = 0;
	form_handler_t * fh;

        is_root = !pp_um_user_get_uid(wp->user);

        if ((fh = pp_hash_get_entry(form_handlers, template)) == NULL) {
	    pp_log("%s(): Unknown subform '%s'\n", ___F, template);
	} else if (is_root) {
	    can_view = can_control = 1;
	} else if (fh->acl_object != NULL) {
	    int is_allowed = PP_SUC == 
                    pp_um_user_has_permission(wp->user, fh->acl_object,
                                              pp_acl_raasip_yes_str);
	    can_view = is_allowed || PP_SUC == 
                pp_um_user_has_permission(wp->user, fh->acl_object,
                                          pp_acl_raasip_view_str);
	    can_control = is_allowed || PP_SUC == 
                pp_um_user_has_permission(wp->user, fh->acl_object,
                                          pp_acl_raasip_control_str);
	}
        if (!strcmp(operation, "begin")) {
	    if(pp_cfg_ldap_prof_is_active(wp->uid) && fh->disable_ldap) {
		ret+=websWrite(wp,disabled_ldap_html);
		++wp->skipOutput;
	    } else {
		if (can_view) {
		    if (can_control) {
			/* activate the apply button */
			websSetVar(wp, "_show_apply", "1");
		    }
		} else {
		    ret += websWrite(wp, no_perm_html);
		    ++wp->skipOutput;
		}
	    }
	} else if (!strcmp(operation, "end") && !can_view) {
	    --wp->skipOutput;
	}
	return ret;
    } else {
	pp_log("%s(): Parameter error\n", ___F);
	return -1;
    }
}

/* usage: button("Text", "filename", "action"[, "translatable"]) */
static int
img_button_asp(int eid UNUSED, webs_t wp, int argc, char **argv)
{
    int len = 0, ret;
    char *line = NULL;
    char *lang = NULL;

#if !defined(PP_FEAT_IMAGE_BUTTONS)
     static const char *rformat = "<input type=\"submit\" class=\"button\" name=\"action_%s\" value=\"%s\" align=\"left\" style=\"vertical-align:middle\">";

     if (argc < 3 || argc > 4) return -1;

     len += strlen(rformat);
     len += strlen(argv[0]);
     len += strlen(argv[2]);             // length of "action"     
     len++;                              // termination 0

     line = malloc(len);
     snprintf(line, len, rformat, argv[2], argv[0]);

     ret = websWrite(wp, line); 
#else
    static const char *format = "<input type=\"image\" src=\"button_%s.%s.gif\" name=\"action_%s\" value=\"%s\" align=\"middle\" style=\"vertical-align:middle\">";
    static const char *format_no_lang = "<input type=\"image\" src=\"button_%s.gif\" name=\"action_%s\" value=\"%s\" align=\"middle\" style=\"vertical-align:middle\">";
    int translatable;

    if (argc < 3 || argc > 4) return -1;

    pp_cfg_get_nodflt(&lang, "language");
    
    translatable = argc == 3 || strcmp(argv[3], "no");
    
    if(translatable) {
        len += strlen(format);          // length of format string
        len += strlen(argv[0]);         // length of "value"
        len += lang ? strlen(lang) : 2; // length of "language"
    } else {
        len += strlen(format_no_lang);  // length of format string
        len += strlen(argv[0]);         // length of "value"
    }
    len += strlen(argv[1]);             // length of first part of "src"
    len += strlen(argv[2]);             // length of "action"
    len++;                              // termination 0

    line = malloc(len);
    if(translatable) {
        snprintf(line, len, format, argv[1], lang ? lang : "en", argv[2],
                 argv[0]);
    } else {
        snprintf(line, len, format_no_lang, argv[1], argv[2], argv[0]);
    }
    ret = websWrite(wp, line);
#endif

    free(lang);
    free(line); 
    return ret;
}

/* usage: button("Text", "filename", "link"[, "translatable"]) */
static int
img_link_button_asp(int eid UNUSED, webs_t wp, int argc, char **argv)
{
    int len = 0, ret;
    char *line = NULL;
    char *lang = NULL;

#if !defined(PP_FEAT_IMAGE_BUTTONS)
     static const char *rformat = 
        "<input type=\"button\" class=\"button\" "
            "name=\"%s\" value=\"%s\" align=\"left\" "
            "style=\"vertical-align:middle\" "
            "onclick=\"window.location.href='%s'\" "
        ">";

     if (argc < 3 || argc > 4) return -1;

     len += strlen(rformat);
     len += 2 * strlen(argv[0]);
     len += strlen(argv[2]);             // length of "action"     
     len++;                              // termination 0

     line = malloc(len);
     snprintf(line, len, rformat, argv[0], argv[0], argv[2]);

     ret = websWrite(wp, line); 
#else
    static const char *format = "<a href=\"%s\"><img src=\"button_%s.%s.gif\" alt=\"%s\" align=\"middle\" style=\"vertical-align:middle\" border=\"0\"></a>";
    static const char *format_no_lang = "<a href=\"%s\"><img src=\"button_%s.gif\" alt=\"%s\" align=\"middle\" style=\"vertical-align:middle\" border=\"0\"></a>";
    int translatable;

    if (argc < 3 || argc > 4) return -1;

    pp_cfg_get_nodflt(&lang, "language");
    
    translatable = argc == 3 || strcmp(argv[3], "no");
    
    if(translatable) {
        len += strlen(format);          // length of format string
        len += strlen(argv[0]);         // length of "value"
        len += lang ? strlen(lang) : 2; // length of "language"
    } else {
        len += strlen(format_no_lang);  // length of format string
        len += strlen(argv[0]);         // length of "value"
    }
    len += strlen(argv[1]);             // length of first part of "filename"
    len += strlen(argv[2]);             // length of "link"
    len++;                              // termination 0

    line = malloc(len);
    if(translatable) {
        snprintf(line, len, format, argv[2], argv[1], lang ? lang : "en",
                 argv[0]);
    } else {
        snprintf(line, len, format_no_lang, argv[2], argv[1], argv[0]);
    }
    ret = websWrite(wp, line);
#endif

    free(lang);
    free(line); 
    return ret;
}

static void
send_support_data(webs_t wp)
{
    static char buf[1024];
    char *serial = NULL;
    char * flashid = pp_hal_common_get_flash_factory_id();
    char * kme_version = NULL;
    char * boardname = eric_misc_get_boardname();
    char *product_esc, *board_esc, *tag_esc;
    char * hw_id = eric_misc_get_hardware_rev_str();
    
    pp_cfg_get_nodflt(&serial, "serial");

#if defined(PP_FEAT_REMOTE_CONSOLE)
    pp_km_comm_proto_ping(0, &kme_version);
#endif

    board_esc = escape_html(boardname);
    product_esc = escape_html(PP_STRINGIFY_EXPANDED(PP_PRODUCT));
    tag_esc = escape_html(pp_firmware_erla_tag);
    
    snprintf(buf, sizeof(buf),
	     "<?xml version=\"1.0\"?>\n"
	     "<support_data>\n"
	     "  <boardname>%s</boardname>\n"
	     "  <producttype>%s</producttype>\n"
	     "  <serial>%s</serial>\n"
	     "  <boardid>%s</boardid>\n"
	     "  <firmware_version>%s</firmware_version>\n"
	     "  <firmware_build_nr>%d</firmware_build_nr>\n"
	     "  <firmware_tag>%s</firmware_tag>\n"
	     "  <hardware_id>%s</hardware_id>\n"
	     "  <kme_firmware_version>%s</kme_firmware_version>\n"
	     "</support_data>\n",
	     board_esc, product_esc, serial ? serial : (flashid ? flashid : ""),
	     serial ? (flashid ? flashid : "") : "",
	     pp_firmware_erla_version, pp_firmware_erla_build_nr, tag_esc,
	     hw_id ? hw_id : "unknown",
	     kme_version ? kme_version : "");
    free(serial);
    free(boardname);
    free(flashid);
    free(kme_version);
    free(product_esc);
    free(board_esc);
    free(tag_esc);

    strcpy(wp->type, "application/octet-stream");
    wp->numbytes = strlen(buf);
    wp->download_image = bstrdup(B_L, buf);
    wp->download_name = "supportdata.xml";
    wp->download_cb = mem_image_download_cb;
}

#if defined(PP_FEAT_WS_MANAGEMENT)

static void send_auth_needed(webs_t wp) {
    websResponse(wp, 401, NULL,
	        "WWW-Authenticate: Basic realm=\"WSManagement\"",
		"<html><head><title>Document Error: Authentication required</title></head>\r\n<body><h2>Access Error: Authentication required</h2>\r\n when trying to access WS Management</body></html>\r\n", NULL);
}

static int handle_wsman(webs_t wp) {
    int ret = -1;
    char * ws_auth = NULL;
    size_t auth_len;
    const char * content_type = websGetVar(wp, "CONTENT_TYPE", "");
    
    free(wp->wsman_response);
    wp->wsman_response = NULL;
    
    if (!(wp->flags & WEBS_AUTH_BASIC) ||
        !wp->wsman_auth ||
        strlen(wp->wsman_auth) < strlen("basic ")) {
    	send_auth_needed(wp);
    	goto bail;
    }
    
    ws_auth = strdup(&wp->wsman_auth[strlen("basic ")]);
    auth_len = decodeBase64(ws_auth);
    ws_auth[auth_len] = '\0';
    wp->wsman_code = 200;
        
    switch (pp_wsman_process_request(ws_auth, &wp->query[wp->post_start],
    	wp->query_len - wp->post_start,
    	&wp->wsman_response, &wp->wsman_response_size,
    	wp->flags & WEBS_SECURE ? 1 : 0,
    	strstr(content_type, "UTF-16") || strstr(content_type, "utf-16"))) {
    	
    	case PP_WSMAN_FAULT:
    	    wp->wsman_code = 500;
    	    /* fall through */
    	case PP_WSMAN_NO_ERROR:
    	    if (wp->wsman_response) {
    	    	wp->numbytes = wp->wsman_response_size;
    	    	wp->download_image = balloc(B_L, wp->numbytes);
    	    	memcpy(wp->download_image, wp->wsman_response, wp->numbytes);
    	    } else {
    	    	wp->numbytes = 0;
    	    	wp->download_image = bstrdup(B_L, "");
    	    }

    	    if (content_type && content_type[0]) {
    	    	snprintf(wp->type, sizeof(wp->type), content_type);
    	    } else {
    	    	snprintf(wp->type, sizeof(wp->type), "application/soap+xml");
    	    }

    	    wp->download_cb = wsman_download_cb;
    	    ret = 0;
    	    break;
    	    
    	case PP_WSMAN_ERROR_AUTH_FAILED:
    	    send_auth_needed(wp);
    	    break;
    	    
    	case PP_WSMAN_ERROR_INTERNAL:
    	    websError(wp, 500, "Error parsing WS Management request");
    	    break;
    }
    
 bail:
    if (ws_auth) free(ws_auth);
    return ret;
}

#endif

#ifdef PP_FEAT_USBPOWERED
static int
get_external_power_status_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    u_char status = pp_gpio_bit_get(PP_GPIO_EXT_POWER_STATUS_DEV, PP_GPIO_EXT_POWER_STATUS);
    char status_text[8];
    
    snprintf(status_text, sizeof(status_text), "%d", status);
    ejSetResult(eid, status_text);
    return 0;
}
#endif

static int
use_raritan_desgin_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char** argv UNUSED)
{
#if defined(PP_FEAT_RARITAN_DESIGN)
    ejSetResult(eid, "1");
    return 0;
#else
    ejSetResult(eid, "0");
    return 0;
#endif
}

static int
sessioninfo_asp(int eid UNUSED, webs_t wp UNUSED, 
                int argc UNUSED, char** argv UNUSED)
{
    time_t mytime;
    char currtime[256];
    char timestr[32];
    char outstr[1024];
    char *utc_offset_opt = NULL;
    int idlesec, offset = 0;
    struct tm tm;
    int n=0;
    char lastloginstr[256];
    u_long lastlogin;

    if (wp->session != 0) {

        idlesec = eric_session_get_idle_time(wp->session);
	pp_cfg_get(&utc_offset_opt, "time.utc_offset");

	if (strchr(utc_offset_opt, '/')) {
	    offset = 0;
	} else if (strchr(utc_offset_opt, '+')) {
	    if (sscanf(utc_offset_opt, "+ %u h", &offset) != 1 || offset > 12) {
		offset = 0;
	    }
	} else if (strchr(utc_offset_opt, '-')) {
	    if (sscanf(utc_offset_opt, "- %u h", &offset) != 1 || offset > 11) {
		offset = 0;
	    } else {
		offset = -offset;
	    }
	}

	time(&mytime);    
	localtime_r(&mytime, &tm);
	tm.tm_hour += offset;

	strftime(currtime, 255, "%d.%m.%Y %T", &tm);

	if (idlesec > 29)
	    snprintf(timestr, 32, _("%2d sec idle"), idlesec);
	else
	    snprintf(timestr, 32, _("active"));
        
        if((PP_ERR == pp_cfg_get_ulong(&lastlogin, "user[%u].last_login_time",
                                       wp->target_uid)) || !lastlogin) {
            snprintf(lastloginstr, 255, _("not set"));
        } else {
	    strftime(lastloginstr, 255, "%d.%m.%Y %T", 
                     localtime_r(&lastlogin, &tm));
        }
           

	snprintf(outstr, sizeof(outstr),
	     "<div class=\"sidebar_row_content\">%s<br><br>"
	     "%s %s<br>"
	     "%s %s<br>"
	     "%s %s<br>"
	     "%s %s<div>",
	     currtime,
   	     _("User : "), 
	     escape_html(eric_session_get_user(wp->session)),
   	     _("State : "), 
	     timestr,
   	     _("Your IP : "), 
     	     eric_session_get_ip_str(wp->session),
             _("Last Login : "),
             lastloginstr);

	n=websWrite(wp, outstr);
    }
    return 0;
}

static int
is_cluster_enabled_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    /* FIXME: wipe off all occurences - it's a kimble relikt */
    ejSetResult(eid, "0");

    return 0;
}

static int
is_cluster_active_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    /* FIXME: wipe off all occurences - it's a kimble relikt */
    ejSetResult(eid, "0");

    return 0;
}

static int
is_rfb_up_on_node_asp(int eid, webs_t wp UNUSED, int argc, char **argv)
{
    int rfb_down = 0;
    if (argc == 1) {
	struct stat st;
	char up_file_path[32];
	snprintf(up_file_path, sizeof(up_file_path), "/var/run/rfb_down_%s", argv[0]);
	rfb_down = (stat(up_file_path, &st) == 0);
    }
    ejSetResult(eid, rfb_down ? "0" : "1");

    return 0;
}

static int is_empty_asp(int eid, webs_t wp UNUSED, int argc, char **argv) {
    int empty = 1;
    if (argc == 1) {
        /* if first char of argv[0] is '\0', variable is empty */
	empty = !*argv[0];
    }
    ejSetResult(eid, empty ? "1" : "0");

    return 0;
}

static int
get_kvm_node_asp(int eid, webs_t wp UNUSED, int argc UNUSED,
		char **argv UNUSED)    
{
    char buf[3];
    
    snprintf(buf, sizeof(buf), "%d", pp_misc_get_kvm_node());
    ejSetResult(eid, buf);
    
    return PP_SUC;
}

static int
get_kvm_node_cnt_asp(int eid, webs_t wp UNUSED, int argc UNUSED,
		char **argv UNUSED)    
{
    char buf[3];
    
    snprintf(buf, sizeof(buf), "%d", pp_misc_get_kvm_node_cnt());
    ejSetResult(eid, buf);
    
    return PP_SUC;
}
