#include <assert.h>
#include <pp/base.h>
#include <pp/ldap_prof.h>
#include <pp/tokenizer.h>
#include <pp/vector.h>
#include <pp/intl.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <pp/intl.h>
#include <pp/xdefs.h>
#include "eric_base.h"
#include "eric_forms.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "wsIntrn.h"
#include "liberic_config.h"

#define PERM_DENIED_PAGE	"/perm_denied.asp"

#define MAX_FORM_ID_LEN 31

static const char cfg_prefix[] = "EC.";
static const char fv_prefix[] = "FV_";
static const char unit_key[] = "unit[%u]";
static const char unit_port_key[] = "unit[%u].port[%u]";
static const char user_key[] = "user[%u]";
static const char group_key[] = "group[%u]";

const char fv_vec_idx_deli = '#';

typedef struct form_flag_entry_s {
    const char * flag_str;
    const u_long flag;
} form_flag_entry_t;

static const char * parse_error_msg = N_("Error during formular evaluation.");
const char * perm_denied_msg = N_("Permission denied.");
const char * internal_error_msg = N_("An internal error occured.");
const char * reg_form_handler_failed_msg = N_("Registering form handler failed");
const char * reg_acl_obj_failed_msg = N_("Registering ACL object failed");
static const char * no_space_msg = N_("No space left for configuration file<br>(Changes have been stored temporarily and will be lost on device reset. Click <a href=\"%s\">here</a> to view memory usage, free some space and apply changes again to store them persistently.)");

pp_hash_t* form_handlers;
static pp_hash_t* form_handlers_short;

static pthread_mutex_t form_handlers_mtx = PTHREAD_MUTEX_INITIALIZER;

static int save_form_var(webs_t wp, form_var_t* fv,
			 pp_profile_type_t prof_type);

static int is_disabled_asp(int eid, webs_t wp, int argc, char** argv);
static int is_disabled_str_asp(int eid UNUSED, webs_t wp, int argc,
			    char** argv);
static int is_default_asp(int eid, webs_t wp, int argc, char** argv);
static int is_default_str_asp(int eid, webs_t wp, int argc, char** argv);
#if !defined(PP_FEAT_DISABLE_ASTERISK_TO_INDICATE_DEFAULT)
static const char* is_default_str_intern(webs_t wp, int argc, char** argv);
#endif
static int get_default_asp(int eid, webs_t wp, int argc, char** argv);
static int get_default_str_asp(int eid, webs_t wp, int argc, char** argv);
static char* get_default_str_intern(webs_t wp, int argc, char** argv);

static int show_required_asterisk(int eid, webs_t wp, int argc, char** argv);

//tmp static const char* form_var_label(pp_cd_as_type_t* type);
static u_long form_var_flags(pp_cd_as_cd_t * cd, const pp_cd_qualified_id_t qid,
                             form_flag_entry_t * flag_map);
//static int cfg_key_flags(const char *key);
static int form_var_str_parse(const char * var, char * tmpl_name,
                              size_t tmpl_name_size);
static int form_var_str_to_idx(const char * var);

static char* cfg_var_to_typename(const char* cfgkey, char* tname,
				 size_t tnamelen);
static char* cfg_var_to_qid(const char* cfgkey, char* tname, size_t tnamelen);
static void form_var_name(const char * tmpl_name, u_int fv_id, char * fv_name, size_t fv_name_len);

/*** form var callback API ***************************************************/
static fv_cb_type_t 
get_form_var_cb_type(const char *cfgkey, char **realkey);

#define INIT_FV_CB_DECL(__name__) \
            inline static char* \
            get_##__name__##_key(webs_t wp, pp_profile_type_t prof_type, \
                                 const char *key, int i, const char *s, \
                                 int *profile); \
            inline static int \
            save_##__name__##_key(webs_t wp, pp_profile_type_t prof_type, \
                                  const char *key, int i, const char *s, \
                                  const char *value); \
            inline static int \
            remove_##__name__##_key(webs_t wp, pp_profile_type_t prof_type, \
                                    const char *key, int i, const char *s); \
            inline static pp_profile_type_t \
            determine_##__name__##_proftype(webs_t wp, const form_var_t *fv);

#define INIT_FV_CB(__name__) \
            _fv_tmpl->get = get_##__name__##_key; \
            _fv_tmpl->save = save_##__name__##_key; \
            _fv_tmpl->remove = remove_##__name__##_key; \
            _fv_tmpl->proftype = determine_##__name__##_proftype;

INIT_FV_CB_DECL(plain)
INIT_FV_CB_DECL(idx)
INIT_FV_CB_DECL(user)
INIT_FV_CB_DECL(user_idx)
INIT_FV_CB_DECL(group)
INIT_FV_CB_DECL(group_idx)
INIT_FV_CB_DECL(group_assoc)
INIT_FV_CB_DECL(unit)
INIT_FV_CB_DECL(unit_idx)
INIT_FV_CB_DECL(unit_port)
INIT_FV_CB_DECL(unit_port_idx)
#if defined(KIRA_RPC)
INIT_FV_CB_DECL(outlet)
#endif /* KIRA_RPC */

form_flag_entry_t form_var_flag_map[] = {
    { "allow_empty",        FV_FLAG_ALLOW_EMPTY },
    { "dont_trim",          FV_FLAG_DONT_TRIM },
    { "tag",                FV_FLAG_TAGABLE },
    { "dont_save",          FV_FLAG_DONT_SAVE },
    { "dont_save_empty",    FV_FLAG_DONT_SAVE_EMPTY },
    { "dont_clean",         FV_FLAG_DONT_CLEAR },
    { "skip_validate",      FV_FLAG_SKIP_VALIDATE },
    { "allow_cond_empty",   FV_FLAG_ALLOW_COND_EMPTY },
    { NULL, 0 }
};

int
forms_init(void)
{
    /* register ASPs */
    websAspDefine("isDisabled", is_disabled_asp);
    websAspDefine("isDisabledStr", is_disabled_str_asp);
    websAspDefine("isDefault", is_default_asp);
    websAspDefine("isDefaultStr", is_default_str_asp);
    websAspDefine("getDefault", get_default_asp);
    websAspDefine("getDefaultStr", get_default_str_asp);
    websAspDefine("ShowRequiredAsterisk", show_required_asterisk);
    
    form_handlers = pp_hash_create(100);
    form_handlers_short = pp_hash_create(100);

    return 0;
}

form_handler_t *
create_form_handler_instance(const char * tmpl_name, const char * acl_object,
			     const form_var_spec_t * fv_spec, size_t fv_cnt)
{
    form_handler_t * fh = (form_handler_t *)calloc(1, sizeof(form_handler_t));
    u_int i;

    assert(tmpl_name);
    assert(acl_object);

    fh->tmpl_name = tmpl_name;
    fh->acl_object = acl_object;
    fh->authorize_hook = default_authorize_hook;
    fh->pre_validate_hook = NULL;
    fh->post_validate_hook = NULL;
    fh->post_save_hook = NULL;
    fh->private_free_hook = free;
    fh->disable_ldap=FALSE;

    fh->fv_cnt = fv_cnt;
    if (fv_cnt > 0) fh->fv_tmpl = calloc(fv_cnt, sizeof(form_var_t));

    /* initialize form var sub-structure */
    for (i = 0; i < fv_cnt; ++i) {
	const form_var_spec_t * _fv_spec = &fv_spec[i];
	form_var_t * _fv_tmpl = &fh->fv_tmpl[i];
	char fv_type[FV_CFGKEY_MAX_LEN + 1];
	pp_cd_as_id_t root_sec;
	pp_cd_as_cd_t * cd = pp_cfg_get_cd();
	pp_cd_as_type_t * type;
        fv_cb_type_t fv_cb_type;
        char *realkey;

        /* determine for var callback type and create "real" cfgkey */
        fv_cb_type = get_form_var_cb_type(_fv_spec->cfgkey, &realkey);
        assert(fv_cb_type);
        
	/* copy fields from formvar template */
	_fv_tmpl->id = _fv_spec->id;
        _fv_tmpl->cfgkey = realkey; // remember to free that one!
	_fv_tmpl->elemkey = _fv_spec->elemkey;

	/* initialize some members, web-name, values */
	form_var_name(tmpl_name, _fv_tmpl->id, _fv_tmpl->fvname, sizeof(_fv_tmpl->fvname));
	_fv_tmpl->val_cnt = 0;
	_fv_tmpl->val.s = NULL;
	_fv_tmpl->val.m = NULL;
	_fv_tmpl->assoc_name = NULL;

	/* resolve the real type */
	cfg_var_to_typename(_fv_tmpl->cfgkey, fv_type, sizeof(fv_type));
	_fv_tmpl->realtype = pp_cd_resolve_type_sym_n_sec(cd, fv_type,
							  PP_CD_RESOLVE_ALIAS, &root_sec);
#if !defined(NDEBUG)
        /**
         * check if form var config key really exists
         * -> remember to install fs-skeleton on config key changes ;-)
         */
	if (_fv_tmpl->realtype == NULL) {
	    pp_log("ERROR: undefined type '%s' in '%s' handler\n", 
                   fv_type, tmpl_name);
	}
	assert(_fv_tmpl->realtype != NULL);
#endif /* !NDEBUG */

	/* resolve the direct type, for the props, name etc */
	type = pp_cd_resolve_type_symbol(cd, fv_type, PP_CD_RESOLVE_SIMPLE);

	/* resolve element type and set val_cnt_max in case of vector */
	if (PP_CD_TYPE_IS_VECTOR_TEMPL(_fv_tmpl->realtype)) {
	    size_t len = strlen(fv_type);
	    if (_fv_tmpl->elemkey == NULL || *_fv_tmpl->elemkey == '\0') {
		/* we want to create elem key but fv-elemkey is NULL,  
		   append default */ 
		snprintf(&fv_type[len], sizeof(fv_type)-len, ".%s",  
			 PP_CD_VECT_ELEMS_COMP_STR);    
            } else {
                snprintf(&fv_type[len], sizeof(fv_type)-len, ".%s",
                         _fv_tmpl->elemkey);
            }
	    _fv_tmpl->elemtype = pp_cd_resolve_type_symbol(cd, 
                                     (const pp_cd_qualified_id_t)fv_type,
				     PP_CD_RESOLVE_VECTOR |
                                     PP_CD_RESOLVE_ALIAS);

#if !defined(NDEBUG)
            if (_fv_tmpl->elemtype == NULL) {
                pp_log("ERROR: undefined type '%s' in '%s' handler\n", 
                       fv_type, tmpl_name);
            }
            assert(_fv_tmpl->elemtype != NULL);
#endif /* !NDEBUG */

            /* init form var callbacks */
            _fv_tmpl->get = NULL;
            _fv_tmpl->save = NULL;
            _fv_tmpl->remove = NULL;
            _fv_tmpl->proftype = NULL;
            if(_fv_tmpl->realtype_vec->idx_type == AS_FUND_TYPE_INT) {
                /* indexed vector */
                _fv_tmpl->val_cnt_max = _fv_tmpl->realtype_vec->bound;
                
                switch(fv_cb_type) {
                    case FV_CB_UNIT_PORT_IDX_KEY:
                        INIT_FV_CB(unit_port_idx);
                        break;
                    case FV_CB_UNIT_IDX_KEY:
                        INIT_FV_CB(unit_idx);
                        break;
                    case FV_CB_USER_IDX_KEY:
                        INIT_FV_CB(user_idx);
                        break;
                    case FV_CB_GROUP_IDX_KEY:
                        INIT_FV_CB(group_idx);
                        break;
                    case FV_CB_IDX_KEY:
                        INIT_FV_CB(idx);
                        break;
                    default:
                        pp_log("invalid form var type for index key %s%s%s\n",
                               _fv_spec->cfgkey, 
                               _fv_spec->elemkey ? "." : "",
                               _fv_spec->elemkey ? : "");                               
                        abort(); // should never happen
                        break;
                }
            } else {
                /* assocative vector */
                switch(fv_cb_type) {
                    case FV_CB_GROUP_ASSOC_KEY:
                        INIT_FV_CB(group_assoc);
                        break;
                    /* not yet implemented */
                    case FV_CB_UNIT_PORT_ASSOC_KEY:
                    case FV_CB_UNIT_ASSOC_KEY:
                    case FV_CB_USER_ASSOC_KEY:
                    case FV_CB_ASSOC_KEY:
                    default:
                        pp_log("invalid form var type for associative key %s%s%s\n",
                               _fv_spec->cfgkey, 
                               _fv_spec->elemkey ? "." : "",
                               _fv_spec->elemkey ? : "");                               
                        abort(); // should never happen
                        break;
                }
            }
	} else {
	    _fv_tmpl->elemtype = NULL;
            _fv_tmpl->val_cnt_max = 1;
            
            switch(fv_cb_type) {
#if defined(KIRA_RPC)
                case FV_CB_OUTLET_KEY:
                    INIT_FV_CB(outlet);
                    break;
#endif /* KIRA_RPC */
                case FV_CB_UNIT_PORT_KEY:
                    INIT_FV_CB(unit_port);
                    break;
                case FV_CB_UNIT_KEY:
                    INIT_FV_CB(unit);
                    break;
                case FV_CB_USER_KEY:
                    INIT_FV_CB(user);
                    break;
                case FV_CB_GROUP_KEY:
                    INIT_FV_CB(group);
                    break;
                case FV_CB_PLAIN_KEY:
                    INIT_FV_CB(plain);
                    break;
                default:
                        pp_log("invalid form var type for key %s%s%s\n",
                               _fv_spec->cfgkey, 
                               _fv_spec->elemkey ? "." : "",
                               _fv_spec->elemkey ? : "");                               
                    abort(); // should never happen
                    break;
            }
	}

        cfg_var_to_qid(fv_type, fv_type, sizeof(fv_type));

	/* cache some members based on the type description, acl, label */
	_fv_tmpl->flags = form_var_flags(cd, fv_type, form_var_flag_map);
	if (root_sec != AS_SECTION_CONFIGS) {
            _fv_tmpl->flags |= FV_FLAG_DONT_SAVE;
        }
        
        /* resolve perm / acl object */
        _fv_tmpl->acl_object = pp_cd_lookup_prop(cd, fv_type, "perm");
        if (!_fv_tmpl->acl_object) {
            /**
             * This warning occures, if no perm flag is set for a config key,
             * displayed in web frontend. In that case, this key is writeable
             * by any user, no matter what permissions the user has or not has!
             *
             * Workaround:
             * - set perm flag for config key
             * - if config key too generic (e.g. "long"), create an alias and
             *   set perm flag, e.g. 'some_number: long { perm: "some_perm"; };'
             */
            pp_log("WARNING! No ACL defined for %s\n", _fv_tmpl->cfgkey);
        }

        _fv_tmpl->label = pp_cd_lookup_prop(cd, fv_type, "desc"); 
    }
    
    return fh;
}

/* destroy the global form handler templates */
static void
destroy_form_handler_instance_global(void * fh)
{
    form_handler_t * _fh = (form_handler_t *)fh;
    // global fh have no private vars except the cfgkey 
    // (see get_form_var_cb_type)
    if (_fh) {
        u_int i;
	for (i = 0; i < _fh->fv_cnt; ++i) {
	    const form_var_t *_fv = &_fh->fv_tmpl[i];
	    free(_fv->cfgkey); // generated by get_form_var_cb_type
	}
        free(_fh->fv_tmpl);
    }
    free(fh);
}

/* destroy a temporary form handler copy that was used for serving one page */
static void
destroy_form_handler_instance_copy(void * fh)
{
    form_handler_t * _fh = (form_handler_t *)fh;
    if (_fh->private_free_hook && _fh->private) {
      _fh->private_free_hook(_fh->private);
    }
    // do NOT free the _fh->fv_tmpl here
    free(fh);
}

int
register_form_handler_instance(form_handler_t * fh)
{
    int ret = 0;
    assert(fh);
    assert(fh->tmpl_name);
    fh->private = NULL; /* this *must* be NULL in the registered handler structure */

    /*
     * We check this here since create_form_handler_instance() must not return
     * NULL because we do not check its return value.
     */
    if (strchr(fh->tmpl_name, fv_vec_idx_deli) != NULL) {
	pp_log("Template names must not contain delimiter '%c'! (%s)\n",
               fv_vec_idx_deli, fh->tmpl_name);
	return -1;
    }

    MUTEX_LOCK(&form_handlers_mtx);
    ret = pp_hash_set_entry(form_handlers, fh->tmpl_name, fh, destroy_form_handler_instance_global);
    MUTEX_UNLOCK(&form_handlers_mtx);
    return ret;
}

form_handler_t*
lookup_form_handler(const char * tmpl_name)
{
    form_handler_t * fh;
    
    MUTEX_LOCK(&form_handlers_mtx);
    fh = pp_hash_get_entry(form_handlers, tmpl_name);
    MUTEX_UNLOCK(&form_handlers_mtx);
    return fh;
}

static const form_var_t *
lookup_form_var(const char * tmpl_name, u_int idx)
{
    form_handler_t * fh = lookup_form_handler(tmpl_name);
    u_int i;

    if (fh) {
	// we need to search for it, as the ids are not necessarily indices
	for (i = 0; i < fh->fv_cnt; ++i) {
	    const form_var_t * fv = &fh->fv_tmpl[i];
	    if (fv->id == idx) return fv;
	}
    }
    return NULL;
}

/* probably unused - remove it?! */
#if 0
static const char *
lookup_form_var_name(const char * tmpl_name, u_int idx)
{
    const form_var_t * fv;
    if ((fv = lookup_form_var(tmpl_name, idx)) == NULL) return NULL;
    return fv->fvname;
}
#endif

static const form_var_t *
lookup_form_var_by_id(const char * var)
{
    char form[MAX_FORM_ID_LEN + 1];
    int id;
    id = form_var_str_parse(var, form, sizeof(form));
    return lookup_form_var(form, id);
}

int
settings_form_header(webs_t wp, const char * acl_object_name,
		     const char * read_perm_name, const char * write_perm_name)
{
    if (!form_was_submitted(wp)) {
	/* check "view" permission */	
	if (acl_object_name &&
	    PP_ERR == pp_um_user_has_permission(wp->user, acl_object_name, 
                                                read_perm_name)) {
	    eric_notify_security_violation(wp->session);
	    show_permission_denied_page(wp);	    
	    return FORM_SETTINGS_PERM_DENIED;
	}
	
	return FORM_SETTINGS_NOT_SUBMITTED; 
    } else {
        /* check "change" permission */
        if (acl_object_name &&
	    PP_ERR == pp_um_user_has_permission(wp->user, acl_object_name,
                                                write_perm_name)) {        
	    eric_notify_security_violation(wp->session);
	    set_response(wp, ERIC_RESPONSE_ERROR, pp_intl_translate(perm_denied_msg));	
	    return FORM_SETTINGS_PERM_DENIED;
        }    
    }    
    return 0;
}

void
show_permission_denied_page(webs_t wp)
{
    websSetRequestPath(wp, "/lib/webpages", PERM_DENIED_PAGE);
}

int
parse_form_vars(webs_t wp)
{
    u_int i, j, sz;
    char fvkey[FV_KEY_MAX_LEN+1];
    const char* fvval;
    form_var_t * fv;

    for (i = 0; wp->form_vars[i].cfgkey != NULL; ++i) {
	fv = &wp->form_vars[i];

	assert(fv->realtype->base.type & AS_TYPE);
	assert(!PP_CD_IS_TYPE(&fv->realtype->base, AS_TYPE_STRUCT));
	assert(!PP_CD_IS_TYPE(&fv->realtype->base, AS_TYPE_FKT));
	assert(!PP_CD_IS_TYPE(&fv->realtype->base, AS_TYPE_CHOICE_TMPL));

	if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
            const pp_cd_as_type_vector_tmpl_t *vect = fv->realtype_vec;
                
            if (vect->idx_type == AS_FUND_TYPE_INT) {
                form_var_sz_name(fv->fvname, fvkey, sizeof(fvkey));
                fvval = websGetVar(wp, fvkey, "0");
                sz = pp_strtoul_10(fvval, UINT_MAX, NULL);
                /* not >= here! we want the actual number of elements which
                   may be equal to bound */
                if (sz > vect->bound) {
                    pp_log("size of %s == %d > bound %d\n", 
                           fv->fvname, sz, vect->bound);
                    set_response(wp, ERIC_RESPONSE_ERROR, pp_intl_translate(parse_error_msg));
                    return -1;
                }
                fv->val_cnt = sz;
                if (sz > 0) fv->val.m = calloc(sz, sizeof(char*));

                for (j = 0; j < sz; ++j) {
                    form_var_vec_name(fv->fvname, j, fvkey, sizeof(fvkey));
                    
                    if (NULL != (fv->val.m[j] = websGetVar(wp, fvkey, NULL))) {
                        /* trim whitespaces if not forbidden */
                        if (!(fv->flags & FV_FLAG_DONT_TRIM)) {
			    websTrimVar(wp, fvkey);
			}
                    } else {
                        websSetVar(wp, fvkey, "");
                        fv->val.m[j] = websGetVar(wp, fvkey, "");
                    }
                }
	    } else {
                /* handle assocative vectors */
                /* now this is a bit tricky...
                   as we don't know the 'index' of an associative vector,
                   walk through complete list of cgi vars and collect that ones
                   starting with the desired name */

                sym_t *sp;
                char *m[vect->bound];
                char *assoc_name[vect->bound];
                int len = strlen(fv->fvname);
                
                fv->val_cnt = 0;
                for (sp = symFirst(wp->cgiVars); 
                     sp && fv->val_cnt < vect->bound; 
                     sp = symNext(wp->cgiVars)) {
                    char *elem;

                    if(sp->content.type == string &&
                       sp->content.value.string &&
                       sp->name.type == string &&
                       sp->name.value.string &&
                       !strncmp(sp->name.value.string, fv->fvname, len) &&
                       sp->name.value.string + len ==
                           (elem = strrchr(sp->name.value.string,
                                           fv_vec_idx_deli))) {
                        /* we got a symbol whose type is string, the name set
                           is prefixed by our form var name, followed by the
                           index delimiter => bingo! */ 
/* FIXME: TODO strdup that? who cares for freeing? */
                        m[fv->val_cnt] = sp->content.value.string;
                        assoc_name[fv->val_cnt] = ++elem;
                        ++fv->val_cnt; /* set one var, get next */
                    }
                }
                
                /* we are done, copy tmp values to real fv */
                if (fv->val_cnt > 0) {
                    fv->val.m = calloc(fv->val_cnt, sizeof(char*));
                    fv->assoc_name = calloc(fv->val_cnt, sizeof(char*));
                    sz = fv->val_cnt * sizeof(char*);
                    memcpy(fv->val.m, m, sz);
                    memcpy(fv->assoc_name, assoc_name, sz);
                }
            }
	} else { // scalar value, such as AS_TYPE_STRING_TMPL etc.
	    if (NULL != (fv->val.s = websGetVar(wp, fv->fvname, NULL))) {
		/* trim whitespaces if not forbidden */
		if (!(fv->flags &  FV_FLAG_DONT_TRIM)) {
		    websTrimVar(wp, fv->fvname);
		}
	    } else {
                /* unchecked checkboxes have no value... set it to "no" */
                if(PP_CD_IS_TYPE(&(fv->realtype->base), AS_TYPE_BOOL))
                    websSetVar(wp, fv->fvname, "no");
                else
                    websSetVar(wp, fv->fvname, "");
                fv->val.s = websGetVar(wp, fv->fvname, "");
	    }
	}
    }

    return 0;
}

int save_form_vars(webs_t wp) {
    u_int i;
    form_var_t* fv;
    pp_profile_type_t prof_type = PP_PROFILE_LOCAL;
    pp_cfg_tx_t* tx;
    char *retstr = NULL;
    int ret;
    int reset_to_default = !strcmp(websGetVar(wp, "__reset_to_defaults__", ""),
                                   "yes");

    tx = pp_cfg_tx_begin(1);
    for (i = 0; wp->form_vars[i].cfgkey != NULL; ++i) {
	fv = &wp->form_vars[i];

	if (((fv->flags & FV_FLAG_DONT_SAVE) && !reset_to_default) ||
            form_var_is_disabled(wp, fv))
        {
	    continue;
	}

	if (PP_FAILED(save_form_var(wp, fv, prof_type))) {
	    /* We got an error - bail out */
            goto error;
	}
    }
    ret = pp_cfg_tx_commit_core(tx, DO_FLUSH, 1, 1, &retstr);
    tx = NULL; /* tx context was destroyed during commit! */
    if (retstr && *retstr) {
        set_response(wp, PP_SUCCED(ret) ? ERIC_RESPONSE_OK : ERIC_RESPONSE_ERROR, retstr);
    }
    free(retstr);

    if (PP_SUCCED(ret)) return PP_SUC;

 error:
    if (tx) pp_cfg_tx_rollback(tx);
    if (wp->response_msg == NULL) {
        if (errno == ENOSPC) {
            set_response(wp, ERIC_RESPONSE_ERROR, pp_intl_translate(no_space_msg),
                         "__fd_usage.asp");
	} else {
            set_response(wp, ERIC_RESPONSE_ERROR, pp_intl_translate(pp_error_string(errno)));
	}
    }
    return PP_ERR;
}

static int save_form_var(webs_t wp, form_var_t* fv,
                         pp_profile_type_t prof_type) {
    int i, ret = PP_SUC;
    int save_empty = !(fv->flags & FV_FLAG_DONT_SAVE_EMPTY);
    int reset_to_default = !strcmp(websGetVar(wp, "__reset_to_defaults__", ""),
                                   "yes");
    pp_profile_type_t def_prof_type = PP_PROFILE_DEFAULTS;
    char *value;

    if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
	char cfgkey[FV_CFGKEY_MAX_LEN+1];
	char szstr[11];
        const char *ck;
        
	ck = cfg_var_vecelem_key(fv->cfgkey, fv->elemkey,
                                 cfgkey, sizeof(cfgkey));
	for (i = 0; i < fv->val_cnt; ++i) {
            assert(fv->val.m[i]);
            if(reset_to_default) {
                value = fv->get(wp, def_prof_type, ck, i, 
                                fv->assoc_name ? fv->assoc_name[i] : NULL,
                                NULL);
                if((value == NULL && fv->val.m[i][0] == '\0') ||
                   !pp_strcmp_safe(value, fv->val.m[i]) ||
                   (fv->flags & FV_FLAG_SKIP_VALIDATE) ||
                   (PP_CD_IS_TYPE(&(fv->realtype->base), AS_TYPE_BOOL) &&
                    !value && !strcmp(fv->val.m[i], "no"))) {
                    /* value is default, remove it from local profile */
                    fv->remove(wp, prof_type, ck, i, 
                               fv->assoc_name ? fv->assoc_name[i] : NULL);
                    free(value);
                    continue;
                }
                free(value);
            }
            if(fv->val.m[i][0] != '\0' || save_empty || reset_to_default) {
                if (PP_FAILED(ret = fv->save(wp, prof_type, ck, i,
                                             fv->assoc_name ? 
                                                 fv->assoc_name[i] : NULL,
                                             fv->val.m[i]))) {
                    break;
                }
	    }
	}
        if(fv->realtype_vec->idx_type == AS_FUND_TYPE_INT) {
            /* save size key only for indexed vectors */
            cfg_var_vecsz_key(fv->cfgkey, cfgkey, sizeof(cfgkey));
            snprintf(szstr, sizeof(szstr), "%u", fv->val_cnt);
            ret = fv->save(wp, prof_type, cfgkey, 0, NULL, szstr);
        }
    } else { /* scalar */
        if(reset_to_default) {
            value = fv->get(wp, def_prof_type, fv->cfgkey, 0, NULL, NULL);
            if((value == NULL && fv->val.s[0] == '\0') ||
               !pp_strcmp_safe(value, fv->val.s) ||
               (fv->flags & FV_FLAG_SKIP_VALIDATE) ||
               (PP_CD_IS_TYPE(&(fv->realtype->base), AS_TYPE_BOOL) &&
                !value && !strcmp(fv->val.s, "no"))) {
                /* value is default, remove it from local profile */
                ret = fv->remove(wp, prof_type, fv->cfgkey, 0, NULL);
                free(value);
                return ret;
            }
            free(value);
        }
        if(fv->val.s[0] != '\0' || save_empty) {
            ret = fv->save(wp, prof_type, fv->cfgkey, 0, NULL, fv->val.s);
        }
    }
    return ret;
}

static void
delete_form_var(webs_t wp, form_var_t * fv)
{
    char fvkey[FV_KEY_MAX_LEN+1];
    u_int j;

    if (fv) {
	if(PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
            if(fv->realtype_vec->idx_type == AS_FUND_TYPE_INT) {
                /* indexed vector */
                form_var_sz_name(fv->fvname, fvkey, sizeof(fvkey));
                symDelete(wp->cgiVars, fvkey);
                for (j = 0; j < fv->val_cnt; ++j) {
                    form_var_vec_name(fv->fvname, j, fvkey, sizeof(fvkey));
                    symDelete(wp->cgiVars, fvkey);
                }
            } else {
                /* associative vector */
                for (j = 0; j < fv->val_cnt; ++j) {
                    form_var_assoc_name(fv->fvname, fv->assoc_name[j],
                                        fvkey, sizeof(fvkey));
                    symDelete(wp->cgiVars, fvkey);
                }
            }
            /* for multi value form vars, we allocated extra space */
            free(fv->val.m);
            fv->val.m = NULL;
            free(fv->assoc_name);
            fv->assoc_name = NULL;
            fv->val_cnt = 0;
	} else {
	    symDelete(wp->cgiVars, fv->fvname);
	}
    }
}

void
delete_form_vars(webs_t wp)
{
    u_int i;

    for (i = 0; wp->form_vars[i].cfgkey != NULL; i++) {
	delete_form_var(wp, &wp->form_vars[i]);
    }
}

char *
get_config_var(webs_t wp, const char * var)
{
    char* ret = NULL;
    const form_var_t* fv;
    int  fvidx;
    char cfgkey[FV_CFGKEY_MAX_LEN+1];
    const char* ck;

    // 1. check whether its a form-var id in current page
    if (!strncmp(var, fv_prefix, sizeof(fv_prefix) - 1)) {
	if ((fv = lookup_form_var_by_id(var)) == NULL)
	    return ret;
	if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
            if(fv->realtype_vec->idx_type == AS_FUND_TYPE_INT) {
                /* indexed vector */
                if (0 > (fvidx = form_var_str_to_idx(var))) { // size requested
                    cfg_var_vecsz_key(fv->cfgkey, cfgkey, sizeof(cfgkey));
                    ret = fv->get(wp, PP_PROFILE_EFFECTIVE, cfgkey,
                                  0, NULL, NULL);
                } else {                                      // value requested
                    ck = cfg_var_vecelem_key(fv->cfgkey, fv->elemkey,
                                             cfgkey, sizeof(cfgkey));
                    ret = fv->get(wp, PP_PROFILE_EFFECTIVE, ck,
                                  fvidx, NULL, NULL);
                }
            } else {
                /* assocative vector */
                char *elem;
                
                if(NULL != (elem = strrchr(var, fv_vec_idx_deli))) {
                    ck = cfg_var_vecelem_key(fv->cfgkey, fv->elemkey,
                                             cfgkey, sizeof(cfgkey));
                    ret = fv->get(wp, PP_PROFILE_EFFECTIVE, ck,
                                  0, ++elem, NULL);
                }
            }
	} else {
            /* scalar */
	    ret = fv->get(wp, PP_PROFILE_EFFECTIVE, fv->cfgkey, 0, NULL, NULL);
	}
    }
    // 2. check for regular config var, expanding embedded webs vars
    else if (!strncmp(var, cfg_prefix, strlen(cfg_prefix))) {
        pp_cfg_get_nodflt(&ret, &var[sizeof(cfg_prefix) - 1]);
	// if error -> ret = NULL;
    }
    // 3. nothing, return the empty
    else return NULL;
    return ret;
}

int
remove_entries(webs_t wp, form_var_t * form_vars, u_int num_entries,
	       int * ids, u_int num_ids,
	       int (*to_be_removed)(form_var_t*, u_int)) {
    u_int i, j, k;

    for (i = 0; i < num_entries; i++) {
	/* remove empty (to_be_removed) entries */
	if (to_be_removed(form_vars, i)) {
            /* find next non empty entry and swap with current (empty) entry */
	    for (j = i + 1; j < num_entries; j++) {
		if (!to_be_removed(form_vars, j)) {
                    /* swap entries i,j */
		    for (k = 0; k < num_ids; k++) {
			const char * s = form_vars[ids[k]].val.m[i];
			form_vars[ids[k]].val.m[i]=form_vars[ids[k]].val.m[j];
			form_vars[ids[k]].val.m[j] = s;
		    }
		    break;
		}
	    }
	    if (j >= num_entries) {
                /* only empty entries remaining, adjust count */
                /* i = index of first empty entry */
		for (k = 0; k < num_ids; k++) {
		    form_vars[ids[k]].val_cnt = i;
		}
		break;
	    }
	}
    }
    /* adjust the webs variables */
    for (k = 0; k < num_ids; k++) {
	if (adjust_webs_var(wp, &form_vars[ids[k]]) == -1) {
	    return -1;
	}
    }

    return 0;
}

void
delete_fh_form_vars(webs_t wp, form_handler_t * fh)
{
    u_int i;

    for (i = 0; i < fh->fv_cnt; ++i) {
	delete_form_var(wp, &fh->fv[i]);
    }
}

void
fh_disable_validate_and_save(form_handler_t * fh, int fv_id_ignore)
{
    u_int i;

    for (i = 0; i < fh->fv_cnt; ++i) {
	if (fv_id_ignore < 0 || i != (u_int)fv_id_ignore) {
	    fh->fv[i].flags |= FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
	}
    }
}

int
adjust_webs_var(webs_t wp, form_var_t * fv)
{
    char fvkey[FV_KEY_MAX_LEN+1];
    char fvszkey[11];
    u_int i;

    if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
	form_var_sz_name(fv->fvname, fvkey, sizeof(fvkey));
	snprintf(fvszkey, sizeof(fvszkey), "%u", fv->val_cnt);
	websSetVar(wp, fvkey, fvszkey);

	fv->val.m = realloc(fv->val.m, fv->val_cnt * sizeof(char *));
	for (i = 0; i < fv->val_cnt; i++) {
	    form_var_vec_name(fv->fvname, i, fvkey, sizeof(fvkey));
	    websSetVar(wp, fvkey, fv->val.m[i]);
	    /* fv->val.m[i] is not valid anymore */
	    fv->val.m[i] = websGetVar(wp, fvkey, "");
	}
    } else {
	websSetVar(wp, fv->fvname, fv->val.s);
	/* fv->val.s is not valid anymore */
	fv->val.s = websGetVar(wp, fv->fvname, "");
    }
    return 0;
}

int
form_was_submitted(webs_t wp)
{
    char ** action_vars;
    int i, action = 0;
    action_vars = websSearchVars(wp, "action_");    
    a_assert(action_vars);
    for (i = 0; action_vars[i] != NULL; i++) {
	if (websGetVar(wp, action_vars[i], "")[0] != '\0') {
	    action = 1;
	    // there was at least one action_ variable with a value
	    // unequal to an empty string
	    break;
	}
    }
    pp_free_dynamic_array(action_vars, -1, free);    
    //pp_log("form_was_submitted=%d\n", action);
    return action;
}

int
form_button_clicked(webs_t wp, const char * var)
{
    char var_x[256];
    const char * val;

    assert(var);
    if (websGetVar(wp, var, NULL) != NULL) {
	/* variable is set --> return 1 */
	return 1;
    } else {
	/* maybe we have a graphical button -> check this ... */
	snprintf(var_x, sizeof(var_x), "%s.x", var);
	if ((val = websGetVar(wp, var_x, NULL)) == NULL) {	    
	    /* nope --> return 0 */
	    return 0;
	} else if (val[0] == '\0') {
	    /* value is empty string --> return 0 */
	    return 0;
	} else {
	    /* valid graphical button -> return 1 */
	    return 1;
	}
    }
}

static int reset_button_clicked(webs_t wp) {
    char ** action_vars;
    int i, reset = 0;
    
    action_vars = websSearchVars(wp, "action_defaults");    
    a_assert(action_vars);
    for (i = 0; action_vars[i] != NULL; i++) {
	if (websGetVar(wp, action_vars[i], "")[0] != '\0') {
            /* we found a reset button */
            reset = 1;
	    break;
	}
    }
    pp_free_dynamic_array(action_vars, -1, free);    
    return reset;
}

int form_var_is_disabled(webs_t wp, const form_var_t* fv)
{
    pp_profile_type_t proftype = fv->proftype(wp, fv);

    /* if ldap-var, disable */
    if (proftype == PP_PROFILE_LDAP_USER || proftype == PP_PROFILE_LDAP_DEV) {
        return 1;
    }
    
    /* for root, all form vars are enabled */
    if (!pp_um_user_get_uid(wp->user)) {
        return 0;
    }
    
    /* if no acl is given, disable the form var! */
    /* acl_object is resolved from cfgkey */
    /*
    printf("form_var_is_disabled: fv->acl_object  %s->%s\n",
             fv->fvname, fv->acl_object);
    */
    if (fv->acl_object
	&& (PP_SUC == pp_um_user_has_permission(wp->user, fv->acl_object,
                                                pp_acl_raasip_yes_str) ||
	    PP_SUC == pp_um_user_has_permission(wp->user, fv->acl_object,
                                                pp_acl_raasip_control_str))) {
        return 0;
    }

    return 1;
}

/*
 * isDisabled(arg0, [arg1[, arg2]])
 *   parameter description --> see source
 *
 * Be careful what you return here, we neither allocate nor free
 * the returned string, so it must be static, or something the
 * caller has control over anyway
 */
static const char *
is_disabled_str_intern(int eid UNUSED, webs_t wp, int argc, char** argv)
{
    const char * tmpl = websGetVar(wp, "__template__", NULL);
    char * idxstr;
    const char * text = "disabled";
    int idx;
    
    if (argc == 1) {
	/* One argument: represents the variable index in the current template */
	idxstr = argv[0];
    } else if (argc == 2) {
	/* Two arguments: two cases ... */
	idx = pp_strtol_10(argv[0], -1, NULL);
	if (idx >= 0) {
	    /* ... first arg represents the variable index in the current
	     * template and the second arg is the text to return if disabled
	     */
	    idxstr = argv[0];
	    text = argv[1];
	} else {
	    /* ... first arg represents the template and the second arg is the
	     * variable index in this template
	     */
	    tmpl = argv[0];
	    idxstr = argv[1];
	}
    } else if (argc == 3) {
	/* Three arguments: The first arg represents the template, the second
	 * arg is the variable index in this template and the third arg is the
	 * text to return if disabled
	 */
	tmpl = argv[0];
	idxstr = argv[1];
	text = argv[2];
    } else {
	assert(0);
	return text;
    }

    if (tmpl && (idx = pp_strtol_10(idxstr, -1, NULL)) >= 0) {
	const form_var_t * var = lookup_form_var(tmpl, idx);
	if (var && !form_var_is_disabled(wp, var)) return "";
    }

    return text;
}

static int
is_disabled_str_asp(int eid, webs_t wp, int argc, char** argv)
{
    ejSetResult(eid, is_disabled_str_intern(eid, wp, argc, argv));
    return 0;
}

static int
is_disabled_asp(int eid UNUSED, webs_t wp, int argc, char** argv)
{
    websWrite(wp, "%s", is_disabled_str_intern(eid, wp, argc, argv));
    return 0;
}

#if defined(PP_FEAT_DISABLE_ASTERISK_TO_INDICATE_DEFAULT)
static int
is_default_asp(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char** argv UNUSED)
{
    return 0;
}

static int
is_default_str_asp(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char** argv UNUSED)
{
    ejSetResult(eid, "");
    return 0;
}
#else /* !PP_FEAT_DISABLE_ASTERISK_TO_INDICATE_DEFAULT */
static int
is_default_asp(int eid UNUSED, webs_t wp, int argc, char** argv)
{
    websWrite(wp, "%s", is_default_str_intern(wp, argc, argv));
    return 0;
}

static int
is_default_str_asp(int eid, webs_t wp, int argc, char** argv)
{
    ejSetResult(eid, is_default_str_intern(wp, argc, argv));
    return 0;
}
#endif

#if !defined(PP_FEAT_DISABLE_ASTERISK_TO_INDICATE_DEFAULT)
/**
 * detect whether a given form var is default
 * this is if form var value is
 * - not set
 * - stored in a profile lower than PP_PROFILE_LOCAL
 * - equal to value at PP_PROFILE_DEFAULTS layer
 * - empty and default is not set
 * @param arvg[0] FV_ID
 * @param argv[1] vector idx, -1 for vector size
 */
static const char* is_default_str_intern(webs_t wp, int argc UNUSED, char** argv) {
    static const char *default_str = PP_FV_DEFAULT_ASTERISK_STR;
    static const char *not_default_str = "";
    const char *ret = not_default_str;
    char *v_effective = NULL, *v_default;
    const char *ck = NULL;
    int prof;
    int idx, fvidx = 0;
    const char *tmpl = websGetVar(wp, "__template__", NULL);
    char cfgkey[FV_CFGKEY_MAX_LEN+1];
    const form_var_t *fv;
    char *elem = NULL;
    
    assert(argc > 0);
    
    if (tmpl && (idx = pp_strtol_10(argv[0], -1, NULL)) >= 0 &&
        (fv = lookup_form_var(tmpl, idx)) != NULL) {
	if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
            if(fv->realtype_vec->idx_type == AS_FUND_TYPE_INT) {
                /* indexed vector */
                assert(argc == 2);
                if (0 > (fvidx = pp_strtol_10(argv[1], -1, NULL))) {
                    // size requested
                    cfg_var_vecsz_key(fv->cfgkey, cfgkey, sizeof(cfgkey));
                    v_effective = fv->get(wp, PP_PROFILE_EFFECTIVE,
                                          cfgkey, 0, NULL, &prof);
                } else { // value requested
                    ck = cfg_var_vecelem_key(fv->cfgkey, fv->elemkey,
                                             cfgkey, sizeof(cfgkey));
                    v_effective = fv->get(wp, PP_PROFILE_EFFECTIVE, 
                                          ck, fvidx, NULL, &prof);
                }
            } else {
                /* assocative vector */
                if(NULL != (elem = strrchr(fv->cfgkey, fv_vec_idx_deli))) {
                    ck = cfg_var_vecelem_key(fv->cfgkey, fv->elemkey,
                                             cfgkey, sizeof(cfgkey));
                    v_effective = fv->get(wp, PP_PROFILE_EFFECTIVE,
                                          ck, 0, ++elem, &prof);
                }
            }
	} else {
	    v_effective = fv->get(wp, PP_PROFILE_EFFECTIVE, 
                                  fv->cfgkey, 0, NULL, &prof);
	}

        if(prof < PP_PROFILE_LOCAL) {
            /* we either didn't get var (-1) or we got var out of a profile
               which is default */
            ret = default_str;
        } else {
            if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
                if(fv->realtype_vec->idx_type == AS_FUND_TYPE_INT) {
                    /* indexed vector */
                    if (0 > fvidx) { // size requested
                        v_default = fv->get(wp, PP_PROFILE_DEFAULTS, 
                                            cfgkey, 0, NULL, NULL);
                    } else { // value requested
                        v_default = fv->get(wp, PP_PROFILE_DEFAULTS,
                                            ck, fvidx, NULL, NULL);
                    }
                } else {
                    /* assocative vector */
                        v_default = fv->get(wp, PP_PROFILE_DEFAULTS,
                                            ck, 0, elem, NULL);
                }
            } else {
                v_default = fv->get(wp, PP_PROFILE_DEFAULTS,
                                    fv->cfgkey, 0, NULL, NULL);
            }
            
            if(!pp_strcmp_safe(v_default, v_effective) ||
               (!v_default && !*v_effective) ||
               (PP_CD_IS_TYPE(&(fv->realtype->base), AS_TYPE_BOOL) &&
                !v_default && !strcmp(v_effective, "no"))) {
                /* stored value is the same as default value */
                ret = default_str;
            }
            
            free(v_default);
        }
        
        free(v_effective);
    }

    return ret;
}
#endif /* !defined(PP_FEAT_DISABLE_ASTERISK_TO_INDICATE_DEFAULT) */

static int
get_default_asp(int eid UNUSED, webs_t wp, int argc, char** argv)
{
    char * s = get_default_str_intern(wp, argc, argv);
    websWrite(wp, "%s", s ? s : "");
    free(s);
    return 0;
}

static int
get_default_str_asp(int eid, webs_t wp, int argc, char** argv)
{
    char * s = get_default_str_intern(wp, argc, argv);
    ejSetResult(eid, s ? s : "");
    free(s);
    return 0;
}

/**
 * gets default of a given form var
 * @param arvg[0] FV_ID
 * @param argv[1] vector idx
 */
static char* get_default_str_intern(webs_t wp, int argc UNUSED, char** argv) {
    char *ret = NULL;
    const char *ck;
    int prof;
    int idx, fvidx;
    const char *tmpl = websGetVar(wp, "__template__", NULL);
    char cfgkey[FV_CFGKEY_MAX_LEN+1];
    const form_var_t *fv;
    
    assert(argc > 0);
    
    if (tmpl && (idx = pp_strtol_10(argv[0], -1, NULL)) >= 0 &&
        (fv = lookup_form_var(tmpl, idx)) != NULL) {
	if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
            assert(argc == 2);
	    if (0 > (fvidx = pp_strtol_10(argv[1], -1, NULL))) {
                // size requested
		cfg_var_vecsz_key(fv->cfgkey, cfgkey, sizeof(cfgkey));
		ret = fv->get(wp, PP_PROFILE_DEFAULTS, cfgkey, 0, NULL, &prof);
	    } else { // value requested
		ck = cfg_var_vecelem_key(fv->cfgkey, fv->elemkey,
					 cfgkey, sizeof(cfgkey));
		ret = fv->get(wp, PP_PROFILE_DEFAULTS, ck, fvidx, NULL, &prof);
	    }
	} else {
	    ret = fv->get(wp, PP_PROFILE_DEFAULTS, fv->cfgkey, 0, NULL, &prof);
	}
    }

    return ret;
}

#if defined(PP_FEAT_DISPLAY_ASTERISK_TO_INDICATE_REQUIRED)
static int
show_required_asterisk(int eid UNUSED, webs_t wp, int argc UNUSED, char** argv UNUSED)
{
    websWrite(wp, " *");
    return 0;
}
#else
static int
show_required_asterisk(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char** argv UNUSED)
{
    return 0;
}
#endif

char *
get_form_var_value_unsaved(webs_t wp, const char * tmpl_name, unsigned int fv_id)
{
    form_handler_t * fh;
    const form_var_t * fv;
    size_t i;

    if (wp->fh_vec) {
	for (i = 0; i < vector_size(wp->fh_vec); ++i) {
	    fh = (form_handler_t *)vector_get(wp->fh_vec, i);
	    if (fh && fh->tmpl_name && !strcmp(fh->tmpl_name, tmpl_name)) {
		if (fh->fv_cnt <= fv_id) return NULL;
		fv = &fh->fv[fv_id];
		if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) return NULL;
		return strdup(fv->val.s);
	    }
	}
    }
    MUTEX_LOCK(&form_handlers_mtx);
    fh = pp_hash_get_entry(form_handlers, tmpl_name);
    MUTEX_UNLOCK(&form_handlers_mtx);
    if (fh) {
	if (fh->fv_cnt <= fv_id) return NULL;
	fv = &fh->fv_tmpl[fv_id];
	if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) return NULL;
        return fv->get(wp, PP_PROFILE_EFFECTIVE, fv->cfgkey, 0, NULL, NULL);
    }
    return NULL;
}

static int reset_to_defaults(webs_t wp) {
    size_t i, j;
    form_var_t *fv;
    pp_profile_type_t prof_type = PP_PROFILE_DEFAULTS;
    char *value;
    
    for (j = 0; wp->form_vars[j].cfgkey != NULL; ++j) {
	fv = &wp->form_vars[j];

	if (form_var_is_disabled(wp, fv)) {
	    continue;
	}
        
        if (PP_CD_TYPE_IS_VECTOR_TEMPL(fv->realtype)) {
            char cfgkey[FV_CFGKEY_MAX_LEN+1];
            char fvkey[FV_KEY_MAX_LEN+1];
            const char *ck;
            
            if(fv->realtype_vec->idx_type == AS_FUND_TYPE_INT) {
                /* no size key for associative vectors */
                cfg_var_vecsz_key(fv->cfgkey, cfgkey, sizeof(cfgkey));
                form_var_sz_name(fv->fvname, fvkey, sizeof(fvkey));
                value = fv->get(wp, prof_type, cfgkey, 0, NULL, NULL);
                if(value) {
                    websSetVar(wp, fvkey, value);
                    free(value);
                } else {
                    websSetVar(wp, fvkey, "0");
                }
            }
            
	    ck = cfg_var_vecelem_key(fv->cfgkey, fv->elemkey, 
                                     cfgkey, sizeof(cfgkey));
            for (i = 0; i < fv->val_cnt; ++i) {
                if(fv->realtype_vec->idx_type == AS_FUND_TYPE_INT) {
                    form_var_vec_name(fv->fvname, i, fvkey, sizeof(fvkey));
                } else {
                    form_var_assoc_name(fv->fvname, fv->assoc_name[i],
                                        fvkey, sizeof(fvkey));
                }
                value = fv->get(wp, prof_type, ck, i, 
                                fv->assoc_name ? fv->assoc_name[i] : NULL,
                                NULL);
                if(value) {
                    websSetVar(wp, fvkey, value);
                    free(value);
                } else {
                    websSetVar(wp, fvkey, "");
                }
            }
        } else { /* scalar */
            value = fv->get(wp, prof_type, fv->cfgkey, 0, NULL, NULL);
            if(value) {
                websSetVar(wp, fv->fvname, value);
                free(value);
            } else {
                websSetVar(wp, fv->fvname, "");
            }
        }
    }
    
    websSetVar(wp, "__reset_to_defaults__", "yes");
    
    return 0;
}

/* ---- main form handler -------------------------------------------------- */

int
form_handler(webs_t wp)
{
    tokenizer_t tokenizer;
    size_t fh_vec_size = 0, i, fv_next_idx, fv_cnt = 0;
    char * templates;
    int do_delete_form_vars = 0;
    int ret = -1;

    /* do nothing if not submitted */
    if (!form_was_submitted(wp)) {
	ret = 0;
	goto finish;
    }

    /* ------------------------------------------------------------------------
     * Determine the used templates and get its form handlers.
     */
    wp->fh_vec = vector_new(NULL, 10, destroy_form_handler_instance_copy);
    templates = strdupa(websGetVar(wp, "__templates__", ""));
    tokenizer_new(&tokenizer, templates, " ", TOKENIZER_DELI_NORET);

    MUTEX_LOCK(&form_handlers_mtx);
 next_token:
    while (tokenizer_has_next(&tokenizer)) {
	form_handler_t * fh, * new_fh;
	char * token = pp_trim_string(tokenizer_get_next(&tokenizer));
        if (!token || strlen(token) == 0) continue; // ignore empty tokens
	if ((fh = pp_hash_get_entry(form_handlers, token)) == NULL) {
	    pp_log("%s(): Unknown subform %s\n", ___F, token);
	    continue;
	}
	/* ignore duplicates */
	for (i = 0; i < vector_size(wp->fh_vec); ++i) {
	    form_handler_t * _fh = (form_handler_t *)vector_get(wp->fh_vec, i);
	    if (_fh && _fh->tmpl_name && !strcmp(token, _fh->tmpl_name)) goto next_token;
	}
	new_fh = (form_handler_t *)malloc(sizeof(form_handler_t));
	memcpy(new_fh, fh, sizeof(form_handler_t));
	vector_add(wp->fh_vec, (void *)new_fh);
	fv_cnt += fh->fv_cnt;
    }
    MUTEX_UNLOCK(&form_handlers_mtx);

    /* ------------------------------------------------------------------------
     * Authorization
     *   If different templates are on one page, only throw security violation
     *   when all autorizations fail. For authorization failures, delete
     *   form_handler form fh_vec.
     */
    fh_vec_size = vector_size(wp->fh_vec);

    if (fh_vec_size > 0) {
        for (i = 0; i < fh_vec_size;) {
            form_handler_t * fh = (form_handler_t *)vector_get(wp->fh_vec, i);
            if (fh && fh->authorize_hook && fh->authorize_hook(wp, fh)) {
                vector_remove(wp->fh_vec, i);
	        --fh_vec_size;
            } else {
	        ++i;
            }
        }

        if (fh_vec_size == 0) {
            eric_notify_security_violation(wp->session);
            set_response(wp, ERIC_RESPONSE_ERROR, pp_intl_translate(perm_denied_msg));
            goto finish;
        }
    } else {
        /* either the page contained no templates (unlikely) or the user *
         * pressed a button before the page was fully loaded             */
    }
    pp_log("Authorization done\n");

    /* ------------------------------------------------------------------------
     * Alloc
     */
    wp->form_vars = (form_var_t *)malloc((fv_cnt + 1) * sizeof(form_var_t));
    for (i = 0, fv_next_idx = 0; i < fh_vec_size; ++i) {
	form_handler_t * fh = (form_handler_t *)vector_get(wp->fh_vec, i);
	if (fh && fh->fv_tmpl) {
	    fh->fv = &wp->form_vars[fv_next_idx];
	    memcpy(fh->fv, fh->fv_tmpl, fh->fv_cnt * sizeof(form_var_t));
	    fv_next_idx += fh->fv_cnt;
	}
    }
    memset(&wp->form_vars[fv_next_idx], 0, sizeof(form_var_t));
    pp_log("Alloc done\n");

    /* ------------------------------------------------------------------------
     * Parse
     */
    if (wp->form_vars) {
	if (parse_form_vars(wp) != 0) goto finish;
    }
    pp_log("Parse done\n");

    /* ------------------------------------------------------------------------
     * Reset to defaults
     */
    if(reset_button_clicked(wp)) {
        reset_to_defaults(wp);
        set_response(wp, ERIC_RESPONSE_OK, _("Values reset to defaults.<br>"
                     "Apply changes to store them persistently."));
        goto finish;
    }
    
    /* ------------------------------------------------------------------------
     * Pre-Validate
     */
    for (i = 0; i < fh_vec_size; ++i) {
	form_handler_t * fh = (form_handler_t *)vector_get(wp->fh_vec, i);
	if (fh && fh->pre_validate_hook) {
	    if (fh->pre_validate_hook(wp, fh)) {
		goto finish;
	    }
	}
    }
    pp_log("Pre Validate done\n");

    /* ------------------------------------------------------------------------
     * Validate
     */
    if (wp->fh_flags & FH_FLAG_ABORT_AT_VALIDATE) {
	ret = 0;
	goto finish;
    }
    if (wp->form_vars) {
	for (i = 0; wp->form_vars[i].cfgkey != NULL; i++) {
	    form_var_t * fv = &wp->form_vars[i];
	    if (fv->flags & FV_FLAG_SKIP_VALIDATE) continue;
	    if (!validate_form_var(wp, fv)) {
		goto finish;
	    }
	}
    }
    pp_log("Validate done\n");

    /* ------------------------------------------------------------------------
     * Post-Validate
     */
    for (i = 0; i < fh_vec_size; ++i) {
	form_handler_t * fh = (form_handler_t *)vector_get(wp->fh_vec, i);
	if (fh && fh->post_validate_hook) {
	    if (fh->post_validate_hook(wp, fh)) {
		goto finish;
	    }
	}
    }
    pp_log("Post Validate done\n");

    /* ------------------------------------------------------------------------
     * Save
     */
    if (wp->fh_flags & FH_FLAG_ABORT_AT_SAVE) {
	ret = 0;
	goto finish;
    }
    if (!(wp->fh_flags & FH_FLAG_SKIP_SAVE)) {
	if (wp->form_vars) {
	    if (save_form_vars(wp) == -1) goto finish;
	}
	/* now after saving, make sure vars will be refetched from config fs */
	do_delete_form_vars = 1;
	pp_log("Save done\n");
    }

    websSetVar(wp, "__reset_to_defaults__", "");
    
    /* ------------------------------------------------------------------------
     * Post-Save
     */
    for (i = 0; i < fh_vec_size; ++i) {
	form_handler_t * fh = (form_handler_t *)vector_get(wp->fh_vec, i);
	if (fh && fh->post_save_hook) {
	    /*
	     * NOTE: We call all post-save-hooks even if they fail! Maybe we
	     *	     should change them to return void.
	     */
	    fh->post_save_hook(wp, fh);
	}
    }
    pp_log("Post Save done\n");

    if (wp->response_msg == NULL) {
	set_response(wp, ERIC_RESPONSE_OK, 
                     _("Operation completed successfully."));
    }

    ret = 0;

 finish:
    if (do_delete_form_vars) delete_form_vars(wp);

    /* prevent freeing of fv_tmpl */
    for (i = 0; i < fh_vec_size; ++i) {
	form_handler_t * fh = (form_handler_t *)vector_get(wp->fh_vec, i);
	if (fh) fh->fv_tmpl = NULL;
    }

/* FIXME: FIXME - this is really crude! Remove it ASAP! */
    /* before returning, wait for net listeners to be reconfigured.
       if these kind of dependencies occure more often, think about a generic
       approach. */
    while (eric_net_listener_reconfiguring()) usleep(200000);

    return ret;
}

/* ---- default hooks ------------------------------------------------------ */

int
default_authorize_hook(webs_t wp, form_handler_t * fh)
{
    /* check "change" permission */
    if (PP_ERR == pp_um_user_has_permission(wp->user, fh->acl_object,
                                            pp_acl_raasip_yes_str)
	&& PP_ERR == pp_um_user_has_permission(wp->user, fh->acl_object,
                                               pp_acl_raasip_control_str)) {
	return -1;
    }
    return 0;
}

/* ---- */

static void
form_var_name(const char * tmpl_name, u_int fv_id, char * fv_name, size_t fv_name_len)
{
    u_int r;
    assert(fv_id < 100);
    r = snprintf(fv_name, fv_name_len, "%s%u_%s", fv_prefix, fv_id, tmpl_name);
    assert(r < fv_name_len);
}

/*
static const char* form_var_label(pp_cd_as_type_t* type) {
    const char* label;
    if (NULL == (label = pp_cd_type_prop_get(type, "desc")))
        label = type->name;
    return label;
}
*/
     
void form_var_sz_name(const char* fvname, char* fvkey, size_t fvklen) {
    unsigned int r;
    r = snprintf(fvkey, fvklen, "%s%csz", fvname, fv_vec_idx_deli);
    assert(r < fvklen);
}

void form_var_vec_name(const char* fvname, u_int idx, 
                       char* fvkey, size_t fvklen) {
    unsigned int r;

    assert(idx < 255); // for PDU, we address 1 byte sensor numbers

    r = snprintf(fvkey, fvklen, "%s%c%u", fvname, fv_vec_idx_deli, idx);
    assert(r < fvklen);
}

void form_var_assoc_name(const char* fvname, const char* idx, 
                         char* fvkey, size_t fvklen) {
    unsigned int r;
    r = snprintf(fvkey, fvklen, "%s%c%s", fvname, fv_vec_idx_deli, idx);
    assert(r < fvklen);
}

void websSetSzVar(webs_t wp, const char* optname, size_t size) {
    char buf[MAX_OPT_KEY_LEN + 1];
    char buf_sz[MAX_OPT_KEY_LEN + 1];
    form_var_sz_name(optname, buf, sizeof(buf));
    snprintf(buf_sz, sizeof(buf_sz), "%u", size);
    return websSetVar(wp, buf, buf_sz);
}

void websSetVecVar(webs_t wp, const char* optname, u_int idx,
                   const char* optvalue) {
    char buf[MAX_OPT_KEY_LEN + 1];
    form_var_vec_name(optname, idx, buf, sizeof(buf));
    return websSetVar(wp, buf, optvalue);
}

void websSetAssocVar(webs_t wp, const char* optname, const char*  idx,
                     const char* optvalue) {
    char buf[MAX_OPT_KEY_LEN + 1];
    form_var_assoc_name(optname, idx, buf, sizeof(buf));
    return websSetVar(wp, buf, optvalue);
}

/**
 * convert int value to char pointer and set value
 *
 * @see websSetVar
 */
void
websSetIntVar(webs_t wp, const char *var, int value) {
    char buf[16];
    
    snprintf(buf, sizeof(buf), "%d", value);
    return websSetVar(wp, var, buf);
}

int form_var_vec_increase(webs_t wp, const char* fvname, int sz, int inc,
			  int bound, const char* fillvalue) {
    char fvkey[FV_KEY_MAX_LEN+1];
    char szstr[11];
    int i, nsz;

    if ((nsz = sz + inc) > bound) nsz = bound;
    for (i = sz; i < nsz; ++i) {
	form_var_vec_name(fvname, i, fvkey, sizeof(fvkey));
	websSetVar(wp, fvkey, fillvalue);
    }
    form_var_sz_name(fvname, fvkey, sizeof(fvkey));
    snprintf(szstr, sizeof(szstr), "%u", nsz);
    websSetVar(wp, fvkey, szstr);
    return nsz;
}

static u_long
form_var_flags(pp_cd_as_cd_t * cd, const pp_cd_qualified_id_t qid, form_flag_entry_t * flag_map)
{
    u_long i, flags = 0;
    
    for (i = 0; flag_map[i].flag_str != NULL; ++i) {
	if (pp_cd_prop_has_flag(cd, qid, "flags", flag_map[i].flag_str))
	    flags |= flag_map[i].flag;
    }
    return flags;
}

#if 0 // not needed at the moment
static int cfg_key_flags(const char *key) {
    char fvtype[FV_CFGKEY_MAX_LEN];
    
    cfg_var_to_typename(key, fvtype, sizeof(fvtype));
    cfg_var_to_qid(fvtype, fvtype, sizeof(fvtype));
    return form_var_flags(pp_cfg_get_cd(), fvtype, form_var_flag_map);
}
#endif

static int
form_var_str_parse(const char * var, char * tmpl_name, size_t tmpl_name_size)
{
    char prefix[10];
    char tmpl_and_idx[64]; /* tmpl_name + index */
    size_t tmpl_name_len;
    u_int id;
    
    if (sscanf(var, "%9[^0-9]%u_%63s", prefix, &id, tmpl_and_idx) < 3 || strcmp(prefix, fv_prefix)) {
	return -1;
    }
    tmpl_name_len = strcspn(tmpl_and_idx, &fv_vec_idx_deli);
    if (tmpl_name_len + 1 > tmpl_name_size) return -1;
    strncpy(tmpl_name, tmpl_and_idx, tmpl_name_len);
    tmpl_name[tmpl_name_len] = '\0';
    return id;
}

/* returns -1 if size tag s is encountered, i.e. FV12_s */
static int
form_var_str_to_idx(const char * var)
{
    const char * p;
    if (NULL == (p = strrchr(var, fv_vec_idx_deli))) return 0;
    if (*++p == 's') return -1;
    return pp_strtol_10(p, 0, NULL);
}

/* looks for last vector and replaces it with the size key of pp_cfg */
const char* cfg_var_vecsz_key(const char* cfgkey,
                              char* szcfgkey, size_t szcfgklen) {
    char* p;
    strncpy(szcfgkey, cfgkey, szcfgklen);
    p = strrchr(szcfgkey, '[');
    assert(p != NULL); // [%u] or [%s] required for vector cfgkey!!
    assert((szcfgklen - (p - szcfgkey)) > 6);
    *p++ = '.';
    strcpy(p, pp_cfg_vect_size_comp);
    return cfgkey;
}

/* skip over all the indices of a cfg key to get the typename */
static char* cfg_var_to_typename(const char* cfgkey, char* tname,
				 size_t tnamelen) {
    const char *p = cfgkey;
    char c, *t = tname, *lt = tname + tnamelen - 1;
    int skipping = 0;
    while((c = *p++) != '\0') {
	if (skipping) {
	    if (c == ']') skipping = 0;
	} else {
	    if (c == '[') skipping = 1;
	    else if (t < lt) *t++ = c;
	    else break;
	}
    }
    *t = '\0';
    return tname;
}

/* skip over all the indices and choice keys of a cfg key to get the typename */
static char* cfg_var_to_qid(const char* cfgkey, char* tname, size_t tnamelen) {
    const char *p = cfgkey;
    char c, *t = tname, *lt = tname + tnamelen - 1;
    int skipping = 0;
    while((c = *p++) != '\0') {
	if (skipping) {
	    if (c == ']') skipping = 0;
	} else {
	    if (c == '[') skipping = 1;
            /* skip choice keys */
            else if (c == '.' && *p == '_' && *(p + 1) == 'c' && 
                     *(p + 2) == '_') p += 3;
	    else if (t < lt) *t++ = c;
	    else break;
	}
    }
    *t = '\0';
    return tname;
}

const char* cfg_var_vecelem_key(const char* veckey, const char* elemkey,
				 char* keybuf, size_t keylen) {
    if (elemkey == NULL || *elemkey == '\0') return veckey;
    snprintf(keybuf, keylen, "%s.%s", veckey, elemkey);
    return keybuf;
}

/*** form var callback API ***************************************************/

/**
 * Now this is still is not a really brilliant peace of code as well, but we
 * do that string compare decisions only once.
 *
 * The following functions are not really a brilliant piece of code.
 * They are very static in order to determine whether a config key
 * needs a target_uid or a target_port. However, for the time beeing,
 * this will work.
 * I could think of much more flexible functions that creates a
 * variable argument stack (or something similar, as this is neither
 * really possible nor sensible, for instance an argument vector)
 * and then calls into libpp_cfg.
 * One solution migth be, to encode dynamic variables in config key itself,
 * e.g. "user[%U]", "unit[%N].port[%P]", ...
 * The problem of passing variable argument lists to cfg lib resists, perhaps
 * use defines for that
 * e.g. #define save_user_key(wp, prof, value, key, args...) \
 *              pp_cfg_set_at_layer(prof, value, key, wp->target_uid, ##args)
 *
 * For ports, only use port magic, if target port is given! (Workaround?)
 */

/**
 * this implements an automaton which determines the form var callback type
 * the "real" cfgkey is created and has to be freed!!!
 */
static fv_cb_type_t 
get_form_var_cb_type(const char *cfgkey, char **realkey) {
    char keybuf[FV_CFGKEY_MAX_LEN + 1];
    char *key = keybuf, *pos;
    fv_cb_type_t type = FV_CB_PLAIN_KEY;

    strncpy(keybuf, cfgkey, sizeof(keybuf));
    
    while(key - keybuf < (int)sizeof(keybuf)) {
        if(NULL == (pos = strchr(key, '%'))) {
            /* so no '%' is left... */
            goto finish;
        }
        
        /* we found '%' at pos, let's see what's next */
        switch (*++pos) {
            case 'U': // uid
                *pos = 'u'; // replace %U by %u
                switch (type) {
                    case FV_CB_PLAIN_KEY:
                        type = FV_CB_USER_KEY;
                        break;
                    default:
                        goto error;
                }
                break;
            case 'G': // gid
                *pos = 'u'; // replace %G by %u
                switch (type) {
                    case FV_CB_PLAIN_KEY:
                        type = FV_CB_GROUP_KEY;
                        break;
                    default:
                        goto error;
                }
                break;
            case 'N': // unit
                *pos = 'u'; // replace %N by %u
                switch (type) {
                    case FV_CB_PLAIN_KEY:
                        type = FV_CB_UNIT_KEY;
                        break;
                    default:
                        goto error;
                }
                break;
            case 'P': // port
                *pos = 'u'; // replace %P by %u
                switch (type) {
                    case FV_CB_UNIT_KEY:
                        type = FV_CB_UNIT_PORT_KEY;
                        break;
                    default:
                        goto error;
                }
                break;
            case 'I': // index
                *pos = 'u'; // replace %I by %u
                switch (type) {
                    case FV_CB_PLAIN_KEY:
                        type = FV_CB_IDX_KEY;
                        break;
                    case FV_CB_USER_KEY:
                        type = FV_CB_USER_IDX_KEY;
                        break;
                    case FV_CB_GROUP_KEY:
                        type = FV_CB_GROUP_IDX_KEY;
                        break;
                    case FV_CB_UNIT_KEY:
                        type = FV_CB_UNIT_IDX_KEY;
                        break;
                    case FV_CB_UNIT_PORT_KEY:
                        type = FV_CB_UNIT_PORT_IDX_KEY;
                        break;
                    default:
                        goto error;
                }
                break;
            case 'S': // association
                *pos = 's'; // replace %S by %s
                switch (type) {
                    case FV_CB_PLAIN_KEY:
                        type = FV_CB_ASSOC_KEY;
                        break;
                    case FV_CB_USER_KEY:
                        type = FV_CB_USER_ASSOC_KEY;
                        break;
                    case FV_CB_GROUP_KEY:
                        type = FV_CB_GROUP_ASSOC_KEY;
                        break;
                    case FV_CB_UNIT_KEY:
                        type = FV_CB_UNIT_ASSOC_KEY;
                        break;
                    case FV_CB_UNIT_PORT_KEY:
                        type = FV_CB_UNIT_PORT_ASSOC_KEY;
                        break;
                    default:
                        goto error;
                }
                break;
#if defined(KIRA_RPC)
            case 'O': // outlet
                *pos = 'u'; // replace %O by %u
                switch (type) {
                    case FV_CB_PLAIN_KEY:
                        type = FV_CB_OUTLET_KEY;
                        break;
                    default:
                        goto error;
                }
                break;
#endif /* KIRA_RPC */
            default:
                goto error;
        }
        key = ++pos;
    }
    
 error:
    type = FV_CB_INVALID_KEY;
    pp_log("ERROR: index identifier '%%%c' in config key '%s' is invalid\n", 
           *pos, cfgkey);
    abort(); // we should never ever get here
    
 finish:
    *realkey = strdup(keybuf);
    return type;
}
 
// get

inline static char* 
get_plain_key(webs_t wp UNUSED, pp_profile_type_t prof_type,
              const char *key, int i UNUSED, const char *s UNUSED,
              int *profile) {
    char *value = NULL;
    pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key);
    return value;
}

inline static char* 
get_idx_key(webs_t wp UNUSED, pp_profile_type_t prof_type,
            const char *key, int i, const char *s UNUSED,
            int *profile) {
    char *value = NULL;
    pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key, i);
    return value;
}

inline static char* 
get_user_key(webs_t wp, pp_profile_type_t prof_type,
             const char *key, int i UNUSED, const char *s UNUSED,
             int *profile) {
    char *value = NULL;
    if (wp->target_uid >= 0) {
        pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                             wp->target_uid);
    }
    return value;
}

inline static char* 
get_user_idx_key(webs_t wp, pp_profile_type_t prof_type,
                 const char *key, int i, const char *s UNUSED,
                 int *profile) {
    char *value = NULL;
    if (wp->target_uid >= 0) {
        pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                             wp->target_uid, i);
    }
    return value;
}

inline static char* 
get_group_key(webs_t wp, pp_profile_type_t prof_type,
              const char *key, int i UNUSED, const char *s UNUSED,
              int *profile) {
    char *value = NULL;
    if (wp->target_gid >= 0) {
        pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                             wp->target_gid);
    }
    return value;
}

inline static char* 
get_group_idx_key(webs_t wp, pp_profile_type_t prof_type,
                  const char *key, int i, const char *s UNUSED,
                  int *profile) {
    char *value = NULL;
    if (wp->target_gid >= 0) {
        pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                             wp->target_gid, i);
    }
    return value;
}

inline static char* 
get_group_assoc_key(webs_t wp, pp_profile_type_t prof_type,
                    const char *key, int i UNUSED, const char *s,
                    int *profile) {
    char *value = NULL;
    if (wp->target_gid >= 0) {
        pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                             wp->target_gid, s);
    }
    return value;
}

inline static char* 
get_unit_key(webs_t wp, pp_profile_type_t prof_type,
             const char *key, int i UNUSED, const char *s UNUSED,
             int *profile) {
    char *value = NULL;
    pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                         wp->target_unit);
    return value;
}

inline static char* 
get_unit_idx_key(webs_t wp, pp_profile_type_t prof_type,
                 const char *key, int i, const char *s UNUSED,
                 int *profile) {
    char *value = NULL;
    pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                         wp->target_unit, i);
    return value;
}

inline static char* 
get_unit_port_key(webs_t wp, pp_profile_type_t prof_type,
                  const char *key, int i UNUSED, const char *s UNUSED,
                  int *profile) {
    char *value = NULL;
    pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                         wp->target_unit, wp->target_port);
    return value;
}

inline static char* 
get_unit_port_idx_key(webs_t wp, pp_profile_type_t prof_type,
                      const char *key, int i, const char *s UNUSED,
                      int *profile) {
    char *value = NULL;
    pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                         wp->target_unit, wp->target_port, i);
    return value;
}

#if defined(KIRA_RPC)
inline static char* 
get_outlet_key(webs_t wp, pp_profile_type_t prof_type,
               const char *key, int i UNUSED, const char *s UNUSED,
               int *profile) {
    char *value = NULL;
    /* outlets are counted 1 - e.g. 8, config keys 0 - 7! */
    pp_cfg_get_at_layer_with_prof_nodflt(prof_type, &value, profile, key,
                                         wp->target_outlet - 1);
    return value;
}
#endif /* KIRA_RPC */

// save

inline static int 
save_plain_key(webs_t wp UNUSED, pp_profile_type_t prof_type,
               const char *key, int i UNUSED, const char *s UNUSED, 
               const char* value) {
    return pp_cfg_set_at_layer(prof_type, value, key);
}

inline static int 
save_idx_key(webs_t wp UNUSED, pp_profile_type_t prof_type,
             const char *key, int i, const char *s UNUSED, 
             const char* value) {
    return pp_cfg_set_at_layer(prof_type, value, key, i);
}

inline static int 
save_user_key(webs_t wp, pp_profile_type_t prof_type,
              const char *key, int i UNUSED, const char *s UNUSED, 
              const char* value) {
    if (wp->target_uid >= 0) {
       return pp_cfg_set_at_layer(prof_type, value, key, wp->target_uid);
    }
    return PP_ERR;
}

inline static int 
save_user_idx_key(webs_t wp, pp_profile_type_t prof_type,
                  const char *key, int i, const char *s UNUSED, 
                  const char* value) {
    if (wp->target_uid >= 0) {
       return pp_cfg_set_at_layer(prof_type, value, key, wp->target_uid, i);
    }
    return PP_ERR;
}

inline static int 
save_group_key(webs_t wp, pp_profile_type_t prof_type,
               const char *key, int i UNUSED, const char *s UNUSED, 
               const char* value) {
    if (wp->target_gid >= 0) {
       return pp_cfg_set_at_layer(prof_type, value, key, wp->target_gid);
    }
    return PP_ERR;
}

inline static int 
save_group_idx_key(webs_t wp, pp_profile_type_t prof_type,
                   const char *key, int i, const char *s UNUSED, 
                   const char* value) {
    if (wp->target_gid >= 0) {
       return pp_cfg_set_at_layer(prof_type, value, key, wp->target_gid, i);
    }
    return PP_ERR;
}

inline static int 
save_group_assoc_key(webs_t wp, pp_profile_type_t prof_type,
                     const char *key, int i UNUSED, const char *s, 
                     const char* value) {
    if (wp->target_gid >= 0) {
       return pp_cfg_set_at_layer(prof_type, value, key, wp->target_gid, s);
    }
    return PP_ERR;
}

inline static int 
save_unit_key(webs_t wp, pp_profile_type_t prof_type,
              const char *key, int i UNUSED, const char *s UNUSED,
              const char* value) {
    return pp_cfg_set_at_layer(prof_type, value, key, wp->target_unit);
}

inline static int 
save_unit_idx_key(webs_t wp, pp_profile_type_t prof_type,
                  const char *key, int i, const char *s UNUSED,
                  const char* value) {
    return pp_cfg_set_at_layer(prof_type, value, key, wp->target_unit, i);
}

inline static int 
save_unit_port_key(webs_t wp, pp_profile_type_t prof_type,
                   const char *key, int i UNUSED, const char *s UNUSED,
                   const char* value) {
    return pp_cfg_set_at_layer(prof_type, value, key, 
                               wp->target_unit, wp->target_port);
}

inline static int 
save_unit_port_idx_key(webs_t wp, pp_profile_type_t prof_type,
                       const char *key, int i, const char *s UNUSED, 
                       const char* value) {
    return pp_cfg_set_at_layer(prof_type, value, key, 
                               wp->target_unit, wp->target_port, i);
}

#if defined(KIRA_RPC)
inline static int 
save_outlet_key(webs_t wp, pp_profile_type_t prof_type,
                const char *key, int i UNUSED, const char *s UNUSED, 
                const char* value) {
    /* outlets are counted 1 - e.g. 8, config keys 0 - 7! */
    return pp_cfg_set_at_layer(prof_type, value, key, wp->target_outlet - 1);
}
#endif /* KIRA_RPC */

// remove

inline static int 
remove_plain_key(webs_t wp UNUSED, pp_profile_type_t prof_type,
                 const char *key, int i UNUSED, const char *s UNUSED) {
    return pp_cfg_remove_at_layer(prof_type, key);
}

inline static int 
remove_idx_key(webs_t wp UNUSED, pp_profile_type_t prof_type,
               const char *key, int i, const char *s UNUSED) {
    return pp_cfg_remove_at_layer(prof_type, key, i);
}

inline static int 
remove_user_key(webs_t wp, pp_profile_type_t prof_type,
                const char *key, int i UNUSED, const char *s UNUSED) {
    if (wp->target_uid >= 0) {
       return pp_cfg_remove_at_layer(prof_type, key, wp->target_uid);
    }
    return PP_ERR;
}

inline static int 
remove_user_idx_key(webs_t wp, pp_profile_type_t prof_type,
                    const char *key, int i, const char *s UNUSED) {
    if (wp->target_uid >= 0) {
       return pp_cfg_remove_at_layer(prof_type, key, wp->target_uid, i);
    }
    return PP_ERR;
}

inline static int 
remove_group_key(webs_t wp, pp_profile_type_t prof_type,
                 const char *key, int i UNUSED, const char *s UNUSED) {
    if (wp->target_gid >= 0) {
       return pp_cfg_remove_at_layer(prof_type, key, wp->target_gid);
    }
    return PP_ERR;
}

inline static int 
remove_group_idx_key(webs_t wp, pp_profile_type_t prof_type,
                     const char *key, int i, const char *s UNUSED) {
    if (wp->target_gid >= 0) {
       return pp_cfg_remove_at_layer(prof_type, key, wp->target_gid, i);
    }
    return PP_ERR;
}

inline static int 
remove_group_assoc_key(webs_t wp, pp_profile_type_t prof_type,
                       const char *key, int i UNUSED, const char *s) {
    if (wp->target_gid >= 0) {
       return pp_cfg_remove_at_layer(prof_type, key, wp->target_gid, s);
    }
    return PP_ERR;
}

inline static int 
remove_unit_key(webs_t wp, pp_profile_type_t prof_type,
                const char *key, int i UNUSED, const char *s UNUSED) {
    return pp_cfg_remove_at_layer(prof_type, key, wp->target_unit);
}

inline static int 
remove_unit_idx_key(webs_t wp, pp_profile_type_t prof_type,
                    const char *key, int i, const char *s UNUSED) {
    return pp_cfg_remove_at_layer(prof_type, key, wp->target_unit, i);
}

inline static int 
remove_unit_port_key(webs_t wp, pp_profile_type_t prof_type,
                     const char *key, int i UNUSED, const char *s UNUSED) {
    return pp_cfg_remove_at_layer(prof_type, key, 
                                  wp->target_unit, wp->target_port);
}

inline static int 
remove_unit_port_idx_key(webs_t wp, pp_profile_type_t prof_type,
                         const char *key, int i, const char *s UNUSED) {
    return pp_cfg_remove_at_layer(prof_type, key, 
                                  wp->target_unit, wp->target_port, i);
}

#if defined(KIRA_RPC)
inline static int 
remove_outlet_key(webs_t wp, pp_profile_type_t prof_type,
                  const char *key, int i UNUSED, const char *s UNUSED) {
    /* outlets are counted 1 - e.g. 8, config keys 0 - 7! */
    return pp_cfg_remove_at_layer(prof_type, key, wp->target_outlet - 1);
}
#endif /* KIRA_RPC */

/**
 * returns the profile type, the requestet form var is stored in
 */

inline static pp_profile_type_t 
determine_plain_proftype(webs_t wp UNUSED, const form_var_t *fv) {
    return pp_cfg_get_prof_type(fv->cfgkey);
}

inline static pp_profile_type_t 
determine_idx_proftype(webs_t wp UNUSED, const form_var_t *fv) {
    int i;
    
    if (0 <= (i = form_var_str_to_idx(fv->fvname))) {
        return pp_cfg_get_prof_type(fv->cfgkey, i);
    }
    return PP_ERR;
}

inline static pp_profile_type_t 
determine_user_proftype(webs_t wp, const form_var_t *fv) {
    if (wp->target_uid >= 0) {
        return pp_cfg_get_prof_type(fv->cfgkey, wp->target_uid);
    }
    return PP_ERR;
}

inline static pp_profile_type_t 
determine_user_idx_proftype(webs_t wp, const form_var_t *fv) {
    int i;
    
    if (0 <= (i = form_var_str_to_idx(fv->fvname)) && wp->target_uid >= 0) {
        return pp_cfg_get_prof_type(fv->cfgkey, wp->target_uid, i);
    }
    return PP_ERR;
}

inline static pp_profile_type_t 
determine_group_proftype(webs_t wp, const form_var_t *fv) {
    if (wp->target_gid >= 0) {
        return pp_cfg_get_prof_type(fv->cfgkey, wp->target_gid);
    }
    return PP_ERR;
}

inline static pp_profile_type_t 
determine_group_idx_proftype(webs_t wp, const form_var_t *fv) {
    int i;
    
    if (0 <= (i = form_var_str_to_idx(fv->fvname)) && wp->target_gid >= 0) {
        return pp_cfg_get_prof_type(fv->cfgkey, wp->target_gid, i);
    }
    return PP_ERR;
}

inline static pp_profile_type_t 
determine_group_assoc_proftype(webs_t wp, const form_var_t *fv) {
    char *s;
                
    if(NULL != (s = strrchr(fv->fvname, fv_vec_idx_deli)) &&
       wp->target_gid >= 0) {
        return pp_cfg_get_prof_type(fv->cfgkey, wp->target_gid, s);
    }
    return PP_ERR;
}

inline static pp_profile_type_t 
determine_unit_proftype(webs_t wp, const form_var_t *fv) {
    return pp_cfg_get_prof_type(fv->cfgkey, wp->target_unit);
}

inline static pp_profile_type_t 
determine_unit_idx_proftype(webs_t wp, const form_var_t *fv) {
    int i;
    
    if (0 <= (i = form_var_str_to_idx(fv->fvname))) {
        return pp_cfg_get_prof_type(fv->cfgkey, wp->target_unit, i);
    }
    return PP_ERR;
}

inline static pp_profile_type_t 
determine_unit_port_proftype(webs_t wp, const form_var_t *fv) {
    return pp_cfg_get_prof_type(fv->cfgkey, wp->target_unit, wp->target_port);
}

inline static pp_profile_type_t 
determine_unit_port_idx_proftype(webs_t wp, const form_var_t *fv) {
    int i;
    
    if (0 <= (i = form_var_str_to_idx(fv->fvname))) {
        return pp_cfg_get_prof_type(fv->cfgkey, 
                                    wp->target_unit, wp->target_port, i);
    }
    return PP_ERR;
}

#if defined(KIRA_RPC)
inline static pp_profile_type_t 
determine_outlet_proftype(webs_t wp, const form_var_t *fv) {
    /* outlets are counted 1 - e.g. 8, config keys 0 - 7! */
    return pp_cfg_get_prof_type(fv->cfgkey, wp->target_outlet - 1);
}
#endif /* KIRA_RPC */

