#include <liberic_config.h>
#include <pp/intl.h>
#include <pp/ipmi.h>
#include "ej.h"
#include "eric_base.h"
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "eric_validate.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#include "tmpl_ipmi_common.h"


/* internal prototypes */
static int ipmi_build_lan_dest_edit_form_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED);
static void print_destination_form(pp_strstream_t* tab, pp_ipmi_lanp_dest_t* filter);
static void display_lan_dest_form(pp_strstream_t* tab, webs_t wp);
static int post_validate_hook(webs_t wp, form_handler_t * fh);
static int pre_validate_hook(webs_t wp, form_handler_t * fh);


FV_SPEC = {
{
    id:             FV_ID_PEF_LAN_DEST_ID,
    cfgkey:         "ipmi_lan_dest_edit.id",
},
{
    id:             FV_ID_PEF_LAN_DEST_ACKNOWLEDGE,
    cfgkey:         "ipmi_lan_dest_edit.acknowledge",
},
{
    id:             FV_ID_PEF_LAN_DEST_TYPE,
    cfgkey:         "ipmi_lan_dest_edit.type",
},
{
    id:             FV_ID_PEF_LAN_DEST_TIMEOUT,
    cfgkey:         "ipmi_lan_dest_edit.timeout",
},
{
    id:             FV_ID_PEF_LAN_DEST_RETRIES,
    cfgkey:         "ipmi_lan_dest_edit.retries",
},
{
    id:             FV_ID_PEF_LAN_DEST_IP_ADDR,
    cfgkey:         "ipmi_lan_dest_edit.ip_addr",
},
{
    id:             FV_ID_PEF_LAN_DEST_MAC_ADDR,
    cfgkey:         "ipmi_lan_dest_edit.mac_addr",
},
{
    id:             FV_ID_PEF_LAN_DEST_GATEWAY,
    cfgkey:         "ipmi_lan_dest_edit.gateway",
},
{
    id:             FV_ID_PEF_LAN_DEST_EMAIL,
    cfgkey:         "ipmi_lan_dest_edit.email",
},
};


int ipmi_lan_dest_edit_tmpl_init(void)
{
    form_handler_t * fh;

    /* register ASPs */
    websAspDefine("ipmiGetLanDestEditForm", ipmi_build_lan_dest_edit_form_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_IPMI_LAN_DEST_EDIT, ACL_OBJ_IPMI_STATUS);
    fh->pre_validate_hook = pre_validate_hook;
    fh->post_validate_hook = post_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int pre_validate_hook(webs_t wp UNUSED, form_handler_t * fh)
{
    if (strcmp(fh->fv[FV_ID_PEF_LAN_DEST_TYPE].val.s, "pet") == 0)
    {
        fh->fv[FV_ID_PEF_LAN_DEST_EMAIL].flags |= FV_FLAG_ALLOW_EMPTY;
    }
    
    if (strcmp(fh->fv[FV_ID_PEF_LAN_DEST_TYPE].val.s, "mail") == 0)
    {
        fh->fv[FV_ID_PEF_LAN_DEST_IP_ADDR].flags |= FV_FLAG_ALLOW_EMPTY;
        fh->fv[FV_ID_PEF_LAN_DEST_MAC_ADDR].flags |= FV_FLAG_ALLOW_EMPTY;
    }
    
    return 0;
}

static int post_validate_hook(webs_t wp, form_handler_t * fh)
{
    pp_ipmi_lanp_dest_t* dest;
    pp_ipmi_parameter_t in_parm;
    int save_form;
    int ib[6];
    int i;

    dest = &(in_parm.data.lan_destination_set);
    memset(dest, 0, sizeof(pp_ipmi_lanp_dest_t));

    save_form = 0;
    if (form_button_clicked(wp, "action_apply")) {
        save_form = 1;
    }

    websSetVar(wp, "dest", fh->fv[FV_ID_PEF_LAN_DEST_ID].val.s);

    if (save_form == 1)
    {
        dest->id = strtol(fh->fv[FV_ID_PEF_LAN_DEST_ID].val.s, NULL, 10);
        
        dest->acked = 0;
        if (strcmp(fh->fv[FV_ID_PEF_LAN_DEST_ACKNOWLEDGE].val.s, "acknowledge") == 0) {
            dest->acked = 1;
        }
        
        dest->type = 0;
        if (strcmp(fh->fv[FV_ID_PEF_LAN_DEST_TYPE].val.s, "mail") == 0) {
            dest->type = 6;
        }
        
        dest->timeout = strtol(fh->fv[FV_ID_PEF_LAN_DEST_TIMEOUT].val.s, NULL, 10);
        dest->retries = strtol(fh->fv[FV_ID_PEF_LAN_DEST_RETRIES].val.s, NULL, 10);
        
        memset(&(dest->ip_addr), 0, 4);
        // checking the sscanf result should not be necessary as ipaddr is validated
        if ( (fh->fv[FV_ID_PEF_LAN_DEST_IP_ADDR].val.s[0] != 0) &&
             (4 == sscanf(fh->fv[FV_ID_PEF_LAN_DEST_IP_ADDR].val.s, "%d.%d.%d.%d", &(ib[0]), &(ib[1]), &(ib[2]), &(ib[3])))
           )
        {
            for (i=0; i<4; i++) {
                dest->ip_addr[i] = (unsigned char)(ib[i]);
            }
        };
        
        strncpy(dest->email, fh->fv[FV_ID_PEF_LAN_DEST_EMAIL].val.s, sizeof(dest->email));
        
        dest->addr_format = 0;
        dest->backup_gateway = 0;
        if (strcmp(fh->fv[FV_ID_PEF_LAN_DEST_GATEWAY].val.s, "backup") == 0) {
            dest->backup_gateway = 1;
        }

        memset(&(dest->mac_addr), 0, 6);
        sscanf(fh->fv[FV_ID_PEF_LAN_DEST_MAC_ADDR].val.s, "%x:%x:%x:%x:%x:%x",
                &(ib[0]), &(ib[1]), &(ib[2]), &(ib[3]), &(ib[4]), &(ib[5]));
        for (i=0; i<6; i++) {
            dest->mac_addr[i] = (unsigned char)(ib[i]);
        }
        
        if (pp_ipmi_send_command(PP_IPMI_CMD_LAN, PP_IPMI_LAN_DESTINATION_SET, &in_parm, NULL, NULL, wp->user)) {
            pp_log("liberic_webs.tmpl_ipmi_lan_dest_edit.c - Error sending ipmi set lan destination command \n");
            set_response(wp, ERIC_RESPONSE_ERROR, "Error sending the IPMI set destination command");
            return -1;
        }
    }

    return 0;
}

static void display_lan_dest_form(pp_strstream_t* tab, webs_t wp) {
    pp_ipmi_return_t ipmi_ret;
    const unsigned char* destNoStr;
    int destNo;
    int err;
    vector_t* destinations;
    pp_ipmi_lanp_dest_t* destination;
    int valid;
    
    valid = 0;
    destination = NULL;
    destinations = NULL;
    destNoStr = websGetVar(wp, "dest", "no_id");
    destNo = pp_strtoul_10(destNoStr, 0, &err);
    if (err) {
        destNo = -1;
        valid = 0;
    }
    else 
    {
        /* read values from ipmi */
        memset(&ipmi_ret, 0, sizeof(ipmi_ret));
        if (!pp_ipmi_send_command(PP_IPMI_CMD_LAN, PP_IPMI_LAN_DESTINATION_GET, NULL, &ipmi_ret, NULL, wp->user))
        {
            destinations = ipmi_ret.data.lan_destination_list;
            if ((destNo < 0) | (destNo > (int)(vector_size(destinations)))) {
                valid = 0;
            } else {
                valid = 1;
                destination = (pp_ipmi_lanp_dest_t*)vector_get(destinations, (int)(destNo));
            }
        }
    }
    
    if (valid) {
        print_destination_form(tab, destination);
    } else {
#if !defined(PP_FEAT_RARITAN_DESIGN)
	pp_strappend(tab, "<table border=\"1\">\n");
#else
	pp_strappend(tab, "<table border=\"0\">\n");
#endif  
        pp_strappend(tab, "<tr><td>&nbsp;<br>The specified LAN alert destination ");
        pp_strappend(tab, destNoStr);
        pp_strappend(tab, " could not be retrieved<br>&nbsp;</td></tr>\n");
        pp_strappend(tab, "</table>\n");
    }
#if defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "\n<div class=\"tmpl_end\"><br></div>\n");    
#endif   
    if (destinations) {
        vector_delete(destinations);
    }
}

static void print_destination_form(pp_strstream_t* tab, pp_ipmi_lanp_dest_t* dest) {
    char buf[25];
    const char* bufp;
    
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<table border=\"1\">\n");
#else
    pp_strappend(tab, "<table border=\"0\">\n");
#endif  

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Destination Number"));
    print_input_cell(tab, FV_ID_PEF_LAN_DEST_ID, "ipmilandestedit", "text", "%d", dest->id, 5, 1);
    pp_strappend(tab, "</tr>\n");
    
    pp_strappend(tab, "<tr>\n");
#if !defined(PP_FEAT_RARITAN_DESIGN)
    print_head(tab, _("Acknowledge"));
#endif
    pp_strappend(tab, "<td class=\"settings_fieldname\">");
    pp_strappend(tab, "<input type=\"checkbox\" ");
    print_fv(tab, FV_ID_PEF_LAN_DEST_ACKNOWLEDGE, "ipmilandestedit");
    pp_strappend(tab, " value=\"acknowledge\"");
    if (dest->acked == 1) {
        pp_strappend(tab, " checked");
    }
    pp_strappend(tab, ">");
    pp_strappend(tab, _("require acknowledge"));
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");
    
    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Timeout"));
    print_input_cell(tab, FV_ID_PEF_LAN_DEST_TIMEOUT, "ipmilandestedit", "text", "%d", dest->timeout, 5, 0);
    pp_strappend(tab, "</tr>\n");
   
    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Retries"));
    print_input_cell(tab, FV_ID_PEF_LAN_DEST_RETRIES, "ipmilandestedit", "text", "%d", dest->retries, 5, 0);
    pp_strappend(tab, "</tr>\n");

   
    /* pet alerts */
    pp_strappend(tab, "<tr>\n");
#if !defined(PP_FEAT_RARITAN_DESIGN)
    print_head(tab, _("Alert Type"));
    pp_strappend(tab, "<td class=\"settings_field\">");
#else
    pp_strappend(tab, "<td class=\"settings_fieldname\">");
    pp_strappend(tab, _("Alert Type"));
    pp_strappend(tab, "</tr><tr>\n");
    pp_strappend(tab, "<td class=\"settings_fieldname_checkbox_en\">\n");    
#endif    
    pp_strappend(tab, "<input type=\"radio\" ");
    print_fv(tab, FV_ID_PEF_LAN_DEST_TYPE, "ipmilandestedit");
    pp_strappend(tab, " value=\"pet\"");
    pp_strappend(tab, " id=\"radio1\" onClick=\"UpdateStates();\"");
    if (dest->type == 0) {
        pp_strappend(tab, " checked");
    }
    pp_strappend(tab, ">&nbsp;\n");
    pp_strappend(tab, _("PET alert"));
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "</tr><tr><td class=\"settings_fieldname\">\n");
#else
    pp_strappend(tab, "<br>\n");
#endif    
    pp_strappend(tab, _("Trap destination:"));
    pp_strappend(tab, "<br>\n");
    snprintf(buf, sizeof(buf), "%d.%d.%d.%d", dest->ip_addr[0], dest->ip_addr[1], dest->ip_addr[2], dest->ip_addr[3]);
    print_input(tab, FV_ID_PEF_LAN_DEST_IP_ADDR, "ipmilandestedit", "text", "%s", buf, 20, 0);

    snprintf(buf, sizeof(buf), "%.02x:%.02x:%.02x:%.02x:%.02x:%.02x", dest->mac_addr[0], dest->mac_addr[1],
                      dest->mac_addr[2], dest->mac_addr[3], dest->mac_addr[4], dest->mac_addr[5]);
    print_input(tab, FV_ID_PEF_LAN_DEST_MAC_ADDR, "ipmilandestedit", "hidden", "%s", buf, 20, 0);
    if (dest->backup_gateway == 0) {
        bufp = "default";
    } else {
        bufp = "backup";
    }
    print_input(tab, FV_ID_PEF_LAN_DEST_GATEWAY, "ipmilandestedit", "hidden", "%s", bufp, 10, 0);
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");

    /* mail alerts */
    pp_strappend(tab, "<tr>\n");
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td>&nbsp;</td><td class=\"settings_field\">");
#else
    pp_strappend(tab, "<td class=\"settings_fieldname_checkbox_en\">");
#endif    
    pp_strappend(tab, "<input type=\"radio\" ");
    print_fv(tab, FV_ID_PEF_LAN_DEST_TYPE, "ipmilandestedit");
    pp_strappend(tab, " value=\"mail\"");
    pp_strappend(tab, " id=\"radio2\" onClick=\"UpdateStates();\"");
    if (dest->type == 6) {
        pp_strappend(tab, " checked");
    }
    pp_strappend(tab, ">&nbsp;\n");
    pp_strappend(tab, _("EMail Alert"));
    pp_strappend(tab, "<br>\n");

    pp_strappend(tab, "<input type=\"text\" ");
    print_fv(tab, FV_ID_PEF_LAN_DEST_EMAIL, "ipmilandestedit");
    pp_strappend(tab, " value=\"");
    pp_strappend(tab, dest->email);
    pp_strappend(tab, "\" size=25 maxlength=64>");
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "</table>\n");
#if defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "\n<div class=\"tmpl_end\"><br></div>\n");    
#endif   

}

static int ipmi_build_lan_dest_edit_form_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED) {
    pp_strstream_t tab = PP_STRSTREAM_INITIALIZER;

    pp_strstream_init(&tab);

    display_lan_dest_form(&tab, wp);

    ejSetResult(eid, tab.buf ? tab.buf : "");

    pp_strstream_free(&tab);

    return 0;
}




